/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-filename-input.c,v 1.14 2005/02/01 02:31:58 hoa Exp $
 */

#include "etpan-filename-input.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-input-common.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <ncurses.h>
#include "etpan-imf-helper.h"
#include <glob.h>
#include "etpan-help-viewer.h"
#include "etpan-tools.h"
#include "etpan-search-input.h"

static void handle_key(struct etpan_subapp * app, int key);
static void handle_resize(struct etpan_subapp * app);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_filename_input_app_driver = {
  .name = "filename-input",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .get_idle_udelay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = handle_resize,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_input_common_app_state common_state;
  
  /* complete */
  int completed_once;

  int completed;
  int tab_count;
  size_t max_common;
  
  char * alloc_complete_str;
  char * complete_str;
  size_t complete_str_len;
  size_t max_complete_list_len;
  carray * complete_list;
  char * default_replace;
  
  int file_type;
};


static int etpan_filename_input_display(struct etpan_subapp * app,
    WINDOW * w);

static void cancel_completion(struct etpan_subapp * app);

static void do_completion(struct etpan_subapp * app);

static void display(struct etpan_subapp * app, WINDOW * w)
{
  etpan_filename_input_display(app, w);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->completed_once = 0;
  state->completed = 0;
  state->tab_count = 0;
  state->alloc_complete_str = NULL;
  state->complete_str = NULL;
  state->default_replace = NULL;
  state->complete_str_len = 0;
  state->complete_list = carray_new(128);
  if (state->complete_list == NULL)
    goto free;
  state->max_complete_list_len = 0;
  state->max_common = 0;
  state->file_type = 0;
  
  r = etpan_input_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free_array;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free_array:
  carray_free(state->complete_list);
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}


struct etpan_subapp * etpan_filename_input_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_filename_input_app_driver);
}

char * etpan_filename_input_get_value(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_input_common_get_value(app, &state->common_state);
}

void etpan_filename_input_set_file_type(struct etpan_subapp * app,
    int file_type)
{
  struct app_state * state;
  
  state = app->data;
  
  state->file_type = file_type;
}

int etpan_filename_input_set(struct etpan_subapp * app,
    char * prompt, size_t size,
    char * default_str,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * data)
{
  struct app_state * state;
  
  state = app->data;
  
  cancel_completion(app);
  etpan_subapp_handle_resize(app);
  
  state->file_type = FILENAME_TYPE_REGULAR;
  
  return etpan_input_common_set(app, &state->common_state,
      prompt, size, default_str, 0, upcall, data);
}


static void replace_complete_string(struct etpan_subapp * app,
    char * str, size_t len)
{
  struct app_state * state;
  char * first;
  size_t offset;
  
  state = app->data;
  
  if (state->common_state.count - state->complete_str_len +
      len < state->common_state.size) {
    /* delete old string */
    offset = state->complete_str - state->alloc_complete_str;
    memmove(state->common_state.data + offset,
        state->common_state.data + offset +
        state->complete_str_len,
        state->common_state.count -
        (offset + state->complete_str_len));
        
    /* insert new string */
    memmove(state->common_state.data + offset + len,
        state->common_state.data + offset,
        state->common_state.count -
        (offset + state->complete_str_len));
      
    first = carray_get(state->complete_list, 0);
        
    /* insert completed string */
    memcpy(state->common_state.data + offset, str, len);
    
    /* update count */
    state->common_state.count -= state->complete_str_len;
    state->common_state.count += len;
    
    /* update pos */
    state->common_state.pos -= state->complete_str_len;
    state->common_state.pos += len;
  }
}

static void complete_string(struct etpan_subapp * app)
{
  struct app_state * state;
  char * first;
  
  state = app->data;
        
  first = carray_get(state->complete_list, 0);
  
  replace_complete_string(app, first, state->max_common);
}

static int show_help(struct etpan_subapp * app);

static void ask_create_directory(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->file_type != FILENAME_TYPE_REGULAR) {
    switch (key) {
    case '\n':
      ask_create_directory(app);
      return;
    }
  }
  
  etpan_input_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
    show_help(app);
    break;

  case '\t':

    if (!state->completed) {
      size_t old_pos;
      
      do_completion(app);
      
      if (state->max_common > 0) {
        old_pos = state->common_state.pos;
        complete_string(app);
        if (old_pos != state->common_state.pos)
          state->completed = 0;
      }
      else {
        replace_complete_string(app,
            state->default_replace, strlen(state->default_replace));
        free(state->default_replace);
        state->default_replace = NULL;
      }
      
      etpan_subapp_handle_resize(app);
    }
    else {
      int lines;
      int count;
      
      lines = app->height - 1;
      count = (carray_count(state->complete_list) + lines - 1) / lines;
      
      if (count != 0) {
        state->tab_count ++;
        state->tab_count %= count;
      }
    }
    break;
    
  default:
    state->completed = 0;
    break;
  }
}

static void handle_resize(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->completed_once) {
    app->top = 0;
    app->left = 0;
    app->height = app->app->height;
    app->width = app->app->width;
    etpan_subapp_set_relative_coord(app);
  }
  else {
    etpan_input_common_handle_resize(app, &state->common_state);
  }
}

static int display_init(struct etpan_subapp * app)
{
  app->show_cursor = 1;
  etpan_subapp_handle_resize(app);
  
  return NO_ERROR;
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_set_color(app, &state->common_state);
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  carray_free(state->complete_list);

  etpan_input_common_done(&state->common_state);

  free(state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;
  
  state->completed_once = 0;
  cancel_completion(app);
  
  etpan_input_common_leave(app, &state->common_state, new_app);
}


/* complete filename */

static int add_match_list(carray * list, char * value)
{
  char * item;
  int res;
  int r;
  
  item = strdup(value);
  if (item == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  r = carray_add(list, item, NULL);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto free;
  }

  return NO_ERROR;
  
 free:
  free(item);
 err:
  return res;
}

enum {
  STATE_NORMAL,
  STATE_FIRST_SLASH,
  STATE_TILDE,
  STATE_SECOND_SLASH,
};

static char * global_quote(char * filename)
{
  char pattern[PATH_MAX];
  char * p;
  size_t remaining;
  int state;
  int end;
  char * p_dest;

  remaining = sizeof(pattern);
  p = filename;
  p_dest = pattern;
  
  state = STATE_FIRST_SLASH;
  
  end = 0;
  while (* p != '\0') {
    int done;
    
    done = 0;
    
    switch (* p) {
    case '/':
      if (state == STATE_NORMAL) {
        state = STATE_FIRST_SLASH;
      }
      else if (state == STATE_FIRST_SLASH) {
        p_dest = pattern;
        state = STATE_NORMAL;
      }
      else if (state == STATE_TILDE) {
        char * home_path;
        char * home_quoted;
        size_t len;

        home_path = etpan_get_home_dir();
        
        home_quoted = global_quote(home_path);;
        if (home_quoted == NULL)
          return NULL;
      
        len = strlen(home_quoted);
        
        p_dest = pattern;
        remaining = sizeof(pattern);
        if (len < remaining) {
          strcpy(p_dest, home_quoted);
          remaining -= len;
          p_dest += len;
          free(home_quoted);
        }
        else {
          free(home_quoted);
          return NULL;
        }
        state = STATE_NORMAL;
        done = 1;
      }
      else
        state = STATE_NORMAL;
      break;
    
    case '~':
      if (state == STATE_FIRST_SLASH)
        state = STATE_TILDE;
      else
        state = STATE_NORMAL;
      break;

    default:
      state = STATE_NORMAL;
      break;
    }
    
    if (!done) {
      if ((* p == '*') || (* p == '?') || (* p == '[') || (* p == ']')) {
        if (remaining <= 1)
          break;
        * p_dest = '\\';
        p_dest ++;
        remaining --;
      }
      if (remaining <= 1)
        break;
      
      * p_dest = * p;
      p_dest ++;
      p ++;
      remaining --;
    }
  }
  
  if (remaining <= 1)
    return NULL;
  
  * p_dest = '\0';
  
  return strdup(pattern);
}

static void do_completion(struct etpan_subapp * app)
{
  int r;
  struct app_state * state;
  char * value;
  size_t max_common;
  size_t value_size;
  char pattern[PATH_MAX];
  char * p;
  unsigned int i;
  glob_t glob_info;
  size_t len;

  cancel_completion(app);
  
  state = app->data;
  
  state->completed_once = 1;
  
  state->completed = 1;
  
  value_size = state->common_state.pos;
  state->alloc_complete_str = malloc(value_size + 1);
  if (state->alloc_complete_str == NULL)
    goto cancel;
  value = state->alloc_complete_str;
  
  strncpy(value, state->common_state.data, value_size);
  value[value_size] = '\0';
  
  state->complete_str = value;
  state->complete_str_len = strlen(state->complete_str);
  
  state->max_complete_list_len = 0;
  
  p = global_quote(state->complete_str);
  if (p == NULL)
    goto cancel;
  
  if (state->default_replace != NULL)
    free(state->default_replace);
  state->default_replace = p;
  
  strncpy(pattern, p, sizeof(pattern));
  pattern[sizeof(pattern) - 1] = '\0';
  
  len = strlen(pattern);
  if (len + 1 >= sizeof(pattern) - 1)
    goto cancel;
  
  pattern[len] = '*';
  pattern[len + 1] = '\0';
  
  r = glob(pattern, GLOB_MARK, NULL, &glob_info);
  if (r == 0) {
    for(i = 0 ; i < (unsigned int) glob_info.gl_pathc ; i ++) {
      r = add_match_list(state->complete_list, glob_info.gl_pathv[i]);
      if (r != NO_ERROR) {
        goto cancel;
      }
    }
  }
  globfree(&glob_info);
  
  max_common = 0;
  if (carray_count(state->complete_list) == 0) {
    /* no match */
  }
  else {
    char * first;
    
    /* find max common initial part of string */
    first = carray_get(state->complete_list, 0);
    
    max_common = strlen(first);
    if (max_common > state->max_complete_list_len)
      state->max_complete_list_len = max_common;

    if (carray_count(state->complete_list) == 1) {
      /* sole completion */
    }
    else {
      unsigned int i;

      /* find possible completion */
      for(i = 1 ; i < carray_count(state->complete_list) ; i ++) {
        char * matched;
        size_t j;
        size_t len;
        
        matched = carray_get(state->complete_list, i);
        
        len = strlen(matched);
        
        if (len < max_common)
          max_common = len;
        
        for(j = 0 ; j < max_common ; j ++)
          if (matched[j] != first[j]) {
            max_common = j;
            break;
          }
      
        if (max_common == 0)
          break;
      }
    }
  }
  
  state->max_common = max_common;
  
  return;
  
 cancel:
  cancel_completion(app);
  ETPAN_APP_LOG((app->app, "complete filename - not enough memory"));
  state->completed = 0;
}

static void cancel_completion(struct etpan_subapp * app)
{
  struct app_state * state;
  unsigned int i;

  state = app->data;

  if (state->alloc_complete_str != NULL)
    free(state->alloc_complete_str);
  state->alloc_complete_str = NULL;
  state->max_common = 0;
  state->completed_once = 0;
  state->completed = 0;
  state->tab_count = 0;
  state->complete_str = NULL;
  state->complete_str_len = 0;
  if (state->default_replace != NULL)
    free(state->default_replace);
  state->default_replace = NULL;
  for(i = 0 ; i < carray_count(state->complete_list) ; i ++) {
    char * item;
    
    item = carray_get(state->complete_list, i);
    free(item);
  }
  carray_set_size(state->complete_list, 0);
  state->max_complete_list_len = 0;
}

static char * get_basename(char * filename)
{
  char * p;
  char * lastp;
  
  lastp = filename;
  while ((p = strchr(filename, '/')) != NULL) {
    lastp = filename;
    filename = p + 1;
  }
  
  if (* filename == '\0')
    filename = lastp;
  
  return filename;
}

static int etpan_filename_input_display(struct etpan_subapp * app,
    WINDOW * w)
{
  struct app_state * state;
  int r;
  char * output;
  char * fill;
  unsigned int i;
  int y;
  int count;
  int lines;
  
  state = app->data;
  
  fill = app->app->fill;
  output = app->app->output;
  
  lines = app->display_height - 1;

  y = 0;
  i = state->tab_count * lines;
    
  count = 0;

  wattron(w, state->common_state.main_attr);
  
  if (carray_count(state->complete_list) == 0) {
    if (y < lines) {
      snprintf(output, app->display_width + 1, "no match%s", fill);
      mvwaddstr(w, y, 0, output);
      y ++;
    }
  }
  else {
    while (i < carray_count(state->complete_list)) {
      char * matched;
      
      if (y >= lines)
        break;
      
      matched = carray_get(state->complete_list, i);
      matched = get_basename(matched);
      
      snprintf(output, app->display_width + 1, "%s%s", matched, fill);
      mvwaddstr(w, y, app->left, output);
      y ++;
      i ++;
    }
    
    if (carray_count(state->complete_list) == 1) {
      if (y < lines) {
        snprintf(output, app->display_width + 1,
            "sole completion%s", fill);
        mvwaddstr(w, y, app->left, output);
        y ++;
      }
    }
  }
  
  while (y < lines) {
    mvwaddstr(w, y, 0, fill);
    y ++;
  }
  
  wattroff(w, state->common_state.main_attr);
  
  r = etpan_input_common_display(app, &state->common_state, w);
  if (r != NO_ERROR)
    return r;

  return NO_ERROR;
}


static void ask_create_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  struct etpan_subapp * app;
  int r;
  char * result;
  char filename[PATH_MAX];
  int fail;
  
  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if (* result != 'y') {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  etpan_app_quit_subapp(input_app);
  
  name = etpan_input_common_get_value(app, &state->common_state);
  
  fail = 0;
  switch (state->file_type) {
  case FILENAME_TYPE_MH:
    r = etpan_mkdir(name);
    if (r < 0)
      fail = 1;
    break;
  case FILENAME_TYPE_MAILDIR:
    snprintf(filename, sizeof(filename), "%s/cur", name);
    r = etpan_mkdir(filename);
    if (r < 0)
      fail = 1;
    snprintf(filename, sizeof(filename), "%s/new", name);
    r = etpan_mkdir(filename);
    if (r < 0)
      fail = 1;
    snprintf(filename, sizeof(filename), "%s/tmp", name);
    r = etpan_mkdir(filename);
    if (r < 0)
      fail = 1;
    break;
  }
  
  if (fail) {
    ETPAN_APP_LOG((app->app, "mailbox could not be created - %s", name));
    return;
  }
  
  ETPAN_APP_LOG((app->app, "mailbox created - %s", name));
  etpan_input_common_handle_key(app, &state->common_state, '\n');
  
  return;
}


static void ask_create_directory(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  char * name;
  struct stat buf_stat;
  
  state = app->data;
  
  name = etpan_input_common_get_value(app, &state->common_state);
  r = stat(name, &buf_stat);
  if (r == 0) {
    if (!S_ISDIR(buf_stat.st_mode)) {
      ETPAN_APP_LOG((app->app, "you entered the name of an existing file"));
      return;
    }
    
    etpan_input_common_handle_key(app, &state->common_state, '\n');
    
    return;
  }
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "do you want to create the mailbox (y/n) ? ", 1,
      NULL, 0,
      ask_create_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "not enough memory"));
}


#define HELP_TEXT \
"\
Help for filename input\n\
-----------------------\n\
\n\
This application will let you edit a filename.\n\
\n\
- [TAB]      : Complete the filename\n\
\n\
- [Enter]    : edition of filename is finished\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
