/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-folder-list-app.c,v 1.33 2006/07/25 08:34:36 hoa Exp $
 */

#include "etpan-folder-list-app.h"
#include "etpan-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-errors.h"
#include "etpan-folder-params.h"
#include "etpan-app-subapp.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include <libetpan/libetpan.h>
#include <libetpan/charconv.h>
#include "etpan-msg-list-app.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-folder-common.h"
#include "etpan-folder-conf.h"
#include "etpan-thread-manager.h"
#include "etpan-config.h"
#include "etpan-help-viewer.h"
#include "etpan-search.h"
#include "etpan-folder-discover.h"
#include "etpan-3paned.h"
#include "etpan-msg-viewer-app.h"

static void idle(struct etpan_subapp * app);
static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void enter(struct etpan_subapp * app, struct etpan_subapp * old_app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);
static int get_idle_delay(struct etpan_subapp * app);
static int get_idle_udelay(struct etpan_subapp * app, struct timeval * delay);

static struct etpan_subapp_driver etpan_folder_list_app_driver = {
  .name = "folder-list",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = get_idle_delay,
  .get_idle_udelay = get_idle_udelay,
  .idle = idle,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = enter,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

#define NEW_SELECTION_DELAY (3 * 100000) /* 0.3 s */

struct app_state {
  struct etpan_folder_common_app_state common_state;
  
  /* poll delay */
  time_t last_poll_date;
  int disable_poll;
  
  int new_selection;
  struct timeval last_sel_date;
};

static int poll_folders(struct etpan_subapp * app);

static void discover_folder_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_discovery_info * info;
  int r;
  struct app_state * state;
  
  info = op->arg;
  
  state = app->data;
  
  ETPAN_APP_DEBUG((app->app, "discover finished %s", info->vpath));
  r = etpan_discover_update_config(app->app->config.discovery_config,
      state->common_state.params, info);
  
  etpan_folder_common_update(&state->common_state);
}

static void discover_folders(struct etpan_subapp * app)
{
  int r;
  chashiter * iter;
  struct app_state * state;
  
  state = app->data;
  
  if (state->disable_poll)
    return;
  
  ETPAN_APP_DEBUG((app->app, "try to discover folders"));
  for(iter = chash_begin(app->app->config.discovery_config->vpath_hash) ;
      iter != NULL ;
      iter = chash_next(app->app->config.discovery_config->vpath_hash, iter)) {
    struct etpan_discovery_info * info;
    chashdatum data;
    
    chash_value(iter, &data);
    info = data.data;
    
    ETPAN_APP_DEBUG((app->app, "poll %s", info->vpath));
    r = etpan_subapp_thread_op_add(app,
        THREAD_ID_FOLDER_DISCOVER,
        ETPAN_THREAD_DISCOVER_FOLDER,
        info->storage,
        NULL, NULL, NULL,
        info,
        discover_folder_callback,
        NULL, NULL);
  }
}

static int get_idle_delay(struct etpan_subapp * app)
{
  struct app_state * state;
  time_t now;
  int diff;
  time_t poll_delay;

  if (!app->enabled)
    return -1;
  
  state = app->data;
  
  poll_delay = app->app->config.global_config->poll_delay;
  
  if (poll_delay == 0)
    return -1;
  
  if (state->last_poll_date == (time_t) -1)
    return 0;
  
  now = time(NULL);
  diff = (int) difftime(state->last_poll_date + poll_delay, now);
  
  if (diff < 0)
    diff = 0;
  
  return diff;
}

static int get_idle_udelay(struct etpan_subapp * app, struct timeval * delay)
{
  struct app_state * state;
  struct timeval now;
  unsigned long d_sec;
  long d_usec;
  
  if (!app->enabled)
    return -1;
  
  state = app->data;
  
  if (!state->new_selection)
    return -1;
  
  if (!etpan_3paned(app->app))
    return -1;
  
  gettimeofday(&now, NULL);
  if (now.tv_sec <= state->last_sel_date.tv_sec)
    d_sec = (state->last_sel_date.tv_sec - now.tv_sec);
  else
    d_sec = 0;
  d_usec = (state->last_sel_date.tv_usec + NEW_SELECTION_DELAY - now.tv_usec);
#if 0
  ETPAN_APP_DEBUG((app->app, "%i %i", d_sec, d_usec));
#endif
  if (d_usec > 1000000) {
    d_sec ++;
    d_usec -= 1000000;
  }
  if (d_usec < 0) {
    if (d_sec == 0) {
      d_usec = 0;
    }
    else {
      d_sec --;
      d_usec += 1000000;
    }
  }
  delay->tv_sec = d_sec;
  delay->tv_usec = d_usec;
#if 0
  ETPAN_APP_DEBUG((app->app, "%i %i", now.tv_sec, now.tv_usec));
  ETPAN_APP_DEBUG((app->app, "%i %i",
                      state->last_sel_date.tv_sec,
                      state->last_sel_date.tv_usec));
  ETPAN_APP_DEBUG((app->app, "%i %i", d_sec, d_usec));
#endif
#if 0
  ETPAN_APP_DEBUG((app->app, "%i %i", d_sec, d_usec));
#endif
  
  return 0;
}

static int open_folder(struct etpan_subapp * app,
    struct mailfolder * folder, int do_switch);

static void idle(struct etpan_subapp * app)
{
  struct app_state * state;
  time_t now;
  time_t poll_delay;
  
  if (!app->enabled)
    return;
  
  state = app->data;
  
  poll_delay = app->app->config.global_config->poll_delay;
  
  if (poll_delay != 0) {
    now = time(NULL);
    if ((state->last_poll_date == (time_t) -1) ||
        ((int) difftime(now, state->last_poll_date) > poll_delay)) {
      ETPAN_APP_DEBUG((app->app, "poll folders %i %i",
                          now - state->last_poll_date,
                          poll_delay));
      poll_folders(app);
      
      discover_folders(app);
    }
  }
  
  if (etpan_3paned(app->app)) {
    if (state->new_selection) {
      struct timeval tv_now;
      unsigned long d_sec;
      long d_usec;
    
      gettimeofday(&tv_now, NULL);
      if (tv_now.tv_sec <= state->last_sel_date.tv_sec) {
        d_sec = (state->last_sel_date.tv_sec - tv_now.tv_sec);
        d_usec = (state->last_sel_date.tv_usec + NEW_SELECTION_DELAY -
            tv_now.tv_usec);
        if (d_usec > 1000000) {
          d_sec ++;
          d_usec -= 1000000;
        }
        if (d_usec < 0) {
          if (d_sec == 0) {
            d_usec = 0;
          }
          else {
            d_sec --;
            d_usec += 1000000;
          }
        }
      }
      else {
        d_sec = 0;
        d_usec = 0;
      }
      
      if ((d_sec == 0) && (d_usec <= 0)) {
        struct mailfolder * folder;
      
        folder = etpan_folder_common_get_selected_folder(&state->common_state);
        if (folder != NULL) {
          if (folder->fld_storage != NULL) {
            int do_poll;
            
            do_poll = etpan_vfolder_do_poll(app->app->config.vfolder_config,
                folder);
            if (do_poll)
              open_folder(app, folder, 0);
          }
        }
        state->new_selection = 0;
      }
    }
  }
}

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

static void thread_status_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  struct etpan_folder_status_result * result;
  
  state = app->data;
  
  if (op->err != NO_ERROR) {
    op->result = NULL;
    return;
  }
  
  result = op->result;
  
  if (((int) result->number) < 0)
    result->number = 0;
  if (((int) result->recent) < 0)
    result->recent = 0;
  if (((int) result->unseen) < 0)
    result->unseen = 0;
  etpan_folder_common_set_stat(&state->common_state, op->data.mailaccess.folder, 1,
      result->number, result->recent, result->unseen);
  
  if (result->unseen > 0)
    etpan_folder_common_set_node_visible(&state->common_state,
        op->data.mailaccess.folder);
    
  if (result->recent != 0) {
    char * folder_name;
      
    folder_name = NULL;
    if (op->data.mailaccess.folder != NULL)
      folder_name = etpan_folder_get_virtual_path(op->data.mailaccess.folder);
      
    if (folder_name != NULL) {
      ETPAN_APP_LOG((app->app, "%i new messages in %s", result->recent,
                        folder_name));
      free(folder_name);
    }
    else
      ETPAN_APP_LOG((app->app, "%i new messages", result->recent));
  }
  free(result);

  op->result = NULL;
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static int update_folder_info(struct etpan_subapp * subapp,
    struct mailfolder * folder);

static void check_folder(struct etpan_subapp * subapp,
    struct mailfolder * folder);

static int match_folder(struct etpan_subapp * msglist_app, void * data)
{
  return (etpan_msg_list_app_get_folder(msglist_app) == data);
}

static int ui_basic_open_folder(struct etpan_subapp * app,
    struct mailfolder * folder, int do_switch)
{
  struct app_state * state;
  struct etpan_subapp * msglist_app;
  
  state = app->data;
  
  update_folder_info(app, folder);
  
  msglist_app = etpan_app_find_subapp(app->app, "msg-list",
    1, match_folder, folder);
  if (msglist_app != NULL) {
    etpan_app_switch_subapp(msglist_app, 0);
    return NO_ERROR;
  }
  
  /* reuse a previous interface */
  msglist_app = etpan_app_find_subapp(app->app, "msg-list",
      0, NULL, NULL);
  if (msglist_app == NULL) {
    /* create new */
    msglist_app = etpan_msg_list_app_new(app->app);
    if (msglist_app == NULL)
      return ERROR_MEMORY;
  }
  
  /* set parent interface */
  etpan_subapp_set_parent(msglist_app, app);
  
  /* set folder and run */
  etpan_msg_list_app_set_folder(msglist_app, folder);
  if (do_switch) {
    etpan_app_switch_subapp(msglist_app, 0);
    ETPAN_APP_DEBUG((app->app, "switch to another folder"));
  }
  
  return NO_ERROR;
}

static int ui_3paned_open_folder(struct etpan_subapp * app,
    struct mailfolder * folder, int do_switch)
{
  struct app_state * state;
  struct etpan_subapp * msglist_app;
  struct mailfolder * old_folder;
  
  state = app->data;
  
  update_folder_info(app, folder);
  
  msglist_app = etpan_3paned_get_msglist(app->app);
  
  old_folder = etpan_msg_list_app_get_folder(msglist_app);
  
  /* set folder and run */
  
  if (old_folder != folder) {
    struct etpan_subapp * msgviewer_app;
    
    etpan_msg_list_app_set_folder(msglist_app, folder);
    ETPAN_APP_DEBUG((app->app, "switch to another folder"));
    
    msgviewer_app = etpan_3paned_get_msgview(app->app);
    etpan_msg_viewer_app_flush(msgviewer_app);
  }
  
  if (do_switch) {
    etpan_app_switch_subapp(msglist_app, 0);
  }
  
  return NO_ERROR;
}

static int open_folder(struct etpan_subapp * app,
    struct mailfolder * folder, int do_switch)
{
  if (etpan_3paned(app->app)) {
    return ui_3paned_open_folder(app, folder, do_switch);
  }
  else {
    return ui_basic_open_folder(app, folder, do_switch);
  }
}

static int catchup_messages(struct etpan_subapp * app);

static int clear_stat_folder(struct etpan_subapp * app);

static int set_poll_stat_folder(struct etpan_subapp * app);

static int edit_folder_conf(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static int search_messages(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  struct mailfolder * folder;
  struct mailfolder * last_selection;
  
  state = app->data;
  
  last_selection =
    etpan_folder_common_get_selected_folder(&state->common_state);
  
  etpan_folder_common_handle_key(app, &state->common_state, key);

  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;

  case 'u':
    set_poll_stat_folder(app);
    folder = etpan_folder_common_get_selected_folder(&state->common_state);
    if (folder != NULL) {
      if (folder->fld_storage != NULL)
        update_folder_info(app, folder);
    }
    break;

  case 'q':
    etpan_app_quit_subapp(app);
    break;
    
  case 'm':
    folder = etpan_folder_common_get_selected_folder(&state->common_state);
    
    etpan_compose_new(app, folder);
    break;
    
  case 'x':
    clear_stat_folder(app);
    break;

  case KEY_CTRL('E'):
    edit_folder_conf(app);
    break;
    
  case '\t':
    if (etpan_3paned(app->app)) {
      struct etpan_subapp * subapp;
      
      subapp = etpan_3paned_get_msglist(app->app);
      etpan_app_switch_subapp(subapp, 0);
    }
    break;
  }
  
  if (!state->disable_poll) {
    switch (key) {
    case '\n':
    case KEY_RIGHT:
      folder = etpan_folder_common_get_selected_folder(&state->common_state);
      if (folder != NULL) {
        if (folder->fld_storage != NULL)
          open_folder(app, folder, 1);
      }
      break;
      
    case 'p':
      poll_folders(app);
      break;

    case ':':
      search_messages(app);
      break;
      
    case KEY_CTRL('C'):
      catchup_messages(app);
      break;
    }
  }
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder != last_selection) {
    gettimeofday(&state->last_sel_date, NULL);
    state->new_selection = 1;
  }
}

static void set_title(struct etpan_subapp * app);

static int display_init(struct etpan_subapp * app)
{
  set_title(app);
  return etpan_app_subapp_display_init(app);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_folder_common_display(app, &state->common_state, w, "q: quit");
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  etpan_folder_common_set_color(app, &state->common_state);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;

  subapp->data = state;
  
  state->last_poll_date = (time_t) -1;
  state->disable_poll = 0;
  
  r = etpan_folder_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free_state;
  
  return NO_ERROR;
  
 free_state:
  free(subapp->data);
  subapp->data = NULL;
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;

  etpan_folder_common_done(&state->common_state);

  free(subapp->data);
}


int etpan_folder_list_set_root_folder(struct etpan_subapp * app,
    struct mailfolder * root)
{
  struct app_state * state;
  unsigned int i;
  int r;
  
  state = app->data;
  
  state->last_poll_date = (time_t) -1;
#if 0
  state->state = STATE_NORMAL;
#endif
  gettimeofday(&state->last_sel_date, NULL);
  state->new_selection = 1;
  
  r = etpan_folder_common_set_root_folder(&state->common_state, root,
      ETPAN_FOLDER_COMMON_SHOW_STAT);

  if (state->common_state.folder_tab != NULL) {
    for (i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
      struct mailfolder * folder;
      
      folder = carray_get(state->common_state.folder_tab, i);
      
      if (folder == app->app->config.vfolder_config->default_folder)
        state->common_state.selected = i;
    }
  }
  
  return r;
}

struct etpan_subapp * etpan_folder_list_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_folder_list_app_driver);
}

void etpan_folder_list_leave_folder(struct etpan_subapp * app,
    struct mailfolder * folder)
{
  struct app_state * state;
  
  state = app->data;

  check_folder(app, folder);
  update_folder_info(app, folder);
}

static void enter(struct etpan_subapp * app, struct etpan_subapp * old_app)
{
  /* do nothing */
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;

  state = app->data;

  etpan_folder_common_flush(app, &state->common_state);
}

/* ***************************************** */
/* implementation */

static int update_folder_info(struct etpan_subapp * subapp,
    struct mailfolder * folder)
{
  int r;
  
  if (folder->fld_storage == NULL)
    return ERROR_INVAL;
  
  if (etpan_subapp_thread_has_match_op(subapp, THREAD_ID_FOLDERLIST_STATUS,
          folder->fld_storage, folder,
          NULL, NULL))
    return NO_ERROR;
  
  r = etpan_subapp_thread_folder_op_add(subapp, THREAD_ID_FOLDERLIST_STATUS,
      ETPAN_THREAD_FOLDER_STATUS, folder,
      NULL, NULL,
      NULL,
      thread_status_callback, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((subapp->app, "folder status - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

static void check_folder(struct etpan_subapp * subapp,
    struct mailfolder * folder)
{
  if (folder->fld_storage != NULL) {
    etpan_subapp_thread_folder_op_add(subapp, THREAD_ID_FOLDERLIST_CHECK,
        ETPAN_THREAD_FOLDER_CHECK, folder,
        NULL, NULL,
        NULL,
        NULL, NULL, NULL);
  }
}



#define TITLE "etPan! - folder list"

static void set_title(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, TITLE);
}

static int check_msglist(struct etpan_subapp * app,
    struct mailfolder * folder,
    carray * msglist)
{
  struct etpan_msglist_check_arg * arg;
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL)
    return ERROR_MEMORY;
  
  arg->msglist = msglist;
  
  return etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_FOLDERLIST_CHECK_MSG, ETPAN_THREAD_MSGLIST_CHECK,
      folder,
      NULL, NULL,
      arg,
      NULL, NULL, NULL);
}

static unsigned int get_env_tree_size(struct mailmessage_tree * node)
{
  unsigned int i;
  unsigned int count;
  
  count = 0;
  if (node->node_msg != NULL)
    count ++;
  
  for(i = 0 ; i < carray_count(node->node_children) ; i ++) {
    count += get_env_tree_size(carray_get(node->node_children, i));
  }
  
  return count;
}

static void catchup_msg(carray * msglist,
    struct mailmessage_tree * node)
{
  unsigned int i;
  
  if (node->node_msg != NULL) {
    struct mail_flags * flags;
    
    flags = node->node_msg->msg_flags;
    
    if (flags != NULL) {
      uint32_t old_flags;
      
      old_flags = flags->fl_flags;
      flags->fl_flags &= ~MAIL_FLAG_NEW;
      flags->fl_flags |= MAIL_FLAG_SEEN;
      
      /* optimization - don't check if no change */
      if (old_flags != flags->fl_flags)
        carray_add(msglist, node->node_msg, NULL);
    }
  }
  
  for(i = 0 ; i < carray_count(node->node_children) ; i ++) {
   catchup_msg(msglist, carray_get(node->node_children, i));
  }
}

static void thread_get_msg_list_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  struct etpan_folder_get_msg_list_result * thread_result;
  struct etpan_folder_free_msg_list_arg * arg;
  int r;
  carray * msglist;
  unsigned int count;

  state = app->data;
  
  if (op->err != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "error while getting list of messages"));
    return;
  }
  
  thread_result = op->result;
  
  count = get_env_tree_size(thread_result->env_tree);
  
  msglist = carray_new(count);
  if (msglist == NULL) {
    ETPAN_APP_LOG((app->app, "change flags - not enough memory"));
    goto free_env_tree;
  }
  
  catchup_msg(msglist, thread_result->env_tree);
  
  r = check_msglist(app, op->data.mailaccess.folder, msglist);
  if (r != NO_ERROR) {
    free(msglist);
    ETPAN_APP_LOG((app->app, "change message flags - not enough memory"));
  }
  
 free_env_tree:
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL) {
    /*
      will create a memory leak, anyway, we're in a case we've got 
       no more memory 
    */
    goto free_result;
  }
#if 0
  arg->msg_params = thread_result->msg_params;
  arg->env_list = thread_result->env_list;
#endif
  arg->env_tree = thread_result->env_tree;
  arg->node_params = thread_result->node_params;
  
  free(thread_result);
  op->result = NULL;

  r = etpan_subapp_thread_op_add(app,
      THREAD_ID_FOLDERLIST_FREE_MSG_LIST,
      ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
      op->data.mailaccess.storage, op->data.mailaccess.folder,
      NULL, NULL, arg, NULL, NULL, NULL);
  if (r != NO_ERROR) {
    free(arg);
    ETPAN_APP_LOG((app->app, "catchup folder - not enough memory"));
    return;
  }
  
  update_folder_info(app, op->data.mailaccess.folder);
  
  return;

 free_result:
  free(thread_result);
  return;
}

static int catchup_messages(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  int r;
  struct app_state * state;
  
  state = app->data;

  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  
  if (folder == NULL)
    return ERROR_INVAL;
  
  if (folder->fld_storage == NULL)
    return ERROR_INVAL;
  
  r = etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_FOLDERLIST_GET_MSG_LIST,
      ETPAN_THREAD_FOLDER_GET_MSG_LIST,
      folder,
      NULL, NULL, NULL, thread_get_msg_list_callback, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "catchup folder - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

static int clear_stat_folder(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  struct app_state * state;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  
  if (folder == NULL)
    return ERROR_INVAL;
  
  etpan_folder_common_set_stat(&state->common_state, folder, 0, 0, 0, 0);
  
  if (!etpan_vfolder_do_poll(app->app->config.vfolder_config, folder))
    return NO_ERROR;
  
  etpan_vfolder_set_poll(app->app->config.vfolder_config, folder, 0);
  etpan_config_write(app->app, &app->app->config);
  
  return NO_ERROR;
}

static int set_poll_stat_folder(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  struct app_state * state;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  
  if (folder == NULL)
    return ERROR_INVAL;
  
  if (etpan_vfolder_do_poll(app->app->config.vfolder_config, folder))
    return NO_ERROR;
  
  etpan_vfolder_set_poll(app->app->config.vfolder_config, folder, 1);
  etpan_config_write(app->app, &app->app->config);
  
  return NO_ERROR;
}

static int poll_folders(struct etpan_subapp * app)
{
  unsigned int i;
  int r;
  struct app_state * state;
  time_t now;
  
  state = app->data;
  
  if (state->disable_poll)
    return NO_ERROR;
  
  for (i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
    struct mailfolder * folder;
    int do_poll;
    
    folder = carray_get(state->common_state.folder_tab, i);
    
    if (folder->fld_storage != NULL) {
      uint32_t num;
      uint32_t recent;
      uint32_t unseen;
      
      r = etpan_folder_common_get_stat(&state->common_state, folder,
          &num, &recent, &unseen);
      do_poll = etpan_vfolder_do_poll(app->app->config.vfolder_config, folder);
      if (r || do_poll)
        update_folder_info(app, folder);
    }
  }
  
  now = time(NULL);
  state->last_poll_date = now;
  
  return NO_ERROR;
}

static void
done_config_finished(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  
  state = app->data;
  
  state->disable_poll = 0;
}

static void
folder_conf_upcall(struct etpan_subapp * folder_conf_app,
    int valid, void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  int r;
  
  app = data;
  
  state = app->data;
  
  if (valid == ETPAN_FOLDER_CONF_VALID) {
    struct etpan_app_config * config;
    
    etpan_subapp_thread_cancel_all(app);
    
    config = malloc(sizeof(* config));
    if (config == NULL) {
      ETPAN_APP_LOG((app->app, "load new config - not enough memory"));
      return;
    }
    
    memcpy(config, &app->app->config, sizeof(* config));
    
    state->disable_poll = 1;
    ETPAN_APP_DEBUG((app->app, "uninitializing old config"));
    r = etpan_subapp_thread_op_add(app,
        THREAD_ID_FOLDERLIST_DONE_CONFIG,
        ETPAN_THREAD_DONE_CONFIG,
        NULL, NULL,
        NULL, NULL, config,
        done_config_finished, NULL,
        NULL);
    if (r < 0) {
      ETPAN_APP_DEBUG((app->app, "uninitializing old config - not enough memory"));
      etpan_app_config_done(&app->app->config);
      free(config);
      state->disable_poll = 0;
    }
    
#if 0
    etpan_app_config_done(&app->app->config);
#endif
    memset(&app->app->config, 0, sizeof(app->app->config));
  
    etpan_app_config_init(app->app->config_path,
        app->app, &app->app->config);
    
    etpan_folder_list_set_root_folder(app,
        app->app->config.vfolder_config->root);
  }
  
  etpan_app_quit_subapp(folder_conf_app);
}

static int edit_folder_conf(struct etpan_subapp * app)
{
  struct etpan_subapp * edit_app;
  
  /* quit all sub application to unref all messages and folders */
  while (1) {
    struct etpan_subapp * subapp;
    
    subapp = etpan_app_find_child_subapp(app, 1);
    if (subapp != NULL)
      etpan_app_leave_subapp(subapp, NULL);
    else
      break;
  }
  
  edit_app = etpan_app_find_subapp(app->app, "folder-conf",
      0, NULL, NULL);
  if (edit_app == NULL)
    edit_app = etpan_app_find_subapp(app->app, "folder-conf",
        1, NULL, NULL);
  if (edit_app == NULL)
    edit_app = etpan_folder_conf_app_new(app->app);
  if (edit_app == NULL)
    return ERROR_MEMORY;
  
  etpan_subapp_set_parent(edit_app, app);
  
  etpan_folder_conf_app_set(edit_app,
      folder_conf_upcall, app);
  etpan_app_switch_subapp(edit_app, 0);
  
  return NO_ERROR;
}

static int search_messages(struct etpan_subapp * app)
{
  struct etpan_subapp * search_app;
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  search_app = etpan_app_find_subapp(app->app, "search-app",
    0, NULL, NULL);
  if (search_app == NULL) {
    search_app = etpan_search_app_new(app->app);
    if (search_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(search_app, app);
  
  etpan_search_app_set(search_app, folder);
  
  etpan_app_switch_subapp(search_app, 0);
  
  return NO_ERROR;
}




int etpan_folder_list_next_unread_message(struct etpan_subapp * app,
    int forward)
{
  struct mailfolder * folder;
  struct etpan_subapp * msglist_app;
  int r;
  struct app_state * state;
  
  state = app->data;
  
  r = etpan_folder_common_next_unread_folder(&state->common_state);
  if (r != NO_ERROR)
    return r;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  
  msglist_app = etpan_app_find_subapp(app->app, "msg-list",
      1, match_folder, folder);
  
  open_folder(app, folder, 1);
  
  if (msglist_app != NULL) {
    etpan_msg_list_first_unread(msglist_app, 1);
  }
  
  return NO_ERROR;
}



#define HELP_TEXT \
"\
Help for folder list view\n\
-------------------------\n\
\n\
This application will let you select the folder which you want to\n\
view the content.\n\
\n\
- left, right,\n\
  Ctrl-W,\n\
  Ctrl-X      : switch between applications\n\
\n\
- up, down    : move cursor\n\
\n\
- u           : update count of messages of selected folder\n\
- x           : clear count of messages of selected folder\n\
- p           : update count of messages of all folders that\n\
                 have the property poll enabled or for which\n\
                 we already requested the count\n\
- Ctrl-C      : mark all messages as read in the selected folder\n\
\n\
- /           : search folder by name\n\
- :           : search messages\n\
\n\
- [Enter]     : view folder content\n\
- m           : compose new message\n\
\n\
- Ctrl-E      : edit folder tree configuration\n\
- q           : quit etPan!\n\
\n\
- ?           : help\n\
- Ctrl-L      : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
