#include "etpan-account-config.h"

#include <string.h>
#include <stdlib.h>

#include "etpan-error.h"
#include "etpan-config-types.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-connection-types.h"
#include "etpan-storage.h"
#include "etpan-storage-imap.h"
#include "etpan-storage-imap-sync.h"
#include "etpan-storage-news.h"
#include "etpan-storage-mbox.h"
#include "etpan-storage-mh.h"
#include "etpan-storage-maildir.h"
#include "etpan-storage-pop.h"
#include "etpan-storage-pop-sync.h"
#include "etpan-sender.h"
#include "etpan-sender-smtp.h"
#include "etpan-sender-news.h"
#include "etpan-sender-sent-folder.h"
#include "etpan-outbox.h"
#include "etpan-utils.h"
#include "etpan-log.h"
#include "etpan-tc.h"
#include "etpan-nls.h"

#define DEFAULT_NEWS_PORT 119
#define DEFAULT_NEWS_TLS_PORT 563

#define DEFAULT_IMAP_PORT 143
#define DEFAULT_IMAP_TLS_PORT 993

#define DEFAULT_POP_PORT 110
#define DEFAULT_POP_TLS_PORT 995

#define DEFAULT_SMTP_PORT 25
#define DEFAULT_SMTP_TLS_PORT 465

static struct {
  int type;
  char * name;
} auth_type_tab[] = {
  { ETPAN_AUTH_NONE, "none" },
  { ETPAN_AUTH_PASSWORD, "password" },
  { ETPAN_AUTH_PLAIN, "plain" },
  { ETPAN_AUTH_CRAM_MD5, "cram-md5" },
  { ETPAN_AUTH_DIGEST_MD5, "digest-md5" },
  { ETPAN_AUTH_KERBEROS4, "kerberos4" },
  { ETPAN_AUTH_GSSAPI, "gssapi" },
  { ETPAN_AUTH_OTP, "otp" },
  { ETPAN_AUTH_PASSDSS_3DES_1, "passdss-3des-1" },
  { ETPAN_AUTH_SRP, "srp" },
  { ETPAN_AUTH_NTLM, "ntlm" },
  { ETPAN_AUTH_ANONYMOUS, "anonymous" },
};

static char * get_auth_type_str(int auth_type)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(auth_type_tab) / sizeof(auth_type_tab[0]) ; i ++) {
    if (auth_type_tab[i].type == auth_type)
      return auth_type_tab[i].name;
  }
  
  return "password";
}

static int get_auth_type(char * str)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(auth_type_tab) / sizeof(auth_type_tab[0]) ; i ++) {
    if (strcasecmp(auth_type_tab[i].name, str) == 0)
      return auth_type_tab[i].type;
  }
  
  return ETPAN_AUTH_PASSWORD;
}

static inline int get_int_entry(tcconf_section_t * conf, char * name)
{
  int r;
  int value;
  
  r = tcconf_getvalue(conf, name, "%i", &value);
  if (r <= 0)
    return -1;
  
  return value;
}

static inline char * get_entry(tcconf_section_t * conf, char * name)
{
  int r;
  char * value;
  
  r = tcconf_getvalue(conf, name, "%s", &value);
  if (r <= 0)
    return NULL;
  
  return value;
}


static void read_config(struct etpan_account_manager * manager,
    tcconf_section_t * config, carray * error_list);

struct etpan_error *
etpan_account_config_read(struct etpan_account_manager * manager,
    char * filename)
{
  tcconf_section_t * config;
  carray * error_list;
  struct etpan_error * error;
  unsigned int i;
  
  config = tcconf_load_file(NULL, filename);
  if (config == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Address book configuration at %s could not be parsed."), filename);
    
    return error;
  }
  
  error_list = carray_new(4);
  if (error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  read_config(manager, config, error_list);
  
  tcfree(config);
  
  etpan_account_manager_notify_modification(manager);
  
  if (carray_count(error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(error_list) == 1) {
    error = carray_get(error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(error_list);
  
  return error;
}

struct etpan_error * etpan_account_config_read_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_account_manager * manager;
  
  manager = etpan_account_manager_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_ACCOUNT_FILE);
  
  return etpan_account_config_read(manager, filename);
}

static void read_account(struct etpan_account_manager * manager,
    tcconf_section_t * account_info, carray * error_list);

static void read_config(struct etpan_account_manager * manager,
    tcconf_section_t * config, carray * error_list)
{
  void * iter;
  
  iter = NULL;
  do {
    tcconf_section_t * account_info;
    
    account_info = tcconf_nextsection(config, "account", &iter);
    if (account_info != NULL) {
      read_account(manager, account_info, error_list);
      tcfree(account_info);
    }
    else {
      break;
    }
  } while (1);
}

static struct etpan_error * add_account(struct etpan_account_manager * manager,
    struct etpan_account  * account);

static int read_account_info(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account ** p_account, carray * error_list);

static void read_account(struct etpan_account_manager * manager,
    tcconf_section_t * account_info, carray * error_list)
{
  int r;
  struct etpan_account * account;
  struct etpan_error * error;
  
  r = read_account_info(account_info, manager, &account, error_list);
  
  if (r == 0) {
    error = add_account(manager, account);
    if (error != NULL) {
      etpan_account_free(account);
      r = carray_add(error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
}

static struct etpan_error * add_account(struct etpan_account_manager * manager,
    struct etpan_account  * account)
{
  char * name;
  struct etpan_account * existing_account;
  struct etpan_error * error;
  
  name = etpan_account_get_id(account);
  
  existing_account = etpan_account_manager_get_account(manager, name);
  if (existing_account != NULL) {
    ETPAN_LOG("address book already exists with this name");
    etpan_crash();
  }
  
  error = etpan_account_setup(account);
  if (error != NULL)
    return error;
  
  etpan_account_manager_add_account(manager, account);
  
  return NULL;
}

static int read_storage_info(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account, carray * error_list);

static void read_sender_info(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static int read_account_info(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account ** p_account, carray * error_list)
{
  int r;
  struct etpan_account * account; 
  char * name;
  char * display_name;
  char * email;
  
  account = etpan_account_new();
  
  name = get_entry(account_info, "name");
  if (name == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Account has no name"));
    etpan_error_strf_long_description(error, _("The account in configuration has no name."));
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    goto free;
  }
  etpan_account_set_id(account, name);
  free(name);
  
  display_name = get_entry(account_info, "display-name");
  etpan_account_set_display_name(account, display_name);
  free(display_name);
  
  email = get_entry(account_info, "email");
  etpan_account_set_mail(account, email);
  free(email);
  
  r = read_storage_info(account_info, manager, account, error_list);
  if (r < 0) {
    goto free;
  }
  
  read_sender_info(account_info, manager, account);
  
  * p_account = account;
  
  return 0;
  
 free:
  etpan_account_free(account);
  return -1;
}

static void read_sender_info(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char * hostname;
  int port;
  int starttls;
  int tls;
  int connection_type;
  char * username;
  char * password;
  struct etpan_outbox * outbox;
  struct etpan_sender * sender;
  char * outbox_path;
  struct etpan_storage * storage;
  struct etpan_storage_driver * storage_driver;
  char * auth_type_str;
  int auth_type;
  (void) manager;
  
  hostname = NULL;
  port = -1;
  connection_type = ETPAN_CONNECTION_PLAIN;
  username = NULL;
  password = NULL;
  auth_type = -1;
  
  hostname = get_entry(account_info, "smtp-hostname");
  
  tls = get_int_entry(account_info, "smtp-tls");
  if (tls != -1) {
    if (tls)
      connection_type = ETPAN_CONNECTION_TLS;
  }
  
  starttls = get_int_entry(account_info, "smtp-starttls");
  if (starttls != -1) {
    if (starttls)
      connection_type = ETPAN_CONNECTION_STARTTLS;
  }
  
  auth_type_str = get_entry(account_info, "smtp-auth-type");
  if (auth_type_str != NULL) {
    auth_type = get_auth_type(auth_type_str);
    free(auth_type_str);
  }
  if (auth_type == -1)
    auth_type = ETPAN_AUTH_NONE;
  
  port = get_int_entry(account_info, "smtp-port");
  
  username = get_entry(account_info, "smtp-username");
  password = get_entry(account_info, "smtp-password");
  
  storage = etpan_account_get_storage(account);
  storage_driver = etpan_storage_get_driver(storage);
  
  /* smtp outbox */
  
  sender = etpan_sender_smtp_new();
  
  etpan_sender_smtp_set_hostname(sender, hostname);
  etpan_sender_smtp_set_port(sender, port);
  etpan_sender_smtp_set_connection_type(sender, connection_type);
  etpan_sender_smtp_set_auth_type(sender, auth_type);
  etpan_sender_smtp_set_username(sender, username);
  etpan_sender_smtp_set_password(sender, password);
  
  outbox = etpan_outbox_new();
  outbox_path = etpan_get_outbox_dir(etpan_account_manager_get_default(),
      account);
  etpan_outbox_set_path(outbox, outbox_path);
  free(outbox_path);
  
  etpan_outbox_set_sender(outbox, sender);
  
  etpan_account_add_outbox(account, outbox);

  /* sent folder outbox
     this is used to append message to 'sent messages' folder, using
     the same mechanism as outbox.
  */
  
  if (strcasecmp(storage_driver->name, "news") != 0) {
    /* XXX - currently, sent folder can't be added for news account */
    sender = etpan_sender_sent_folder_new();
    etpan_sender_sent_folder_set_storage(sender, storage);
    outbox = etpan_outbox_new();
    outbox_path = etpan_get_outbox_sent_folder_dir(etpan_account_manager_get_default(),
        account);
    etpan_outbox_set_path(outbox, outbox_path);
    free(outbox_path);
    
    etpan_outbox_set_sender(outbox, sender);
    
    etpan_account_add_outbox(account, outbox);
  }
  
  if (strcasecmp(storage_driver->name, "news") == 0) {
    /* news, add a second outbox */
    
    sender = etpan_sender_news_new();
    
    etpan_sender_news_set_storage(sender, storage);
    
    outbox = etpan_outbox_new();
    outbox_path = etpan_get_outbox_news_dir(etpan_account_manager_get_default(),
        account);
    etpan_outbox_set_path(outbox, outbox_path);
    free(outbox_path);
    
    etpan_outbox_set_sender(outbox, sender);
    
    etpan_account_add_outbox(account, outbox);
  }
  
  free(hostname);
  free(password);
  free(username);
}

static void read_imap_storage(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static void read_pop_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static void read_news_storage(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static void read_mbox_storage(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static void read_mh_storage(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static void read_maildir_storage(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account);

static int read_storage_info(tcconf_section_t  * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account, carray * error_list)
{
  char * account_type;
  int r;
  
  account_type = get_entry(account_info, "type");
  if (account_type == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Account has no type"));
    etpan_error_strf_long_description(error, _("The account %s has no type."),
        etpan_account_get_id(account));
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    return -1;
  }
  
  if (strcasecmp(account_type, "imap") == 0) {
    read_imap_storage(account_info, manager, account);
  }
  else if (strcasecmp(account_type, "news") == 0) {
    read_news_storage(account_info, manager, account);
  }
  else if (strcasecmp(account_type, "mbox") == 0) {
    read_mbox_storage(account_info, manager, account);
  }
  else if (strcasecmp(account_type, "mh") == 0) {
    read_mh_storage(account_info, manager, account);
  }
  else if (strcasecmp(account_type, "maildir") == 0) {
    read_maildir_storage(account_info, manager, account);
  }
  else if (strcasecmp(account_type, "pop") == 0) {
    read_pop_storage(account_info, manager, account);
  }
  else {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Account has an invalid type"));
    etpan_error_strf_long_description(error, _("The account %s has an invalid type (%s)."), etpan_account_get_id(account), account_type);
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    free(account_type);
    return -1;
  }
  
  free(account_type);
  
  return 0;
}

static void read_host_info(tcconf_section_t  * account_info,
    char ** p_hostname, int * p_port, int * p_connection_type,
    int * p_auth_type,
    char ** p_username, char ** p_password)
{
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  int tls;
  int starttls;
  int auth_type;
  char * auth_type_str;
  
  hostname = NULL;
  port = -1;
  connection_type = ETPAN_CONNECTION_PLAIN;
  username = NULL;
  password = NULL;
  auth_type = -1;
  
  hostname = get_entry(account_info, "hostname");
  
  tls = get_int_entry(account_info, "tls");
  if (tls != -1) {
    if (tls)
      connection_type = ETPAN_CONNECTION_TLS;
  }
  
  starttls = get_int_entry(account_info, "starttls");
  if (starttls != -1) {
    if (starttls)
      connection_type = ETPAN_CONNECTION_STARTTLS;
  }
  
  auth_type_str = get_entry(account_info, "auth-type");
  if (auth_type_str != NULL) {
    auth_type = get_auth_type(auth_type_str);
    free(auth_type_str);
  }
  
  port = get_int_entry(account_info, "port");
  
  username = get_entry(account_info, "username");
  password = get_entry(account_info, "password");
  
  * p_hostname = hostname;
  * p_port = port;
  * p_connection_type = connection_type;
  * p_auth_type = auth_type;
  * p_username = username;
  * p_password = password;
}

static void read_imap_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  char * base_location;
  char * cache_path;
  int auth_type;
  
  storage = etpan_storage_imap_sync_new();
  if (storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  read_host_info(account_info, &hostname, &port, &connection_type,
      &auth_type, &username, &password);
  if (auth_type == -1)
    auth_type = ETPAN_AUTH_PASSWORD;
  
  etpan_storage_imap_sync_set_hostname(storage, hostname);
  etpan_storage_imap_sync_set_connection_type(storage, connection_type);
  etpan_storage_imap_sync_set_auth_type(storage, auth_type);
  
  if (port != -1) {
    etpan_storage_imap_sync_set_port(storage, port);
  }
  else {
    if (connection_type == ETPAN_CONNECTION_TLS)
      etpan_storage_imap_sync_set_port(storage, DEFAULT_IMAP_TLS_PORT);
    else
      etpan_storage_imap_sync_set_port(storage, DEFAULT_IMAP_PORT);
  }
  
  etpan_storage_imap_sync_set_username(storage, username);
  etpan_storage_imap_sync_set_password(storage, password);

  base_location = get_entry(account_info, "base-location");
  etpan_storage_imap_sync_set_base_location(storage, base_location);
  free(base_location);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  etpan_account_set_storage(account, storage);
  
  free(hostname);
  free(username);
  free(password);
}

static void read_pop_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  char * path;
  char * cache_path;
  char * flags_path;
  int auth_type;
  
  storage = etpan_storage_pop_sync_new();
  if (storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  read_host_info(account_info, &hostname, &port, &connection_type,
      &auth_type, &username, &password);
  if (auth_type == -1)
    auth_type = ETPAN_AUTH_PASSWORD;
  
  etpan_storage_pop_sync_set_hostname(storage, hostname);
  etpan_storage_pop_sync_set_connection_type(storage, connection_type);
  etpan_storage_pop_sync_set_auth_type(storage, auth_type);
  
  if (port != -1) {
    etpan_storage_pop_sync_set_port(storage, port);
  }
  else {
    if (connection_type == ETPAN_CONNECTION_TLS)
      etpan_storage_pop_sync_set_port(storage, DEFAULT_POP_TLS_PORT);
    else
      etpan_storage_pop_sync_set_port(storage, DEFAULT_POP_PORT);
  }
  
  etpan_storage_pop_sync_set_username(storage, username);
  etpan_storage_pop_sync_set_password(storage, password);

  path = get_entry(account_info, "path");
  etpan_storage_pop_sync_set_path(storage, path);
  free(path);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  etpan_account_set_storage(account, storage);
  
  free(hostname);
  free(username);
  free(password);
}


static void read_news_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  char * cache_path;
  char * flags_path;
  int auth_type;
  
  storage = etpan_storage_news_new();
  if (storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  read_host_info(account_info, &hostname, &port, &connection_type,
      &auth_type, &username, &password);
  if (auth_type == -1)
    auth_type = ETPAN_AUTH_NONE;
  
  etpan_storage_news_set_hostname(storage, hostname);
  etpan_storage_news_set_connection_type(storage, connection_type);
  
  if (port != -1) {
    etpan_storage_news_set_port(storage, port);
  }
  else {
    if (connection_type == ETPAN_CONNECTION_TLS)
      etpan_storage_news_set_port(storage, DEFAULT_NEWS_TLS_PORT);
    else
      etpan_storage_news_set_port(storage, DEFAULT_NEWS_PORT);
  }
  
  etpan_storage_news_set_username(storage, username);
  etpan_storage_news_set_password(storage, password);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  etpan_account_set_storage(account, storage);
  
  free(hostname);
  free(username);
  free(password);
}


static void read_mbox_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * path;
  char * cache_path;
  char * flags_path;
  
  storage = etpan_storage_mbox_new();
  
  path = get_entry(account_info, "path");
  etpan_storage_mbox_set_path(storage, path);
  free(path);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  etpan_account_set_storage(account, storage);
}


static void read_mh_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * path;
  char * cache_path;
  char * flags_path;
  
  storage = etpan_storage_mh_new();
  
  path = get_entry(account_info, "path");
  etpan_storage_mh_set_path(storage, path);
  free(path);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  etpan_account_set_storage(account, storage);
}

static void read_maildir_storage(tcconf_section_t * account_info,
    struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * path;
  char * cache_path;
  char * flags_path;
  
  storage = etpan_storage_maildir_new();
  
  path = get_entry(account_info, "path");
  etpan_storage_maildir_set_path(storage, path);
  free(path);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  etpan_account_set_storage(account, storage);
}

static void pop_account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;  
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * path;
  
  fprintf(f, "  type \"pop\"\n");
  
  storage = etpan_account_get_storage(account);
  hostname = etpan_storage_pop_sync_get_hostname(storage);
  if (hostname != NULL) {
    char * quoted_hostname;
    
    quoted_hostname = etpan_quote_string(hostname);
    fprintf(f, "  hostname \"%s\"\n", quoted_hostname);
    free(quoted_hostname);
  }
  
  port = etpan_storage_pop_sync_get_port(storage);
  fprintf(f, "  port %i\n", port);
  
  connection_type = etpan_storage_pop_sync_get_connection_type(storage);
  switch (connection_type) {
  case ETPAN_CONNECTION_TLS:
    fprintf(f, "  tls 1\n");
    break;
  case ETPAN_CONNECTION_STARTTLS:
    fprintf(f, "  starttls 1\n");
    break;
  }
  
  auth_type = etpan_storage_pop_sync_get_auth_type(storage);
  fprintf(f, "  auth-type \"%s\"\n", get_auth_type_str(auth_type));
  
  username = etpan_storage_pop_sync_get_username(storage);
  if (username != NULL) {
    char * quoted_username;
    
    quoted_username = etpan_quote_string(username);
    fprintf(f, "  username \"%s\"\n", quoted_username);
    free(quoted_username);
  }
  
  password = etpan_storage_pop_sync_get_password(storage);
  if (password != NULL) {
    char * quoted_password;
    
    quoted_password = etpan_quote_string(password);
    fprintf(f, "  password \"%s\"\n", quoted_password);
    free(quoted_password);
  }

  path = etpan_storage_pop_sync_get_path(storage);
  if (path != NULL) {
    char * quoted_path;
    
    quoted_path = etpan_quote_string(path);
    fprintf(f, "  path \"%s\"\n", quoted_path);
    free(quoted_path);
  }
}

static void imap_account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;  
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * base_location;
  
  fprintf(f, "  type \"imap\"\n");
  
  storage = etpan_account_get_storage(account);
  hostname = etpan_storage_imap_sync_get_hostname(storage);
  if (hostname != NULL) {
    char * quoted_hostname;
    
    quoted_hostname = etpan_quote_string(hostname);
    fprintf(f, "  hostname \"%s\"\n", quoted_hostname);
    free(quoted_hostname);
  }
  
  port = etpan_storage_imap_sync_get_port(storage);
  fprintf(f, "  port %i\n", port);
  
  connection_type = etpan_storage_imap_sync_get_connection_type(storage);
  switch (connection_type) {
  case ETPAN_CONNECTION_TLS:
    fprintf(f, "  tls 1\n");
    break;
  case ETPAN_CONNECTION_STARTTLS:
    fprintf(f, "  starttls 1\n");
    break;
  }
  
  auth_type = etpan_storage_imap_sync_get_auth_type(storage);
  fprintf(f, "  auth-type \"%s\"\n", get_auth_type_str(auth_type));
  
  username = etpan_storage_imap_sync_get_username(storage);
  if (username != NULL) {
    char * quoted_username;
    
    quoted_username = etpan_quote_string(username);
    fprintf(f, "  username \"%s\"\n", quoted_username);
    free(quoted_username);
  }
  
  password = etpan_storage_imap_sync_get_password(storage);
  if (password != NULL) {
    char * quoted_password;
    
    quoted_password = etpan_quote_string(password);
    fprintf(f, "  password \"%s\"\n", quoted_password);
    free(quoted_password);
  }

  base_location = etpan_storage_imap_sync_get_base_location(storage);
  if (base_location != NULL) {
    char * quoted_base_location;
    
    quoted_base_location = etpan_quote_string(base_location);
    fprintf(f, "  base-location \"%s\"\n", quoted_base_location);
    free(quoted_base_location);
  }
}

static void news_account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;  
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;

  fprintf(f, "  type \"news\"\n");
  
  storage = etpan_account_get_storage(account);
  hostname = etpan_storage_news_get_hostname(storage);
  if (hostname != NULL) {
    char * quoted_hostname;
    
    quoted_hostname = etpan_quote_string(hostname);
    fprintf(f, "  hostname \"%s\"\n", quoted_hostname);
    free(quoted_hostname);
  }
  
  port = etpan_storage_news_get_port(storage);
  fprintf(f, "  port %i\n", port);
  
  connection_type = etpan_storage_news_get_connection_type(storage);
  switch (connection_type) {
  case ETPAN_CONNECTION_TLS:
    fprintf(f, "  tls 1\n");
    break;
  case ETPAN_CONNECTION_STARTTLS:
    fprintf(f, "  starttls 1\n");
    break;
  }
  
  username = etpan_storage_news_get_username(storage);
  if (username != NULL) {
    char * quoted_username;
    
    quoted_username = etpan_quote_string(username);
    fprintf(f, "  username \"%s\"\n", quoted_username);
    free(quoted_username);
  }
  
  password = etpan_storage_news_get_password(storage);
  if (password != NULL) {
    char * quoted_password;
    
    quoted_password = etpan_quote_string(password);
    fprintf(f, "  password \"%s\"\n", quoted_password);
    free(quoted_password);
  }
}

static void maildir_account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;  
  char * path;
  
  fprintf(f, "  type \"maildir\"\n");
  storage = etpan_account_get_storage(account);
  path = etpan_storage_maildir_get_path(storage);
  fprintf(f, "  path \"%s\"\n", path);
}

static void mh_account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;  
  char * path;
  
  fprintf(f, "  type \"mh\"\n");
  storage = etpan_account_get_storage(account);
  path = etpan_storage_mh_get_path(storage);
  {
    char * quoted_path;
    
    quoted_path = etpan_quote_string(path);
    fprintf(f, "  path \"%s\"\n", quoted_path);
    free(quoted_path);
  }
}

static void mbox_account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;  
  char * path;
  
  fprintf(f, "  type \"mbox\"\n");
  storage = etpan_account_get_storage(account);
  path = etpan_storage_mbox_get_path(storage);
  {
    char * quoted_path;
    
    quoted_path = etpan_quote_string(path);
    fprintf(f, "  path \"%s\"\n", quoted_path);
    free(quoted_path);
  }
}

static void smtp_sender_save(struct etpan_sender * sender, FILE * f)
{
  char * hostname;
  int connection_type;
  char * auth_type_str;
  int auth_type;
  int port;
  char * username;
  char * password;
  
  hostname = etpan_sender_smtp_get_hostname(sender);
  if (hostname != NULL) {
    char * quoted_hostname;
    
    quoted_hostname = etpan_quote_string(hostname);
    fprintf(f, "  smtp-hostname \"%s\"\n", quoted_hostname);
    free(quoted_hostname);
  }
  
  connection_type = etpan_sender_smtp_get_connection_type(sender);
  switch (connection_type) {
  case ETPAN_CONNECTION_TLS:
    fprintf(f, "  tls 1\n");
    break;
  case ETPAN_CONNECTION_STARTTLS:
    fprintf(f, "  starttls 1\n");
    break;
  }
  
  auth_type = etpan_sender_smtp_get_auth_type(sender);
  auth_type_str = get_auth_type_str(auth_type);
  fprintf(f, "  smtp-auth-type \"%s\"\n", auth_type_str);
  
  port = etpan_sender_smtp_get_port(sender);
  fprintf(f, "  smtp-port %i\n", port);
  
  username = etpan_sender_smtp_get_username(sender);
  if (username != NULL) {
    char * quoted_username;
    
    quoted_username = etpan_quote_string(username);
    fprintf(f, "  smtp-username \"%s\"\n", quoted_username);
    free(quoted_username);
  }
  
  password = etpan_sender_smtp_get_password(sender);
  if (password != NULL) {
    char * quoted_password;
    
    quoted_password = etpan_quote_string(password);
    fprintf(f, "  smtp-password \"%s\"\n", quoted_password);
    free(quoted_password);
  }
}

static void account_save(struct etpan_account * account, FILE * f)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  char * name;
  char * display_name;
  char * email;
  carray * outbox_list;
  unsigned int i;
  
  storage = etpan_account_get_storage(account);
  if (storage == NULL) {
    ETPAN_LOG("invalid account");
    etpan_crash();
    return;
  }
  
  driver = etpan_storage_get_driver(storage);
  
  fprintf(f, "account [\n");
  name = etpan_account_get_id(account);
  if (name != NULL) {
    char * quoted_name;
    
    quoted_name = etpan_quote_string(name);
    fprintf(f, "  name \"%s\"\n", quoted_name);
    free(quoted_name);
  }
  
  /* get name and mail */
  display_name = etpan_account_get_display_name(account);
  if (display_name != NULL) {
    char * quoted_display_name;
    
    quoted_display_name = etpan_quote_string(display_name);
    fprintf(f, "  display-name \"%s\"\n", quoted_display_name);
    free(quoted_display_name);
  }
  
  email = etpan_account_get_mail(account);
  if (email != NULL) {
    char * quoted_email;
    
    quoted_email = etpan_quote_string(email);
    fprintf(f, "  email \"%s\"\n", quoted_email);
    free(quoted_email);
  }
  
  /* storage */
  if (strcasecmp(driver->name, "pop-sync") == 0)
    pop_account_save(account, f);
  else if (strcasecmp(driver->name, "imap-sync") == 0)
    imap_account_save(account, f);
  else if (strcasecmp(driver->name, "news") == 0)
    news_account_save(account, f);
  else if (strcasecmp(driver->name, "maildir") == 0)
    maildir_account_save(account, f);
  else if (strcasecmp(driver->name, "mh") == 0)
    mh_account_save(account, f);
  else if (strcasecmp(driver->name, "mbox") == 0)
    mbox_account_save(account, f);
  
  /* SMTP */
  outbox_list = etpan_account_get_outbox_list(account);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    struct etpan_sender * sender;
    struct etpan_sender_driver * driver;
    
    outbox = carray_get(outbox_list, i);
    sender = etpan_outbox_get_sender(outbox);
    if (sender == NULL)
      continue;
    
    driver = etpan_sender_get_driver(sender);
    if (strcmp(driver->name, "smtp") == 0) {
      smtp_sender_save(sender, f);
    }
  }
  
  fprintf(f, "]\n");
  fprintf(f, "\n");
}

struct etpan_error *
etpan_account_config_save(struct etpan_account_manager * manager,
    char * filename)
{
  carray * list;
  unsigned int i;
  FILE * f;
  (void) manager;
  
  f = fopen(filename, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Configuration file could not be written"));
    etpan_error_strf_long_description(error, _("Account configuration file %s could not be written to disk."), filename);
    
    return error;
  }
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_account * account;
    
    account = carray_get(list, i);
    account_save(account, f);
  }
  
  fclose(f);
  
  return NULL;
}

struct etpan_error * etpan_account_config_save_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_account_manager * manager;
  
  manager = etpan_account_manager_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_ACCOUNT_FILE);
  
  return etpan_account_config_save(manager, filename);
}
