#include "etpan-account-manager.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-account.h"
#include "etpan-error.h"
#include "etpan-signal.h"
#include "etpan-log.h"

struct etpan_account_manager * etpan_account_manager_new(void)
{
  struct etpan_account_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->account_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager->account_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->ordered_list = carray_new(4);
  if (manager->ordered_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->path = NULL;

  manager->stop_remaining = 0;
  manager->stop_cb_data = NULL;
  manager->stop_callback = NULL;
  
  return manager;
}

void etpan_account_manager_free(struct etpan_account_manager * manager)
{
  chashiter * iter;
  
  free(manager->path);
  
  for(iter = chash_begin(manager->account_hash) ; iter != NULL ;
      iter = chash_next(manager->account_hash, iter)) {
    chashdatum value;
    struct etpan_account * account;
    
    chash_value(iter, &value);
    account = value.data;
    
    etpan_account_free(account);
  }
  
  carray_free(manager->ordered_list);
  chash_free(manager->account_hash);
  free(manager);
}

chash * etpan_account_manager_get_account_hash(struct etpan_account_manager *
    manager)
{
  return manager->account_hash;
}

struct etpan_account *
etpan_account_manager_get_account(struct etpan_account_manager * manager,
    char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_get(manager->account_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

void etpan_account_manager_add_account(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  chashdatum key;
  chashdatum value;
  int r;
  char * name;
  
  name = etpan_account_get_id(account);
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_get(manager->account_hash, &key, &value);
  if (r == 0) {
    ETPAN_LOG("account with id %s already exists", name);
    etpan_crash();
  }
  
  value.data = account;
  value.len = 0;
  
  r = chash_set(manager->account_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(manager->ordered_list, account, NULL);
  if (r < 0) {
    chash_delete(manager->account_hash, &key, NULL);
    ETPAN_LOG_MEMORY_ERROR;
  }
}

void etpan_account_manager_remove_account(struct etpan_account_manager *
    manager, struct etpan_account * account)
{
  chashdatum key;
  int r;
  char * name;
  unsigned int i;
  
  name = etpan_account_get_id(account);
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_delete(manager->account_hash, &key, NULL);
  /* ignore error */
  
  for(i = 0 ; i < carray_count(manager->ordered_list) ; i ++) {
    struct etpan_account * current_account;
    
    current_account = carray_get(manager->ordered_list, i);
    if (current_account == account) {
      carray_delete_slow(manager->ordered_list, i);
      break;
    }
  }
}

static struct etpan_account_manager * default_manager = NULL;

void etpan_account_manager_set_default(struct etpan_account_manager * manager)
{
  default_manager = manager;
}

struct etpan_account_manager * etpan_account_manager_get_default(void)
{
  return default_manager;
}

carray * etpan_account_manager_get_ordered_list(struct etpan_account_manager * manager)
{
  return manager->ordered_list;
}

void etpan_account_manager_notify_modification(struct etpan_account_manager * manager)
{
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      manager, NULL);
}


char *
etpan_account_manager_get_path(struct etpan_account_manager * manager)
{
  return manager->path;
}

void etpan_account_manager_set_path(struct etpan_account_manager * manager,
    char * path)
{
  if (path != manager->path) {
    free(manager->path);
    if (path != NULL) {
      manager->path = strdup(path);
      if (manager->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      manager->path = NULL;
  }
}

static void account_stop_callback(void * cb_data)
{
  struct etpan_account_manager * manager;
  
  manager = cb_data;
  
  manager->stop_remaining --;
  if (manager->stop_remaining > 0)
    return;
  
  if (manager->stop_callback != NULL)
    manager->stop_callback(manager->stop_cb_data);
  manager->stop_callback = NULL;
  manager->stop_cb_data = NULL;
}

void etpan_account_manager_stop(struct etpan_account_manager * manager,
    void (* callback)(void *), void * cb_data)
{
  chashiter * iter;
  chash * account_hash;
  
  if (manager->stop_remaining) {
    ETPAN_LOG("account manager stop called twice");
    etpan_crash();
  }
  
  manager->stop_cb_data = cb_data;
  manager->stop_callback = callback;
  manager->stop_remaining = 0;
  
  account_hash = etpan_account_manager_get_account_hash(manager);
  if (chash_count(account_hash) == 0) {
    manager->stop_remaining = 1;
    account_stop_callback(manager);
    return;
  }
  
  for(iter = chash_begin(account_hash) ; iter != NULL ;
      iter = chash_next(account_hash, iter)) {
    chashdatum value;
    struct etpan_account * account;
    
    chash_value(iter, &value);
    account = value.data;
    
    manager->stop_remaining ++;
    etpan_account_stop(account, account_stop_callback, manager);
  }
}

void etpan_account_manager_setup(struct etpan_account_manager * manager)
{
  (void) manager;
}

void etpan_account_manager_unsetup(struct etpan_account_manager * manager)
{
  chashiter * iter;
  chash * account_hash;
  
  account_hash = etpan_account_manager_get_account_hash(manager);
  
  for(iter = chash_begin(account_hash) ; iter != NULL ;
      iter = chash_next(account_hash, iter)) {
    chashdatum value;
    struct etpan_account * account;
    
    chash_value(iter, &value);
    account = value.data;
    
    etpan_account_unsetup(account);
  }
}


