#include "etpan-account.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-outbox.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-sender.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-thread-manager-app.h"

struct etpan_account * etpan_account_new(void)
{
  struct etpan_account * account;
  
  account = malloc(sizeof(* account));
  if (account == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  account->ref_count = 1;
  account->id = NULL;
  account->display_name = NULL;
  account->mail = NULL;
  
  account->storage = NULL;
  account->outbox_list = carray_new(2);
  if (account->outbox_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  account->special_folder_list = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYALL);
  if (account->special_folder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  account->stop_remaining = 0;
  account->stop_cb_data = NULL;
  account->stop_callback = NULL;
  
  return account;
}

void etpan_account_free(struct etpan_account * account)
{
  unsigned int i;
  
  chash_free(account->special_folder_list);
  for(i = 0 ; i < carray_count(account->outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    
    outbox = carray_get(account->outbox_list, i);
    etpan_outbox_free(outbox);
  }
  carray_free(account->outbox_list);
  if (account->storage != NULL)
    etpan_storage_free(account->storage);
  free(account->mail);
  free(account->display_name);
  free(account->id);
  free(account);
}

static void set_storage_id(struct etpan_account * account,
    struct etpan_storage * storage, char * id)
{
  (void) account;
  
  if (storage != NULL)
    etpan_storage_set_id(storage, id);
}

static void set_outbox_id(struct etpan_account * account,
    struct etpan_outbox * outbox, char * id)
{
  (void) account;
  
  if (outbox != NULL)
    etpan_outbox_set_id(outbox, id);
}

void etpan_account_set_storage(struct etpan_account * account,
    struct etpan_storage * storage)
{
  if (account->storage != storage) {
    set_storage_id(account, storage, account->id);
    
    if (account->storage != NULL)
      etpan_storage_free(account->storage);
    
    account->storage = storage;
    etpan_storage_set_account(account->storage, account);
  }
}

void etpan_account_set_id(struct etpan_account * account, char * id)
{
  if (id != account->id) {
    char * oldid;
    unsigned int i;
    
    oldid = account->id;
    
    if (id != NULL) {
      account->id = strdup(id);
      if (account->id == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      account->id = NULL;
    }
    
    set_storage_id(account, account->storage, id);
    for(i = 0 ; i < carray_count(account->outbox_list) ; i ++) {
      struct etpan_outbox * outbox;
      
      outbox = carray_get(account->outbox_list, i);
      
      set_outbox_id(account, outbox, id);
    }
    
    free(oldid);
  }
}

char * etpan_account_get_id(struct etpan_account * account)
{
  return account->id;
}

char * etpan_account_get_display_name(struct etpan_account * account)
{
  return account->display_name;
}

char * etpan_account_get_mail(struct etpan_account * account)
{
  return account->mail;
}

void etpan_account_set_display_name(struct etpan_account * account,
    char * display_name)
{
  if (display_name != account->display_name) {
    free(account->display_name);
    if (display_name != NULL) {
      account->display_name = strdup(display_name);
      if (account->display_name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      account->display_name = NULL;
  }
}

void etpan_account_set_mail(struct etpan_account * account, char * mail)
{
  if (mail != account->mail) {
    free(account->mail);
    if (mail != NULL) {
      account->mail = strdup(mail);
      if (account->mail == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      account->mail = NULL;
  }
}

void etpan_account_set_special_folder(struct etpan_account * account,
    char * name, struct etpan_folder * folder)
{
  etpan_account_set_special_folder_by_location(account,
      name, folder->location);
}

void etpan_account_set_special_folder_by_location(struct etpan_account * account,
    char * name, char * location)
{
  chashdatum key;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  
  if (location == NULL) {
    chash_delete(account->special_folder_list, &key, NULL);
  }
  else {
    chashdatum value;
    
    value.data = location;
    value.len = strlen(location + 1);
    
    r = chash_set(account->special_folder_list, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

void etpan_account_set_draft_by_location(struct etpan_account * account,
    char * draft)
{
  etpan_account_set_special_folder_by_location(account,
      SPECIAL_FOLDER_DRAFT, draft);
}

void etpan_account_set_sent_by_location(struct etpan_account * account,
    char * sent)
{
  etpan_account_set_special_folder_by_location(account,
      SPECIAL_FOLDER_SENT, sent);
}

void etpan_account_set_trash_by_location(struct etpan_account * account,
    char * trash)
{
  etpan_account_set_special_folder_by_location(account,
      SPECIAL_FOLDER_TRASH, trash);
}

void etpan_account_set_draft(struct etpan_account * account,
    struct etpan_folder * draft)
{
  etpan_account_set_special_folder(account,
      SPECIAL_FOLDER_DRAFT, draft);
}

void etpan_account_set_sent(struct etpan_account * account,
    struct etpan_folder * sent)
{
  etpan_account_set_special_folder(account,
      SPECIAL_FOLDER_SENT, sent);
}

void etpan_account_set_trash(struct etpan_account * account,
    struct etpan_folder * trash)
{
  etpan_account_set_special_folder(account,
      SPECIAL_FOLDER_TRASH, trash);
}

struct etpan_storage *
etpan_account_get_storage(struct etpan_account * account)
{
  return account->storage;
}

struct etpan_folder *
etpan_account_get_special_folder(struct etpan_account * account,
    char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  char * location;
  
  key.data = name;
  key.len = strlen(name);
  
  r = chash_get(account->special_folder_list, &key, &value);
  if (r < 0)
    return NULL;
  
  location = value.data;
  
  return etpan_storage_get_folder(account->storage, location);
}

struct etpan_folder * etpan_account_get_draft(struct etpan_account * account)
{
  return etpan_account_get_special_folder(account, SPECIAL_FOLDER_DRAFT);
}

struct etpan_folder * etpan_account_get_sent(struct etpan_account * account)
{
  return etpan_account_get_special_folder(account, SPECIAL_FOLDER_SENT);
}

struct etpan_folder * etpan_account_get_trash(struct etpan_account * account)
{
  return etpan_account_get_special_folder(account, SPECIAL_FOLDER_TRASH);
}

carray * etpan_account_get_outbox_list(struct etpan_account * account)
{
  return account->outbox_list;
}

void etpan_account_add_outbox(struct etpan_account * account,
    struct etpan_outbox * outbox)
{
  int r;
  
  set_outbox_id(account, outbox, account->id);
  
  r = carray_add(account->outbox_list, outbox, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_outbox_set_account(outbox, account);
}

void etpan_account_remove_outbox(struct etpan_account * account,
    struct etpan_outbox * outbox)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(account->outbox_list) ; i ++) {
    struct etpan_outbox * current_outbox;
    
    current_outbox = carray_get(account->outbox_list, i);
    if (current_outbox == outbox) {
      set_outbox_id(account, outbox, NULL);
      etpan_outbox_set_account(outbox, NULL);
      carray_delete(account->outbox_list, i);
      break;
    }
  }
}

static void stop_calling_callback(struct etpan_account * account)
{
  account->stop_remaining --;
  if (account->stop_remaining > 0)
    return;
  
  if (account->stop_callback != NULL)
    account->stop_callback(account->stop_cb_data);
  account->stop_callback = NULL;
 account->stop_cb_data = NULL;
}

static void outbox_stop_callback(struct etpan_outbox * outbox, void * cb_data)
{
  struct etpan_account * account;
  (void) outbox;
  
  account = cb_data;
  stop_calling_callback(account);
}

static void storage_disconnect_callback(int cancelled,
    void * dummy, void * cb_data)
{
  struct etpan_account * account;
  (void) cancelled;
  (void) dummy;
  
  ETPAN_LOG("storage stopped");
  account = cb_data;
  stop_calling_callback(account);
}

void etpan_account_stop(struct etpan_account * account,
    void (* callback)(void *), void * cb_data)
{
  struct etpan_storage * storage;
  carray * outbox_list;
  unsigned int i;
  
  if (account->stop_remaining) {
    ETPAN_LOG("account stop called twice");
    etpan_crash();
  }
  
  account->stop_cb_data = cb_data;
  account->stop_callback = callback;
  account->stop_remaining = 0;
  
  storage = etpan_account_get_storage(account);
  account->stop_remaining ++;
  etpan_storage_disconnect(etpan_thread_manager_app_get_default(), storage,
      storage_disconnect_callback, account);
  
  outbox_list = etpan_account_get_outbox_list(account);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    
    outbox = carray_get(outbox_list, i);
    account->stop_remaining ++;
    etpan_outbox_stop(outbox, outbox_stop_callback, account);
  }
}

struct etpan_error * etpan_account_setup(struct etpan_account * account)
{
  struct etpan_storage * storage;
  carray * outbox_list;
  unsigned int i;
  struct etpan_error * error;
  
  storage = etpan_account_get_storage(account);
  
  error = etpan_storage_setup(storage);
  if (error != NULL)
    return error;
  
  outbox_list = etpan_account_get_outbox_list(account);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    
    outbox = carray_get(outbox_list, i);
    error = etpan_outbox_setup(outbox);
    if (error != NULL)
      return error;
  }
  
  return NULL;
}

void etpan_account_unsetup(struct etpan_account * account)
{
  struct etpan_storage * storage;
  carray * outbox_list;
  unsigned int i;
  
  storage = etpan_account_get_storage(account);
  
  outbox_list = etpan_account_get_outbox_list(account);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    
    outbox = carray_get(outbox_list, i);
    etpan_outbox_unsetup(outbox);
  }
  
  etpan_storage_unsetup(storage);
}
