#include "etpan-backend-main.h"

#include <signal.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>

#include "etpan-lep.h"
#include "etpan-utils.h"
#include "etpan-nls.h"
#include "etpan-error.h"
#include "etpan-config-types.h"
#include "etpan-log.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-global-config.h"
#include "etpan-message-color.h"
#include "etpan-abook-manager.h"
#include "etpan-account-manager.h"
#include "etpan-bookmark.h"
#include "etpan-mbox-cache.h"
#include "etpan-account-config.h"
#include "etpan-abook-config.h"
#include "etpan-bookmark-config.h"
#include "etpan-message.h"
#include "etpan-folder.h"
#include "etpan-mail-manager.h"
#include "etpan-filter-config.h"
#include "etpan-filter.h"

static char log_filename[PATH_MAX];

static void log_callback(const char * str)
{
  FILE * f;
  
  f = fopen(log_filename, "a");
  if (f == NULL)
    return;
  
  fprintf(f, "%s\n", str);
  
  fclose(f);
}

struct etpan_error * etpan_backend_init(void)
{
  struct etpan_account_manager * account_manager;
  struct etpan_thread_manager_app * thread_manager;
  struct etpan_signal_manager * signal_manager;
  struct sigaction act;
  struct etpan_bookmark_manager * bookmark_manager;
  char dirname[PATH_MAX];
  char filename[PATH_MAX];
  char time_str[PATH_MAX];
  char * home;
  struct etpan_mbox_cache * mbox_cache;
  struct etpan_message_color * msg_color;
  struct etpan_global_config * global_config;
  struct etpan_abook_manager * abook_manager;
  struct etpan_filter_config * filter_config;
  struct tm time_data;
  time_t time_value;
  char * enable_log_file_str;
  int enable_log_file;
  struct etpan_error * error;
  char * enable_network_log_str;
  int enable_network_log;
  struct etpan_mail_manager * mail_manager;
  
  /* ***** system init ***** */
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    goto err;
  }
  
  /* signal */
  act.sa_handler = SIG_IGN;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  sigaction(SIGPIPE, &act, NULL);
  etpan_crash_report_setup();
  
  /* randomize initialization for Message-ID generation */
  srandom(getpid());
  
  /* temp dir */
  snprintf(dirname, sizeof(dirname), "%s/%s", home, ETPAN_TMP_DIR);
  etpan_set_tmp_dir(dirname);
  
  /* ***** libetpan init ***** */
  
  /* 1 minute for network timeout */
  mailstream_network_delay.tv_sec = 60;
  mailstream_network_delay.tv_usec = 0;
  
  error = etpan_lep_engine_setup();
  if (error != NULL) {
    goto err;
  }
  
  /* ***** backend init ***** */
  
  /* logs */
  etpan_log_init();
  
  enable_log_file_str = getenv("ETPAN_ENABLE_LOG_FILE");
  enable_log_file = 0;
  if (enable_log_file_str != NULL) {
    enable_log_file = strtoul(enable_log_file_str, NULL, 10);
  }
  if (enable_log_file) {
    snprintf(log_filename, sizeof(log_filename), "%s/%s", home, ETPAN_LOG_DIR);
    etpan_make_dir(log_filename);
    time(&time_value);
    localtime_r(&time_value, &time_data);
    strftime(time_str, sizeof(time_str), "%Y-%m-%d--%H-%M-%S.log", &time_data);
    snprintf(log_filename, sizeof(log_filename), "%s/%s/%s",
        home, ETPAN_LOG_DIR, time_str);
    etpan_log_callback = log_callback;
  }
  
  enable_network_log_str = getenv("ETPAN_NETWORK_LOG");
  enable_network_log = 0;
  if (enable_network_log_str != NULL)
    enable_network_log = strtoul(enable_network_log_str, NULL, 10);
  mailstream_debug = enable_network_log;
  
  /* notifications */
  signal_manager = etpan_signal_manager_new();
  etpan_signal_manager_set_default(signal_manager);
  
  /* thread manager */
  thread_manager = etpan_thread_manager_app_new();
  etpan_thread_manager_app_set_default(thread_manager);
  
  /* misc config */
  global_config = etpan_global_config_new();
  etpan_global_config_set_default(global_config);
  
  /* msg color */
  msg_color = etpan_message_color_new();
  etpan_message_color_set_default(msg_color);
  
  /* abook */
  abook_manager = etpan_abook_manager_new();
  etpan_abook_manager_set_default(abook_manager);
  
  /* account */
  account_manager = etpan_account_manager_new();
  snprintf(dirname, sizeof(dirname), "%s/%s", home, ETPAN_ACCOUNT_DIR);
  etpan_make_dir(dirname);
  etpan_account_manager_set_path(account_manager, dirname);
  etpan_account_manager_set_default(account_manager);
  
  /* mail manager */
  mail_manager = etpan_mail_manager_new();
  etpan_mail_manager_set_default(mail_manager);
  
  /* bookmark */
  bookmark_manager = etpan_bookmark_manager_new();
  etpan_bookmark_manager_set_default(bookmark_manager);
  
  /* filter */
  etpan_filter_init();
  filter_config = etpan_filter_config_new();
  etpan_filter_config_set_default(filter_config);
  
  /* mbox cache */
  snprintf(filename, sizeof(filename), "%s/%s/%s",
      home, ETPAN_CACHE_DIR, ETPAN_MBOX_CACHE_FILE);
  mbox_cache = etpan_mbox_cache_new(filename);
  etpan_mbox_cache_set_default(mbox_cache);
  
  return NULL;
  
 err:
  return error;
}

void etpan_backend_done(void)
{
  struct etpan_account_manager * account_manager;
  struct etpan_thread_manager_app * thread_manager;
  struct etpan_signal_manager * signal_manager;
  struct etpan_bookmark_manager * bookmark_manager;
  struct etpan_mbox_cache * mbox_cache;
  struct etpan_message_color * msg_color;
  struct etpan_global_config * global_config;
  struct etpan_abook_manager * abook_manager;
  struct etpan_filter_config * filter_config;
  struct etpan_mail_manager * mail_manager;
  
  mbox_cache = etpan_mbox_cache_get_default();
  etpan_mbox_cache_free(mbox_cache);
  etpan_mbox_cache_set_default(NULL);
  filter_config = etpan_filter_config_get_default();
  etpan_filter_config_free(filter_config);
  etpan_filter_config_set_default(NULL);
  etpan_filter_done();
  bookmark_manager = etpan_bookmark_manager_get_default();
  etpan_bookmark_manager_free(bookmark_manager);
  etpan_bookmark_manager_set_default(NULL);
  mail_manager = etpan_mail_manager_get_default();
  etpan_mail_manager_free(mail_manager);
  etpan_mail_manager_set_default(NULL);
  account_manager = etpan_account_manager_get_default();
  etpan_account_manager_free(account_manager);
  etpan_account_manager_set_default(NULL);
  abook_manager = etpan_abook_manager_get_default();
  etpan_abook_manager_free(abook_manager);
  etpan_abook_manager_set_default(NULL);
  msg_color = etpan_message_color_get_default();
  etpan_message_color_free(msg_color);
  etpan_message_color_set_default(NULL);
  global_config = etpan_global_config_get_default();
  etpan_global_config_free(global_config);
  etpan_global_config_set_default(NULL);
  thread_manager = etpan_thread_manager_app_get_default();
  etpan_thread_manager_app_free(thread_manager);
  etpan_thread_manager_app_set_default(NULL);
  signal_manager = etpan_signal_manager_get_default();
  etpan_signal_manager_free(signal_manager);
  etpan_signal_manager_set_default(NULL);
  etpan_lep_engine_unsetup();
  
  etpan_folder_print_stats();
  etpan_message_print_stats();
  
  etpan_log_done();
  etpan_crash_report_unsetup();
}

void etpan_backend_setup(void)
{
  struct etpan_mbox_cache * mbox_cache;
  struct etpan_thread_manager_app * thread_manager;
  struct etpan_account_manager * account_manager;
  struct etpan_abook_manager * abook_manager;
  struct etpan_mail_manager * mail_manager;
  struct etpan_error * error;
  
  error = etpan_global_config_read_default();
  ETPAN_ERROR_IGNORE(error);
  error = etpan_message_color_read_default();
  ETPAN_ERROR_IGNORE(error);
  error = etpan_account_config_read_default();
  ETPAN_ERROR_IGNORE(error);
  error = etpan_abook_config_read_default();
  ETPAN_ERROR_IGNORE(error);
  error = etpan_bookmark_config_read_default();
  ETPAN_ERROR_IGNORE(error);
  error = etpan_filter_config_read_default();
  ETPAN_ERROR_IGNORE(error);
  
  /* mbox cache setup */
  mbox_cache = etpan_mbox_cache_get_default();
  etpan_mbox_cache_setup(mbox_cache);
  
  /* start thread manager */
  thread_manager = etpan_thread_manager_app_get_default();
  etpan_thread_manager_app_start(thread_manager);
  
  /* abook manager */
  abook_manager = etpan_abook_manager_get_default();
  etpan_abook_manager_setup(abook_manager);
  
  /* account manager */
  account_manager = etpan_account_manager_get_default();
  etpan_account_manager_setup(account_manager);
  
  /* mail manager */
  mail_manager = etpan_mail_manager_get_default();
  etpan_mail_manager_setup(mail_manager);
}

static void mail_manager_stop_callback(void * cb_data);
static void manager_stop_callback(void * cb_data);
static int manager_remaining = 0;
static void (* stop_callback)(void * cb_data) = NULL;
static void * stop_cb_data = NULL;

void etpan_backend_stop(void (* callback)(void * cb_data), void * cb_data)
{
  struct etpan_mail_manager * mail_manager;
  
  stop_callback = callback;
  stop_cb_data = cb_data;
  
  /* stop mail manager */
  mail_manager = etpan_mail_manager_get_default();
  ETPAN_LOG("stopping mail manager");
  etpan_mail_manager_stop(mail_manager, mail_manager_stop_callback, NULL);
}

static void mail_manager_stop_callback(void * cb_data)
{
  struct etpan_account_manager * account_manager;
  struct etpan_abook_manager * abook_manager;
  (void) cb_data;
  
  manager_remaining = 2;
  
  /* stop account */
  account_manager = etpan_account_manager_get_default();
  ETPAN_LOG("stopping account manager");
  etpan_account_manager_stop(account_manager, manager_stop_callback, NULL);
  
  /* stop abook */
  abook_manager = etpan_abook_manager_get_default();
  ETPAN_LOG("stopping abook manager");
  etpan_abook_manager_stop(abook_manager, manager_stop_callback, NULL);
}

static void manager_stop_callback(void * cb_data)
{
  (void) cb_data;
  
  manager_remaining --;
  
  if (manager_remaining > 0)
    return;
  
  ETPAN_LOG("stop callback");
  if (stop_callback != NULL)
    stop_callback(stop_cb_data);
}

void etpan_backend_unsetup(void)
{
  struct etpan_mbox_cache * mbox_cache;
  struct etpan_thread_manager_app * thread_manager;
  struct etpan_account_manager * account_manager;
  struct etpan_abook_manager * abook_manager;
  struct etpan_mail_manager * mail_manager;
  
  mbox_cache = etpan_mbox_cache_get_default();
  etpan_mbox_cache_unsetup(mbox_cache);
  
  ETPAN_LOG("mail manager off");
  mail_manager = etpan_mail_manager_get_default();
  etpan_mail_manager_unsetup(mail_manager);
  
  ETPAN_LOG("account manager off");
  account_manager = etpan_account_manager_get_default();
  etpan_account_manager_unsetup(account_manager);
  
  ETPAN_LOG("abook manager off");
  abook_manager = etpan_abook_manager_get_default();
  etpan_abook_manager_unsetup(abook_manager);
  
  /* stop thread manager */
  thread_manager = etpan_thread_manager_app_get_default();
  etpan_thread_manager_app_stop(thread_manager, NULL, NULL);
  
  etpan_thread_manager_app_join(thread_manager);
}
