#include "etpan-bookmark-config.h"

#include <stdlib.h>

#include "etpan-config-types.h"
#include "etpan-tc.h"
#include "etpan-error.h"
#include "etpan-account-manager.h"
#include "etpan-bookmark.h"
#include "etpan-nls.h"
#include "etpan-log.h"

static inline char * get_entry(tcconf_section_t * conf, char * name)
{
  int r;
  char * value;
  
  r = tcconf_getvalue(conf, name, "%s", &value);
  if (r <= 0)
    return NULL;
  
  return value;
}

static void read_config(struct etpan_bookmark_manager * manager,
    tcconf_section_t * config, carray * error_list);

struct etpan_error *
etpan_bookmark_config_read(struct etpan_bookmark_manager * manager,
    char * filename)
{
  tcconf_section_t * config;
  carray * error_list;
  struct etpan_error * error;
  unsigned int i;
  
  config = tcconf_load_file(NULL, filename);
  if (config == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Bookmark configuration at %s could not be parsed."), filename);
    
    return error;
  }
  
  error_list = carray_new(4);
  if (error_list != NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  read_config(manager, config, error_list);
  
  tcfree(config);
  
  if (carray_count(error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(error_list) == 1) {
    error = carray_get(error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(error_list);
  
  return error;
}

struct etpan_error * etpan_bookmark_config_read_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_bookmark_manager * manager;
  
  manager = etpan_bookmark_manager_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_BOOKMARK_FILE);
  
  return etpan_bookmark_config_read(manager, filename);
}


static void read_bookmark(struct etpan_bookmark_manager * manager,
    tcconf_section_t * info, carray * error_list);

static void read_config(struct etpan_bookmark_manager * manager,
    tcconf_section_t * config, carray * error_list)
{
  void * iter;
  
  iter = NULL;
  do {
    tcconf_section_t * bookmark_info;
    
    bookmark_info = tcconf_nextsection(config, "bookmark", &iter);
    if (bookmark_info != NULL) {
      read_bookmark(manager, bookmark_info, error_list);
      tcfree(bookmark_info);
    }
    else {
      break;
    }
  } while (1);
}

static void read_bookmark(struct etpan_bookmark_manager * manager,
    tcconf_section_t * info, carray * error_list)
{
  struct etpan_bookmark * item;
  char * display_name;
  char * storage;
  char * location;
  (void) error_list;
  
  item = etpan_bookmark_new();
  
  display_name = get_entry(info, "display-name");
  etpan_bookmark_set_display_name(item, display_name);
  free(display_name);
  
  storage = get_entry(info, "storage");
  etpan_bookmark_set_storage(item, storage);
  free(storage);
  
  location = get_entry(info, "location");
  etpan_bookmark_set_location(item, location);
  free(location);
  
  etpan_bookmark_manager_add(manager, item);
}
