#include "etpan-bookmark.h"

#include <string.h>
#include <stdlib.h>

#include "etpan-error.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-storage.h"
#include "etpan-log.h"

struct etpan_bookmark_manager * etpan_bookmark_manager_new(void)
{
  struct etpan_bookmark_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->bookmark_list = carray_new(16);
  if (manager->bookmark_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return manager;
}

void etpan_bookmark_manager_free(struct etpan_bookmark_manager * manager)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->bookmark_list) ; i ++) {
    struct etpan_bookmark * item;
    
    item = carray_get(manager->bookmark_list, i);
    etpan_bookmark_free(item);
  }
  carray_free(manager->bookmark_list);
  free(manager);
}

void etpan_bookmark_manager_add(struct etpan_bookmark_manager * manager,
    struct etpan_bookmark * item)
{
  int r;
  
  r = carray_add(manager->bookmark_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_bookmark_manager_remove(struct etpan_bookmark_manager * manager,
    struct etpan_bookmark * item)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->bookmark_list) ; i ++) {
    struct etpan_bookmark * current_item;
    
    current_item = carray_get(manager->bookmark_list, i);
    if (item == current_item) {
      carray_delete_slow(manager->bookmark_list, i);
      etpan_bookmark_free(item);
      break;
    }
  }
}

carray *
etpan_bookmark_manager_get_list(struct etpan_bookmark_manager * manager)
{
  return manager->bookmark_list;
}

struct etpan_bookmark * etpan_bookmark_new(void)
{
  struct etpan_bookmark * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    return NULL;
  
  item->display_name = NULL;
  item->storage = NULL;
  item->location = NULL;
  
  return item;
}

void etpan_bookmark_free(struct etpan_bookmark * item)
{
  free(item->display_name);
  free(item->storage);
  free(item->location);
  free(item);
}

void etpan_bookmark_set_display_name(struct etpan_bookmark * bookmark,
    char * display_name)
{
  if (display_name != bookmark->display_name) {
    free(bookmark->display_name);
    if (display_name != NULL) {
      bookmark->display_name = strdup(display_name);
      if (bookmark->display_name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      bookmark->display_name = NULL;
  }
}

char * etpan_bookmark_get_display_name(struct etpan_bookmark * bookmark)
{
  return bookmark->display_name;
}

void etpan_bookmark_set_storage(struct etpan_bookmark * bookmark,
    char * storage)
{
  if (storage != bookmark->storage) {
    free(bookmark->storage);
    if (storage != NULL) {
      bookmark->storage = strdup(storage);
      if (bookmark->storage == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      bookmark->storage = NULL;
  }
}

char * etpan_bookmark_get_storage(struct etpan_bookmark * bookmark)
{
  return bookmark->storage;
}

void etpan_bookmark_set_location(struct etpan_bookmark * bookmark,
    char * location)
{
  if (location != bookmark->location) {
    free(bookmark->location);
    if (location != NULL) {
      bookmark->location = strdup(location);
      if (bookmark->location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      bookmark->location = NULL;
  }
}

char * etpan_bookmark_get_location(struct etpan_bookmark * bookmark)
{
  return bookmark->location;
}


static struct etpan_bookmark_manager * default_manager = NULL;

void etpan_bookmark_manager_set_default(struct etpan_bookmark_manager *
    manager)
{
  default_manager = manager;
}

struct etpan_bookmark_manager * etpan_bookmark_manager_get_default(void)
{
  return default_manager;
}


struct etpan_folder *
etpan_bookmark_get_folder(struct etpan_account_manager * manager,
    struct etpan_bookmark * bookmark)
{
  struct etpan_account * account;
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  
  if (bookmark->storage == NULL)
    return NULL;
  
  account = etpan_account_manager_get_account(manager, bookmark->storage);
  if (account == NULL)
    return NULL;

  storage = etpan_account_get_storage(account);
  if (storage == NULL)
    return NULL;
  
  if (bookmark->location == NULL)
    return NULL;
  
  folder = etpan_storage_get_folder(storage, bookmark->location);
  
  return folder;
}
