/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-db-read.c,v 1.8 2005/02/01 02:31:58 hoa Exp $
 */

#include "etpan-db-read.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "etpan-errors.h"

#define MAX_LINE 1024
#define DEFAULT_ENTRY_SIZE 4
#define DEFAULT_ARRAY_SIZE 16

static void etpan_chomp(char * str)
{
  char * p;
  size_t len;

  p = strchr(str, '\r');
  if (p != NULL)
    * p = 0;

  p = strchr(str, '\n');
  if (p != NULL)
    * p = 0;
}

static void etpan_strip_string(char * str)
{
  char * p;
  size_t len;

  p = strchr(str, '#');
  if (p != NULL)
    * p = 0;
  
  p = strchr(str, '\r');
  if (p != NULL)
    * p = 0;

  p = strchr(str, '\n');
  if (p != NULL)
    * p = 0;

  p = str;
  while ((* p == ' ') || (* p == '\t')) {
    p ++;
  }
  
  len = strlen(p);
  memmove(str, p, len);
  str[len] = 0;
  
  if (len == 0)
    return;
  
  p = str;
  len = len - 1;
  while ((p[len] == ' ') || (p[len] == '\t')) {
    p[len] = '\0';
    
    if (len == 0)
      break;
    
    len --;
  }
}

static chash * entry_new(void)
{
  return chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
}

static void entry_free(chash * entry)
{
  chash_free(entry);
}

char * etpan_db_entry_get_value(chash * entry, char * name)
{
  chashdatum key;
  chashdatum value;
  char * lower_name;
  char * p;
  int r;

  lower_name = strdup(name);
  if (lower_name == NULL)
    return NULL;
  for(p = lower_name ; * p != '\0' ; p ++) {
    * p = (char) tolower((unsigned char) * p);
  }
  
  key.data = lower_name;
  key.len = strlen(lower_name) + 1;

  r = chash_get(entry, &key, &value);
  free(lower_name);
  
  if (r < 0)
    return NULL;
  else
    return value.data;
}

static struct etpan_db * etpan_db_new(void)
{
  carray * data;
  struct etpan_db * db;
  
  data = carray_new(DEFAULT_ARRAY_SIZE);
  if (data == NULL)
    goto err;
  
  db = malloc(sizeof(* db));
  if (db == NULL)
    goto free_data;

  db->data = data;
  
  return db;
  
 free_data:
  carray_free(data);
 err:
  return NULL;
}

static int etpan_db_set(struct etpan_db * db, chash * entry)
{
  int r;
  int res;

  r = carray_add(db->data, entry, NULL);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}

void etpan_db_free(struct etpan_db * db)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    
    entry = carray_get(db->data, i);
    entry_free(entry);
  }
  carray_free(db->data);
  free(db);
}

int etpan_read_config(char * filename, struct etpan_db ** result)
{
  FILE * f;
  struct etpan_db * db;
  char s[MAX_LINE];
  chash * entry;
  char id[MAX_LINE];
  int res;
  int r;
  chashdatum key;
  chashdatum d_value;
    
  f = fopen(filename, "r");
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }

  db = etpan_db_new();
  if (db == NULL) {
    res = ERROR_MEMORY;
    goto close;
  }
  
  /* create a new empty entry */
  entry = entry_new();
  if (entry == NULL) {
    res = ERROR_MEMORY;
    goto free;
  }
  
  * id = '\0';
  
  while (fgets(s, MAX_LINE, f) != NULL) {
    char * p;
    char * name;
    char * value;
    
    etpan_chomp(s);
    
    if (* s == '\0') {
      size_t len;
      
      r = etpan_db_set(db, entry);
      if (r < 0) {
        entry_free(entry);
        res = ERROR_MEMORY;
        goto free;
      }
      
      entry = entry_new();
      if (entry == NULL) {
        res = ERROR_MEMORY;
        goto free;
      }
    }
    
    etpan_strip_string(s);
    
    /* split the string in two parts */
    p = strchr(s, '=');
    if (p == NULL)
      continue;
    
    * p = '\0';
    name = s;
    value = p + 1;

    etpan_strip_string(name);
    for(p = name ; * p != '\0' ; p ++) {
      * p = (char) tolower(((unsigned char) * p));
    }
    etpan_strip_string(value);
    
#if 0
    /* id=value will set a new entry */
    if (strcasecmp(name, "id") == 0) {
      size_t len;
      
      if (* id != '\0') {
        r = etpan_db_set(db, entry);
        if (r < 0) {
          entry_free(entry);
          res = ERROR_MEMORY;
          goto free;
        }
      }
      else {
        entry_free(entry);
      }
      
      len = strlen(value);
      if (len > sizeof(id)) {
        len = sizeof(id);
      }
      strncpy(id, value, len);
      id[sizeof(id) - 1] = 0;

      entry = entry_new();
      if (entry == NULL) {
        res = ERROR_MEMORY;
        goto free;
      }
    }
#endif
    
    key.data = name;
    key.len = strlen(name) + 1;
    d_value.data = value;
    d_value.len = strlen(value) + 1;
    
    chash_set(entry, &key, &d_value, NULL);
  }
  
#if 0
  if (* id != '\0') {
    r = etpan_db_set(db, entry);
    if (r < 0) {
      entry_free(entry);
      res = ERROR_MEMORY;
      goto free;
    }
  }
  else {
    entry_free(entry);
  }
#endif
  if (chash_count(entry) != 0) {
    r = etpan_db_set(db, entry);
    if (r < 0) {
      entry_free(entry);
      res = ERROR_MEMORY;
      goto free;
    }
  }
  else {
    entry_free(entry);
  }
  
  * result = db;

  fclose(f);
  
  return NO_ERROR;
  
 free:
  etpan_db_free(db);
 close:
  fclose(f);
 err:
  return res;
}
