#include "etpan-error.h"

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "etpan-log.h"
#include "etpan-nls.h"

struct etpan_error * etpan_error_new(void)
{
  struct etpan_error * error;
  
  error = malloc(sizeof(* error));
  if (error == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  error->code = ERROR_INTERNAL;
  error->short_description = NULL;
  error->long_description = NULL;
  error->children = NULL;
  
  return error;
}

void etpan_error_free(struct etpan_error * error)
{
  if (error->children != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(error->children) ; i ++) {
      struct etpan_error * child;
      
      child = carray_get(error->children, i);
      etpan_error_free(child);
    }
    carray_free(error->children);
  }
  
  free(error->long_description);
  free(error->short_description);
  free(error);
}

void etpan_error_set_code(struct etpan_error * error, int code)
{
  error->code = code;
}

int etpan_error_get_code(struct etpan_error * error)
{
  return error->code;
}

void etpan_error_set_short_description(struct etpan_error * error,
    char * short_description)
{
  if (short_description != error->short_description) {
    free(error->short_description);
    if (short_description != NULL) {
      error->short_description = strdup(short_description);
      if (error->short_description == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
    else
      error->short_description = NULL;
  }
}

char * etpan_error_get_short_description(struct etpan_error * error)
{
  return error->short_description;
}

void etpan_error_set_long_description(struct etpan_error * error,
    char * long_description)
{
  if (long_description != error->long_description) {
    free(error->long_description);
    if (long_description != NULL) {
      error->long_description = strdup(long_description);
      if (error->long_description == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
    else
      error->long_description = NULL;
  }
}

char * etpan_error_get_long_description(struct etpan_error * error)
{
  return error->long_description;
}

void etpan_error_add_child(struct etpan_error * parent,
    struct etpan_error * child)
{
  int r;
  
  if (parent->children == NULL) {
    parent->children = carray_new(4);
    if (parent->children == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  r = carray_add(parent->children, child, NULL);
  if (r < 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
}

carray * etpan_error_get_children(struct etpan_error * parent)
{
  return parent->children;
}


/* helper */

void etpan_error_strf_long_description(struct etpan_error * error,
    char * format, ...)
{
  va_list argp;
  char description[1024];
  
  va_start(argp, format);
  vsnprintf(description, sizeof(description), format, argp);
  etpan_error_set_long_description(error, description);
  
  va_end(argp);
}

struct etpan_error *
etpan_error_set_from_error(struct etpan_error * previous_error,
    char * format, ...)
{
  char * previous_long_description;
  struct etpan_error * new_error;
  va_list argp;
  char description[1024];
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(previous_error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(previous_error));
  
  va_start(argp, format);
  previous_long_description = etpan_error_get_long_description(previous_error);
  vsnprintf(description, sizeof(description), format, argp);
  etpan_error_strf_long_description(new_error, "%s\n%s",
      description, previous_long_description);
  
  va_end(argp);
  
  return new_error;
}

struct etpan_error * etpan_error_dup(struct etpan_error * error)
{
  struct etpan_error * new_error;
  unsigned int i;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  etpan_error_set_long_description(new_error,
      etpan_error_get_long_description(error));
  if (error->children != NULL) {
    for(i = 0 ; i < carray_count(error->children) ; i ++) {
      struct etpan_error * suberror;
    
      suberror = carray_get(error->children, i);
      suberror = etpan_error_dup(suberror);
      etpan_error_add_child(error, suberror);
    }
  }
  
  return new_error;
}

int etpan_error_is_cancelled(struct etpan_error * error)
{
  return etpan_error_has_code(error, ERROR_CANCELLED);
}

int etpan_error_has_code(struct etpan_error * error, int error_code)
{
  unsigned int i;
  
  if (error == NULL) {
    ETPAN_LOG("used error_has_code with NULL error");
    return 0;
  }
  
  if (etpan_error_get_code(error) == error_code)
    return 1;
  
  if (error->children == NULL)
    return 0;
  
  for(i = 0 ; i < carray_count(error->children) ; i ++) {
    struct etpan_error * suberror;
    
    suberror = carray_get(error->children, i);
    if (etpan_error_has_code(suberror, error_code))
      return 1;
  }
  
  return 0;
}

struct etpan_error * etpan_error_internal(char * long_description)
{
  struct etpan_error * error;
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_INTERNAL);
  etpan_error_set_short_description(error, _("An internal error has occurred"));
  etpan_error_set_long_description(error, long_description);
  
  return error;
}

struct etpan_error * etpan_error_internal_strf(char * format, ...)
{
  va_list argp;
  char description[1024];
  struct etpan_error * error;
  
  va_start(argp, format);
  vsnprintf(description, sizeof(description), format, argp);
  
  error = etpan_error_internal(description);
  
  va_end(argp);
  
  return error;
}

struct etpan_error * etpan_error_multiple(void)
{
  struct etpan_error * error;
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_MULTIPLE);
  etpan_error_set_short_description(error, _("Some errors have occurred"));
  
  return error;
}

void etpan_error_log(struct etpan_error * error)
{
  ETPAN_WARN_LOG("ERROR: summary: (%i) %s", etpan_error_get_code(error),
      etpan_error_get_short_description(error));
  ETPAN_WARN_LOG("ERROR: details: %s", etpan_error_get_long_description(error));
  
  if (etpan_error_get_code(error) == ERROR_MULTIPLE) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(error->children) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error->children, i);
      etpan_error_log(suberror);
    }
  }
}
