#include "etpan-folder-copy.h"

#include <stdlib.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-message-copy.h"
#include "etpan-folder-create.h"
#include "etpan-folder-delete.h"
#include "etpan-nls.h"
#include "etpan-mail-manager.h"

/* sub errors should be collected */

enum {
  STATE_IDLE,
  STATE_CREATING,
  STATE_COPYING_SUBFOLDER,
  STATE_COPYING_MSGLIST_FETCH_MSGLIST,
  STATE_COPYING_MSGLIST,
  STATE_DELETING,
};

static struct etpan_error * rewrite_error(struct etpan_folder * src_folder,
    struct etpan_folder * dst_folder, int delete,
    struct etpan_error * error);

static void etpan_folder_copy_unsetup(struct etpan_folder_copy * folder_copy);
static void etpan_folder_copy_free(struct etpan_folder_copy * folder_copy);

static void run(struct etpan_folder_copy * folder_copy);
static void created_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void copy_next_subfolder(struct etpan_folder_copy * folder_copy);
static void copy_subfolder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void copy_subfolder_finished(struct etpan_folder_copy * folder_copy);
static void fetch_msg_list(struct etpan_folder_copy * folder_copy);
static void cancel_fetch_msg_list(struct etpan_folder_copy * folder_copy);
static void msg_list_fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void copy_msg_list(struct etpan_folder_copy * folder_copy);
static void copy_msglist_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void delete_folder(struct etpan_folder_copy * folder_copy);
static void deleted_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void copy_finished(struct etpan_folder_copy * folder_copy);

struct etpan_folder_copy * etpan_folder_copy_new(void)
{
  struct etpan_folder_copy * folder_copy;
  
  folder_copy = malloc(sizeof(* folder_copy));
  if (folder_copy == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_copy->source_folder = NULL;
  folder_copy->dest_storage = NULL;
  folder_copy->dest_folder = NULL;
  folder_copy->created_folder = NULL;
  folder_copy->msg_copy = NULL;
  folder_copy->subfolder_copy = NULL;
  folder_copy->opened = 0;
  folder_copy->foldertab = NULL;
  folder_copy->next_folder = 0;
  folder_copy->delete = 0;
  folder_copy->dest_location = NULL;
  folder_copy->error = NULL;
  folder_copy->error_list = NULL;
  folder_copy->state = STATE_IDLE;
  folder_copy->ref_count = 1;
  
  return folder_copy;
}

static void etpan_folder_copy_free(struct etpan_folder_copy * folder_copy)
{
  etpan_folder_copy_unsetup(folder_copy);
  
  ETPAN_ERROR_FREE(folder_copy->error);
  folder_copy->error = NULL;
  
  if (folder_copy->dest_location != NULL)
    free(folder_copy->dest_location);
  
  if (folder_copy->created_folder != NULL)
    etpan_folder_unref(folder_copy->created_folder);
  if (folder_copy->dest_folder != NULL)
    etpan_folder_unref(folder_copy->dest_folder);
  if (folder_copy->source_folder != NULL)
    etpan_folder_unref(folder_copy->source_folder);
  
  free(folder_copy);
}

void etpan_folder_copy_ref(struct etpan_folder_copy * folder_copy)
{
  folder_copy->ref_count ++;
}

void etpan_folder_copy_unref(struct etpan_folder_copy * folder_copy)
{
  folder_copy->ref_count --;
  if (folder_copy->ref_count == 0)
    etpan_folder_copy_free(folder_copy);
}

void etpan_folder_copy_set_delete(struct etpan_folder_copy * folder_copy,
    int delete)
{
  folder_copy->delete = delete;
}

int etpan_folder_copy_get_delete(struct etpan_folder_copy * folder_copy)
{
  return folder_copy->delete;
}

void etpan_folder_copy_set_source(struct etpan_folder_copy * folder_copy,
    struct etpan_folder * source_folder)
{
  if (source_folder != folder_copy->source_folder) {
    if (folder_copy->source_folder != NULL)
      etpan_folder_unref(folder_copy->source_folder);
    folder_copy->source_folder = source_folder;
    if (source_folder != NULL) {
      etpan_folder_ref(source_folder);
    }
    else
      folder_copy->source_folder = NULL;
  }
}

struct etpan_folder *
etpan_folder_copy_get_source(struct etpan_folder_copy * folder_copy)
{
  return folder_copy->source_folder;
}

void etpan_folder_copy_set_dest_storage(struct etpan_folder_copy * folder_copy,
    struct etpan_storage * dest_storage)
{
  folder_copy->dest_storage = dest_storage;
}

void etpan_folder_copy_set_dest_folder(struct etpan_folder_copy * folder_copy,
    struct etpan_folder * dest_folder)
{
  if (dest_folder != folder_copy->dest_folder) {
    if (folder_copy->dest_folder != NULL)
      etpan_folder_unref(folder_copy->dest_folder);
    folder_copy->dest_folder = dest_folder;
    if (dest_folder != NULL) {
      etpan_folder_ref(dest_folder);
    }
    else
      folder_copy->dest_folder = NULL;
  }
}

struct etpan_folder *
etpan_folder_copy_get_dest_folder(struct etpan_folder_copy * folder_copy)
{
  return folder_copy->dest_folder;
}

struct etpan_storage *
etpan_folder_copy_get_dest_storage(struct etpan_folder_copy * folder_copy)
{
  return folder_copy->dest_storage;
}

void etpan_folder_copy_setup(struct etpan_folder_copy * folder_copy)
{
  (void) folder_copy;
}

static void etpan_folder_copy_unsetup(struct etpan_folder_copy * folder_copy)
{
  etpan_folder_copy_cancel(folder_copy);
}

void etpan_folder_copy_run(struct etpan_folder_copy * folder_copy)
{
  run(folder_copy);
}

void etpan_folder_copy_cancel(struct etpan_folder_copy * folder_copy)
{
  switch (folder_copy->state) {
  case STATE_IDLE:
    break;
  case STATE_CREATING:
    etpan_folder_create_cancel(folder_copy->folder_create);
    break;
  case STATE_COPYING_SUBFOLDER:
    etpan_folder_copy_cancel(folder_copy->subfolder_copy);
    break;
  case STATE_COPYING_MSGLIST_FETCH_MSGLIST:
    cancel_fetch_msg_list(folder_copy);
    break;
  case STATE_COPYING_MSGLIST:
    etpan_message_copy_cancel(folder_copy->msg_copy);
    break;
  case STATE_DELETING:
    etpan_folder_delete_cancel(folder_copy->folder_delete);
    break;
  }
}

static void run(struct etpan_folder_copy * folder_copy)
{
  char * source_name;
  
  folder_copy->error_list = carray_new(4);
  if (folder_copy->error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  source_name = etpan_folder_get_ui_name(folder_copy->source_folder);
  
  folder_copy->state = STATE_CREATING;
  ETPAN_LOG("create new folder");
  /* create folder */
  folder_copy->folder_create = etpan_folder_create_new();
  etpan_folder_create_set_folder_name(folder_copy->folder_create, source_name);
  etpan_folder_create_set_storage(folder_copy->folder_create,
      folder_copy->dest_storage);
  etpan_folder_create_set_parent_folder(folder_copy->folder_create,
      folder_copy->dest_folder);
  etpan_folder_create_setup(folder_copy->folder_create);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL,
      folder_copy->folder_create, folder_copy,
      created_callback);
  
  etpan_folder_create_run(folder_copy->folder_create);
}

static void created_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_copy * folder_copy;
  struct etpan_folder_create * folder_create;
  struct etpan_error * error;
  int r;
  struct etpan_folder * created_folder;
  carray * subfolder_list;
  struct etpan_storage * source_storage;
  chash * folder_hash;
  chashiter * iter;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  folder_copy = user_data;
  folder_create = folder_copy->folder_create;
  folder_copy->folder_create = NULL;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
    ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, folder_copy,
    created_callback);
  
  error = etpan_folder_create_get_error(folder_create);
  
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      etpan_folder_create_unref(folder_create);
    
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      if (folder_copy->delete) {
        etpan_error_set_short_description(error,
            _("Move of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Move of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      else {
        etpan_error_set_short_description(error,
            _("Copy of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Copy of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
    }
    else {
      error = etpan_error_dup(error);
    }
    
    r = carray_add(folder_copy->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    copy_finished(folder_copy);
    return;
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_copy->source_folder,
        folder_copy->dest_folder, folder_copy->delete, error);
    
    etpan_folder_create_unref(folder_create);
    
    r = carray_add(folder_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    copy_finished(folder_copy);
    return;
  }
  
  /* get created folder */
  created_folder = etpan_folder_create_get_created_folder(folder_create);
  ETPAN_LOG("created folder %p", created_folder);
  etpan_folder_ref(created_folder);
  etpan_folder_create_unref(folder_create);
  folder_copy->created_folder = created_folder;
  
  ETPAN_LOG("destination subfolder %s", etpan_folder_get_ui_path(folder_copy->created_folder));
  
  /* get sub folders */
  subfolder_list = carray_new(8);
  if (subfolder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  source_storage = etpan_folder_get_storage(folder_copy->source_folder);
  folder_hash = etpan_storage_get_folder_list(source_storage);
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    folder = value.data;
    if (etpan_folder_is_sub_folder(folder_copy->source_folder,
            folder)) {
      r = carray_add(subfolder_list, folder, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      
      etpan_folder_ref(folder);
    }
  }
  folder_copy->foldertab = subfolder_list;
  folder_copy->next_folder = 0;
  
  folder_copy->state = STATE_COPYING_SUBFOLDER;
  
  ETPAN_LOG("copy subfolders start");
  copy_next_subfolder(folder_copy);
}

static void copy_next_subfolder(struct etpan_folder_copy * folder_copy)
{
  struct etpan_folder * source_subfolder;
  
  if (folder_copy->next_folder >= carray_count(folder_copy->foldertab)) {
    copy_subfolder_finished(folder_copy);
    return;
  }
  
  /* copy next sub folder */
  folder_copy->subfolder_copy = etpan_folder_copy_new();
  etpan_folder_copy_set_delete(folder_copy->subfolder_copy,
      folder_copy->delete);
  source_subfolder = carray_get(folder_copy->foldertab,
      folder_copy->next_folder);
  etpan_folder_copy_set_source(folder_copy->subfolder_copy,
      source_subfolder);
  etpan_folder_copy_set_dest_storage(folder_copy->subfolder_copy,
    folder_copy->dest_storage);
  etpan_folder_copy_set_dest_folder(folder_copy->subfolder_copy,
    folder_copy->created_folder);
  etpan_folder_copy_setup(folder_copy->subfolder_copy);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_COPY_FINISHED_SIGNAL,
      folder_copy->subfolder_copy, folder_copy,
      copy_subfolder_finished_handler);
  
  ETPAN_LOG("copy subfolder %s", etpan_folder_get_ui_path(source_subfolder));
  etpan_folder_copy_run(folder_copy->subfolder_copy);
}

static void copy_subfolder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_copy * folder_copy;
  struct etpan_folder_copy * subfolder_copy;
  int r;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  
  folder_copy = user_data;
  subfolder_copy = sender;
  folder_copy->subfolder_copy = NULL;
  
  ETPAN_LOG("copy subfolder %s done",
      etpan_folder_get_ui_path(etpan_folder_copy_get_source(subfolder_copy)));
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_COPY_FINISHED_SIGNAL,
      subfolder_copy, folder_copy,
      copy_subfolder_finished_handler);
  
  error = etpan_folder_copy_get_error(subfolder_copy);
  
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      etpan_folder_copy_unref(subfolder_copy);
    
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      if (folder_copy->delete) {
        etpan_error_set_short_description(error,
            _("Move of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Move of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      else {
        etpan_error_set_short_description(error,
            _("Copy of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Copy of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
    
      r = carray_add(folder_copy->error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      copy_finished(folder_copy);
      return;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_copy->source_folder,
        folder_copy->dest_folder, folder_copy->delete, error);
    
    r = carray_add(folder_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (folder_copy->delete)
      copy_finished(folder_copy);
  }
  
  ETPAN_LOG("copy subfolder %s finished", etpan_folder_get_ui_path(etpan_folder_copy_get_source(subfolder_copy)));
  etpan_folder_copy_unref(subfolder_copy);
  
  folder_copy->next_folder ++;
  
  copy_next_subfolder(folder_copy);
}

static void copy_subfolder_finished(struct etpan_folder_copy * folder_copy)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(folder_copy->foldertab) ; i ++) {
    struct etpan_folder * folder;
    
    folder = carray_get(folder_copy->foldertab, i);
    etpan_folder_unref(folder);
  }
  carray_free(folder_copy->foldertab);
  folder_copy->foldertab = NULL;
  
  ETPAN_LOG("copy subfolders done");
  
  ETPAN_LOG("fetch message list");
  folder_copy->state = STATE_COPYING_MSGLIST_FETCH_MSGLIST;
  
  /* fetch message list */
  fetch_msg_list(folder_copy);
}

static void fetch_msg_list(struct etpan_folder_copy * folder_copy)
{
  folder_copy->opened = 1;
  etpan_mail_manager_folder_open(etpan_mail_manager_get_default(),
    folder_copy->source_folder);
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL,
      msg_list_fetched_handler, folder_copy);
  etpan_mail_manager_folder_update(etpan_mail_manager_get_default(),
      folder_copy->source_folder);
}

static void cancel_fetch_msg_list(struct etpan_folder_copy * folder_copy)
{
  int r;
  struct etpan_error * error;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL,
      msg_list_fetched_handler, folder_copy);
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_CANCELLED);
  if (folder_copy->delete) {
    etpan_error_set_short_description(error,
        _("Move of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Move of the folder %s to %s was interrupted."),
        etpan_folder_get_ui_path(folder_copy->source_folder),
        etpan_folder_get_ui_path(folder_copy->dest_folder));
  }
  else {
    etpan_error_set_short_description(error,
        _("Copy of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Copy of the folder %s to %s was interrupted."),
        etpan_folder_get_ui_path(folder_copy->source_folder),
        etpan_folder_get_ui_path(folder_copy->dest_folder));
  }
      
  r = carray_add(folder_copy->error_list, error, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
    
  copy_finished(folder_copy);
}

static void msg_list_fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_copy * folder_copy;
  struct etpan_error * error;
  int r;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  folder_copy = user_data;
  
  if (folder_copy->source_folder != signal_data)
    return;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL,
      msg_list_fetched_handler, folder_copy);
  error = etpan_mail_manager_folder_get_error(etpan_mail_manager_get_default(),
      folder_copy->source_folder);
  
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      if (folder_copy->delete) {
        etpan_error_set_short_description(error,
            _("Move of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Move of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      else {
        etpan_error_set_short_description(error,
            _("Copy of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Copy of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      
      r = carray_add(folder_copy->error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      copy_finished(folder_copy);
      return;
    }
  }

  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_copy->source_folder,
        folder_copy->dest_folder, folder_copy->delete, error);
    
    r = carray_add(folder_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    copy_finished(folder_copy);
    return;
  }
  
  copy_msg_list(folder_copy);
}

static void copy_msg_list(struct etpan_folder_copy * folder_copy)
{
  chash * msg_hash;
  
  /* copy messages */
  
  ETPAN_LOG("copy messages");
  
  folder_copy->state = STATE_COPYING_MSGLIST;
  
  folder_copy->copy_msg_id = etpan_mail_manager_get_next_copy_messages_id(etpan_mail_manager_get_default());
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_COPY_MSG_FINISHED_SIGNAL,
      copy_msglist_finished_handler, folder_copy);
  
  msg_hash = etpan_folder_get_message_hash(folder_copy->source_folder);
  if (folder_copy->delete)
    etpan_mail_manager_move_messages(etpan_mail_manager_get_default(),
        folder_copy->created_folder, msg_hash);
  else
    etpan_mail_manager_copy_messages(etpan_mail_manager_get_default(),
        folder_copy->created_folder, msg_hash);
}

static void copy_msglist_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_copy * folder_copy;
  int r;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  unsigned int * copy_msg_id;
  
  folder_copy = user_data;
  copy_msg_id = signal_data;
  
  if (* copy_msg_id != folder_copy->copy_msg_id) {
    return;
  }
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_COPY_MSG_FINISHED_SIGNAL,
      copy_msglist_finished_handler, folder_copy);
  
  ETPAN_LOG("copy messages done");
  
  error = etpan_mail_manager_copy_messages_get_error(etpan_mail_manager_get_default());
  
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      if (folder_copy->delete) {
        etpan_error_set_short_description(error,
            _("Move of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Move of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      else {
        etpan_error_set_short_description(error,
            _("Copy of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Copy of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      
      r = carray_add(folder_copy->error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      
      copy_finished(folder_copy);
      return;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_copy->source_folder,
        folder_copy->dest_folder, folder_copy->delete, error);
    
    r = carray_add(folder_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    copy_finished(folder_copy);
    return;
  }
  
  if (!folder_copy->delete) {
    copy_finished(folder_copy);
    return;
  }
  
  if (carray_count(folder_copy->error_list) != 0) {
    copy_finished(folder_copy);
    return;
  }
  
  delete_folder(folder_copy);
}  

static void delete_folder(struct etpan_folder_copy * folder_copy)
{
  /* delete folder */
  folder_copy->state = STATE_DELETING;
  
  folder_copy->folder_delete = etpan_folder_delete_new();
  etpan_folder_delete_set_folder(folder_copy->folder_delete,
      folder_copy->source_folder);
  etpan_folder_delete_setup(folder_copy->folder_delete);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_DELETE_FINISHED_SIGNAL,
      folder_copy->folder_delete, folder_copy,
      deleted_callback);
  
  etpan_folder_delete_run(folder_copy->folder_delete);
  ETPAN_LOG("delete folder");
}

static void deleted_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_delete * folder_delete;
  int r;
  struct etpan_folder_copy * folder_copy;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  folder_copy = user_data;
  folder_delete = folder_copy->folder_delete;
  folder_copy->folder_delete = NULL;
  
  error = etpan_folder_delete_get_error(folder_delete);
  
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      etpan_folder_delete_unref(folder_delete);
    
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      if (folder_copy->delete) {
        etpan_error_set_short_description(error,
            _("Move of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Move of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
      else {
        etpan_error_set_short_description(error,
            _("Copy of folder cancelled"));
        etpan_error_strf_long_description(error,
            _("Copy of the folder %s to %s was interrupted."),
            etpan_folder_get_ui_path(folder_copy->source_folder),
            etpan_folder_get_ui_path(folder_copy->dest_folder));
      }
    
      r = carray_add(folder_copy->error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      copy_finished(folder_copy);
      return;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_copy->source_folder,
        folder_copy->dest_folder, folder_copy->delete, error);
    
    r = carray_add(folder_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_folder_delete_unref(folder_delete);
    copy_finished(folder_copy);
    return;
  }
  
  copy_finished(folder_copy);
}

static void copy_finished(struct etpan_folder_copy * folder_copy)
{
  struct etpan_error * error;
  unsigned int i;
  
  folder_copy->state = STATE_IDLE;
  
  if (folder_copy->opened)
    etpan_mail_manager_folder_close(etpan_mail_manager_get_default(),
        folder_copy->source_folder);
  
  if (carray_count(folder_copy->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(folder_copy->error_list) == 1) {
    error = carray_get(folder_copy->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(folder_copy->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(folder_copy->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(folder_copy->error_list);
  folder_copy->error_list = NULL;
  folder_copy->error = error;
  
  /* notify */
  ETPAN_LOG("folder copy finished");
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_COPY_FINISHED_SIGNAL, folder_copy, NULL);
}

struct etpan_error *
etpan_folder_copy_get_error(struct etpan_folder_copy * folder_copy)
{
  return folder_copy->error;
}

struct etpan_folder *
etpan_folder_copy_get_created_folder(struct etpan_folder_copy * folder_copy)
{
  return folder_copy->created_folder;
}

static struct etpan_error * rewrite_error(struct etpan_folder * src_folder,
    struct etpan_folder * dst_folder, int delete,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  if (delete) {
    etpan_error_strf_long_description(new_error,
        _("An error occurred during move of the folder %s to %s.\n"
            "%s"),
        etpan_folder_get_ui_path(src_folder),
        etpan_folder_get_ui_path(dst_folder),
        previous_long_description);
  }
  else {
    etpan_error_strf_long_description(new_error,
        _("An error occurred during copy of the folder %s to %s.\n"
            "%s"),
        etpan_folder_get_ui_path(src_folder),
        etpan_folder_get_ui_path(dst_folder),
        previous_long_description);
  }
  
  return new_error;
}
