#include "etpan-folder-create.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-nls.h"
#include "etpan-mail-manager.h"

enum {
  STATE_IDLE,
  STATE_CREATING,
  STATE_FETCHING_FOLDERLIST,
};

static struct etpan_error *
rewrite_error(char * location, struct etpan_error * error);

static void etpan_folder_create_free(struct etpan_folder_create * folder_create);
static void etpan_folder_create_unsetup(struct etpan_folder_create * folder_create);
static void notify_finished(struct etpan_folder_create * folder_create);

static void created_callback(int cancelled,
    struct etpan_storage_create_folder_result * result,
    void * cb_data);
static void cancel_fetch_folder_list(struct etpan_folder_create * folder_create);
static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
#if 0
static void folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data);
#endif
static void run(struct etpan_folder_create * folder_create);


struct etpan_folder_create * etpan_folder_create_new(void)
{
  struct etpan_folder_create * folder_create;
  
  folder_create = malloc(sizeof(* folder_create));
  if (folder_create == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_create->ref_count = 1;
  folder_create->name = NULL;
  folder_create->parent_folder = NULL;
  folder_create->storage = NULL;
  folder_create->location = NULL;
  folder_create->error = NULL;
  folder_create->op = NULL;
  folder_create->state = STATE_IDLE;
  folder_create->created_folder = NULL;
  
  return folder_create;
}

static void etpan_folder_create_free(struct etpan_folder_create * folder_create)
{
  etpan_folder_create_unsetup(folder_create);
  
  ETPAN_ERROR_FREE(folder_create->error);
  folder_create->error = NULL;
  
  if (folder_create->created_folder != NULL)
    etpan_folder_unref(folder_create->created_folder);
  free(folder_create->location);
  if (folder_create->parent_folder != NULL)
    etpan_folder_unref(folder_create->parent_folder);
  free(folder_create->name);
  free(folder_create);
}

void etpan_folder_create_ref(struct etpan_folder_create * folder_create)
{
  folder_create->ref_count ++;
}

void etpan_folder_create_unref(struct etpan_folder_create * folder_create)
{
  folder_create->ref_count --;
  if (folder_create->ref_count == 0)
    etpan_folder_create_free(folder_create);
}

void etpan_folder_create_set_folder_name(struct etpan_folder_create * folder_create, char * name)
{
  if (name != folder_create->name) {
    free(folder_create->name);
    if (name != NULL) {
      folder_create->name = strdup(name);
      if (folder_create->name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder_create->name = NULL;
  }
}

char * etpan_folder_create_get_folder_name(struct etpan_folder_create * folder_create)
{
  return folder_create->name;
}

void etpan_folder_create_set_parent_folder(struct etpan_folder_create * folder_create,
    struct etpan_folder * parent_folder)
{
  if (parent_folder != folder_create->parent_folder) {
    if (folder_create->parent_folder != NULL)
      etpan_folder_unref(folder_create->parent_folder);
    folder_create->parent_folder = parent_folder;
    if (parent_folder != NULL) {
      etpan_folder_ref(parent_folder);
    }
    else
      folder_create->parent_folder = NULL;
  }
}

struct etpan_folder *
etpan_folder_create_get_parent_folder(struct etpan_folder_create * folder_create)
{
  return folder_create->parent_folder;
}

void etpan_folder_create_set_storage(struct etpan_folder_create * folder_create,
    struct etpan_storage * storage)
{
  folder_create->storage = storage;
}

struct etpan_storage *
etpan_folder_create_get_storage(struct etpan_folder_create * folder_create)
{
  return folder_create->storage;
}


void etpan_folder_create_setup(struct etpan_folder_create * folder_create)
{
  (void) folder_create;
}

static void etpan_folder_create_unsetup(struct etpan_folder_create * folder_create)
{
  etpan_folder_create_cancel(folder_create);
}

void etpan_folder_create_run(struct etpan_folder_create * folder_create)
{
  run(folder_create);
}

void etpan_folder_create_cancel(struct etpan_folder_create * folder_create)
{
  switch (folder_create->state) {
  case STATE_IDLE:
    break;
  case STATE_CREATING:
    etpan_thread_op_cancel(folder_create->op);
    break;
  case STATE_FETCHING_FOLDERLIST:
#if 0
    etpan_thread_op_cancel(folder_create->op);
#endif
    cancel_fetch_folder_list(folder_create);
    break;
  }
}

struct etpan_error *
etpan_folder_create_get_error(struct etpan_folder_create * folder_create)
{
  return folder_create->error;
}

struct etpan_folder *
etpan_folder_create_get_created_folder(struct etpan_folder_create * folder_create)
{
  ETPAN_LOG("created %p", folder_create);
  return folder_create->created_folder;
}

static void run(struct etpan_folder_create * folder_create)
{
  char * location;
  
  location = etpan_storage_get_sub_folder_location(folder_create->storage,
      folder_create->parent_folder, folder_create->name);
  folder_create->location = location;
  
  folder_create->state = STATE_CREATING;
  
  ETPAN_LOG("create new folder");
  /* create folder */
  folder_create->op =
    etpan_storage_create_folder(etpan_thread_manager_app_get_default(),
        folder_create->storage, location,
        created_callback,
        folder_create);
}

static void created_callback(int cancelled,
    struct etpan_storage_create_folder_result * result,
    void * cb_data)
{
  struct etpan_folder_create * folder_create;
  
  folder_create = cb_data;
  folder_create->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Creation of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Creation of the folder %s was interrupted."),
        folder_create->location);
    
    folder_create->error = error;
    notify_finished(folder_create);
    return;
  }
  
  if (result->error != NULL) {
    folder_create->error = result->error;
    result->error = NULL;
    
    notify_finished(folder_create);
    return;
  }
  
  folder_create->state = STATE_FETCHING_FOLDERLIST;
  /* fetch folder list */
#if 0
  folder_create->op =
    etpan_storage_fetch_folder_list(etpan_thread_manager_app_get_default(),
        folder_create->storage, folder_list_fetched_callback,
        folder_create);
#endif
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_create);
  etpan_mail_manager_storage_update(etpan_mail_manager_get_default(),
      folder_create->storage);
}

static void cancel_fetch_folder_list(struct etpan_folder_create * folder_create)
{
  int r;
  struct etpan_error * error;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_create);
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_CANCELLED);
  etpan_error_set_short_description(error,
      _("Creation of folder cancelled"));
  etpan_error_strf_long_description(error,
      _("Creation of the folder %s was interrupted."),
      folder_create->location);
  folder_create->error = error;
  notify_finished(folder_create);
}

static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_create * folder_create;
  struct etpan_error * error;
  
  folder_create = user_data;
  if (signal_data != folder_create->storage)
    return;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_create);
  
  error = etpan_mail_manager_storage_get_error(etpan_mail_manager_get_default(),folder_create->storage);
  if (error != NULL) {
    folder_create->error = rewrite_error(folder_create->location,
        error);
    notify_finished(folder_create);
    return;
  }
  
  folder_create->state = STATE_IDLE;
  
  /* get new folder */
  folder_create->created_folder =
    etpan_storage_get_folder(folder_create->storage,
        folder_create->location);
  if (folder_create->created_folder == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error,
        _("Creation of folder failed"));
    etpan_error_strf_long_description(error,
        _("Folder %s could not be found."),
        folder_create->location);
    
    folder_create->error = error;
    notify_finished(folder_create);
    return;
  }
  
  etpan_folder_ref(folder_create->created_folder);
  
  ETPAN_LOG("created %p", folder_create);
  notify_finished(folder_create);
}

#if 0
static void folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data)
{
  struct etpan_folder_create * folder_create;
  struct etpan_error * error;
  
  folder_create = cb_data;
  folder_create->op = NULL;
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Creation of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Creation of the folder %s was interrupted."),
        folder_create->location);
    
    folder_create->error = error;
    notify_finished(folder_create);
    return;
  }
  
  if (result->error != NULL) {
    folder_create->error = rewrite_error(folder_create->location,
        result->error);
    notify_finished(folder_create);
    return;
  }
  
  error = etpan_storage_update_folder_list(folder_create->storage,
      result->folder_list);
  ETPAN_ERROR_IGNORE(error);
  etpan_storage_remove_lost_folders(folder_create->storage);
  
  folder_create->state = STATE_IDLE;
  
  /* get new folder */
  folder_create->created_folder =
    etpan_storage_get_folder(folder_create->storage,
        folder_create->location);
  if (folder_create->created_folder == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error,
        _("Creation of folder failed"));
    etpan_error_strf_long_description(error,
        _("Folder %s could not be found."),
        folder_create->location);
    
    folder_create->error = error;
    notify_finished(folder_create);
    return;
  }
  
  etpan_folder_ref(folder_create->created_folder);
  
  ETPAN_LOG("created %p", folder_create);
  notify_finished(folder_create);
}
#endif

static void notify_finished(struct etpan_folder_create * folder_create)
{
  ETPAN_LOG("folder create finished");
  folder_create->state = STATE_IDLE;
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, NULL);
}

static struct etpan_error *
rewrite_error(char * location, struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred during creation of the folder %s.\n"
          "%s"),
      location,
      previous_long_description);
  
  return new_error;
}
