#include "etpan-folder-delete.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-nls.h"
#include "etpan-mail-manager.h"

enum {
  STATE_IDLE,
  STATE_FETCHING_FOLDERLIST,
  STATE_DELETING_SUBFOLDERS,
  STATE_DELETING,
  STATE_FETCHING_NEW_FOLDERLIST,
};

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    struct etpan_error * error);

static void etpan_folder_delete_unsetup(struct etpan_folder_delete * folder_delete);
static void run(struct etpan_folder_delete * folder_delete);
static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void cancel_fetch_folder_list(struct etpan_folder_delete * folder_delete);
#if 0
static void folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data);
#endif
static void delete_next_subfolder(struct etpan_folder_delete * folder_delete);
static void delete_subfolder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void delete_subfolder_finished(struct etpan_folder_delete * folder_delete);
static void delete_folder_done_callback(int cancelled,
    struct etpan_storage_delete_folder_result * result,
    void * cb_data);
#if 0
static void final_folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data);
#endif
static void final_folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void cancel_final_fetch_folder_list(struct etpan_folder_delete * folder_delete);
static void notify_finished(struct etpan_folder_delete * folder_delete);

struct etpan_folder_delete * etpan_folder_delete_new(void)
{
  struct etpan_folder_delete * folder_delete;
  
  folder_delete = malloc(sizeof(* folder_delete));
  if (folder_delete == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_delete->ref_count = 1;
  folder_delete->folder = NULL;
  folder_delete->op = NULL;
  folder_delete->subfolder_delete = NULL;
  folder_delete->foldertab = NULL;
  folder_delete->next_folder = 0;
  folder_delete->state = STATE_IDLE;
  folder_delete->error = NULL;
  folder_delete->error_list = NULL;
  
  return folder_delete;
}

static void etpan_folder_delete_free(struct etpan_folder_delete * folder_delete)
{
  etpan_folder_delete_unsetup(folder_delete);
  
  ETPAN_ERROR_FREE(folder_delete->error);
  folder_delete->error = NULL;
  
  if (folder_delete->folder != NULL)
    etpan_folder_unref(folder_delete->folder);
  free(folder_delete);
}


void etpan_folder_delete_ref(struct etpan_folder_delete * folder_delete)
{
  folder_delete->ref_count ++;
}

void etpan_folder_delete_unref(struct etpan_folder_delete * folder_delete)
{
  folder_delete->ref_count --;
  if (folder_delete->ref_count == 0)
    etpan_folder_delete_free(folder_delete);
}

void etpan_folder_delete_set_folder(struct etpan_folder_delete * folder_delete,
    struct etpan_folder * folder)
{
  if (folder != folder_delete->folder) {
    if (folder_delete->folder != NULL)
      etpan_folder_unref(folder_delete->folder);
    folder_delete->folder = folder;
    if (folder != NULL) {
      etpan_folder_ref(folder);
    }
    else
      folder_delete->folder = NULL;
  }
}

struct etpan_folder *
etpan_folder_delete_get_folder(struct etpan_folder_delete * folder_delete)
{
  return folder_delete->folder;
}

void etpan_folder_delete_setup(struct etpan_folder_delete * folder_delete)
{
  (void) folder_delete;
}

static void etpan_folder_delete_unsetup(struct etpan_folder_delete * folder_delete)
{
  etpan_folder_delete_cancel(folder_delete);
}

void etpan_folder_delete_run(struct etpan_folder_delete * folder_delete)
{
  run(folder_delete);
}

void etpan_folder_delete_cancel(struct etpan_folder_delete * folder_delete)
{
  switch (folder_delete->state) {
  case STATE_IDLE:
    break;
  case STATE_FETCHING_FOLDERLIST:
    cancel_fetch_folder_list(folder_delete);
    break;
  case STATE_DELETING_SUBFOLDERS:
    etpan_folder_delete_cancel(folder_delete->subfolder_delete);
    break;
  case STATE_DELETING:
    etpan_thread_op_cancel(folder_delete->op);
    break;
  case STATE_FETCHING_NEW_FOLDERLIST:
    cancel_final_fetch_folder_list(folder_delete);
    break;
  }
}

struct etpan_error *
etpan_folder_delete_get_error(struct etpan_folder_delete * folder_delete)
{
  return folder_delete->error;
}


static void run(struct etpan_folder_delete * folder_delete)
{
  struct etpan_storage * storage;
  
  folder_delete->state = STATE_FETCHING_FOLDERLIST;
  folder_delete->error_list = carray_new(4);
  
  /* recursive delete */
  ETPAN_LOG("fetch folder list");
  /* fetch folder list */
  
  storage = etpan_folder_get_storage(folder_delete->folder);
#if 0
  folder_delete->op =
    etpan_storage_fetch_folder_list(etpan_thread_manager_app_get_default(),
        storage, folder_list_fetched_callback,
        folder_delete);
#endif
  
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_delete);
  etpan_mail_manager_storage_update(etpan_mail_manager_get_default(),
      storage);
}

static void cancel_fetch_folder_list(struct etpan_folder_delete * folder_delete)
{
  int r;
  struct etpan_error * error;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_delete);
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_CANCELLED);
  etpan_error_set_short_description(error,
      _("Deletion of folder cancelled"));
  etpan_error_strf_long_description(error,
      _("Deletion of the folder %s was interrupted."),
      etpan_folder_get_ui_path(folder_delete->folder));
  r = carray_add(folder_delete->error_list, error, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  notify_finished(folder_delete);
}

static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_delete * folder_delete;
  struct etpan_storage * storage;
  chash * folder_hash;
  chashiter * iter;
  carray * subfolder_list;
  struct etpan_error * error;
  int r;
  
  folder_delete = user_data;
  storage = etpan_folder_get_storage(folder_delete->folder);
  if (signal_data != storage)
    return;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_delete);

  error = etpan_mail_manager_storage_get_error(etpan_mail_manager_get_default(),
      storage);
  if (error != NULL) {
    error = rewrite_error(folder_delete->folder, error);
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  /* get sub folders */
  subfolder_list = carray_new(8);
  if (subfolder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_hash = etpan_storage_get_folder_list(storage);
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    folder = value.data;
    if (etpan_folder_is_sub_folder(folder_delete->folder,
            folder)) {
      r = carray_add(subfolder_list, folder, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      etpan_folder_ref(folder);
    }
  }
  folder_delete->foldertab = subfolder_list;
  folder_delete->next_folder = 0;
  
  folder_delete->state = STATE_DELETING_SUBFOLDERS;
  
  ETPAN_LOG("delete subfolders start");
  delete_next_subfolder(folder_delete);
}

#if 0
static void folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data)
{
  chash * folder_hash;
  chashiter * iter;
  struct etpan_storage * storage;
  carray * subfolder_list;
  struct etpan_folder_delete * folder_delete;
  int r;
  struct etpan_error * error;

  folder_delete = cb_data;
  folder_delete->op = NULL;
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Deletion of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Deletion of the folder %s was interrupted."),
        etpan_folder_get_ui_path(folder_delete->folder));
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  if (result->error != NULL) {
    error = rewrite_error(folder_delete->folder, result->error);
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  storage = etpan_folder_get_storage(folder_delete->folder);
  
  error = etpan_storage_update_folder_list(storage, result->folder_list);
  ETPAN_ERROR_IGNORE(error);
  etpan_storage_remove_lost_folders(storage);
  
  /* get sub folders */
  subfolder_list = carray_new(8);
  if (subfolder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_hash = etpan_storage_get_folder_list(storage);
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    folder = value.data;
    if (etpan_folder_is_sub_folder(folder_delete->folder,
            folder)) {
      r = carray_add(subfolder_list, folder, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      etpan_folder_ref(folder);
    }
  }
  folder_delete->foldertab = subfolder_list;
  folder_delete->next_folder = 0;
  
  folder_delete->state = STATE_DELETING_SUBFOLDERS;
  
  ETPAN_LOG("delete subfolders start");
  delete_next_subfolder(folder_delete);
}
#endif

static void delete_next_subfolder(struct etpan_folder_delete * folder_delete)
{
  struct etpan_folder * subfolder;
  
  if (folder_delete->next_folder >= carray_count(folder_delete->foldertab)) {
    delete_subfolder_finished(folder_delete);
    return;
  }
  
  /* delete next sub folder */
  folder_delete->subfolder_delete = etpan_folder_delete_new();
  subfolder = carray_get(folder_delete->foldertab,
      folder_delete->next_folder);
  etpan_folder_delete_set_folder(folder_delete->subfolder_delete,
      subfolder);
  etpan_folder_delete_setup(folder_delete->subfolder_delete);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_DELETE_FINISHED_SIGNAL,
      folder_delete->subfolder_delete, folder_delete,
      delete_subfolder_finished_handler);
  
  ETPAN_LOG("delete subfolder %s", etpan_folder_get_ui_path(subfolder));
  etpan_folder_delete_run(folder_delete->subfolder_delete);
}

static void delete_subfolder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_delete * folder_delete;
  struct etpan_folder_delete * subfolder_delete;
  int r;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  
  folder_delete = user_data;
  subfolder_delete = sender;
  folder_delete->subfolder_delete = NULL;
  
  ETPAN_LOG("delete subfolder %s done",
      etpan_folder_get_ui_path(etpan_folder_delete_get_folder(subfolder_delete)));
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_DELETE_FINISHED_SIGNAL,
      subfolder_delete, folder_delete,
      delete_subfolder_finished_handler);
  
  error = etpan_folder_delete_get_error(subfolder_delete);
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      etpan_folder_delete_unref(subfolder_delete);
    
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      etpan_error_set_short_description(error,
          _("Deletion of folder cancelled"));
      etpan_error_strf_long_description(error,
          _("Deletion of the folder %s was interrupted."),
          etpan_folder_get_ui_path(folder_delete->folder));
      
      r = carray_add(folder_delete->error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      notify_finished(folder_delete);
      return;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_delete->folder, error);
    r = carray_add(folder_delete->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  ETPAN_LOG("delete subfolder %s finished", etpan_folder_get_ui_path(etpan_folder_delete_get_folder(subfolder_delete)));
  etpan_folder_delete_unref(subfolder_delete);
  
  folder_delete->next_folder ++;
  
  delete_next_subfolder(folder_delete);
}

static void delete_subfolder_finished(struct etpan_folder_delete * folder_delete)
{
  unsigned int i;
  struct etpan_storage * storage;
  
  for(i = 0 ; i < carray_count(folder_delete->foldertab) ; i ++) {
    struct etpan_folder * folder;
    
    folder = carray_get(folder_delete->foldertab, i);
    etpan_folder_unref(folder);
  }
  carray_free(folder_delete->foldertab);
  folder_delete->foldertab = NULL;
  
  ETPAN_LOG("delete subfolders done");
  
  ETPAN_LOG("delete folder %s",
      etpan_folder_get_location(folder_delete->folder));
  folder_delete->state = STATE_DELETING;
  
  storage = etpan_folder_get_storage(folder_delete->folder);
  
  folder_delete->op =
    etpan_storage_delete_folder(etpan_thread_manager_app_get_default(),
        storage, etpan_folder_get_location(folder_delete->folder),
        delete_folder_done_callback, folder_delete);
}

static void delete_folder_done_callback(int cancelled,
    struct etpan_storage_delete_folder_result * result,
    void * cb_data)
{
  struct etpan_folder_delete * folder_delete;
  struct etpan_storage * storage;
  int r;
  
  folder_delete = cb_data;
  folder_delete->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Deletion of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Deletion of the folder %s was interrupted."),
        etpan_folder_get_ui_path(folder_delete->folder));
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  ETPAN_LOG("delete folder done");
  
  if (result->error != NULL) {
    r = carray_add(folder_delete->error_list, result->error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    result->error = NULL;
    
    notify_finished(folder_delete);
    return;
  }
  
  folder_delete->state = STATE_FETCHING_NEW_FOLDERLIST;
  
  /* fetch folder list */
  storage = etpan_folder_get_storage(folder_delete->folder);

#if 0
  folder_delete->op =
    etpan_storage_fetch_folder_list(etpan_thread_manager_app_get_default(),
        storage, final_folder_list_fetched_callback,
        folder_delete);
#endif
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      final_folder_list_updated_handler, folder_delete);
  etpan_mail_manager_storage_update(etpan_mail_manager_get_default(),
      storage);
  
  ETPAN_LOG("fetch folder list");
}

static void cancel_final_fetch_folder_list(struct etpan_folder_delete * folder_delete)
{
  struct etpan_error * error;  
  int r;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      final_folder_list_updated_handler, folder_delete);
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_CANCELLED);
  etpan_error_set_short_description(error,
      _("Deletion of folder cancelled"));
  etpan_error_strf_long_description(error,
      _("Deletion of the folder %s was interrupted."),
      etpan_folder_get_ui_path(folder_delete->folder));
  r = carray_add(folder_delete->error_list, error, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
    
  notify_finished(folder_delete);
}

static void final_folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_delete * folder_delete;
  struct etpan_storage * storage;
  int r;
  struct etpan_error * error;
  
  folder_delete = user_data;
  storage = etpan_folder_get_storage(folder_delete->folder);
  if (signal_data != storage)
    return;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      final_folder_list_updated_handler, folder_delete);
  
  ETPAN_LOG("fetch folder list done");
  
  error = etpan_mail_manager_storage_get_error(etpan_mail_manager_get_default(),
      storage);
  if (error != NULL) {
    error = rewrite_error(folder_delete->folder, error);
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  notify_finished(folder_delete);
}

#if 0
static void final_folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data)
{
  struct etpan_folder_delete * folder_delete;
  struct etpan_storage * storage;
  int r;
  struct etpan_error * error;
  
  folder_delete = cb_data;
  folder_delete->op = NULL;
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Deletion of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Deletion of the folder %s was interrupted."),
        etpan_folder_get_ui_path(folder_delete->folder));
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
  }
  
  ETPAN_LOG("fetch folder list done");
  
  if (result->error != NULL) {
    error = rewrite_error(folder_delete->folder, result->error);
    r = carray_add(folder_delete->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    notify_finished(folder_delete);
    return;
  }
  
  storage = etpan_folder_get_storage(folder_delete->folder);
  error = etpan_storage_update_folder_list(storage,
      result->folder_list);
  ETPAN_ERROR_IGNORE(error);
  etpan_storage_remove_lost_folders(storage);
  
  notify_finished(folder_delete);
}
#endif

static void notify_finished(struct etpan_folder_delete * folder_delete)
{
  struct etpan_error * error;
  unsigned int i;
  
  if (carray_count(folder_delete->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(folder_delete->error_list) == 1) {
    error = carray_get(folder_delete->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(folder_delete->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(folder_delete->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(folder_delete->error_list);
  folder_delete->error_list = NULL;
  folder_delete->error = error;
  
  ETPAN_LOG("folder delete finished");
  folder_delete->state = STATE_IDLE;
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_DELETE_FINISHED_SIGNAL, folder_delete, NULL);
}

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred during deletion of the folder %s.\n"
          "%s"),
      etpan_folder_get_ui_path(folder),
      previous_long_description);
  
  return new_error;
}
