#include "etpan-folder-indexer.h"

#include <string.h>
#include <stdlib.h>
#include <libetpan/libetpan.h>

#include "etpan-folder.h"
#include "etpan-indexer.h"
#include "etpan-error.h"
#include "etpan-signal.h"
#include "etpan-log.h"
#include "etpan-message.h"
#include "etpan-message-header.h"
#include "etpan-address.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-message-fetcher.h"
#include "etpan-part.h"
#include "etpan-utils.h"
#include "etpan-storage.h"
#include "etpan-account-manager.h"
#include "etpan-nls.h"

enum {
  INDEXER_IDLE,
  INDEXER_GETTING_INDEXER_STATE,
  INDEXER_INDEXING,
  INDEXER_INDEXING_BODY,
  INDEXER_CANCELLING_INDEXING_BODY,
};

struct get_indexer_state_result {
  carray * indexed_messages;
  carray * body_indexed_messages;
};

static struct etpan_error * rewrite_error(struct etpan_folder_indexer * fi,
    struct etpan_error * error);

static void open_index(struct etpan_folder_indexer * indexer);
static void close_index(struct etpan_folder_indexer * indexer);
static void update(struct etpan_folder_indexer * fi);
static void cancel(struct etpan_folder_indexer * fi);
static void set_state(struct etpan_folder_indexer * fi, int state);
static void folder_content_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void start_indexing_body(struct etpan_folder_indexer * fi);
static void cancel_indexing(struct etpan_folder_indexer * fi);
static void
start_indexing(struct etpan_folder_indexer * fi,
    void (* callback)(int, void *, void *),
    void * cb_data);
static void indexing_callback(int cancelled,
    void * dummy, void * cb_data);
static void index_next_body(struct etpan_folder_indexer * fi);
static void index_next_body_fetch_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void body_indexing(struct etpan_folder_indexer * fi,
    void (* callback)(int, void *, void *),
    void * cb_data);
static void index_next_body_indexed_callback(int cancelled, void * dummy,
    void * cb_data);
static void cancel_indexing_body(struct etpan_folder_indexer * fi);
static void indexing_body_finished(struct etpan_folder_indexer * fi);
static void get_indexer_state(struct etpan_folder_indexer * fi);
static void start_get_indexer_state(struct etpan_folder_indexer * fi,
    void (* callback)(int, struct get_indexer_state_result *, void *),
    void * cb_data);
static void get_indexer_state_callback(int cancelled,
    struct get_indexer_state_result * result, void * cb_data);
static void cancel_get_indexer_state(struct etpan_folder_indexer * fi);

struct etpan_folder_indexer * etpan_folder_indexer_new(void)
{
  struct etpan_folder_indexer * fi;
  
  fi = malloc(sizeof(* fi));
  if (fi == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fi->path = NULL;
  fi->folder = NULL;
  
  fi->dirty = 0;
  fi->state = INDEXER_IDLE;
  fi->indexer_state_initialized = 0;
  
  fi->current_messages_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (fi->current_messages_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fi->new_messages_hash = NULL;
  fi->deleted_messages_list = NULL;
  fi->indexing_op = NULL;
  fi->indexer = NULL;
  
  fi->body_current_messages_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (fi->body_current_messages_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fi->body_new_messages_hash = NULL;
  fi->body_new_messages_list = NULL;
  fi->body_deleted_messages_list = NULL;
  fi->body_next_msg = 0;
  fi->fetcher = NULL;
  fi->body_indexing_op = NULL;
  fi->body_indexer = NULL;
  
  fi->search_op = NULL;

  fi->error_list = NULL;
  fi->error = NULL;
  
  return fi;
}

void etpan_folder_indexer_free(struct etpan_folder_indexer * fi)
{
  chash_free(fi->current_messages_hash);
  chash_free(fi->body_current_messages_hash);
  free(fi->path);
  free(fi);
}

void etpan_folder_indexer_setup(struct etpan_folder_indexer * fi)
{
  open_index(fi);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_MSGLIST_UPDATED,
      fi->folder, fi,
      folder_content_changed_handler);
  if (etpan_folder_has_message_list(fi->folder)) {
    update(fi);
  }
}

void etpan_folder_indexer_unsetup(struct etpan_folder_indexer * fi)
{
  cancel(fi);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_MSGLIST_UPDATED, fi->folder, fi,
      folder_content_changed_handler);
  
  close_index(fi);
  etpan_folder_indexer_set_folder(fi, NULL);
}

void etpan_folder_indexer_set_path(struct etpan_folder_indexer * fi,
    char * path)
{
  if (path != fi->path) {
    free(fi->path);
    if (path != NULL) {
      fi->path = strdup(path);
      if (fi->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      fi->path = NULL;
  }
}

char * etpan_folder_indexer_get_path(struct etpan_folder_indexer * fi)
{
  return fi->path;
}

static void folder_content_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_indexer * fi;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  fi = user_data;
  if (!etpan_folder_has_message_list(fi->folder))
    return;
  
  ETPAN_FOLDER_INDEXER_LOG("folder content changed");
  update(fi);
}

void etpan_folder_indexer_set_folder(struct etpan_folder_indexer * fi,
    struct etpan_folder * folder)
{
  if (folder != fi->folder) {
    if (fi->folder != NULL) {
      etpan_folder_unref_msg_list(fi->folder);
      etpan_folder_unref(fi->folder);
    }
    fi->folder = folder;
    if (fi->folder != NULL) {
      etpan_folder_ref(fi->folder);
      etpan_folder_ref_msg_list(fi->folder);
    }
  }
}

struct etpan_folder * etpan_folder_indexer_get_folder(struct etpan_folder_indexer * fi)
{
  return fi->folder;
}

static void open_index(struct etpan_folder_indexer * fi)
{
  char path[PATH_MAX];
  
  if (fi->path != NULL) {
    ETPAN_FOLDER_INDEXER_LOG("index: %s", fi->path);
    etpan_make_dir(fi->path);
    snprintf(path, sizeof(path), "%s/headers.index", fi->path);
    fi->indexer = etpan_indexer_new(path);
  }
  else {
    ETPAN_FOLDER_INDEXER_LOG("index: %s", fi->path);
    fi->indexer = etpan_indexer_new(NULL);
  }
  
  if (fi->path != NULL) {
    etpan_make_dir(fi->path);
    snprintf(path, sizeof(path), "%s/body.index", fi->path);
    fi->body_indexer = etpan_indexer_new(path);
  }
  else {
    fi->body_indexer = etpan_indexer_new(NULL);
  }
}





/* **************** */

struct close_indexer_param {
  struct etpan_indexer * indexer;
  struct etpan_indexer * body_indexer;
};

static void close_indexer_cleanup(struct etpan_thread_op * op)
{
  struct close_indexer_param * param;
  
  param = op->param;
  free(op->param);
  op->param = NULL;
}

static void close_indexer_run(struct etpan_thread_op * op)
{
  struct close_indexer_param * param;
  
  param = op->param;  
  etpan_indexer_unref(param->indexer);
  etpan_indexer_unref(param->body_indexer);
  param->indexer = NULL;
  param->body_indexer = NULL;
}

static void start_close_indexer(struct etpan_folder_indexer * fi,
    void (* callback)(int, void *, void *),
    void * cb_data)
{
  struct close_indexer_param * param;
  struct etpan_thread_op * op;
  
  ETPAN_FOLDER_INDEXER_LOG("closing indexer");
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->indexer = fi->indexer;
  param->body_indexer = fi->body_indexer;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  op->cancellable = 1;
  op->run = close_indexer_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = close_indexer_cleanup;
  
  ETPAN_FOLDER_INDEXER_LOG("start closing indexer op");
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
}

/* **************** */



static void close_index(struct etpan_folder_indexer * fi)
{
  ETPAN_FOLDER_INDEXER_LOG("close index");
  
  start_close_indexer(fi, NULL, fi);
  
#if 0
  etpan_indexer_unref(fi->body_indexer);
  fi->body_indexer = NULL;
  etpan_indexer_unref(fi->indexer);
  fi->indexer = NULL;
#endif
  fi->body_indexer = NULL;
  fi->indexer = NULL;
}

static carray * split_keywords_list(const char * str)
{
  char * dup_str;
  char * current_str;
  carray * keywords_list;
  int r;
  
  keywords_list = carray_new(4);
  if (keywords_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  dup_str = strdup(str);
  if (dup_str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  current_str = dup_str;
  while (1) {
    char * p;
    
    while (* current_str == ' ') {
      current_str ++;
    }
    
    if (* current_str == '\0') {
      break;
    }
    
    p = strchr(current_str, ' ');
    if (p != NULL) {
      char * keyword;
      
      * p = '\0';
      
      keyword = strdup(current_str);
      if (keyword == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      r = carray_add(keywords_list, keyword, NULL);
      if (r < 0) {
        free(keyword);
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      p ++;
      current_str = p;
    }
    else {
      char * keyword;
      
      keyword = strdup(current_str);
      if (keyword == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      r = carray_add(keywords_list, keyword, NULL);
      if (r < 0) {
        free(keyword);
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      break;
    }
  }
  
  free(dup_str); 
  
  return keywords_list;
}

static char * build_expr(int filter_type, const char * keywords_list_str)
{
  MMAPString * str;
  unsigned int i;
  char * dup_str;
  carray * keywords_list;
  
  str = mmap_string_new("");
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  keywords_list = split_keywords_list(keywords_list_str);
  
  for(i = 0 ; i < carray_count(keywords_list) ; i ++) {
    char * keyword;
    char * quoted_keyword;
    char * subexpr;
    size_t len;
    
    keyword = carray_get(keywords_list, i);
    quoted_keyword = etpan_indexer_quote(keyword);
    
    len = strlen(quoted_keyword) + 2;
    subexpr = malloc(len);
    if (subexpr == NULL) {
      free(quoted_keyword);
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    snprintf(subexpr, len, "%s*", quoted_keyword);
    
    if (i != 0) {
      if (mmap_string_append(str, " AND ") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    
    if (mmap_string_append(str, "(") == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    switch (filter_type) {
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL:
      if (mmap_string_append(str, "from:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, " OR subject:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, " OR recipient:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM_AND_SUBJECT:
      if (mmap_string_append(str, "from:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, " OR subject:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT_AND_SUBJECT:
      if (mmap_string_append(str, "recipient:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, " OR subject:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM:
      if (mmap_string_append(str, "from:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT:
      if (mmap_string_append(str, "recipient:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_SUBJECT:
      if (mmap_string_append(str, "subject:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_BODY:
      if (mmap_string_append(str, "body:") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, subexpr) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      break;
    }
    mmap_string_append(str, ")");
    
    free(subexpr);
    free(quoted_keyword);
  }
  
  dup_str = strdup(str->str);
  if (dup_str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(keywords_list) ; i ++) {
    char * keyword;
    
    keyword = (char *) carray_get(keywords_list, i);
    free(keyword);
  }
  mmap_string_free(str);
  
  return dup_str;
}

void etpan_folder_search_simple(struct etpan_folder_indexer * fi,
    int search_type, char * text,
    void (* callback)(int cancelled,
        struct etpan_folder_indexer_search_result * result, void * cb_data),
    void * cb_data)
{
  char * expr_headers;
  char * expr_body;
  
  expr_headers = NULL;
  expr_body = NULL;
  
  expr_headers = build_expr(search_type, text);
  
  if (search_type == ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL) {
    expr_body = build_expr(ETPAN_FOLDER_INDEXER_SEARCH_TYPE_BODY, text);
  }
  
  ETPAN_FOLDER_INDEXER_LOG("expr: %s %s", expr_headers, expr_body);
  
  etpan_folder_search(fi, expr_headers, expr_body, callback, cb_data);
  
  free(expr_body);
  free(expr_headers);
}

void etpan_folder_indexer_update(struct etpan_folder_indexer * fi)
{
  update(fi);
}



/* indexing */

struct indexing_elt {
  char * doc_id;
  char * type;
  char * value;
};

static void collect_for_indexing(carray * indexing_data,
    char * doc_id, char * type, char * value)
{
  struct indexing_elt * elt;
  int r;
  
  elt = malloc(sizeof(* elt));
  if (elt == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  elt->doc_id = strdup(doc_id);
  elt->type = strdup(type);
  elt->value = strdup(value);
  
  if (elt->doc_id == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (elt->type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (elt->value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(indexing_data, elt, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void collect_for_indexing_address(carray * indexing_data,
    char * doc_id, char * type, struct etpan_address * addr)
{
  char * display_name;
  char * address;
  
  display_name = etpan_address_get_display_name(addr);
  address = etpan_address_get_address(addr);
  
  if (display_name != NULL) {
    collect_for_indexing(indexing_data, doc_id, type, display_name);
  }
  if (address != NULL) {
    char * p;
    
    collect_for_indexing(indexing_data, doc_id, type, address);
    /* index domain */
    p = strchr(address, '@');
    if (p != NULL) {
      address = p + 1;
      
      while (1) {
        collect_for_indexing(indexing_data, doc_id, type, address);
        
        p = strchr(address, '.');
        if (p == NULL)
          break;
        
        address = p + 1;
      }
    }
  }
}

static void collect_for_indexing_address_list(carray * indexing_data,
    char * doc_id, char * type, carray * addr_list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(addr_list) ; i ++) {
    struct etpan_address * address;
    
    address = carray_get(addr_list, i);
    collect_for_indexing_address(indexing_data, doc_id, type, address);
  }
}

static void collect_for_indexing_message(carray * indexing_data,
    struct etpan_message * msg)
{
  struct etpan_message_header * header;
  char * uid;
  
  uid = etpan_message_get_uid(msg);
  header = etpan_message_get_header(msg);
  if (header != NULL) {
    carray * from;
    carray * to;
    carray * cc;
    char * subject;
    
    from = etpan_message_header_get_from(header);
    collect_for_indexing_address_list(indexing_data,
        uid, "from", from);
    
    to = etpan_message_header_get_to(header);
    collect_for_indexing_address_list(indexing_data,
        uid, "recipient", to);
    
    cc = etpan_message_header_get_cc(header);
    collect_for_indexing_address_list(indexing_data,
        uid, "recipient", cc);
    
    subject = etpan_message_header_get_subject(header);
    if (subject != NULL) {
      collect_for_indexing(indexing_data,
          uid, "subject", subject);
    }
  }
}



static void get_msg_diff(struct etpan_folder * folder,
    chash * current_messages_hash,
    chash * new_messages_hash, carray * deleted_messages_list)
{
  chash * msg_list;
  chashiter * iter;
  int r;
  
  msg_list = etpan_folder_get_message_hash(folder);
  for(iter = chash_begin(msg_list) ; iter != NULL ;
      iter = chash_next(msg_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    char * uid;
    
    chash_value(iter, &value);
    msg = value.data;
    uid = etpan_message_get_uid(msg);
    key.data = uid;
    key.len = strlen(uid) + 1;
    r = chash_get(current_messages_hash, &key, &value);
    if (r < 0) {
      value.data = msg;
      value.len = 0;
      r = chash_set(new_messages_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  for(iter = chash_begin(current_messages_hash) ; iter != NULL ;
      iter = chash_next(current_messages_hash, iter)) {
    chashdatum key;
    chashdatum value;
    
    chash_key(iter, &key);
    r = chash_get(msg_list, &key, &value);
    if (r < 0) {
      char * uid;
      
      uid = strdup(key.data);
      r = carray_add(deleted_messages_list, uid, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
}

static void update(struct etpan_folder_indexer * fi)
{
  switch (fi->state) {
  case INDEXER_IDLE:
    get_indexer_state(fi);
    break;
  case INDEXER_GETTING_INDEXER_STATE:
    /* do nothing */
    break;
  case INDEXER_INDEXING:
    fi->dirty = 1;
    break;
  case INDEXER_INDEXING_BODY:
    set_state(fi, INDEXER_CANCELLING_INDEXING_BODY);
    break;
  case INDEXER_CANCELLING_INDEXING_BODY:
    /* do nothing */
    break;
  }
}

static void set_state(struct etpan_folder_indexer * fi, int state)
{
  ETPAN_FOLDER_INDEXER_LOG("switch to state %i", state);
  fi->state = state;
  ETPAN_SIGNAL_SEND(fi, ETPAN_FOLDER_INDEXER_STATE_CHANGED);
}

static void cancel(struct etpan_folder_indexer * fi)
{
  etpan_folder_indexer_cancel_search(fi);
  
  switch (fi->state) {
  case INDEXER_IDLE:
    /* do nothing */
    break;
  case INDEXER_GETTING_INDEXER_STATE:
    cancel_get_indexer_state(fi);
    break;
  case INDEXER_INDEXING:
    cancel_indexing(fi);
    break;
  case INDEXER_INDEXING_BODY:
  case INDEXER_CANCELLING_INDEXING_BODY:
    cancel_indexing_body(fi);
    break;
  }
}

/* get indexer state */

static void get_indexer_state(struct etpan_folder_indexer * fi)
{
  ETPAN_SIGNAL_SEND(fi, ETPAN_FOLDER_INDEXER_START);
  
  fi->error_list = carray_new(4);
  if (fi->error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fi->cancelled = 0;
  
  if (!fi->indexer_state_initialized) {
    start_get_indexer_state(fi, get_indexer_state_callback, fi);
    return;
  }
  
  start_indexing(fi, indexing_callback, fi);
}

struct get_indexer_state_param {
  struct etpan_indexer * indexer;
  struct etpan_indexer * body_indexer;
};

static void get_indexer_state_cleanup(struct etpan_thread_op * op)
{
  if (op->result) {
    struct get_indexer_state_result * result;
    unsigned int i;
    
    result = op->result;
    if (result->indexed_messages != NULL) {
      for(i = 0 ; i < carray_count(result->indexed_messages) ; i ++) {
        char * uid;
      
        uid = carray_get(result->indexed_messages, i);
        free(uid);
      }
      carray_free(result->indexed_messages);
    }
    
    if (result->body_indexed_messages != NULL) {
      for(i = 0 ; i < carray_count(result->body_indexed_messages) ; i ++) {
        char * uid;
      
        uid = carray_get(result->body_indexed_messages, i);
        free(uid);
      }
      carray_free(result->body_indexed_messages);
    }
    
    free(op->result);
    op->result = NULL;
  }
  
  if (op->param) {
    struct get_indexer_state_param * param;
    
    param = op->param;
    etpan_indexer_unref(param->body_indexer);
    etpan_indexer_unref(param->indexer);
    
    free(op->param);
    op->param = NULL;
  }
}

static void get_indexer_state_run(struct etpan_thread_op * op)
{
  struct get_indexer_state_param * param;
  struct get_indexer_state_result * result;
  
  param = op->param;  
  result = op->result;
  
  result->indexed_messages = etpan_indexer_get_keys(param->indexer);
  result->body_indexed_messages = etpan_indexer_get_keys(param->body_indexer);
}

static void start_get_indexer_state(struct etpan_folder_indexer * fi,
    void (* callback)(int, struct get_indexer_state_result *, void *),
    void * cb_data)
{
  struct get_indexer_state_param * param;
  struct get_indexer_state_result * result;
  struct etpan_thread_op * op;
  
  set_state(fi, INDEXER_GETTING_INDEXER_STATE);
  fi->dirty = 0;
  ETPAN_FOLDER_INDEXER_LOG("getting indexer state");
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->indexer = fi->indexer;
  etpan_indexer_ref(param->indexer);
  param->body_indexer = fi->body_indexer;
  etpan_indexer_ref(param->body_indexer);
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->indexed_messages = NULL;
  result->body_indexed_messages = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  op->cancellable = 1;
  op->run = get_indexer_state_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = get_indexer_state_cleanup;
  
  ETPAN_FOLDER_INDEXER_LOG("start getting indexer state op");
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
  
  fi->get_indexer_state_op = op;
}


static void get_indexer_state_callback(int cancelled,
    struct get_indexer_state_result * result, void * cb_data)
{
  struct etpan_folder_indexer * fi;
  unsigned int i;
  int r;
  
  fi = cb_data;
  fi->get_indexer_state_op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    fi->cancelled = 1;
    set_state(fi, INDEXER_IDLE);

    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Indexation cancelled"));
    etpan_error_strf_long_description(error,
        _("Indexation of the folder %s was interrupted."),
        etpan_folder_get_ui_path(fi->folder));
    
    r = carray_add(fi->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    indexing_body_finished(fi);
    return;
  }
  
  /* copying state */
  for(i = 0 ; i < carray_count(result->indexed_messages) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * uid;
    
    uid = carray_get(result->indexed_messages, i);
    key.data = uid;
    key.len = strlen(uid) + 1;
    r = chash_set(fi->current_messages_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(result->body_indexed_messages) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * uid;
    
    uid = carray_get(result->body_indexed_messages, i);
    key.data = uid;
    key.len = strlen(uid) + 1;
    r = chash_set(fi->body_current_messages_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  fi->indexer_state_initialized = 1;
  
  start_indexing(fi, indexing_callback, fi);
}

static void cancel_get_indexer_state(struct etpan_folder_indexer * fi)
{
  etpan_thread_op_cancel(fi->get_indexer_state_op);
  fi->get_indexer_state_op = NULL;
}

/* header indexer */

struct indexer_param {
  carray * indexing_data;
  carray * unindexing_data;
  struct etpan_indexer * indexer;
};

static void indexer_cleanup(struct etpan_thread_op * op)
{
  if (op->param) {
    struct indexer_param * param;
    unsigned int i;
    
    param = op->param;
    for(i = 0 ; i < carray_count(param->indexing_data) ; i ++) {
      struct indexing_elt * elt;
      
      elt = carray_get(param->indexing_data, i);
      
      free(elt->value);
      free(elt->type);
      free(elt->doc_id);
      free(elt);
    }
    carray_free(param->indexing_data);
    
    for(i = 0 ; i < carray_count(param->unindexing_data) ; i ++) {
      char * doc_id;
      
      doc_id = carray_get(param->unindexing_data, i);
      free(doc_id);
    }
    carray_free(param->unindexing_data);
    
    etpan_indexer_unref(param->indexer);
    
    free(op->param);
    op->param = NULL;
  }
}

static void indexer_run(struct etpan_thread_op * op)
{
  struct indexer_param * param;
  unsigned int i;
  int count;
  
  param = op->param;  
  
  count = 0;
  for(i = 0 ; i < carray_count(param->indexing_data) ; i ++) {
    struct indexing_elt * elt;
    
    /* every 100 messages */
    if (count >= 100) {
      if (etpan_thread_op_cancelled(op))
        return;
      count = 0;
    }
    
    elt = carray_get(param->indexing_data, i);
    etpan_indexer_set(param->indexer, elt->doc_id, elt->type, elt->value);
    
    count ++;
  }
  
  count = 0;
  for(i = 0 ; i < carray_count(param->unindexing_data) ; i ++) {
    char * doc_id;
    
    /* every 100 messages */
    if (count >= 100) {
      if (etpan_thread_op_cancelled(op))
        return;
      count = 0;
    }
    
    doc_id = carray_get(param->unindexing_data, i);
    etpan_indexer_delete(param->indexer, doc_id);
  }
}

static void
start_indexing(struct etpan_folder_indexer * fi,
    void (* callback)(int, void *, void *),
    void * cb_data)
{
  carray * indexing_data;
  carray * unindexing_data;
  unsigned int i;
  chashiter * iter;
  struct indexer_param * param;
  struct etpan_thread_op * op;
  int r;
  
  set_state(fi, INDEXER_INDEXING);
  fi->dirty = 0;
  ETPAN_FOLDER_INDEXER_LOG("prepare indexer for headers");
  /* build indexing data */
  
  fi->new_messages_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (fi->new_messages_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fi->deleted_messages_list = carray_new(16);
  if (fi->deleted_messages_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  get_msg_diff(fi->folder, fi->current_messages_hash,
      fi->new_messages_hash, fi->deleted_messages_list);
  
  indexing_data = carray_new(chash_count(fi->new_messages_hash));
  if (indexing_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  unindexing_data = carray_new(carray_count(fi->deleted_messages_list));
  if (unindexing_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_FOLDER_INDEXER_LOG("message count : %i", chash_count(fi->new_messages_hash));
  for(iter = chash_begin(fi->new_messages_hash) ; iter != NULL ;
      iter = chash_next(fi->new_messages_hash, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    
    chash_value(iter, &value);
    
    msg = value.data;
    collect_for_indexing_message(indexing_data, msg);
  }
  
  for(i = 0 ; i < carray_count(fi->deleted_messages_list) ; i ++) {
    char * uid;
    
    uid = carray_get(fi->deleted_messages_list, i);
    uid = strdup(uid);
    r = carray_add(unindexing_data, uid, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  ETPAN_FOLDER_INDEXER_LOG("indexing headers");
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->indexing_data = indexing_data;
  param->unindexing_data = unindexing_data;
  param->indexer = fi->indexer;
  etpan_indexer_ref(param->indexer);
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  op->cancellable = 1;
  op->run = indexer_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = indexer_cleanup;
  
  ETPAN_FOLDER_INDEXER_LOG("start indexer op");
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
  
  fi->indexing_op = op;
}

static void cancel_indexing(struct etpan_folder_indexer * fi)
{
  etpan_thread_op_cancel(fi->indexing_op);
  fi->indexing_op = NULL;
}

static void indexing_callback(int cancelled,
    void * dummy, void * cb_data)
{
  chashiter * iter;
  unsigned int i;
  struct etpan_folder_indexer * fi;
  int r;
  (void) dummy;
  
  ETPAN_FOLDER_INDEXER_LOG("indexing headers ended");
  fi = cb_data;
  fi->indexing_op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    fi->cancelled = 1;
    set_state(fi, INDEXER_IDLE);
    carray_free(fi->deleted_messages_list);
    chash_free(fi->new_messages_hash);
    fi->deleted_messages_list = NULL;
    fi->new_messages_hash = NULL;

    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Indexation cancelled"));
    etpan_error_strf_long_description(error,
        _("Indexation of the folder %s was interrupted."),
        etpan_folder_get_ui_path(fi->folder));
    
    r = carray_add(fi->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    indexing_body_finished(fi);
    return;
  }
  
  /* synchronize current message hash */
  
  for(i = 0 ; i < carray_count(fi->deleted_messages_list) ; i ++) {
    char * uid;
    chashdatum key;
    
    uid = carray_get(fi->deleted_messages_list, i);
    key.data = uid;
    key.len = strlen(uid) + 1;
    chash_delete(fi->current_messages_hash, &key, NULL);
    free(uid);
  }
  
  for(iter = chash_begin(fi->new_messages_hash) ; iter != NULL ;
      iter = chash_next(fi->new_messages_hash, iter)) {
    chashdatum key;
    chashdatum value;
    
    chash_key(iter, &key);
    value.data = NULL;
    value.len = 0;
    r = chash_set(fi->current_messages_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  carray_free(fi->deleted_messages_list);
  chash_free(fi->new_messages_hash);
  
  fi->deleted_messages_list = NULL;
  fi->new_messages_hash = NULL;
  
  if (fi->dirty) {
    ETPAN_SIGNAL_SEND(fi, ETPAN_FOLDER_INDEXER_UPDATED_RESULT);
    start_indexing(fi, indexing_callback, fi);
  }
  else {
    ETPAN_SIGNAL_SEND(fi, ETPAN_FOLDER_INDEXER_UPDATED_RESULT);
    start_indexing_body(fi);
  }
}

/* indexing body */
/* XXX should be rewritten using msg list iterator */

static void start_indexing_body(struct etpan_folder_indexer * fi)
{
  chashiter * iter;
  int r;
  
  set_state(fi, INDEXER_INDEXING_BODY);
  ETPAN_FOLDER_INDEXER_LOG("indexing body");
  
  fi->dirty = 0;
  
  fi->body_new_messages_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (fi->body_new_messages_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fi->body_new_messages_list = carray_new(16);
  if (fi->body_new_messages_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fi->body_deleted_messages_list = carray_new(16);
  if (fi->body_deleted_messages_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  get_msg_diff(fi->folder, fi->body_current_messages_hash,
      fi->body_new_messages_hash, fi->body_deleted_messages_list);
  
  for(iter = chash_begin(fi->body_new_messages_hash) ; iter != NULL ;
      iter = chash_next(fi->body_new_messages_hash, iter)) {
    chashdatum key;
    char * uid;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    uid = key.data;
    uid = strdup(uid);
    r = carray_add(fi->body_new_messages_list, uid, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  fi->body_next_msg = 0;
  
  if (carray_count(fi->body_new_messages_list) == 0)
    indexing_body_finished(fi);
  else
    index_next_body(fi);
}

static void indexing_body_finished(struct etpan_folder_indexer * fi)
{
  unsigned int i;
  int restart_indexing;
  struct etpan_error * error;
  
  if (fi->body_deleted_messages_list != NULL) {
    for(i = 0 ; i < carray_count(fi->body_deleted_messages_list) ; i ++) {
      char * uid;
      
      uid = carray_get(fi->body_deleted_messages_list, i);
      free(uid);
    }
  }
  
  if (fi->body_new_messages_list != NULL) {
    for(i = 0 ; i < carray_count(fi->body_new_messages_list) ; i ++) {
      char * uid;
    
      uid = carray_get(fi->body_new_messages_list, i);
      free(uid);
    }
  }
  
  if (fi->body_deleted_messages_list != NULL) {
    carray_free(fi->body_deleted_messages_list);
  }
  
  if (fi->body_new_messages_list != NULL) {
    carray_free(fi->body_new_messages_list);
  }
  if (fi->body_new_messages_hash != NULL) {
    chash_free(fi->body_new_messages_hash);
  }
  
  fi->body_deleted_messages_list = NULL;
  fi->body_new_messages_list = NULL;
  fi->body_new_messages_hash = NULL;
  
  restart_indexing = (fi->state == INDEXER_CANCELLING_INDEXING_BODY);
  
  if (restart_indexing) {
    start_indexing(fi, indexing_callback, fi);
  }
  else {
    set_state(fi, INDEXER_IDLE);
    
    if (carray_count(fi->error_list) == 0) {
      error = NULL;
    }
    else if (carray_count(fi->error_list) == 1) {
      error = carray_get(fi->error_list, 0);
    }
    else {
      error = etpan_error_multiple();
      for(i = 0 ; i < carray_count(fi->error_list) ; i ++) {
        struct etpan_error * suberror;
      
        suberror = carray_get(fi->error_list, i);
        etpan_error_add_child(error, suberror);
      }
    }
    carray_free(fi->error_list);
    fi->error_list = NULL;
    fi->error = error;
    
    ETPAN_SIGNAL_SEND(fi, ETPAN_FOLDER_INDEXER_FINISHED);
    
    ETPAN_ERROR_FREE(fi->error);
    fi->error = NULL;
  }
}

static void index_next_body(struct etpan_folder_indexer * fi)
{
  char * uid;
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_message * msg;
  
  uid = carray_get(fi->body_new_messages_list, fi->body_next_msg);
  key.data = uid;
  key.len = strlen(uid) + 1;
  r = chash_get(fi->body_new_messages_hash, &key, &value);
  if (r < 0) {
    ETPAN_FOLDER_INDEXER_LOG("index_next_body consistency error");
    etpan_crash();
  }
  msg = value.data;
  
  fi->fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fi->fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET);
  etpan_message_fetcher_set_message(fi->fetcher, msg);
  etpan_message_fetcher_setup(fi->fetcher);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fi->fetcher, fi,
      index_next_body_fetch_callback);
  etpan_message_fetcher_run(fi->fetcher);
}

static void index_next_body_fetch_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_indexer * fi;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  fi = user_data;
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fi->fetcher, fi,
      index_next_body_fetch_callback);
  
  body_indexing(fi, index_next_body_indexed_callback, fi);
}

static void body_indexing(struct etpan_folder_indexer * fi,
    void (* callback)(int, void *, void *),
    void * cb_data)
{
  carray * indexing_data;
  carray * unindexing_data;
  unsigned int i;
  struct indexer_param * param;
  struct etpan_thread_op * op;
  int r;
  struct etpan_message * msg;
  carray * list;
  char * uid;
  struct etpan_error * error;
  
  error = etpan_message_fetcher_get_error(fi->fetcher);
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      struct etpan_error * new_error;
    
      fi->cancelled = 1;
      set_state(fi, INDEXER_IDLE);
    
      new_error = etpan_error_new();
      etpan_error_set_code(new_error, ERROR_CANCELLED);
      etpan_error_set_short_description(new_error,
					_("Indexation cancelled"));
      etpan_error_strf_long_description(new_error,
					_("Indexation of the folder %s was interrupted."),
					etpan_folder_get_ui_path(fi->folder));
    
      r = carray_add(fi->error_list, new_error, NULL);
      if (r < 0)
	ETPAN_LOG_MEMORY_ERROR;
      
      etpan_message_fetcher_unref(fi->fetcher);
      fi->fetcher = NULL;
      
      indexing_body_finished(fi);
      return;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(fi, error);
    r = carray_add(fi->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_message_fetcher_unref(fi->fetcher);
    fi->fetcher = NULL;
    
    index_next_body_indexed_callback(0, NULL, fi);
    return;
  }
  
  ETPAN_FOLDER_INDEXER_LOG("body indexing");
  /* build indexing data */
  
  indexing_data = carray_new(16);
  if (indexing_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  unindexing_data = carray_new(16);
  if (unindexing_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg = etpan_message_fetcher_get_message(fi->fetcher);
  uid = etpan_message_get_uid(msg);
  list = etpan_message_fetcher_get_part_list(fi->fetcher);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_part_fetch_info * part_info;
    int type;
    struct etpan_part * part;
    int part_type;
    
    part_info = carray_get(list, i);
    type = etpan_part_fetch_info_get_type(part_info);
    part = etpan_part_fetch_info_get_part(part_info);
    part_type = etpan_part_get_type(part);
    
    if ((type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) &&
        (part_type == ETPAN_PART_TYPE_SINGLE)) {
      char * content;
      size_t content_length;
      etpan_part_fetch_info_get_content(part_info,
          &content, &content_length);
      collect_for_indexing(indexing_data, uid, "body", content);
    }
  }
  
  if (fi->body_deleted_messages_list != NULL) {
    for(i = 0 ; i < carray_count(fi->body_deleted_messages_list) ; i ++) {
      uid = carray_get(fi->body_deleted_messages_list, i);
      uid = strdup(uid);
      if (uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      r = carray_add(unindexing_data, uid, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  etpan_message_fetcher_unref(fi->fetcher);
  fi->fetcher = NULL;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->indexing_data = indexing_data;
  param->unindexing_data = unindexing_data;
  param->indexer = fi->body_indexer;
  etpan_indexer_ref(param->indexer);
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  op->cancellable = 1;
  op->run = indexer_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = indexer_cleanup;
  
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
  
  fi->body_indexing_op = op;
}

static void index_next_body_indexed_callback(int cancelled, void * dummy,
    void * cb_data)
{
  struct etpan_folder_indexer * fi;
  char * uid;
  chashdatum key;
  chashdatum value;
  int r;
  (void) dummy;
  
  fi = cb_data;
  
  fi->body_indexing_op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    fi->cancelled = 1;
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Indexation cancelled"));
    etpan_error_strf_long_description(error,
        _("Indexation of the folder %s was interrupted."),
        etpan_folder_get_ui_path(fi->folder));
    
    r = carray_add(fi->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    indexing_body_finished(fi);
    return;
  }
  
  /* synchronize current message hash */
  
  if (fi->body_deleted_messages_list != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(fi->body_deleted_messages_list) ; i ++) {
      uid = carray_get(fi->body_deleted_messages_list, i);
      key.data = uid;
      key.len = strlen(uid) + 1;
      chash_delete(fi->body_current_messages_hash, &key, NULL);
      free(uid);
    }
    carray_free(fi->body_deleted_messages_list);
    fi->body_deleted_messages_list = NULL;
  }
  
  /* add message */
  uid = carray_get(fi->body_new_messages_list, fi->body_next_msg);
  key.data = uid;
  key.len = strlen(uid) + 1;
  value.data = NULL;
  value.len = 0;
  r = chash_set(fi->body_current_messages_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_SIGNAL_SEND(fi, ETPAN_FOLDER_INDEXER_UPDATED_RESULT);
  
  if (fi->state == INDEXER_CANCELLING_INDEXING_BODY) {
    indexing_body_finished(fi);
    return;
  }
  
  /* next message */
  fi->body_next_msg ++;
  ETPAN_FOLDER_INDEXER_LOG("%i/%i",
      fi->body_next_msg, carray_count(fi->body_new_messages_list));
  if (fi->body_next_msg < carray_count(fi->body_new_messages_list))
    index_next_body(fi);
  else
    indexing_body_finished(fi);
}

static void cancel_indexing_body(struct etpan_folder_indexer * fi)
{
  if (fi->fetcher != NULL) {
    etpan_message_fetcher_cancel(fi->fetcher);
  }
  if (fi->body_indexing_op != NULL) {
    etpan_thread_op_cancel(fi->body_indexing_op);
    fi->body_indexing_op = NULL;
  }
}

char * etpan_folder_indexer_get_path_for_folder(struct etpan_folder_indexer * fi)
{
  struct etpan_storage * storage;
  struct etpan_account * account;
  char * location;
  char * index_dir;
  char result[PATH_MAX];
  char * qmb;
  char * value;
  
  location = etpan_folder_get_location(fi->folder);
  
  storage = etpan_folder_get_storage(fi->folder);
  account = etpan_storage_get_account(storage);
  
  index_dir = etpan_get_index_dir(etpan_account_manager_get_default(),
      account);
  qmb = etpan_quote_mailbox(location);
  
  snprintf(result, sizeof(result), "%s/%s", index_dir, qmb);
  
  free(qmb);
  free(index_dir);
  
  value = strdup(result);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}


/* search */

struct etpan_folder_indexer_search_param {
  struct etpan_indexer * indexer;
  struct etpan_indexer * body_indexer;
  char * expr_headers;
  char * expr_body;
  void (* callback)(int cancelled,
      struct etpan_folder_indexer_search_result * result, void * cb_data);
  void * cb_data;
  struct etpan_folder_indexer * folder_indexer;
};

static void search_cleanup(struct etpan_thread_op * op)
{
  if (op->result) {
    struct etpan_folder_indexer_search_result * result;
    unsigned int i;
    
    result = op->result;
    if (result->messages != NULL) {
      for(i = 0 ; i < carray_count(result->messages) ; i ++) {
        char * uid;
      
        uid = carray_get(result->messages, i);
        free(uid);
      }
      carray_free(result->messages);
    }
    
    free(op->result);
    op->result = NULL;
  }
  
  if (op->param) {
    struct etpan_folder_indexer_search_param * param;
    
    param = op->param;
    etpan_indexer_unref(param->body_indexer);
    etpan_indexer_unref(param->indexer);
    free(param->expr_body);
    free(param->expr_headers);
    
    free(op->param);
    op->param = NULL;
  }
}

static void search_run(struct etpan_thread_op * op)
{
  struct etpan_folder_indexer_search_param * param;
  struct etpan_folder_indexer_search_result * result;
  unsigned int count;
  carray * headers_messages;
  carray * body_messages;
  int r;
  
  param = op->param;  
  result = op->result;
  
  headers_messages = NULL;
  if (param->expr_headers != NULL)
    headers_messages = etpan_indexer_search(param->indexer,
        param->expr_headers);
  body_messages = NULL;
  if (param->expr_body != NULL)
    body_messages = etpan_indexer_search(param->body_indexer,
        param->expr_body);
  
  count = 0;
  if (headers_messages != NULL)
    count += carray_count(headers_messages);
  if (body_messages != NULL)
    count += carray_count(body_messages);
  
  result->messages = carray_new(count);
  if (result->messages == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  /* XXX - should remove duplicates */
  if (headers_messages != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(headers_messages) ; i ++) {
      char * uid;
      
      uid = carray_get(headers_messages, i);
      r = carray_add(result->messages, uid, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    r = carray_set_size(headers_messages, 0);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    carray_free(headers_messages);
  }
  if (body_messages != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(body_messages) ; i ++) {
      char * uid;
      
      uid = carray_get(body_messages, i);
      r = carray_add(result->messages, uid, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    r = carray_set_size(body_messages, 0);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    carray_free(body_messages);
  }
  result->error = NULL;
}

static void search_callback(int cancelled,
    struct etpan_folder_indexer_search_result * result, void * cb_data);

void etpan_folder_search(struct etpan_folder_indexer * fi,
    char * expr_headers,
    char * expr_body,
    void (* callback)(int cancelled,
        struct etpan_folder_indexer_search_result * result, void * cb_data),
    void * cb_data)
{
  struct etpan_folder_indexer_search_param * param;
  struct etpan_folder_indexer_search_result * result;
  struct etpan_thread_op * op;
  unsigned int count;
  
  if (fi->search_op != NULL) {
    ETPAN_FOLDER_INDEXER_LOG("already searching");
    etpan_crash();
  }
  
  etpan_indexer_set_max_response_count(UINT_MAX);
  ETPAN_FOLDER_INDEXER_LOG("searching");
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->indexer = fi->indexer;
  etpan_indexer_ref(param->indexer);
  param->body_indexer = fi->body_indexer;
  etpan_indexer_ref(param->body_indexer);
  if (expr_headers == NULL) {
    param->expr_headers = NULL;
  }
  else {
    param->expr_headers = strdup(expr_headers);
    if (param->expr_headers == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  if (expr_body == NULL) {
    param->expr_body = NULL;
  }
  else {
    param->expr_body = strdup(expr_body);
    if (param->expr_body == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  param->callback = callback;
  param->cb_data = cb_data;
  param->folder_indexer = fi;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->messages = NULL;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  op->cancellable = 1;
  op->run = search_run;
  op->callback = (void (*)(int, void *, void *)) search_callback;
  op->callback_data = param;
  op->cleanup = search_cleanup;
  
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
  
  fi->search_op = op;
}

static void search_callback(int cancelled,
    struct etpan_folder_indexer_search_result * result, void * cb_data)
{
  struct etpan_folder_indexer_search_param * param;
  struct etpan_folder_indexer * fi;
  
  param = cb_data;
  fi = param->folder_indexer;
  fi->search_op = NULL;
  
  param->callback(cancelled, result, param->cb_data);
}

void etpan_folder_indexer_cancel_search(struct etpan_folder_indexer * fi)
{
  if (fi->search_op != NULL) {
    etpan_thread_op_cancel(fi->search_op);
  }
}

int etpan_folder_indexer_indexing_state(struct etpan_folder_indexer * fi)
{
  switch (fi->state) {
  case INDEXER_IDLE:
    return ETPAN_FOLDER_INDEXER_STATE_IDLE;
  case INDEXER_GETTING_INDEXER_STATE:
  case INDEXER_INDEXING:
    return ETPAN_FOLDER_INDEXER_STATE_INDEXING_HEADERS;
  case INDEXER_INDEXING_BODY:
  case INDEXER_CANCELLING_INDEXING_BODY:
    return ETPAN_FOLDER_INDEXER_STATE_INDEXING_BODY;
  }
  return ETPAN_FOLDER_INDEXER_STATE_IDLE;
}

struct etpan_error *
etpan_folder_indexer_get_error(struct etpan_folder_indexer * fi)
{
  return fi->error;
}

static struct etpan_error * rewrite_error(struct etpan_folder_indexer * fi,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred during indexation of %s.\n"
          "%s"),
      etpan_folder_get_ui_path(fi->folder),
      previous_long_description);
  
  return new_error;
}

int etpan_folder_indexer_is_searching(struct etpan_folder_indexer * fi)
{
  return (fi->search_op != NULL);
}
