#include "etpan-folder-rename.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-nls.h"
#include "etpan-mail-manager.h"

enum {
  STATE_IDLE,
  STATE_RENAMING,
  STATE_FETCHING_FOLDERLIST,
};

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    struct etpan_error * error);

static void etpan_folder_rename_unsetup(struct etpan_folder_rename * folder_rename);
static void run(struct etpan_folder_rename * folder_rename);
static void rename_folder_done_callback(int cancelled,
    struct etpan_storage_rename_folder_result * result,
    void * cb_data);
#if 0
static void folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data);
#endif
static void cancel_fetch_folder_list(struct etpan_folder_rename * folder_rename);
static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void notify_finished(struct etpan_folder_rename * folder_rename);

struct etpan_folder_rename * etpan_folder_rename_new(void)
{
  struct etpan_folder_rename * folder_rename;
  
  folder_rename = malloc(sizeof(* folder_rename));
  if (folder_rename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_rename->ref_count = 1;
  folder_rename->folder = NULL;
  folder_rename->op = NULL;
  folder_rename->new_name = NULL;
  folder_rename->state = STATE_IDLE;
  folder_rename->error = NULL;
  folder_rename->renamed_folder = NULL;
  
  return folder_rename;
}

static void etpan_folder_rename_free(struct etpan_folder_rename * folder_rename)
{
  etpan_folder_rename_unsetup(folder_rename);
  
  ETPAN_ERROR_FREE(folder_rename->error);
  folder_rename->error = NULL;
  
  if (folder_rename->renamed_folder != NULL)
    etpan_folder_unref(folder_rename->renamed_folder);
  free(folder_rename->new_name);
  if (folder_rename->folder != NULL)
    etpan_folder_unref(folder_rename->folder);
  free(folder_rename);
}


void etpan_folder_rename_ref(struct etpan_folder_rename * folder_rename)
{
  folder_rename->ref_count ++;
}

void etpan_folder_rename_unref(struct etpan_folder_rename * folder_rename)
{
  folder_rename->ref_count --;
  if (folder_rename->ref_count == 0)
    etpan_folder_rename_free(folder_rename);
}

void etpan_folder_rename_set_folder(struct etpan_folder_rename * folder_rename,
    struct etpan_folder * folder)
{
  if (folder != folder_rename->folder) {
    if (folder_rename->folder != NULL)
      etpan_folder_unref(folder_rename->folder);
    folder_rename->folder = folder;
    if (folder != NULL) {
      etpan_folder_ref(folder);
    }
    else
      folder_rename->folder = NULL;
  }
}

struct etpan_folder *
etpan_folder_rename_get_folder(struct etpan_folder_rename * folder_rename)
{
  return folder_rename->folder;
}

void etpan_folder_rename_set_new_name(struct etpan_folder_rename * folder_rename, char * new_name)
{
  if (new_name != folder_rename->new_name) {
    free(folder_rename->new_name);
    if (new_name != NULL) {
      folder_rename->new_name = strdup(new_name);
      if (folder_rename->new_name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder_rename->new_name = NULL;
  }
}

char * etpan_folder_rename_get_new_name(struct etpan_folder_rename * folder_rename)
{
  return folder_rename->new_name;
}

void etpan_folder_rename_setup(struct etpan_folder_rename * folder_rename)
{
  (void) folder_rename;
}

static void etpan_folder_rename_unsetup(struct etpan_folder_rename * folder_rename)
{
  etpan_folder_rename_cancel(folder_rename);
}

void etpan_folder_rename_run(struct etpan_folder_rename * folder_rename)
{
  run(folder_rename);
}

void etpan_folder_rename_cancel(struct etpan_folder_rename * folder_rename)
{
  switch (folder_rename->state) {
  case STATE_IDLE:
    break;
  case STATE_RENAMING:
    etpan_thread_op_cancel(folder_rename->op);
    break;
  case STATE_FETCHING_FOLDERLIST:
    cancel_fetch_folder_list(folder_rename);
    break;
  }
}

struct etpan_error *
etpan_folder_rename_get_error(struct etpan_folder_rename * folder_rename)
{
  return folder_rename->error;
}

static void run(struct etpan_folder_rename * folder_rename)
{
  struct etpan_storage * storage;
  char * location;
  char * new_location;
  
  folder_rename->state = STATE_RENAMING;
  
  storage = etpan_folder_get_storage(folder_rename->folder);
  location = etpan_folder_get_location(folder_rename->folder);
  new_location = folder_rename->new_name;
  
  folder_rename->op =
    etpan_storage_rename_folder(etpan_thread_manager_app_get_default(),
        storage, location, new_location,
        rename_folder_done_callback, folder_rename);
}

static void rename_folder_done_callback(int cancelled,
    struct etpan_storage_rename_folder_result * result,
    void * cb_data)
{
  struct etpan_folder_rename * folder_rename;
  struct etpan_storage * storage;
  
  folder_rename = cb_data;
  folder_rename->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Change of name of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Change of the name of the folder %s was interrupted."),
        etpan_folder_get_ui_path(folder_rename->folder));
    
    folder_rename->error = error;
    
    notify_finished(folder_rename);
    return;
  }
  
  ETPAN_LOG("rename folder done");
  
  if (result->error != NULL) {
    folder_rename->error = result->error;
    result->error = NULL;
    
    notify_finished(folder_rename);
    return;
  }
  
  folder_rename->state = STATE_FETCHING_FOLDERLIST;
  
  /* fetch folder list */
  storage = etpan_folder_get_storage(folder_rename->folder);
#if 0
  folder_rename->op =
    etpan_storage_fetch_folder_list(etpan_thread_manager_app_get_default(),
        storage, folder_list_fetched_callback,
        folder_rename);
  ETPAN_LOG("fetch folder list");
#endif
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_rename);
  etpan_mail_manager_storage_update(etpan_mail_manager_get_default(),
      storage);
}

static void cancel_fetch_folder_list(struct etpan_folder_rename * folder_rename)
{
  int r;
  struct etpan_error * error;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_rename);
  
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_CANCELLED);
  etpan_error_set_short_description(error,
      _("Change of name of folder cancelled"));
  etpan_error_strf_long_description(error,
      _("Change of the name of the folder %s was interrupted."),
      etpan_folder_get_ui_path(folder_rename->folder));
  folder_rename->error = error;
  notify_finished(folder_rename);
}

static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_rename * folder_rename;
  struct etpan_error * error;
  struct etpan_storage * storage;
  
  folder_rename = user_data;
  storage = etpan_folder_get_storage(folder_rename->folder);
  if (signal_data != storage)
    return;

  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_rename);
  
  error = etpan_mail_manager_storage_get_error(etpan_mail_manager_get_default(),storage);
  if (error != NULL) {
    error = rewrite_error(folder_rename->folder, error);
    folder_rename->error = error;
    
    notify_finished(folder_rename);
    return;
  }

  /* get new folder */
  folder_rename->renamed_folder =
    etpan_storage_get_folder(storage, folder_rename->new_name);
  if (folder_rename->renamed_folder == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, "Folder not found");
    etpan_error_strf_long_description(error, "Folder is not found through its new name %s.", folder_rename->new_name);
    
    folder_rename->error = error;
    notify_finished(folder_rename);
    return;
  }
  
  etpan_folder_ref(folder_rename->renamed_folder);
  
  notify_finished(folder_rename);
}

#if 0
static void folder_list_fetched_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data)
{
  struct etpan_folder_rename * folder_rename;
  struct etpan_storage * storage;
  struct etpan_error * error;
  
  folder_rename = cb_data;
  folder_rename->op = NULL;
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Change of name of folder cancelled"));
    etpan_error_strf_long_description(error,
        _("Change of the name of the folder %s was interrupted."),
        etpan_folder_get_ui_path(folder_rename->folder));
    
    folder_rename->error = error;
    
    notify_finished(folder_rename);
    return;
  }
  
  ETPAN_LOG("fetch folder list done");
  
  if (result->error != NULL) {
    error = rewrite_error(folder_rename->folder, result->error);
    folder_rename->error = error;
    
    notify_finished(folder_rename);
    return;
  }
  
  storage = etpan_folder_get_storage(folder_rename->folder);
  error = etpan_storage_update_folder_list(storage,
      result->folder_list);
  ETPAN_ERROR_IGNORE(error);
  etpan_storage_remove_lost_folders(storage);
  
  /* get new folder */
  folder_rename->renamed_folder =
    etpan_storage_get_folder(storage, folder_rename->new_name);
  if (folder_rename->renamed_folder == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, "Folder not found");
    etpan_error_strf_long_description(error, "Folder is not found through its new name %s.", folder_rename->new_name);
    
    folder_rename->error = error;
    notify_finished(folder_rename);
    return;
  }
  
  etpan_folder_ref(folder_rename->renamed_folder);
  
  notify_finished(folder_rename);
}
#endif

static void notify_finished(struct etpan_folder_rename * folder_rename)
{
  ETPAN_LOG("folder rename finished");
  folder_rename->state = STATE_IDLE;
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_RENAME_FINISHED_SIGNAL, folder_rename, NULL);
}

struct etpan_folder *
etpan_folder_rename_get_renamed_folder(struct etpan_folder_rename * folder_rename)
{
  return folder_rename->renamed_folder;
}

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred while changing the name of the folder %s.\n"
          "%s"),
      etpan_folder_get_ui_path(folder),
      previous_long_description);
  
  return new_error;
}
