#include "etpan-folder-tree.h"

#include <stdlib.h>
#include <libetpan/libetpan.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-folder.h"
#include "etpan-log.h"

struct etpan_folder_tree * etpan_folder_tree_new(void)
{
  struct etpan_folder_tree * folder_tree;
  
  folder_tree = malloc(sizeof(* folder_tree));
  if (folder_tree == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_tree->ui_path = NULL;
  folder_tree->folder = NULL;
  folder_tree->parent = NULL;
  folder_tree->children = carray_new(16);
  if (folder_tree->children == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return folder_tree;
}

void etpan_folder_tree_free(struct etpan_folder_tree * folder_tree)
{
  if (folder_tree->folder != NULL)
    etpan_folder_unref(folder_tree->folder);
  carray_free(folder_tree->children);
  free(folder_tree->ui_path);
  free(folder_tree);
}

void etpan_folder_tree_free_recursive(struct etpan_folder_tree * folder_tree)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(folder_tree->children) ; i ++) {
    struct etpan_folder_tree * sub;
    
    sub = carray_get(folder_tree->children, i);
    etpan_folder_tree_free_recursive(sub);
  }
  etpan_folder_tree_free(folder_tree);
}

void etpan_folder_tree_set_folder(struct etpan_folder_tree * folder_tree,
    struct etpan_folder * folder)
{
  if (folder != folder_tree->folder) {
    if (folder_tree->folder != NULL) {
      etpan_folder_unref(folder_tree->folder);
    }
    folder_tree->folder = folder;
    if (folder_tree->folder != NULL) {
      etpan_folder_ref(folder_tree->folder);
    }
  }
}

struct etpan_folder *
etpan_folder_tree_get_folder(struct etpan_folder_tree * folder_tree)
{
  return folder_tree->folder;
}

struct etpan_folder_tree *
etpan_folder_tree_get_parent(struct etpan_folder_tree * child)
{
  return child->parent;
}

void etpan_folder_tree_set_parent(struct etpan_folder_tree * child,
    struct etpan_folder_tree * parent)
{
  child->parent = parent;
}

void etpan_folder_tree_add_child(struct etpan_folder_tree * parent,
    struct etpan_folder_tree * child)
{
  int r;
  
  r = carray_add(parent->children, child, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

carray * etpan_folder_tree_get_children(struct etpan_folder_tree * parent)
{
  return parent->children;
}

void etpan_folder_tree_set_ui_path(struct etpan_folder_tree * folder_tree,
    char * ui_path)
{
  if (ui_path != folder_tree->ui_path) {
    free(folder_tree->ui_path);
    if (ui_path != NULL) {
      folder_tree->ui_path = strdup(ui_path);
      if (folder_tree->ui_path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder_tree->ui_path = NULL;
  }
}

char * etpan_folder_tree_get_ui_path(struct etpan_folder_tree * folder_tree)
{
  return folder_tree->ui_path;
}

static char * get_parent_folder_name(char * folder_name)
{
  char * p;
  char * dup_folder_name;
  
  dup_folder_name = strdup(folder_name);
  p = strrchr(dup_folder_name, '/');
  if (p == NULL) {
    free(dup_folder_name);
    return NULL;
  }
  
  * p = '\0';
  
  return dup_folder_name;
}

struct etpan_folder_tree * etpan_folder_tree_get_node(struct etpan_folder_tree * root, char * folder_name)
{
  carray * children;
  char * parent_folder_name;
  struct etpan_folder_tree * parent;
  unsigned int i;
  
  parent_folder_name = get_parent_folder_name(folder_name);
  if (parent_folder_name == NULL)
    return root;
  
  parent = etpan_folder_tree_get_node(root, parent_folder_name);
  free(parent_folder_name);
  
  if (parent == NULL) {
    return NULL;
  }
  
  children = etpan_folder_tree_get_children(parent);
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_folder_tree * child;
    char * ui_path;
    
    child = carray_get(children, i);
    ui_path = etpan_folder_tree_get_ui_path(child);
    if (strcmp(ui_path, folder_name) == 0)
      return child;
  }
  
  return NULL;
}
