#include "etpan-folder-view.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-msg-list-view.h"
#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-signal.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-signal.h"
#include "etpan-mail-manager.h"
#include "etpan-folder-indexer.h"
#include "etpan-message.h"

struct etpan_folder_view * etpan_folder_view_new(void)
{
  struct etpan_folder_view * view;
  
  view = malloc(sizeof(* view));
  if (view == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  view->msg_list_view = etpan_msg_list_view_new();
  view->folder_list = carray_new(4);
  if (view->folder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  view->folder_for_orphans = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (view->folder_for_orphans == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  view->has_message_list = 0;
  view->has_filter = 0;
  
  view->searching = 0;
  view->message_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (view->message_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return view;
}

void etpan_folder_view_free(struct etpan_folder_view * view)
{
  unsigned int i;
  
  chash_free(view->message_list);
  chash_free(view->folder_for_orphans);
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder * folder;
    
    folder = carray_get(view->folder_list, i);
    etpan_folder_unref(folder);
  }
  carray_free(view->folder_list);
  etpan_msg_list_view_free(view->msg_list_view);
  free(view);
}

struct etpan_msg_list_view *
etpan_folder_view_get_msg_list_view(struct etpan_folder_view * view)
{
  return view->msg_list_view;
}

static void updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_folder_view_setup(struct etpan_folder_view * view)
{
  unsigned int i;
  chashiter * iter;
  
  etpan_msg_list_view_setup(view->msg_list_view);
  
  for(iter = chash_begin(view->folder_for_orphans) ; iter != NULL ;
      iter = chash_next(view->folder_for_orphans, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    folder = value.data;
    
    etpan_msg_list_view_add_folder_for_orphans(view->msg_list_view, folder);
  }
  
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder * folder;
    
    folder = carray_get(view->folder_list, i);
    etpan_mail_manager_folder_open(etpan_mail_manager_get_default(), folder);
  }
  
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL,
      updated_handler, view);
}

void etpan_folder_view_unsetup(struct etpan_folder_view * view)
{
  unsigned int i;
  chashiter * iter;
  
  etpan_folder_view_cancel_search(view);
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL,
      updated_handler, view);
  
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder * folder;
    
    folder = carray_get(view->folder_list, i);
    etpan_mail_manager_folder_close(etpan_mail_manager_get_default(), folder);
  }

  for(iter = chash_begin(view->folder_for_orphans) ; iter != NULL ;
      iter = chash_next(view->folder_for_orphans, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    folder = value.data;
    
    etpan_msg_list_view_remove_folder_for_orphans(view->msg_list_view, folder);
  }
  
  etpan_msg_list_view_unsetup(view->msg_list_view);
}

static void update_msg_list(struct etpan_folder_view * view,
    struct etpan_folder * folder, int force);
static void build(struct etpan_folder_view * view);

static void updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_view * view;
  struct etpan_folder * folder;
  int found;
  unsigned int i;
  (void) signal_name;
  (void) sender;
  
  view = user_data;
  
  found = 0;
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    folder = carray_get(view->folder_list, i);
    if (signal_data == folder) {
      found = 1;
      break;
    }
  }
  
  if (!found)
    return;
  
  folder = signal_data;
  
  if (etpan_mail_manager_folder_get_error(etpan_mail_manager_get_default(),
          folder) == NULL)
    view->has_message_list = 1;
  update_msg_list(view, folder, 1);
  build(view);
}

static void update_msg_list(struct etpan_folder_view * view,
    struct etpan_folder * folder, int force)
{
  chash * msg_hash;
#if 0
  chash * pending_addition;
#endif
  chash * pending_deleted;
  chashiter * iter;
  chash * filtered_hash; /* messages to show */
  int r;
  carray * to_remove;
  
#if 0
  pending_addition = etpan_mail_manager_get_pending_added_messages(etpan_mail_manager_get_default(), view->folder);
#endif
  pending_deleted = etpan_mail_manager_get_pending_deleted_messages(etpan_mail_manager_get_default());
  
  msg_hash = etpan_folder_get_message_hash(folder);
  
  to_remove = carray_new(chash_count(msg_hash));
  if (to_remove == NULL)
    ETPAN_LOG_MEMORY_ERROR;
    
  filtered_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (filtered_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(msg_hash) ; iter != NULL ;
      iter = chash_next(msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    if (view->has_filter) {
      r = chash_get(view->message_list, &key, &value);
      if (r < 0) {
        r = carray_add(to_remove, key.data, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        continue;
      }
    }
    
#if 0
    if (!etpan_folder_get_show_unfiltered(folder)) {
      /* message that were not filtered will be removed */
      r = chash_get(filtered_msg_hash, &key, &value);
      if (r < 0) {
        continue;
      }
    }
#endif
    
    r = chash_get(pending_deleted, &key, &value);
    if (r == 0) {
      r = carray_add(to_remove, key.data, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      continue;
    }
    
    chash_value(iter, &value);
    msg = value.data;
    if (!etpan_message_is_filtered(msg)) {
      continue;
    }
    
    r = chash_set(filtered_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  ETPAN_LOG("XXX - update");
  etpan_log_stack();
  if (force) {
    etpan_msg_list_view_reset(view->msg_list_view);
  }
  etpan_msg_list_view_remove_messages(view->msg_list_view, to_remove);
  etpan_msg_list_view_add_messages(view->msg_list_view, filtered_hash);
  
  chash_free(filtered_hash);
  carray_free(to_remove);
  
  chash_free(pending_deleted);
#if 0
  chash_free(pending_addition);
#endif
}

static void build(struct etpan_folder_view * view)
{
  ETPAN_LOG("build");
  etpan_msg_list_view_update(view->msg_list_view);
}

void etpan_folder_view_add_folder(struct etpan_folder_view * view,
    struct etpan_folder * folder)
{
  int r;
  
  r = carray_add(view->folder_list, folder, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  etpan_folder_ref(folder);
}

void etpan_folder_view_remove_folder(struct etpan_folder_view * view,
    struct etpan_folder * folder)
{
  unsigned int i;
  int found;
  unsigned int found_index;
  chashdatum key;
  
  found_index = 0;
  found = 0;
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    if (carray_get(view->folder_list, i) == folder) {
      found = 1;
      found_index = i;
      break;
    }
  }
  
  if (!found)
    return;
  
  etpan_folder_unref(folder);
  etpan_msg_list_view_add_folder_for_orphans(view->msg_list_view, folder);
  carray_delete(view->folder_list, found_index);
  
  key.data = &folder;
  key.len = sizeof(folder);
  chash_delete(view->folder_for_orphans, &key, NULL);
}

void etpan_folder_view_set_folder(struct etpan_folder_view * view,
    struct etpan_folder * folder)
{
  int r;
  
  ETPAN_LOG("XXX - set folder");
  while (carray_count(view->folder_list) > 0) {
    struct etpan_folder * cur_folder;
    
    cur_folder = carray_get(view->folder_list, 0);
    etpan_folder_view_remove_folder(view, cur_folder);
  }
  
  if (folder != NULL) {
    struct etpan_storage * storage;
    struct etpan_folder * sent_folder;
    
    etpan_folder_view_add_folder(view, folder);
    
    storage = etpan_folder_get_storage(folder);
    sent_folder = etpan_storage_get_folder(storage, "Sent Messages");
    if (sent_folder != NULL) {
      if (sent_folder != folder) {
        chashdatum key;
        chashdatum value;
      
        etpan_folder_view_add_folder(view, sent_folder);
      
        key.data = &sent_folder;
        key.len = sizeof(sent_folder);
        value.data = sent_folder;
        value.len = 0;
        r = chash_set(view->folder_for_orphans, &key, &value, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
}

#if 0
void etpan_folder_view_set_filter(struct etpan_folder_view * view,
    carray * uid_list)
{
  unsigned int i;
  int r;
  
  view->has_filter = 1;
  chash_clear(view->filter_set);
  for(i = 0 ; i < carray_count(uid_list) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * uid;
    
    uid = carray_get(uid_list, i);
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = NULL;
    value.len = 0;
    r = chash_set(view->filter_set, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (view->has_message_list) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
      struct etpan_folder * folder;
      
      folder = carray_get(view->folder_list, i);
      update_msg_list(view, folder);
    }
    build(view);
  }
}

void etpan_folder_view_clear_filter(struct etpan_folder_view * view)
{
  view->has_filter = 0;
  chash_clear(view->filter_set);
  if (view->has_message_list) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
      struct etpan_folder * folder;
      
      folder = carray_get(view->folder_list, i);
      update_msg_list(view, folder);
    }
    build(view);
  }
}
#endif


static void search_callback(int cancelled,
    struct etpan_folder_indexer_search_result * result, void * cb_data);

struct search_callback_data {
  struct etpan_folder_view * view;
  struct etpan_folder * folder;
};

void etpan_folder_view_search_simple(struct etpan_folder_view * view,
    int search_type, char * text)
{
  unsigned int i;
  
  if (view->searching > 0) {
    ETPAN_CRASH("already searching");
  }
  
  view->searching = carray_count(view->folder_list);
  
  chash_clear(view->message_list);
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder_indexer * fi;
    struct etpan_folder * folder;
    struct search_callback_data * cb_data;
    
    folder = carray_get(view->folder_list, i);
    fi = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
        folder);
    cb_data = malloc(sizeof(* cb_data));
    if (cb_data == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    cb_data->view = view;
    cb_data->folder = folder;
    etpan_folder_search_simple(fi, search_type, text,
        search_callback, cb_data);
  }
}

void etpan_folder_view_search(struct etpan_folder_view * view,
    char * expr_headers,
    char * expr_body)
{
  unsigned int i;
  
  if (view->searching > 0) {
    ETPAN_CRASH("already searching");
  }
  
  view->searching = carray_count(view->folder_list);
  
  chash_clear(view->message_list);
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder_indexer * fi;
    struct etpan_folder * folder;
    struct search_callback_data * cb_data;
    
    folder = carray_get(view->folder_list, i);
    fi = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
        folder);
    cb_data = malloc(sizeof(* cb_data));
    if (cb_data == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    cb_data->view = view;
    cb_data->folder = folder;
    etpan_folder_search(fi, expr_headers, expr_body,
      search_callback, view);
  }
}

static void search_callback(int cancelled,
    struct etpan_folder_indexer_search_result * result, void * cb_data)
{
  struct etpan_folder_view * view;
  unsigned int i;
  int r;
  struct search_callback_data * data;
  struct etpan_folder * folder;
  
  data = cb_data;
  view = data->view;
  folder = data->folder;
  free(data);
  
  view->searching --;
  
  if (cancelled)
    return;
  
  view->has_filter = 1;
  /* add current result to global result */
  for(i = 0 ; i < carray_count(result->messages) ; i ++) {
    char * uid;
    chashdatum key;
    chashdatum value;
    
    uid = carray_get(result->messages, i);
    
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = NULL;
    value.len = 0;
    r = chash_set(view->message_list, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  update_msg_list(view, folder, 0);
  build(view);
  
  ETPAN_SIGNAL_SEND(view, ETPAN_FOLDER_VIEW_SEARCH_UPDATED_RESULT);
  
  if (view->searching == 0) {
    ETPAN_SIGNAL_SEND(view, ETPAN_FOLDER_VIEW_SEARCH_FINISHED);
  }
}

void etpan_folder_view_cancel_search(struct etpan_folder_view * view)
{
  unsigned int i;
  
  view->has_filter = 0;
  chash_clear(view->message_list);
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder_indexer * fi;
    struct etpan_folder * folder;
    
    folder = carray_get(view->folder_list, i);
    fi = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
        folder);
    
    if (etpan_folder_indexer_is_searching(fi))
      etpan_folder_indexer_cancel_search(fi);
    update_msg_list(view, folder, 0);
  }
  build(view);
}

int etpan_folder_view_is_searching(struct etpan_folder_view * view)
{
  return (view->searching > 0);
}

int etpan_folder_view_is_indexing(struct etpan_folder_view * view)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder_indexer * fi;
    struct etpan_folder * folder;
    int indexer_state;
    
    folder = carray_get(view->folder_list, i);
    fi = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
        folder);
    indexer_state = etpan_folder_indexer_indexing_state(fi);
    if (indexer_state == ETPAN_FOLDER_INDEXER_STATE_INDEXING_HEADERS)
      return 1;
  }
  
  return 0;
}

int etpan_folder_view_is_indexing_body(struct etpan_folder_view * view)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(view->folder_list) ; i ++) {
    struct etpan_folder_indexer * fi;
    struct etpan_folder * folder;
    int indexer_state;
    
    folder = carray_get(view->folder_list, i);
    fi = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
        folder);
    indexer_state = etpan_folder_indexer_indexing_state(fi);
    if ((indexer_state == ETPAN_FOLDER_INDEXER_STATE_INDEXING_BODY) ||
        (indexer_state == ETPAN_FOLDER_INDEXER_STATE_INDEXING_HEADERS))
      return 1;
  }
  
  return 0;
}
