#include "etpan-folder.h"
#include "etpan-folder-private.h"

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "etpan-nls.h"
#include "etpan-message.h"
#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-storage-private.h"
#include "etpan-message.h"
#include "etpan-message-private.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"

#define ETPAN_MODULE_LOG_NAME "FOLDER"
#include "etpan-log.h"

static void etpan_folder_free(struct etpan_folder * folder);
static void etpan_folder_unsetup(struct etpan_folder * folder);

static struct etpan_error * folder_connect(struct etpan_folder * folder);
static void folder_disconnect(struct etpan_folder * folder);

static int folder_count = 0;

void etpan_folder_print_stats(void)
{
  ETPAN_WARN_LOG("folder count : %i", folder_count);
}

struct etpan_folder * etpan_folder_new(void)
{
  struct etpan_folder * folder;
  
  folder_count ++;
  
  folder = malloc(sizeof(* folder));
  if (folder == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder->ref_count = 1;
  folder->storage = NULL;
  folder->location = NULL;
  folder->threaded_location = NULL;
  folder->uid = NULL;
  folder->show_unfiltered = 1;
  folder->max = 0;
  folder->lost = 0;
  folder->default_recipient = NULL;
  folder->special_folder_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (folder->special_folder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder->ui_name = NULL;
  folder->ui_path = NULL;
  folder->poll = 0;
  folder->info_fetched = 0;
  folder->total = 0;
  folder->recent = 0;
  folder->unseen = 0;
  
  folder->message_list_fetched = 0;
  folder->message_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder->message_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder->data = NULL;
  folder->driver = NULL;
  
  folder->ref_msg_list = 0;
  
  return folder;
}

static void message_list_free(chash * msglist);

static void etpan_folder_free(struct etpan_folder * folder)
{
  folder_count --;
  if (folder_count < 0)
    ETPAN_WARN_LOG("folder count under zero");
  
  etpan_folder_unsetup(folder);
  
  folder_disconnect(folder);
  
  if (folder->driver->free_data != NULL)
    folder->driver->free_data(folder);
  
  if (folder->ref_msg_list > 0) {
    ETPAN_WARN_LOG("ref msg list is > 0 while freeing folder %s", folder->ui_path);
    etpan_log_stack();
  }
  
  message_list_free(folder->message_list);
  
  free(folder->ui_path);
  free(folder->ui_name);
  
  chash_free(folder->special_folder_list);
  
  free(folder->uid);
  free(folder->threaded_location);
  free(folder->location);
  
  free(folder);
}

void etpan_folder_ref(struct etpan_folder * folder)
{
  folder->ref_count ++;
#if 0
  ETPAN_LOG("folder ref %s %i", etpan_folder_get_ui_path(folder),
      folder->ref_count);
  etpan_log_stack();
#endif
}

void etpan_folder_unref(struct etpan_folder * folder)
{
  folder->ref_count --;
#if 0
  ETPAN_LOG("folder unref %s %i",
      etpan_folder_get_ui_path(folder), folder->ref_count);
  etpan_log_stack();
#endif
  if (folder->ref_count == 0)
    etpan_folder_free(folder);
}

void etpan_folder_set_data(struct etpan_folder * folder,
    void * data)
{
  folder->data = data;
}

void * etpan_folder_get_data(struct etpan_folder * folder)
{
  return folder->data;
}

void etpan_folder_set_driver(struct etpan_folder * folder,
    struct etpan_folder_driver * driver)
{
  folder->driver = driver;
}

struct etpan_folder_driver *
etpan_folder_get_driver(struct etpan_folder * folder)
{
  return folder->driver;
}

void etpan_folder_set_storage(struct etpan_folder * folder,
    struct etpan_storage * storage)
{
  folder->storage = storage;
}

void etpan_folder_set_location(struct etpan_folder * folder, char * location)
{
  if (location != folder->location) {
    free(folder->location);
    if (location != NULL) {
      folder->location = strdup(location);
      if (folder->location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder->location = NULL;
  }
}

void etpan_folder_set_default_recipient(struct etpan_folder * folder,
    char * default_recipient)
{
  if (default_recipient != folder->default_recipient) {
    free(folder->default_recipient);
    if (default_recipient != NULL) {
      folder->default_recipient = strdup(default_recipient);
      if (folder->default_recipient == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder->default_recipient = NULL;
  }
}

void etpan_folder_set_max(struct etpan_folder * folder,
    unsigned int max)
{
  folder->max = max;
}

void etpan_folder_set_special_folder(struct etpan_folder * folder,
    char * name, struct etpan_folder * special_folder)
{
  etpan_folder_set_special_folder_by_location(folder,
      name, special_folder->location);
}

void etpan_folder_set_special_folder_by_location(struct etpan_folder * folder,
    char * name, char * location)
{
  chashdatum key;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  
  if (location == NULL) {
    chash_delete(folder->special_folder_list, &key, NULL);
  }
  else {
    chashdatum value;
    
    value.data = location;
    value.len = strlen(location + 1);
    
    r = chash_set(folder->special_folder_list, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

void etpan_folder_set_draft_by_location(struct etpan_folder * folder,
    char * draft)
{
  etpan_folder_set_special_folder_by_location(folder,
      SPECIAL_FOLDER_DRAFT, draft);
}

void etpan_folder_set_sent_by_location(struct etpan_folder * folder,
    char * sent)
{
  etpan_folder_set_special_folder_by_location(folder,
      SPECIAL_FOLDER_SENT, sent);
}

void etpan_folder_set_trash_by_location(struct etpan_folder * folder,
    char * trash)
{
  etpan_folder_set_special_folder_by_location(folder,
      SPECIAL_FOLDER_TRASH, trash);
}

void etpan_folder_set_draft(struct etpan_folder * folder,
    struct etpan_folder * draft)
{
  etpan_folder_set_special_folder(folder,
      SPECIAL_FOLDER_DRAFT, draft);
}

void etpan_folder_set_sent(struct etpan_folder * folder,
    struct etpan_folder * sent)
{
  etpan_folder_set_special_folder(folder,
      SPECIAL_FOLDER_SENT, sent);
}

void etpan_folder_set_trash(struct etpan_folder * folder,
    struct etpan_folder * trash)
{
  etpan_folder_set_special_folder(folder,
      SPECIAL_FOLDER_TRASH, trash);
}

void etpan_folder_set_ui_path(struct etpan_folder * folder,
    char * ui_path)
{
  char * ui_name;
  
  if (ui_path != folder->ui_path) {
    free(folder->ui_path);
    if (ui_path != NULL) {
      folder->ui_path = strdup(ui_path);
      if (folder->ui_path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
   }
    else
      folder->ui_path = NULL;
  }
  
  /* generate ui_name */
  ui_name = strrchr(folder->ui_path, '/');
  if (ui_name == NULL)
    ui_name = folder->ui_path;
  else
    ui_name ++;
  
  if (folder->ui_name != NULL)
    free(folder->ui_name);
  folder->ui_name = strdup(ui_name);
  if (folder->ui_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_folder_set_poll(struct etpan_folder * folder,
    int do_poll)
{
  folder->poll = do_poll;
}

void etpan_folder_set_count(struct etpan_folder * folder,
    unsigned int total,
    unsigned int unseen,
    unsigned int recent)
{
  folder->info_fetched = 1; 
  folder->total = total;
  folder->unseen = unseen;
  folder->recent = recent;
  ETPAN_SIGNAL_SEND(folder, ETPAN_FOLDER_COUNT_UPDATED);
}

void etpan_folder_clear_count(struct etpan_folder * folder)
{
  folder->info_fetched = 0;
  ETPAN_SIGNAL_SEND(folder, ETPAN_FOLDER_COUNT_UPDATED);
}

void etpan_folder_set_lost(struct etpan_folder * folder,
    int lost)
{
  folder->lost = lost;
}

struct etpan_storage * etpan_folder_get_storage(struct etpan_folder * folder)
{
  return folder->storage;
}

char * etpan_folder_get_location(struct etpan_folder * folder)
{
  return folder->location;
}

char * etpan_folder_get_default_recipient(struct etpan_folder * folder)
{
  return folder->default_recipient;
}

unsigned int etpan_folder_get_max(struct etpan_folder * folder)
{
  return folder->max;
}

struct etpan_folder *
etpan_folder_get_special_folder(struct etpan_folder * folder,
    char * name)
{
  char * location;
  
  location = etpan_folder_get_special_folder_location(folder, name);
  
  return etpan_storage_get_folder(folder->storage, location);
}

char * etpan_folder_get_special_folder_location(struct etpan_folder * folder,
    char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  
  r = chash_get(folder->special_folder_list, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

char * etpan_folder_get_draft_location(struct etpan_folder * folder)
{
  return etpan_folder_get_special_folder_location(folder, SPECIAL_FOLDER_DRAFT);
}

char * etpan_folder_get_sent_location(struct etpan_folder * folder)
{
  return etpan_folder_get_special_folder_location(folder, SPECIAL_FOLDER_SENT);
}

char * etpan_folder_get_trash_location(struct etpan_folder * folder)
{
  return etpan_folder_get_special_folder_location(folder, SPECIAL_FOLDER_TRASH);
}

struct etpan_folder * etpan_folder_get_draft(struct etpan_folder * folder)
{
  return etpan_folder_get_special_folder(folder,
      SPECIAL_FOLDER_DRAFT);
}

struct etpan_folder * etpan_folder_get_sent(struct etpan_folder * folder)
{
  return etpan_folder_get_special_folder(folder,
      SPECIAL_FOLDER_SENT);
}

struct etpan_folder * etpan_folder_get_trash(struct etpan_folder * folder)
{
  return etpan_folder_get_special_folder(folder,
      SPECIAL_FOLDER_TRASH);
}

char * etpan_folder_get_ui_path(struct etpan_folder * folder)
{
  return folder->ui_path;
}

char * etpan_folder_get_ui_name(struct etpan_folder * folder)
{
  return folder->ui_name;
}

int etpan_folder_get_poll(struct etpan_folder * folder)
{
  return folder->poll;
}

int etpan_folder_get_count(struct etpan_folder * folder,
    unsigned int * p_total,
    unsigned int * p_unseen,
    unsigned int * p_recent)
{
  if (!folder->info_fetched)
    return 0;
  
  * p_total = folder->total;
  * p_unseen = folder->unseen;
  * p_recent = folder->recent;
  
  return 1;
}

int etpan_folder_is_lost(struct etpan_folder * folder)
{
  return folder->lost;
}

void etpan_folder_set_uid(struct etpan_folder * folder, char * uid)
{
  if (uid != folder->uid) {
    free(folder->uid);
    if (uid != NULL) {
      folder->uid = strdup(uid);
      if (folder->uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder->uid = NULL;
  }
}

char * etpan_folder_get_uid(struct etpan_folder * folder)
{
  return folder->uid;
}

void etpan_folder_set_show_unfiltered(struct etpan_folder * folder, int value)
{
  folder->show_unfiltered = value;
}

int etpan_folder_get_show_unfiltered(struct etpan_folder * folder)
{
  return folder->show_unfiltered;
}

/* network access */

/* get msg list */

struct etpan_folder_fetch_msg_list_param {
  struct etpan_folder * folder;
};

static void message_list_free(chash * msglist)
{
  chashiter * iter;
  
  for(iter = chash_begin(msglist) ; iter != NULL ;
      iter = chash_next(msglist, iter)) {
    chashdatum value;
    struct etpan_message * message;
    
    chash_value(iter, &value);
    message = value.data;
    etpan_message_unref(message);
  }
  chash_free(msglist);
}

static void
fetch_msg_list_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_fetch_msg_list_param * param;
  struct etpan_folder_fetch_msg_list_result * result;
  
  param = op->param;
  etpan_folder_unref(param->folder);
  free(param);
  
  result = op->result;
  if (result->current_message_list != NULL)
    message_list_free(result->current_message_list);
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static void
fetch_msg_list_run(struct etpan_thread_op * op)
{
  struct etpan_folder_fetch_msg_list_param * param;
  struct etpan_folder_fetch_msg_list_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    if (param->folder->driver->fetch_msg_list != NULL) {
      error = param->folder->driver->fetch_msg_list(param->folder,
          result->current_message_list);
      if (error == NULL) {
        chashiter * iter;
        for(iter = chash_begin(result->current_message_list) ;
            iter != NULL ;
            iter = chash_next(result->current_message_list, iter)) {
          chashdatum value;
          struct etpan_message * msg;
          
          chash_value(iter, &value);
          msg = value.data;
          etpan_message_compute_description(msg);
        }
      }
    }
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("fetch-msg-list is not implemented for %s"),
          param->folder->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_folder_fetch_msg_list(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    void (* callback)(int, struct etpan_folder_fetch_msg_list_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_fetch_msg_list_param * param;
  struct etpan_folder_fetch_msg_list_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result->current_message_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (result->current_message_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_msg_list_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = fetch_msg_list_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

void etpan_folder_update_msg_list(struct etpan_folder * folder,
    chash * current_message_list)
{
  chashiter * iter;
  int r;
  unsigned int new_count;
  
  ETPAN_LOG("folder update msg list: %i", chash_count(current_message_list));
  new_count = 0;
  for(iter = chash_begin(current_message_list) ; iter != NULL ;
      iter = chash_next(current_message_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    
    r = chash_get(folder->message_list, &key, &value);
    if (r == 0) {
      struct etpan_message * old_msg;
      
      old_msg = value.data;
      if (!etpan_message_has_dirty_flags(old_msg)) {
        struct etpan_message_flags * flags;
        
        chash_value(iter, &value);
        msg = value.data;
        flags = etpan_message_get_flags(msg);
        etpan_message_set_flags(old_msg, flags);
        etpan_message_set_clean(old_msg);
      }
      continue;
    }
    
    chash_value(iter, &value);
    msg = value.data;
    
    etpan_message_ref(msg);
    r = chash_set(folder->message_list, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    new_count ++;
  }
  
  for(iter = chash_begin(folder->message_list) ; iter != NULL ;
      iter = chash_next(folder->message_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    
    chash_value(iter, &value);
    msg = value.data;
    
    r = chash_get(current_message_list, &key, &value);
    if (r == 0)
      continue;
    
    etpan_message_set_lost(msg, 1);
  }
  
#if 0 /* XXX */
  if (folder->message_list_fetched == 0)
    folder->message_list_fetched ++;
#endif
  folder->message_list_fetched = 1;
  if (new_count > 0) {
    ETPAN_SIGNAL_SEND(folder, ETPAN_FOLDER_MSGLIST_UPDATED);
  }
}

int etpan_folder_has_message_list(struct etpan_folder * folder)
{
  return folder->message_list_fetched;
}

void etpan_folder_remove_lost_messages(struct etpan_folder * folder)
{
  chashiter * iter;
  carray * msg_to_remove;
  unsigned int i;
  int r;
  
  msg_to_remove = carray_new(16);
  if (msg_to_remove == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(iter = chash_begin(folder->message_list) ; iter != NULL ;
      iter = chash_next(folder->message_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    char * uid;
    
    chash_key(iter, &key);
    uid = key.data;
    
    chash_value(iter, &value);
    msg = value.data;

    if (!etpan_message_is_lost(msg))
      continue;
    
    etpan_message_unref(msg);
    
    r = carray_add(msg_to_remove, uid, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  for(i = 0 ; i < carray_count(msg_to_remove) ; i ++) {
    char * uid;
    chashdatum key;
    
    uid = carray_get(msg_to_remove, i);
    
    key.data = uid;
    key.len = strlen(uid) + 1;
    chash_delete(folder->message_list, &key, NULL);
  }
  
  if (carray_count(msg_to_remove) > 0) {
    ETPAN_SIGNAL_SEND(folder, ETPAN_FOLDER_MSGLIST_UPDATED);
  }
  
  carray_free(msg_to_remove);
}

struct etpan_thread_op *
unref_in_thread(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    carray * msg_to_unref,
    void (* callback)(int, void *, void *),
    void * cb_data);

static void etpan_folder_clear_msg_list(struct etpan_folder * folder)
{
  chashiter * iter;
  carray * msg_to_remove;
  unsigned int i;
  int r;

#if 0
  folder->message_list_fetched --;
  
  if (folder->message_list_fetched > 0)
    return;
#endif
  folder->message_list_fetched = 0;
  
  msg_to_remove = carray_new(16);
  if (msg_to_remove == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(iter = chash_begin(folder->message_list) ; iter != NULL ;
      iter = chash_next(folder->message_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    char * uid;
    
    chash_key(iter, &key);
    uid = key.data;
    
    chash_value(iter, &value);
    msg = value.data;

    r = carray_add(msg_to_remove, msg, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  for(i = 0 ; i < carray_count(msg_to_remove) ; i ++) {
    char * uid;
    chashdatum key;
    struct etpan_message * msg;
    
    msg = carray_get(msg_to_remove, i);
    uid = etpan_message_get_uid(msg);
    
    key.data = uid;
    key.len = strlen(uid) + 1;
    chash_delete(folder->message_list, &key, NULL);
    
  }
  
  unref_in_thread(etpan_thread_manager_app_get_default(),
      folder, msg_to_remove, NULL, NULL);
  
#if 0
  for(i = 0 ; i < carray_count(msg_to_remove) ; i ++) {
    struct etpan_message * msg;
    
    msg = carray_get(msg_to_remove, i);
    etpan_message_unref(msg);
  }
#endif
  
  carray_free(msg_to_remove);
}

struct etpan_unref_param {
  carray * msg_to_unref;
};

static void
folder_unref_cleanup(struct etpan_thread_op * op)
{
  struct etpan_unref_param * param;
    
  param = op->param;
  carray_free(param->msg_to_unref);
  free(param);
}

static void
folder_unref_run(struct etpan_thread_op * op)
{
  struct etpan_unref_param * param;
  unsigned int i;
  
  param = op->param;
  for(i = 0 ; i < carray_count(param->msg_to_unref) ; i ++) {
    struct etpan_message * msg;
    
    msg = carray_get(param->msg_to_unref, i);
    etpan_message_unref(msg);
  }
}

struct etpan_thread_op *
unref_in_thread(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    carray * msg_to_unref,
    void (* callback)(int, void *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_unref_param * param;
  int r;
  unsigned int i;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->msg_to_unref = carray_new(carray_count(msg_to_unref));
  if (param->msg_to_unref == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(msg_to_unref) ; i ++) {
    r = carray_add(param->msg_to_unref, carray_get(msg_to_unref, i), NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = folder_unref_run;
  op->callback = callback;
  op->callback_data = cb_data;
  op->cleanup = folder_unref_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

void etpan_folder_remove_msg(struct etpan_folder * folder,
    struct etpan_message * msg)
{
  chashdatum key;
  
  key.data = msg->uid;
  key.len = strlen(msg->uid) + 1;
  
  chash_delete(folder->message_list, &key, NULL);
  
  etpan_message_unref(msg);
}

/* check */

struct etpan_folder_check_param {
  struct etpan_folder * folder;
};

static void
folder_check_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_check_param * param;
  struct etpan_folder_check_result * result;
  
  param = op->param;
  etpan_folder_unref(param->folder);
  free(param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static void
folder_check_run(struct etpan_thread_op * op)
{
  struct etpan_folder_check_param * param;
  struct etpan_folder_check_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    if (param->folder->driver->check != NULL)
      error = param->folder->driver->check(param->folder);
    else
      error = NULL;
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_folder_check(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    void (* callback)(int, struct etpan_folder_check_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_check_param * param;
  struct etpan_folder_check_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = folder_check_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = folder_check_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

/* status */

struct etpan_folder_status_param {
  struct etpan_folder * folder;
};

static void
folder_status_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_status_result * result;
  struct etpan_folder_status_param * param;
  
  param = op->param;
  etpan_folder_unref(param->folder);
  free(param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static void
folder_status_run(struct etpan_thread_op * op)
{
  struct etpan_folder_status_param * param;
  struct etpan_folder_status_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    if (param->folder->driver->status != NULL)
      error = param->folder->driver->status(param->folder,
          &result->count, &result->unseen, &result->recent);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("status is not implemented for %s"),
          param->folder->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_folder_status(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    void (* callback)(int, struct etpan_folder_status_result *,
        void *), void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_status_param * param;
  struct etpan_folder_status_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  result->count = 0;
  result->unseen = 0;
  result->recent = 0;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = folder_status_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = folder_status_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op); 
  
  return op;
}

/* expunge */

struct etpan_folder_expunge_param {
  struct etpan_folder * folder;
};


static void
folder_expunge_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_expunge_param * param;
  struct etpan_folder_expunge_result * result;
  
  param = op->param;
  etpan_folder_unref(param->folder);
  free(param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static void
folder_expunge_run(struct etpan_thread_op * op)
{
  struct etpan_folder_expunge_param * param;
  struct etpan_folder_expunge_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    if (param->folder->driver->expunge != NULL)
      error = param->folder->driver->expunge(param->folder);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("expunge is not implemented for %s"),
          param->folder->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}


struct etpan_thread_op *
etpan_folder_expunge(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    void (* callback)(int, struct etpan_folder_expunge_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_expunge_param * param;
  struct etpan_folder_expunge_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = folder_expunge_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = folder_expunge_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

/* append */

struct etpan_folder_append_msg_param {
  struct etpan_folder * folder;
  MMAPString * message;
  struct etpan_message_flags * flags;
};

static void
folder_append_msg_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_append_msg_param * param;
  struct etpan_folder_append_msg_result * result;
  
  param = op->param;
  etpan_message_flags_free(param->flags);
  mmap_string_free(param->message);
  etpan_folder_unref(param->folder);
  free(param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static void
folder_append_msg_run(struct etpan_thread_op * op)
{
  struct etpan_folder_append_msg_param * param;
  struct etpan_folder_append_msg_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    if (param->folder->driver->append_msg != NULL)
      error = param->folder->driver->append_msg(param->folder,
          param->message->str,
          param->message->len,
          param->flags);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("append-msg is not implemented for %s"),
          param->folder->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_folder_append(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    char * message, size_t length,
    struct etpan_message_flags * flags,
    void (* callback)(int, struct etpan_folder_append_msg_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_append_msg_param * param;
  struct etpan_folder_append_msg_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  param->message = mmap_string_new_len(message, length);
  if (param->message == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->flags = etpan_message_flags_dup(flags);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = folder_append_msg_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = folder_append_msg_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

/* append file */

struct etpan_folder_append_file_msg_param {
  struct etpan_folder * folder;
  char * filename;
  struct etpan_message_flags * flags;
};

static void
folder_append_file_msg_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_append_file_msg_param * param;
  struct etpan_folder_append_msg_result * result;
  
  param = op->param;
  etpan_message_flags_free(param->flags);
  free(param->filename);
  etpan_folder_unref(param->folder);
  free(op->param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static struct etpan_error * append_file(struct etpan_folder * folder,
    char * filename, struct etpan_message_flags * flags)
{
  int r;
  struct stat stat_buf;
  int fd;
  char * data;
  size_t len;
  int res;
  struct etpan_error * error;
  
  r = stat(filename, &stat_buf);
  if (r < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not add a message"));
    etpan_error_strf_long_description(error,
        _("The file (%s) containing the message to add was not found."),
        filename);
    res = ERROR_FILE;
    goto err;
  }
  
  len = stat_buf.st_size;
  
  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not add a message"));
    etpan_error_strf_long_description(error,
        _("The file (%s) containing the message could not be read."),
        filename);
    res = ERROR_FILE;
    goto err;
  }
  
  data = mmap(NULL, len, PROT_READ, MAP_PRIVATE, fd, 0);
  if (data == MAP_FAILED) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not add a message"));
    etpan_error_strf_long_description(error,
        _("The file (%s) containing the message could not be read."),
        filename);
    res = ERROR_FILE;
    goto close;
  }
  
  error = folder->driver->append_msg(folder, data, len, flags);
  if (error != NULL) {
    goto unmap;
  }
  
  munmap(data, len);
  close(fd);
  
  return NULL;
  
 unmap:
  munmap(data, len);
 close:
  close(fd);
 err:
  return error;
}

static void
folder_append_file_msg_run(struct etpan_thread_op * op)
{
  struct etpan_folder_append_file_msg_param * param;
  struct etpan_folder_append_msg_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    if (param->folder->driver->append_msg != NULL) {
      error = append_file(param->folder, param->filename, param->flags);
    }
    else {
      error = etpan_error_internal_strf(_("append-msg is not implemented for %s"),
          param->folder->driver->name);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_folder_append_file(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    char * filename,
    struct etpan_message_flags * flags,
    void (* callback)(int, struct etpan_folder_append_msg_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_append_file_msg_param * param;
  struct etpan_folder_append_msg_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  param->filename = strdup(filename);
  if (param->filename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->flags = etpan_message_flags_dup(flags);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = folder_append_file_msg_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = folder_append_file_msg_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

/* check msg list */

struct etpan_folder_check_msg_list_param {
  struct etpan_folder * folder;
  chash * flags_hash;
  chash * msg_list;
};

static void
check_msg_list_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_check_msg_list_param * param;
  chashiter * iter;
  struct etpan_folder_check_msg_list_result * result; 
  
  /* free param */
  param = op->param;
  if (param->msg_list != NULL) {
    message_list_free(param->msg_list);
  }
  
  for(iter = chash_begin(param->flags_hash) ; iter != NULL ;
      iter = chash_next(param->flags_hash, iter)) {
    chashdatum value;
    struct etpan_message_flags * flags;
    
    chash_value(iter, &value);
    flags = value.data;
    etpan_message_flags_free(flags);
  }
  chash_free(param->flags_hash);
  
  etpan_folder_unref(param->folder);
  free(param);
  
  /* free result */
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}


static void
check_msg_list_run(struct etpan_thread_op * op)
{
  struct etpan_folder_check_msg_list_param * param;
  struct etpan_folder_check_msg_list_result * result; 
  chashiter * iter;
  int r;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->folder);
  if (error == NULL) {
    for(iter = chash_begin(param->msg_list) ; iter != NULL ;
        iter = chash_next(param->msg_list, iter)) {
      chashdatum key;
      chashdatum value;
      struct etpan_message * msg;
      struct etpan_message_flags * flags;
      struct etpan_error * msg_error;
      
      chash_key(iter, &key);
      
      r = chash_get(param->flags_hash, &key, &value);
      if (r < 0)
        continue;
      
      flags = value.data;
      chash_value(iter, &value);
      msg = value.data;
      msg_error = etpan_message_check(msg, flags);
      ETPAN_ERROR_IGNORE(msg_error);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_folder_check_msg_list(struct etpan_thread_manager_app * manager,
    struct etpan_folder * folder,
    chash * message_list,
    void (* callback)(int, struct etpan_folder_check_msg_list_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_check_msg_list_param * param;
  struct etpan_folder_check_msg_list_result * result;
  chashiter * iter;
  int r;
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->folder = folder;
  etpan_folder_ref(param->folder);
  param->msg_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (param->msg_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(iter = chash_begin(message_list) ; iter != NULL ;
      iter = chash_next(message_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    etpan_message_ref(msg);
    r = chash_set(param->msg_list, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  param->flags_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (param->flags_hash == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(iter = chash_begin(param->msg_list) ; iter != NULL ;
      iter = chash_next(param->msg_list, iter)) {
    struct etpan_message_flags * flags;
    struct etpan_message_flags * dup_flags;
    int r;
    chashdatum key;
    chashdatum value;
    char * uid;
    struct etpan_message * msg;
    
    chash_value(iter, &value);
    msg = value.data;
    
    uid = etpan_message_get_uid(msg);
    if (uid == NULL)
      continue;
    
    if (!etpan_message_has_dirty_flags(msg))
      continue;
    
    flags = etpan_message_get_flags(msg);
    if (flags == NULL)
      continue;

    dup_flags = etpan_message_flags_dup(flags);
    
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = dup_flags;
    value.len = 0;
    r = chash_set(param->flags_hash, &key, &value, NULL);
    if (r < 0) {
      etpan_message_flags_free(dup_flags);
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    etpan_message_set_clean(msg);
  }
  ETPAN_LOG("%i messages to check", chash_count(param->flags_hash));
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = check_msg_list_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = check_msg_list_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, folder->storage, op);
  
  return op;
}

struct etpan_message *
etpan_folder_get_message(struct etpan_folder * folder, char * uid)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = uid;
  key.len = strlen(uid) + 1;
  
  r = chash_get(folder->message_list, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

chash *
etpan_folder_get_message_hash(struct etpan_folder * folder)
{
  return folder->message_list;
}


struct etpan_error * etpan_folder_setup(struct etpan_folder * folder)
{
  struct etpan_error * error;
  
  if (folder->driver->setup != NULL) {
    error = folder->driver->setup(folder);
    if (error != NULL)
      return error;
  }
  
  free(folder->threaded_location);
  if (folder->location != NULL) {
    folder->threaded_location = strdup(folder->location);
    if (folder->threaded_location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return NULL;
}

static void etpan_folder_unsetup(struct etpan_folder * folder)
{
  (void) folder;
}

char * etpan_folder_get_threaded_location(struct etpan_folder * folder)
{
  return folder->threaded_location;
}

static struct etpan_error * folder_connect(struct etpan_folder * folder)
{
  struct etpan_error * error;
  
  error = etpan_storage_connect_nt(folder->storage);
  if (error == NULL) {
    /* try to connect */
    if (folder->driver->connect != NULL)
      error = folder->driver->connect(folder);
    else
      error = NULL;
    
    if (error == NULL)
      etpan_storage_mark_connected_folder(folder->storage,
          folder, 1);
  }
  
  /* disconnect everything if it failed */
  if (error != NULL) {
    if (etpan_error_has_code(error, ERROR_STREAM)) {
      ETPAN_ERROR_FREE(error);
    
      ETPAN_LOG("reconnection of %s", etpan_folder_get_ui_path(folder));
    
      etpan_storage_disconnect_nt(folder->storage);
    
      error = etpan_storage_connect_nt(folder->storage);
      if (error == NULL) {
	/* retry connection */
	if (folder->driver->connect != NULL)
	  error = folder->driver->connect(folder);
	else
	  error = NULL;
	ETPAN_LOG("reconnection of %s", etpan_folder_get_ui_path(folder));
      
	if (error == NULL)
	  etpan_storage_mark_connected_folder(folder->storage,
					      folder, 1);
      }
    }
  }
  
  return error;
}

static void folder_disconnect(struct etpan_folder * folder)
{
  if (folder->driver->disconnect != NULL)
    folder->driver->disconnect(folder);
  
  etpan_storage_mark_connected_folder(folder->storage,
      folder, 0);
}

struct etpan_error * etpan_folder_connect_nt(struct etpan_folder * folder)
{
  return folder_connect(folder);
}

void etpan_folder_disconnect_nt(struct etpan_folder * folder)
{
  folder_disconnect(folder);
}

int etpan_folder_is_sub_folder(struct etpan_folder * parent,
    struct etpan_folder * child)
{
  if (parent->storage != child->storage)
    return 0;
  
  if (parent->driver->is_sub_folder == NULL)
    return 0;
  
  return parent->driver->is_sub_folder(parent, child);
}

void etpan_folder_ref_msg_list(struct etpan_folder * folder)
{
  folder->ref_msg_list ++;
}

void etpan_folder_unref_msg_list(struct etpan_folder * folder)
{
  folder->ref_msg_list --;
  if (folder->ref_msg_list == 0)
    etpan_folder_clear_msg_list(folder);
}
