#include "etpan-global-config.h"

#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <paths.h>

#include "etpan-tc.h"
#include "etpan-error.h"
#include "etpan-config-types.h"
#include "etpan-nls.h"
#include "etpan-log.h"

static struct etpan_global_config * default_config = NULL;

void etpan_global_config_set_default(struct etpan_global_config * config)
{
  default_config = config;
}

struct etpan_global_config * etpan_global_config_get_default(void)
{
  return default_config;
}

static inline char * get_entry(tcconf_section_t * conf, char * name)
{
  int r;
  char * value;
  
  r = tcconf_getvalue(conf, name, "%s", &value);
  if (r <= 0)
    return NULL;
  
  return value;
}

struct etpan_error *
etpan_global_config_read(struct etpan_global_config * config,
    char * filename)
{
  tcconf_section_t * sec;
  char * attachment_folder;
  
  sec = tcconf_load_file(NULL, filename);
  if (sec == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Global configuration at %s could not be parsed."), filename);
    
    return error;
  }
  
  attachment_folder = get_entry(sec, "attachment-folder");
  etpan_global_config_set_attachment_folder(config, attachment_folder);
  free(attachment_folder);
  
  tcfree(sec);
  
  return NULL;
}

static void set_default_values(struct etpan_global_config * config)
{
  char * home;
  
  /* set default values */
  
  home = getenv("HOME");
  if (home == NULL) {
    etpan_global_config_set_attachment_folder(config, _PATH_TMP);
    return;
  }
  
  if (config->attachment_folder == NULL) {
    char path[PATH_MAX];
    
    snprintf(path, sizeof(path), "%s/Desktop", home);
    etpan_global_config_set_attachment_folder(config, path);
  }
}

struct etpan_error * etpan_global_config_read_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_global_config * config;
  struct etpan_error * error;
  
  config = etpan_global_config_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s",
      home, ETPAN_GLOBAL_CONFIG_FILE);
  error = etpan_global_config_read(config, filename);
  
  set_default_values(config);
  
  return error;
}

struct etpan_global_config * etpan_global_config_new(void)
{
  struct etpan_global_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  config->attachment_folder = NULL;
  
  return config;
}

void etpan_global_config_free(struct etpan_global_config * config)
{
  free(config->attachment_folder);
  free(config);
}

void etpan_global_config_set_attachment_folder(struct etpan_global_config * config,
    char * attachment_folder)
{
  if (attachment_folder != config->attachment_folder) {
    free(config->attachment_folder);
    if (attachment_folder != NULL) {
      config->attachment_folder = strdup(attachment_folder);
      if (config->attachment_folder == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      config->attachment_folder = NULL;
  }
}

char * etpan_global_config_get_attachment_folder(struct etpan_global_config * config)
{
  return config->attachment_folder;
}


struct etpan_error *
etpan_global_config_save(struct etpan_global_config * config,
    char * filename)
{
  FILE * f;
  char * quoted_attachment_folder;
  
  f = fopen(filename, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Configuration file could not be written"));
    etpan_error_strf_long_description(error, _("Global configuration file %s could not be written to disk."), filename);
    
    return error;
  }
  
  quoted_attachment_folder = etpan_quote_string(config->attachment_folder);
  fprintf(f, "  attachment-folder \"%s\"\n", quoted_attachment_folder);
  free(quoted_attachment_folder);
  
  fclose(f);
  
  return NULL;
}

struct etpan_error * etpan_global_config_save_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_global_config * config;
  struct etpan_error * error;
  
  config = etpan_global_config_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s",
      home, ETPAN_GLOBAL_CONFIG_FILE);
  error = etpan_global_config_save(config, filename);
  
  return error;
}
