#include "etpan-lep.h"

#include <string.h>
#include <stdlib.h>
#include <pthread.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-message-lep.h"
#include "etpan-address.h"
#include "etpan-message-header.h"
#include "etpan-utils.h"
#include "etpan-config-types.h"
#include "etpan-uri.h"
#include "etpan-nls.h"
#include "etpan-folder.h"
#include "etpan-part-header.h"

#define DEFAULT_INCOMING_CHARSET "iso-8859-1"
#define DEFAULT_DISPLAY_CHARSET "utf-8"

enum {
  ERROR_DOMAIN_FETCH_MSG_LIST,
  ERROR_DOMAIN_FETCH_MSG,
};

static struct etpan_error *
folder_translate_error(struct etpan_folder * folder,
    int domain, int error_code);
static struct etpan_error * message_translate_error(int domain, int error_code);

/* fetch msg list */

struct etpan_error * etpan_lep_fetch_msg_list(struct etpan_folder * folder,
    struct mailfolder * ep_folder,
    chash * current_msg_list)
{
  struct mailmessage_list * msg_list;
  unsigned int count;
  int r;
  struct etpan_error * error;
  
  r = mailfolder_get_messages_list(ep_folder, &msg_list);
  if (r != MAIL_NO_ERROR) {
    error = folder_translate_error(folder, ERROR_DOMAIN_FETCH_MSG_LIST, r);
    goto err;
  }
  
  r = mailfolder_get_envelopes_list(ep_folder, msg_list);
  if (r != MAIL_NO_ERROR) {
    error = folder_translate_error(folder, ERROR_DOMAIN_FETCH_MSG_LIST, r);
    goto free_msg_list;
  }
  
  /* convert data types */
  count = carray_count(msg_list->msg_tab);
  while (count > 0) {
    struct mailmessage * ep_msg;
    
    ep_msg = carray_get(msg_list->msg_tab, 0);
    carray_delete(msg_list->msg_tab, 0);
    count --;
    
    if ((ep_msg->msg_flags->fl_flags & MAIL_FLAG_CANCELLED) != 0) {
      continue;
    }
    
    if (ep_msg->msg_uid != NULL) {
      chashdatum key;
      chashdatum value;
      struct etpan_message * msg;
      char * uid;
      
      msg = etpan_message_lep_new();
      
      /* this must ref the message */
      etpan_message_set_folder(msg, folder);
      etpan_message_lep_set_msg(msg, ep_msg);
      
      uid = etpan_message_get_uid(msg); 
      key.data = uid;
      key.len = strlen(uid) + 1;
      value.data = msg;
      value.len = 0;
      
      r = chash_set(current_msg_list, &key, &value, NULL);
      if (r < 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  carray_set_size(msg_list->msg_tab, 0);
  mailmessage_list_free(msg_list);
  
  return NULL;
  
 free_msg_list:
  mailmessage_list_free(msg_list);
 err:
  return error;
}

/* engine */

static struct mailprivacy * _ep_privacy = NULL;

struct mailprivacy * etpan_lep_privacy(void)
{
  return _ep_privacy;
}

struct etpan_error * etpan_lep_engine_setup(void)
{
  struct mailprivacy * privacy;
  int r;
  char * tmp_dir;
  char path[PATH_MAX];
  char * home;
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  tmp_dir = etpan_get_tmp_dir();
  r = etpan_make_dir(tmp_dir);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not initialize"));
    
    etpan_error_strf_long_description(error,
        _("The directory %s could not be created."), tmp_dir);
    
    return error;
  }
  
  privacy = mailprivacy_new(tmp_dir, 1);
  if (privacy == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailprivacy_gnupg_init(privacy);
  if (r != MAIL_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
#if 1
  r = mailprivacy_smime_init(privacy);
  if (r != MAIL_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  snprintf(path, sizeof(path), "%s/%s", home, ETPAN_SMIME_CERT_DIR);
  mailprivacy_smime_set_cert_dir(privacy, path);
  
  snprintf(path, sizeof(path), "%s/%s", home, ETPAN_SMIME_CA_DIR);
  mailprivacy_smime_set_CA_dir(privacy, path);
  
  snprintf(path, sizeof(path), "%s/%s", home, ETPAN_SMIME_PRIVATE_DIR);
  mailprivacy_smime_set_private_keys_dir(privacy, path);
  
  mailprivacy_smime_set_CA_check(privacy, 1);
  mailprivacy_smime_set_store_cert(privacy, 1);
#endif
  
  _ep_privacy = privacy;
  
  return NULL;
}

void etpan_lep_engine_unsetup(void)
{
  struct mailprivacy * privacy;
  
  privacy = _ep_privacy;
  
#if 1
  mailprivacy_smime_done(privacy);
#endif
  mailprivacy_gnupg_done(privacy);
  
  mailprivacy_free(privacy);
  
  _ep_privacy = NULL;
}

/* flags */

static int etpan_basic_flags_to_lep(int value)
{
  int ep_value;
  
  ep_value = 0;
  
  if ((value & ETPAN_FLAGS_SEEN) != 0)
    ep_value |= MAIL_FLAG_SEEN;
  
  if ((value & ETPAN_FLAGS_FLAGGED) != 0)
    ep_value |= MAIL_FLAG_FLAGGED;
  
  if ((value & ETPAN_FLAGS_DELETED) != 0)
    ep_value |= MAIL_FLAG_DELETED;
  
  if ((value & ETPAN_FLAGS_ANSWERED) != 0)
    ep_value |= MAIL_FLAG_ANSWERED;
  
  if ((value & ETPAN_FLAGS_FORWARDED) != 0)
    ep_value |= MAIL_FLAG_FORWARDED;
  
  return ep_value;
}

static int etpan_basic_flags_from_lep(int lep_value)
{
  int value;
  
  value = 0;
  
  if ((lep_value & MAIL_FLAG_SEEN) != 0)
    value |= ETPAN_FLAGS_SEEN;
  
  if ((lep_value & MAIL_FLAG_FLAGGED) != 0)
    value |= ETPAN_FLAGS_FLAGGED;
  
  if ((lep_value & MAIL_FLAG_DELETED) != 0)
    value |= ETPAN_FLAGS_DELETED;
  
  if ((lep_value & MAIL_FLAG_ANSWERED) != 0)
    value |= ETPAN_FLAGS_ANSWERED;
  
  if ((lep_value & MAIL_FLAG_FORWARDED) != 0)
    value |= ETPAN_FLAGS_FORWARDED;
  
  return value;
}

struct mail_flags *
etpan_lep_flags_to_lep(struct etpan_message_flags * flags)
{
  struct mail_flags * ep_flags;
  carray * ext_list;
  unsigned int i;
  int basic_flags;
  int r;
  
  ep_flags = mail_flags_new_empty();
  if (ep_flags == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  basic_flags = etpan_message_flags_get_value(flags);
  ep_flags->fl_flags = etpan_basic_flags_to_lep(basic_flags);
  
  ext_list = etpan_message_flags_get_ext(flags);
  for(i = 0 ; i < carray_count(ext_list) ; i ++) {
    char * ext;
    
    ext = carray_get(ext_list, i);
    r = mail_flags_add_extension(ep_flags, ext);
    if (r != MAIL_NO_ERROR) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return ep_flags;
}

struct etpan_message_flags *
etpan_lep_flags_from_lep(struct mail_flags * lep_flags)
{
  struct etpan_message_flags * flags;
  clistiter * iter;
  
  flags = etpan_message_flags_new();
  etpan_message_flags_set_value(flags,
      etpan_basic_flags_from_lep(lep_flags->fl_flags));
  
  for(iter = clist_begin(lep_flags->fl_extension) ; iter != NULL ;
      iter = clist_next(iter)) {
    char * ext;
    
    ext = clist_content(iter);
    etpan_message_flags_add_ext(flags, ext);
  }
  
  return flags;
}


/* mailbox conversion */

static struct mailimf_mailbox * etpan_dup_mailbox(struct mailimf_mailbox * mb);

static struct mailimf_mailbox_list *
etpan_dup_mailbox_list(struct mailimf_mailbox_list * mb_list);

struct mailimf_mailbox_list *
etpan_address_to_mailbox_list(struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  struct mailimf_mailbox_list * mb_list;
  int r;
  
  mb_list = mailimf_mailbox_list_new_empty();
  if (mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox_list * new_mb_list;
    
    addr = clist_content(cur);
    switch (addr->ad_type) {
    case MAILIMF_ADDRESS_MAILBOX:
      mb = etpan_dup_mailbox(addr->ad_data.ad_mailbox);
      r = mailimf_mailbox_list_add(mb_list, mb);
      if (r < 0) {
        mailimf_mailbox_free(mb);
        ETPAN_LOG_MEMORY_ERROR;
      }
      break;
      
    case MAILIMF_ADDRESS_GROUP:
      if (addr->ad_data.ad_group->grp_mb_list != NULL) {
        new_mb_list =
          etpan_dup_mailbox_list(addr->ad_data.ad_group->grp_mb_list);
        
        etpan_append_mailbox_list(mb_list, new_mb_list);
        mailimf_mailbox_list_free(new_mb_list);
      }
      break;
    }
  }
  
  return mb_list;
}

static struct mailimf_mailbox * etpan_dup_mailbox(struct mailimf_mailbox * mb)
{
  struct mailimf_mailbox * new_mb;
  char * display_name;
  char * addr_spec;
  
  display_name = NULL;
  if (mb->mb_display_name != NULL) {
    display_name = strdup(mb->mb_display_name);
    if (display_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  addr_spec = NULL;
  if (mb->mb_addr_spec != NULL) {
    addr_spec = strdup(mb->mb_addr_spec);
    if (addr_spec == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  new_mb = mailimf_mailbox_new(display_name, addr_spec);
  if (new_mb == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return new_mb;
}

static struct mailimf_mailbox_list *
etpan_dup_mailbox_list(struct mailimf_mailbox_list * mb_list)
{
  struct mailimf_mailbox_list * new_mb_list;
  clist * list;
  clistiter * cur;
  int r;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox * new_mb;
    
    mb = clist_content(cur);
    new_mb = etpan_dup_mailbox(mb);
    
    r = clist_append(list, new_mb);
    if (r < 0) {
      mailimf_mailbox_free(new_mb);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  new_mb_list = mailimf_mailbox_list_new(list);
  if (new_mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return new_mb_list;
}

static struct mailimf_address_list *
etpan_mailbox_to_address_list(struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  clist * list;
  struct mailimf_address_list * addr_list;
  int r;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox * new_mb;
  
    mb = clist_content(cur);
    new_mb = etpan_dup_mailbox(mb);

    addr = mailimf_address_new(MAILIMF_ADDRESS_MAILBOX, new_mb, NULL);
    if (addr == NULL) {
      mailimf_mailbox_free(new_mb);
      ETPAN_LOG_MEMORY_ERROR;
    }

    r = clist_append(list, addr);
    if (r < 0) {
      mailimf_address_free(addr);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }

  addr_list = mailimf_address_list_new(list);
  if (addr_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return addr_list;
}

static struct mailimf_group * etpan_dup_group(struct mailimf_group * group)
{
  struct mailimf_group * new_group;
  struct mailimf_mailbox_list * mb_list;
  char * dsp_name;
  
  mb_list = NULL;
  if (group->grp_mb_list != NULL) {
    mb_list = etpan_dup_mailbox_list(group->grp_mb_list);
  }
  
  dsp_name = strdup(group->grp_display_name);
  if (dsp_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  new_group = mailimf_group_new(strdup(group->grp_display_name), mb_list);
  if (new_group == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return new_group;
}

static struct mailimf_address * etpan_dup_address(struct mailimf_address * addr)
{
  struct mailimf_address * new_addr;
  struct mailimf_mailbox * new_mb;
  struct mailimf_group * group;
	
  new_mb = NULL;
  group = NULL;
  
  switch (addr->ad_type) {
  case MAILIMF_ADDRESS_MAILBOX:
    new_mb = etpan_dup_mailbox(addr->ad_data.ad_mailbox);
    break;
    
  case MAILIMF_ADDRESS_GROUP:
    group = etpan_dup_group(addr->ad_data.ad_group);
    break;
  }
  
  new_addr = mailimf_address_new(addr->ad_type, new_mb, group);
  if (new_addr == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return new_addr;
}

static void etpan_append_address_list(struct mailimf_address_list * new_addr_list,
    struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  int r;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    struct mailimf_address * new_addr;
    
    addr = clist_content(cur);
    
    new_addr = etpan_dup_address(addr);
    
    r = clist_append(new_addr_list->ad_list, new_addr);
    if (r < 0) {
      mailimf_address_free(new_addr);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
}

void etpan_append_mailbox_list(struct mailimf_mailbox_list * new_mb_list,
    struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  int r;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox * new_mb;
    
    mb = clist_content(cur);
    
    new_mb = etpan_dup_mailbox(mb);

    r = clist_append(new_mb_list->mb_list, new_mb);
    if (r < 0) {
      mailimf_mailbox_free(new_mb);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
}

static struct mailimf_address_list *
etpan_dup_address_list(struct mailimf_address_list * addr_list)
{
  clist * list;
  struct mailimf_address_list * new_addr_list;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  new_addr_list = mailimf_address_list_new(list);
  if (new_addr_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  if (addr_list != NULL) {
    etpan_append_address_list(new_addr_list, addr_list);
  }

  return new_addr_list;
}

static void etpan_mailbox_encode(struct mailimf_mailbox * mb);
static void etpan_mailbox_decode(struct mailimf_mailbox * mb);

static struct etpan_address * lep_address_from_lep_decode(struct mailimf_mailbox * mb,
    int decode)
{
  struct etpan_address * addr;
  struct mailimf_mailbox * dup_mb;
  
  dup_mb = etpan_dup_mailbox(mb);
  
  if (decode) {
    etpan_mailbox_decode(dup_mb);
  }
  
  addr = etpan_address_new();
  
  etpan_address_set_display_name(addr, dup_mb->mb_display_name);
  etpan_address_set_address(addr, dup_mb->mb_addr_spec);
  
  mailimf_mailbox_free(dup_mb);
  
  return addr;
}

struct etpan_address * etpan_lep_address_from_lep(struct mailimf_mailbox * mb)
{
  return lep_address_from_lep_decode(mb, 1);
}

struct mailimf_mailbox * etpan_lep_address_to_lep(struct etpan_address * addr)
{
  struct mailimf_mailbox * new_mb;
  char * display_name;
  char * addr_spec;
  
  display_name = etpan_address_get_display_name(addr);
  if (display_name != NULL) {
    display_name = strdup(display_name);
    if (display_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  addr_spec = etpan_address_get_address(addr);
  if (addr_spec != NULL) {
    addr_spec = strdup(addr_spec);
    if (addr_spec == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  new_mb = mailimf_mailbox_new(display_name, addr_spec);
  if (new_mb == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_mailbox_encode(new_mb);
  
  return new_mb;
}

struct mailimf_mailbox_list *
etpan_lep_address_list_to_lep_mb(carray * addr_list)
{
  unsigned int i;
  struct mailimf_mailbox_list * lep_mb_list;
  int r;
  
  lep_mb_list = mailimf_mailbox_list_new_empty();
  if (lep_mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(addr_list) ; i ++) {
    struct mailimf_mailbox * lep_mb;
    struct etpan_address * addr;
    
    addr = carray_get(addr_list, i);
    lep_mb = etpan_lep_address_to_lep(addr);
    
    r = mailimf_mailbox_list_add(lep_mb_list, lep_mb);
    if (r != MAILIMF_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return lep_mb_list;
}

struct mailimf_address_list *
etpan_lep_address_list_to_lep_addr(carray * addr_list)
{
  struct mailimf_mailbox_list * lep_mb_list;
  struct mailimf_address_list * lep_addr_list;
  
  lep_mb_list = etpan_lep_address_list_to_lep_mb(addr_list);
  
  lep_addr_list = etpan_mailbox_to_address_list(lep_mb_list);
  
  mailimf_mailbox_list_free(lep_mb_list);
  
  return lep_addr_list;
}


static carray * lep_address_list_from_lep_mb_decode(struct mailimf_mailbox_list *
    lep_mb_list, int decode)
{
  clistiter * cur;
  carray * addr_list;
  int r;
  
  addr_list = etpan_address_list_new();
 
  for(cur = clist_begin(lep_mb_list->mb_list) ;
      cur != NULL ; cur = clist_next(cur)) {
    struct mailimf_mailbox * lep_mb;
    struct etpan_address * addr;
    
    lep_mb = clist_content(cur);
    
    addr = lep_address_from_lep_decode(lep_mb, decode);
    
    r = carray_add(addr_list, addr, NULL);
    if (r < 0) {
      etpan_address_free(addr);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return addr_list;
}

carray * etpan_lep_address_list_from_lep_mb(struct mailimf_mailbox_list *
    lep_mb_list)
{
  return lep_address_list_from_lep_mb_decode(lep_mb_list, 1);
}

static carray * lep_address_list_from_lep_addr_decode(struct mailimf_address_list *
    lep_addr_list, int decode)
{
  struct mailimf_mailbox_list * lep_mb_list;
  carray * addr_list;
  
  lep_mb_list = etpan_address_to_mailbox_list(lep_addr_list);
  
  addr_list = lep_address_list_from_lep_mb_decode(lep_mb_list, decode);
  
  mailimf_mailbox_list_free(lep_mb_list);
  
  return addr_list;
}

carray * etpan_lep_address_list_from_lep_addr(struct mailimf_address_list *
    lep_addr_list)
{
  return lep_address_list_from_lep_addr_decode(lep_addr_list, 1);
}



#ifndef WRONG
#define WRONG	(-1)
#endif /* !defined WRONG */

static int tmcomp(struct tm * atmp, struct tm * btmp)
{
  register int	result;
  
  if ((result = (atmp->tm_year - btmp->tm_year)) == 0 &&
      (result = (atmp->tm_mon - btmp->tm_mon)) == 0 &&
      (result = (atmp->tm_mday - btmp->tm_mday)) == 0 &&
      (result = (atmp->tm_hour - btmp->tm_hour)) == 0 &&
      (result = (atmp->tm_min - btmp->tm_min)) == 0)
    result = atmp->tm_sec - btmp->tm_sec;
  return result;
}

static time_t mkgmtime(struct tm * tmp)
{
  register int			dir;
  register int			bits;
  register int			saved_seconds;
  time_t				t;
  struct tm			yourtm, mytm;
  
  yourtm = *tmp;
  saved_seconds = yourtm.tm_sec;
  yourtm.tm_sec = 0;
  /*
  ** Calculate the number of magnitude bits in a time_t
  ** (this works regardless of whether time_t is
  ** signed or unsigned, though lint complains if unsigned).
  */
  for (bits = 0, t = 1; t > 0; ++bits, t <<= 1)
    ;
  /*
  ** If time_t is signed, then 0 is the median value,
  ** if time_t is unsigned, then 1 << bits is median.
  */
  t = (t < 0) ? 0 : ((time_t) 1 << bits);
  for ( ; ; ) {
    gmtime_r(&t, &mytm);
    dir = tmcomp(&mytm, &yourtm);
    if (dir != 0) {
      if (bits-- < 0)
	return WRONG;
      if (bits < 0)
	--t;
      else if (dir > 0)
	t -= (time_t) 1 << bits;
      else	t += (time_t) 1 << bits;
      continue;
    }
    break;
  }
  t += saved_seconds;
  return t;
}

static time_t
etpan_lep_header_get_date(struct mailimf_single_fields * single_fields)
{
  struct tm tmval;
  time_t timeval;
  struct mailimf_date_time * date_time;
  
  if (single_fields->fld_orig_date == NULL)
    return (time_t) -1;
  
  date_time = single_fields->fld_orig_date->dt_date_time;
  
  tmval.tm_sec  = date_time->dt_sec;
  tmval.tm_min  = date_time->dt_min;
  tmval.tm_hour = date_time->dt_hour;
  tmval.tm_sec  = date_time->dt_sec;
  tmval.tm_mday = date_time->dt_day;
  tmval.tm_mon  = date_time->dt_month - 1;
  tmval.tm_year = date_time->dt_year - 1900;
  
  timeval = mkgmtime(&tmval);
  
  timeval -= date_time->dt_zone * 36;
  
  return timeval;
}

struct etpan_error * etpan_fetch_bodystructure(mailmessage * ep_msg)
{
  struct mailmime * mime;
  int r;
  
  if (ep_msg->msg_mime != NULL) {
    ETPAN_LOG("WARNING already fetched");
    etpan_crash();
  }
  
  r = mailprivacy_msg_get_bodystructure(etpan_lep_privacy(),
      ep_msg, &mime);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r != MAIL_NO_ERROR) {
    struct etpan_error * error;
    
    error = message_translate_error(ERROR_DOMAIN_FETCH_MSG, r);
    
    return error;
  }
  
  return NULL;
}

void etpan_flush_bodystructure(mailmessage * ep_msg)
{
  mailprivacy_msg_flush(etpan_lep_privacy(), ep_msg);
}




/* extracted from etpan-mime-tools */


static inline int to_be_quoted(char * word, size_t size)
{
  int do_quote;
  char * cur;
  size_t i;

  do_quote = 0;
  cur = word;
  for(i = 0 ; i < size ; i ++) {
    switch (* cur) {
    case ',':
    case ':':
    case '!':
    case '"':
    case '#':
    case '$':
    case '@':
    case '[':
    case '\\':
    case ']':
    case '^':
    case '`':
    case '{':
    case '|':
    case '}':
    case '~':
    case '=':
    case '?':
    case '_':
      do_quote = 1;
      break;
    default:
      if (((unsigned char) * cur) >= 128)
        do_quote = 1;
      break;
    }
    cur ++;
  }

  return do_quote;
}

#define MAX_IMF_LINE 72

static inline void quote_word(char * display_charset,
    MMAPString * mmapstr, char * word, size_t size)
{
  char * cur;
  size_t i;
  char hex[4];
  int col;
  
  if (mmap_string_append(mmapstr, "=?") == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (mmap_string_append(mmapstr, display_charset) == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (mmap_string_append(mmapstr, "?Q?") == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  col = mmapstr->len;
  
  cur = word;
  for(i = 0 ; i < size ; i ++) {
    int do_quote_char;

    if (col + 2 /* size of "?=" */
        + 3 /* max size of newly added character */
        + 1 /* minimum column of string in a
               folded header */ >= MAX_IMF_LINE) {
      int old_pos;
      /* adds a concatened encoded word */
      
      if (mmap_string_append(mmapstr, "?=") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      if (mmap_string_append(mmapstr, " ") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      old_pos = mmapstr->len;
      
      if (mmap_string_append(mmapstr, "=?") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(mmapstr, display_charset) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(mmapstr, "?Q?") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      col = mmapstr->len - old_pos;
    }
    
    do_quote_char = 0;
    switch (* cur) {
    case ',':
    case ':':
    case '!':
    case '"':
    case '#':
    case '$':
    case '@':
    case '[':
    case '\\':
    case ']':
    case '^':
    case '`':
    case '{':
    case '|':
    case '}':
    case '~':
    case '=':
    case '?':
    case '_':
      do_quote_char = 1;
      break;

    default:
      if (((unsigned char) * cur) >= 128)
        do_quote_char = 1;
      break;
    }

    if (do_quote_char) {
      snprintf(hex, 4, "=%2.2X", (unsigned char) * cur);
      if (mmap_string_append(mmapstr, hex) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      col += 3;
    }
    else {
      if (* cur == ' ') {
        if (mmap_string_append_c(mmapstr, '_') == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }
      else {
        if (mmap_string_append_c(mmapstr, * cur) == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }
      col += 3;
    }
    cur ++;
  }

  if (mmap_string_append(mmapstr, "?=") == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

static inline void get_word(char * begin, char ** pend, int * pto_be_quoted)
{
  char * cur;
  
  cur = begin;

  while ((* cur != ' ') && (* cur != '\t') && (* cur != '\0')) {
    cur ++;
  }

  if (cur - begin +
      1  /* minimum column of string in a
            folded header */ > MAX_IMF_LINE)
    * pto_be_quoted = 1;
  else
    * pto_be_quoted = to_be_quoted(begin, cur - begin);
  
  * pend = cur;
}

static char * etpan_make_quoted_printable(char * display_charset,
    char * phrase)
{
  char * str;
  char * cur;
  MMAPString * mmapstr;

  mmapstr = mmap_string_new("");
  if (mmapstr == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  cur = phrase;
  while (* cur != '\0') {
    char * begin;
    char * end;
    int do_quote;
    int quote_words;

    begin = cur;
    end = begin;
    quote_words = 0;
    do_quote = 1;

    while (* cur != '\0') {
      get_word(cur, &cur, &do_quote);
      if (do_quote) {
        quote_words = 1;
        end = cur;
      }
      else
        break;
      if (* cur != '\0')
        cur ++;
    }

    if (quote_words) {
      quote_word(display_charset, mmapstr, begin, end - begin);

      if ((* end == ' ') || (* end == '\t')) {
        if (mmap_string_append_c(mmapstr, * end) == NULL)
          ETPAN_LOG_MEMORY_ERROR;
        end ++;
      }

      if (* end != '\0') {
        if (mmap_string_append_len(mmapstr, end, cur - end) == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
    else {
      if (mmap_string_append_len(mmapstr, begin, cur - begin) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }

    if ((* cur == ' ') || (* cur == '\t')) {
      if (mmap_string_append_c(mmapstr, * cur) == 0)
        ETPAN_LOG_MEMORY_ERROR;
      cur ++;
    }
  }

  str = strdup(mmapstr->str);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  mmap_string_free(mmapstr);
  
  return str;
}



/* encode decode of MIME header values */

char * etpan_encode_mime_header(char * phrase)
{
  return
    etpan_make_quoted_printable(DEFAULT_DISPLAY_CHARSET,
        phrase);
}


char * etpan_decode_mime_header(char * phrase)
{
  int r;
  size_t cur_token;
  char * decoded;
  
  if (* phrase == '\0') {
    decoded = strdup("");
    if (decoded == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    return decoded;
  }
  
  cur_token = 0;
  r = mailmime_encoded_phrase_parse(DEFAULT_INCOMING_CHARSET,
      phrase, strlen(phrase),
      &cur_token, DEFAULT_DISPLAY_CHARSET,
      &decoded);
  
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_LOG("%i %s", r, phrase);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return decoded;
}


/* ************ decode / encode MIME in headers ****************** */

static void etpan_mailbox_encode(struct mailimf_mailbox * mb)
{
  char * decoded;
  
  if (mb->mb_display_name == NULL)
    return;
  
  decoded = etpan_encode_mime_header(mb->mb_display_name);
  
  /* replace display name */
  free(mb->mb_display_name);
  mb->mb_display_name = decoded;
}

static void etpan_mailbox_list_encode(struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    
    mb = clist_content(cur);
    etpan_mailbox_encode(mb);
  }
}

static void etpan_group_encode(struct mailimf_group * group)
{
  char * decoded;
  
  decoded = etpan_encode_mime_header(group->grp_display_name);
  free(group->grp_display_name);
  group->grp_display_name = decoded;
  
  if (group->grp_mb_list != NULL)
    etpan_mailbox_list_encode(group->grp_mb_list);
}

static void etpan_addr_encode(struct mailimf_address * addr)
{
  switch (addr->ad_type) {
  case MAILIMF_ADDRESS_MAILBOX:
    etpan_mailbox_encode(addr->ad_data.ad_mailbox);
    break;
  case MAILIMF_ADDRESS_GROUP:
    etpan_group_encode(addr->ad_data.ad_group);
    break;
  }
}

static void etpan_addr_list_encode(struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    
    addr = clist_content(cur);
    etpan_addr_encode(addr);
  }
}

static void etpan_mailbox_decode(struct mailimf_mailbox * mb)
{
  if (mb->mb_display_name != NULL) {
    char * decoded;
    
    decoded =  etpan_decode_mime_header(mb->mb_display_name);
    
    /* replace display name */
    free(mb->mb_display_name);
    mb->mb_display_name = decoded;
  }
  
  if (mb->mb_addr_spec != NULL) {
    char * decoded;
    int r;
    
    r = charconv("utf-8", "utf-8", mb->mb_addr_spec, strlen(mb->mb_addr_spec),
        &decoded);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    free(mb->mb_addr_spec);
    mb->mb_addr_spec = decoded;
  }
}

static void etpan_mailbox_list_decode(struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    
    mb = clist_content(cur);
    etpan_mailbox_decode(mb);
  }
}

static void etpan_group_decode(struct mailimf_group * group)
{
  char * decoded;
  
  decoded = etpan_decode_mime_header(group->grp_display_name);
  if (decoded != NULL) {
    free(group->grp_display_name);
    group->grp_display_name = decoded;
  }
  
  if (group->grp_mb_list != NULL)
    etpan_mailbox_list_decode(group->grp_mb_list);
}

static void etpan_addr_decode(struct mailimf_address * addr)
{
  switch (addr->ad_type) {
  case MAILIMF_ADDRESS_MAILBOX:
    etpan_mailbox_decode(addr->ad_data.ad_mailbox);
    break;
  case MAILIMF_ADDRESS_GROUP:
    etpan_group_decode(addr->ad_data.ad_group);
    break;
  }
}

static void etpan_addr_list_decode(struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    
    addr = clist_content(cur);
    etpan_addr_decode(addr);
  }
}


static char * get_discrete_type(struct mailmime_discrete_type * discrete_type)
{
  switch (discrete_type->dt_type) {
  case MAILMIME_DISCRETE_TYPE_TEXT:
    return "text";

  case MAILMIME_DISCRETE_TYPE_IMAGE:
    return "image";

  case MAILMIME_DISCRETE_TYPE_AUDIO:
    return "audio";

  case MAILMIME_DISCRETE_TYPE_VIDEO:
    return "video";

  case MAILMIME_DISCRETE_TYPE_APPLICATION:
    return "application";

  case MAILMIME_DISCRETE_TYPE_EXTENSION:
    return discrete_type->dt_extension;
  }

  return NULL;
}

static char *
get_composite_type(struct mailmime_composite_type * composite_type)
{
  switch (composite_type->ct_type) {
  case MAILMIME_COMPOSITE_TYPE_MESSAGE:
    return "message";

  case MAILMIME_COMPOSITE_TYPE_MULTIPART:
    return "multipart";

  case MAILMIME_COMPOSITE_TYPE_EXTENSION:
    return composite_type->ct_token;
  }

  return NULL;
}


char * etpan_get_content_type_str(struct mailmime_content * content)
{
  char * str;
  char * result;

  str = "unknown";

  switch (content->ct_type->tp_type) {
  case MAILMIME_TYPE_DISCRETE_TYPE:
    str = get_discrete_type(content->ct_type->tp_data.tp_discrete_type);
    break;

  case MAILMIME_TYPE_COMPOSITE_TYPE:
    str = get_composite_type(content->ct_type->tp_data.tp_composite_type);
    break;
  }

  if (str == NULL)
    str = "unknown";
  
  result = malloc(strlen(str) + strlen(content->ct_subtype) + 2);
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  strcpy(result, str);
  strcat(result, "/");
  strcat(result, content->ct_subtype);

  return result;
}


static int etpan_mime_is_text(struct mailmime * build_info)
{
  if (build_info->mm_type == MAILMIME_SINGLE) {
    if (build_info->mm_content_type != NULL) {
      if (build_info->mm_content_type->ct_type->tp_type ==
          MAILMIME_TYPE_DISCRETE_TYPE) {
        if (build_info->mm_content_type->ct_type->tp_data.tp_discrete_type->dt_type ==
            MAILMIME_DISCRETE_TYPE_TEXT)
          return 1;
      }
    }
    else
      return 1;
  }

  return 0;
}


#if 0
static int etpan_address_list_write(FILE * f, int * col,
    struct mailimf_address_list * addr_list)
{
  struct mailimf_address_list * decoded_addr_list;
  int r;

  decoded_addr_list = etpan_dup_address_list(addr_list);
  if (decoded_addr_list == NULL)
    return ERROR_MEMORY;
  
  etpan_addr_list_decode(decoded_addr_list);
  
  r = mailimf_address_list_write(f, col, decoded_addr_list);
  mailimf_address_list_free(decoded_addr_list);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;
  
  return NO_ERROR;
}

static int etpan_mailbox_list_write(FILE * f, int * col,
    struct mailimf_mailbox_list * mb_list)
{
  struct mailimf_mailbox_list * decoded_mb_list;
  int r;

  decoded_mb_list = etpan_dup_mailbox_list(mb_list);
  if (decoded_mb_list == NULL)
    return ERROR_MEMORY;
  
  etpan_mailbox_list_decode(decoded_mb_list);
  
  r = mailimf_mailbox_list_write(f, col, decoded_mb_list);
  mailimf_mailbox_list_free(decoded_mb_list);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;
  
  return NO_ERROR;
}
#endif

static carray * etpan_lep_msgid_list_from_lep(clist * ep_msgidlist)
{
  carray * msgidlist;
  clistiter * iter;
  
  msgidlist = etpan_message_id_list_new();
  
  for(iter = clist_begin(ep_msgidlist) ; iter != NULL ;
      iter = clist_next(iter)) {
    char * item;
    
    item = clist_content(iter);
    etpan_message_id_list_add(msgidlist, item);
  }
  
  return msgidlist;
}

static void lep_msgidlist_free(clist * lep_msgidlist)
{
  clistiter * iter;
  
  for(iter = clist_begin(lep_msgidlist) ; iter != NULL ;
      iter = clist_next(iter)) {
    char * item;
    
    item = clist_content(iter);
    free(item);
  }
  clist_free(lep_msgidlist);
}

static clist * etpan_lep_msgid_list_to_lep(carray * msgidlist)
{
  clist * lep_msgidlist;
  int r;
  unsigned int i;
  
  lep_msgidlist = clist_new();
  if (lep_msgidlist == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(msgidlist) ; i ++) {
    char * item;
    char * dup_item;
    
    item = carray_get(msgidlist, i);
    dup_item = strdup(item);
    if (dup_item == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = clist_append(lep_msgidlist, dup_item);
    if (r < 0) {
      free(dup_item);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return lep_msgidlist;
}

struct etpan_message_header *
etpan_header_from_lep_header(struct mailimf_fields * fields)
{
  struct etpan_message_header * header;
  time_t date;
  char * subject;
  struct mailimf_mailbox_list * ep_from;
  carray * from;
  struct mailimf_address_list * ep_replyto;
  carray * replyto;
  struct mailimf_address_list * ep_to;
  carray * to;
  struct mailimf_address_list * ep_cc;
  carray * cc;
  struct mailimf_address_list * ep_bcc;
  carray * bcc;
  char * decoded_subject;
  struct mailimf_single_fields single_fields;
  char * msgid;
  carray * references;
  clist * ep_references;
  carray * inreplyto; 
  clist * ep_inreplyto;
  clistiter * iter;
  
  mailimf_single_fields_init(&single_fields, fields);
  
  header = etpan_message_header_new();
  if (header == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  /* date */
  date = etpan_lep_header_get_date(&single_fields);
  etpan_message_header_set_date(header, date);
  
  /* subject */
  subject = NULL;
  if (single_fields.fld_subject != NULL)
    subject = single_fields.fld_subject->sbj_value;
  
  if (subject != NULL)
    decoded_subject = etpan_decode_mime_header(subject);
  else
    decoded_subject = NULL;
  
  etpan_message_header_set_subject(header, decoded_subject);
  free(decoded_subject);
  
  /* from */
  ep_from = NULL;
  if (single_fields.fld_from != NULL)
    ep_from = single_fields.fld_from->frm_mb_list;
  
  from = NULL;
  if (ep_from != NULL) {
    from = etpan_lep_address_list_from_lep_mb(ep_from);
  }
  
  if (from != NULL) {
    etpan_message_header_set_from(header, from);
    etpan_address_list_free(from);
  }
  
  /* replyto */
  ep_replyto = NULL;
  if (single_fields.fld_reply_to != NULL)
    ep_replyto = single_fields.fld_reply_to->rt_addr_list;
  
  replyto = NULL;
  if (ep_replyto != NULL) {
    replyto = etpan_lep_address_list_from_lep_addr(ep_replyto);
  }
  
  if (replyto != NULL) {
    etpan_message_header_set_replyto(header, replyto);
    etpan_address_list_free(replyto);
  }
  
  /* to */
  ep_to = NULL;
  if (single_fields.fld_to != NULL)
    ep_to = single_fields.fld_to->to_addr_list;
  
  to = NULL;
  if (ep_to != NULL) {
    to = etpan_lep_address_list_from_lep_addr(ep_to);
  }
  
  if (to != NULL) {
    etpan_message_header_set_to(header, to);
    etpan_address_list_free(to);
  }

  /* cc */
  ep_cc = NULL;
  if (single_fields.fld_cc != NULL)
    ep_cc = single_fields.fld_cc->cc_addr_list;
  
  cc = NULL;
  if (ep_cc != NULL) {
    cc = etpan_lep_address_list_from_lep_addr(ep_cc);
  }
  
  if (cc != NULL) {
    etpan_message_header_set_cc(header, cc);
    etpan_address_list_free(cc);
  }
  
  /* bcc */
  ep_bcc = NULL;
  if (single_fields.fld_bcc != NULL)
    ep_bcc = single_fields.fld_bcc->bcc_addr_list;
  
  bcc = NULL;
  if (ep_bcc != NULL) {
    bcc = etpan_lep_address_list_from_lep_addr(ep_bcc);
  }
  
  if (bcc != NULL) {
    etpan_message_header_set_bcc(header, bcc);
    etpan_address_list_free(bcc);
  }
  
  /* msgid */
  msgid = NULL;
  if (single_fields.fld_message_id != NULL)
    msgid = single_fields.fld_message_id->mid_value;
  
  etpan_message_header_set_msgid(header, msgid);
  
  /* references */
  ep_references = NULL;
  if (single_fields.fld_references != NULL)
    ep_references = single_fields.fld_references->mid_list;
  
  references = NULL;
  if (ep_references != NULL) {
    references = etpan_lep_msgid_list_from_lep(ep_references);
  }
  
  if (references != NULL) {
    etpan_message_header_set_references(header, references);
    etpan_message_id_list_free(references);
  }
  
  /* inreplyto */
  ep_inreplyto = NULL;
  if (single_fields.fld_in_reply_to != NULL)
    ep_inreplyto = single_fields.fld_in_reply_to->mid_list;
  
  inreplyto = NULL;
  if (ep_inreplyto != NULL) {
    inreplyto = etpan_lep_msgid_list_from_lep(ep_inreplyto);
  }
  
  if (inreplyto != NULL) {
    etpan_message_header_set_inreplyto(header, inreplyto);
    etpan_message_id_list_free(inreplyto);
  }
  
  /* extract Newsgroups */
  for(iter = clist_begin(fields->fld_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailimf_field * field;
    
    field = clist_content(iter);
    if (field->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD) {
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name, "Newsgroups") == 0) {
        char * newsgroups;
        
        newsgroups = field->fld_data.fld_optional_field->fld_value;
        etpan_message_header_set_newsgroups(header, newsgroups);
        break;
      }
      else if (strcasecmp(field->fld_data.fld_optional_field->fld_name, "List-Post") == 0) {
        struct etpan_uri * uri;
        char * value;
        size_t len;
        char * dup_value;
        char * scheme;
        carray * addr_list;
        struct etpan_address * addr;
        
        value = field->fld_data.fld_optional_field->fld_value;
        len = strlen(value);
        
        if ((value[0] != '<') && (value[len - 1] != '>'))
          continue;
        
        dup_value = strdup(value + 1);
        if (dup_value == NULL)
          ETPAN_LOG_MEMORY_ERROR;
        
        dup_value[len - 2] = '\0';
        
        uri = etpan_uri_parse(dup_value);
        if (uri == NULL)
          goto no_listpost;
        
        scheme = etpan_uri_get_scheme(uri);
        if (scheme == NULL)
          goto free_uri;
        
        if (strcasecmp(scheme, "mailto") != 0)
          goto free_uri;
        
        addr_list = etpan_address_list_to_from_str(etpan_uri_get_opaque(uri));
        
        if (carray_count(addr_list) != 1)
          goto free_addr_list;
        
        addr = carray_get(addr_list, 0);
        etpan_message_header_set_listpost(header, addr);
        
      free_addr_list:
        etpan_address_list_free(addr_list);
      free_uri:
        etpan_uri_free(uri);
      no_listpost:
        free(dup_value);
      }
    }
  }
  
  return header;
}

static struct mailimf_date_time * get_date(time_t timeval)
{
  struct tm gmt;
  struct tm lt;
  int off;
  struct mailimf_date_time * date_time;

  if (gmtime_r(&timeval, &gmt) == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  if (localtime_r(&timeval, &lt) == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  off = (mkgmtime(&lt) - mkgmtime(&gmt)) / (60 * 60) * 100;

  date_time = mailimf_date_time_new(lt.tm_mday, lt.tm_mon + 1,
      lt.tm_year + 1900,
      lt.tm_hour, lt.tm_min, lt.tm_sec,
      off);
  if (date_time == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return date_time;
}

struct mailimf_fields *
etpan_header_to_lep_header(struct etpan_message_header * header)
{
  time_t date;
  struct mailimf_date_time * lep_date;
  char * subject;
  char * lep_subject;
  carray * from;
  struct mailimf_mailbox_list * lep_from;
  carray * replyto;
  struct mailimf_address_list * lep_replyto;
  carray * to;
  struct mailimf_address_list * lep_to;
  carray * cc;
  struct mailimf_address_list * lep_cc;
  carray * bcc;
  struct mailimf_address_list * lep_bcc;
  struct mailimf_fields * fields;
  char * msgid;
  char * lep_msgid;
  carray * references;
  clist * lep_references;
  carray * inreplyto;
  clist * lep_inreplyto;
  char * newsgroups;
  char * lep_newsgroups;
  int r;
  
  lep_date = NULL;
  lep_subject = NULL;
  lep_newsgroups = NULL;
  lep_from = NULL;
  lep_replyto = NULL;
  lep_to = NULL;
  lep_cc = NULL;
  lep_bcc = NULL;
  lep_msgid = NULL;
  lep_inreplyto = NULL;
  lep_references = NULL;
  
  date = etpan_message_header_get_date(header);
  if (date != (time_t) -1) {
    lep_date = get_date(date);
  }
  else {
    lep_date = mailimf_get_current_date();
    if (lep_date == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  subject = etpan_message_header_get_subject(header);
  if (subject != NULL) {
    lep_subject = etpan_encode_mime_header(subject);
  }
  
  newsgroups = etpan_message_header_get_newsgroups(header);
  if (newsgroups != NULL) {
    lep_newsgroups = strdup(newsgroups);
    if (lep_newsgroups == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  from = etpan_message_header_get_from(header);
  if (carray_count(from) > 0) {
    lep_from = etpan_lep_address_list_to_lep_mb(from);
  }
  
  replyto = etpan_message_header_get_replyto(header);
  if (carray_count(replyto) > 0) {
    lep_replyto = etpan_lep_address_list_to_lep_addr(replyto);
  }
  
  to = etpan_message_header_get_to(header);
  if (carray_count(to) > 0) {
    lep_to = etpan_lep_address_list_to_lep_addr(to);
  }
  
  cc = etpan_message_header_get_cc(header);
  if (carray_count(cc) > 0) {
    lep_cc = etpan_lep_address_list_to_lep_addr(cc);
  }
  
  bcc = etpan_message_header_get_bcc(header);
  if (carray_count(bcc) > 0) {
    lep_bcc = etpan_lep_address_list_to_lep_addr(bcc);
  }
  
  msgid = etpan_message_header_get_msgid(header);
  if (msgid != NULL) {
    lep_msgid = strdup(msgid);
    if (lep_msgid == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  inreplyto = etpan_message_header_get_inreplyto(header);
  if (carray_count(inreplyto) > 0) {
    lep_inreplyto = etpan_lep_msgid_list_to_lep(inreplyto);
  }

  references = etpan_message_header_get_references(header);
  if (carray_count(references) > 0) {
    lep_references = etpan_lep_msgid_list_to_lep(references);
  }
  
  fields = mailimf_fields_new_with_data_all(lep_date,
      lep_from, NULL, lep_replyto, lep_to, lep_cc,
      lep_bcc, lep_msgid, lep_inreplyto, lep_references,
      lep_subject);
  if (fields == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (lep_newsgroups != NULL) {
    struct mailimf_field * field;
    char * name;
    
    name = strdup("Newsgroups");
    if (name == NULL) {
      free(lep_newsgroups);
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    field = mailimf_field_new_custom(name, lep_newsgroups);
    if (field == NULL) {
      free(lep_newsgroups);
      free(name);
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    r = mailimf_fields_add(fields, field);
    if (r != MAILIMF_NO_ERROR) {
      mailimf_field_free(field);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return fields;
}



void
etpan_single_resent_fields_init(struct mailimf_single_fields * single_fields,
    struct mailimf_fields * fields)
{
  clistiter * cur;

  memset(single_fields, 0, sizeof(struct mailimf_single_fields));

  cur = clist_begin(fields->fld_list);
  while (cur != NULL) {
    struct mailimf_field * field;

    field = clist_content(cur);

    switch (field->fld_type) {
    case MAILIMF_FIELD_RESENT_DATE:
      if (single_fields->fld_orig_date == NULL)
        single_fields->fld_orig_date = field->fld_data.fld_resent_date;
      cur = clist_next(cur);
      break;
    case MAILIMF_FIELD_RESENT_FROM:
      if (single_fields->fld_from == NULL) {
        single_fields->fld_from = field->fld_data.fld_resent_from;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_from->frm_mb_list->mb_list,
                     field->fld_data.fld_resent_from->frm_mb_list->mb_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_SENDER:
      if (single_fields->fld_sender == NULL)
        single_fields->fld_sender = field->fld_data.fld_resent_sender;
      cur = clist_next(cur);
      break;
    case MAILIMF_FIELD_RESENT_TO:
      if (single_fields->fld_to == NULL) {
        single_fields->fld_to = field->fld_data.fld_resent_to;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_to->to_addr_list->ad_list,
                     field->fld_data.fld_resent_to->to_addr_list->ad_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_CC:
      if (single_fields->fld_cc == NULL) {
        single_fields->fld_cc = field->fld_data.fld_resent_cc;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_cc->cc_addr_list->ad_list, 
                     field->fld_data.fld_resent_cc->cc_addr_list->ad_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_BCC:
      if (single_fields->fld_bcc == NULL) {
        single_fields->fld_bcc = field->fld_data.fld_resent_bcc;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_bcc->bcc_addr_list->ad_list,
                     field->fld_data.fld_resent_bcc->bcc_addr_list->ad_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_MSG_ID:
      if (single_fields->fld_message_id == NULL)
        single_fields->fld_message_id = field->fld_data.fld_resent_msg_id;
      cur = clist_next(cur);
      break;
    default:
      cur = clist_next(cur);
      break;
    }
  }
}

static pthread_mutex_t lep_folder_hash_lock = PTHREAD_MUTEX_INITIALIZER;
static chash * lep_folder_hash = NULL;

struct mailfolder * etpan_lep_folder_from_folder(struct etpan_folder * folder)
{
  struct mailfolder * result;
  
  pthread_mutex_lock(&lep_folder_hash_lock);
  if (lep_folder_hash == NULL) {
    result = NULL;
  }
  else {
    chashdatum key;
    chashdatum value;
    int r;
    
    key.data = &folder;
    key.len = sizeof(folder);
    r = chash_get(lep_folder_hash, &key, &value);
    if (r < 0)
      result = NULL;
    else
      result = value.data;
  }
  pthread_mutex_unlock(&lep_folder_hash_lock);
  
  return result;
}

void etpan_set_lep_folder(struct etpan_folder * folder,
    struct mailfolder * lep_folder)
{
  chashdatum key;
  chashdatum value;
  
  pthread_mutex_lock(&lep_folder_hash_lock);
  if (lep_folder != NULL) {
    if (lep_folder_hash == NULL) {
      lep_folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
      if (lep_folder_hash == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  if (lep_folder_hash != NULL) {
    key.data = &folder;
    key.len = sizeof(folder);
    value.data = lep_folder;
    value.len = 0;
    if (lep_folder == NULL) {
      chash_delete(lep_folder_hash, &key, NULL);
      if (chash_count(lep_folder_hash) == 0) {
        chash_free(lep_folder_hash);
        lep_folder_hash = NULL;
      }
    }
    else {
      chash_set(lep_folder_hash, &key, &value, NULL);
    }
  }
  
  pthread_mutex_unlock(&lep_folder_hash_lock);
}

carray * etpan_address_list_to_from_str(char * str)
{
  struct mailimf_address_list * ep_list;
  carray * list;
  int r;
  size_t index;
  
  index = 0;
  r = mailimf_address_list_parse(str, strlen(str), &index, &ep_list);
  if (r == MAILIMF_ERROR_PARSE) {
    return etpan_address_list_new();
  }
  
  if (r != MAILIMF_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  list = lep_address_list_from_lep_addr_decode(ep_list, 0);
  
  mailimf_address_list_free(ep_list);
  
  return list;
}


static struct etpan_error * folder_translate_error(struct etpan_folder * folder,
    int domain, int error_code)
{
  struct etpan_error * error;
  
  error = etpan_error_new();
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH_MSG_LIST:
    switch (error_code) {
    case MAIL_ERROR_PARSE:
      etpan_error_set_code(error, ERROR_PARSE);
      etpan_error_set_short_description(error, "Parse error");
      etpan_error_strf_long_description(error,
          _("Retrieval of message list of folder %s failed."
              "The application did not understand what the server sent."),
          etpan_folder_get_ui_path(folder));
      
      break;
      
    case MAIL_ERROR_STREAM:
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Retrieval of message list of folder %s failed."
              "The connection closed unexpectedly."),
          etpan_folder_get_ui_path(folder));
      
      break;

    case MAIL_ERROR_FETCH:
      etpan_error_set_code(error, ERROR_FETCH);
      etpan_error_set_short_description(error, "Could not get message list");
      etpan_error_strf_long_description(error,
          _("Retrieval of message list of folder %s failed."
              "The data could not be fetched."),
          etpan_folder_get_ui_path(folder));
      
      break;
      
    default:
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("Retrieval of message list of folder %s failed."
              "An unexpected error (libetpan: %i) occurred."),
          etpan_folder_get_ui_path(folder), error_code);
      break;
    }
    break;
    
  default:
    etpan_error_set_code(error, ERROR_INTERNAL);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An unexpected error (libetpan: %i) occurred while working on folder %s."),
        error_code, etpan_folder_get_ui_path(folder));
    
    break;
  }
  
  return error;
}

static struct etpan_error * message_translate_error(int domain, int error_code)
{
  struct etpan_error * error;
  
  error = etpan_error_new();
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH_MSG:
    switch (error_code) {
    case MAIL_ERROR_STREAM:
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("The connection closed unexpectedly."));
      break;

    case MAIL_ERROR_FETCH:
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Could not get message data");
      etpan_error_strf_long_description(error,
          _("The message data could not be fetched."));
      break;
      
    default:
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("An unexpected error (libetpan: %i) occurred."),
          error_code);
      break;
    }
    break;
    
  default:
    etpan_error_set_code(error, ERROR_INTERNAL);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An unexpected error (libetpan: %i) occurred."),
        error_code);
    break;
  }
  
  return error;
}

struct etpan_part_header *
etpan_mime_header_from_lep(struct mailmime_fields * mime_fields,
    struct mailmime_content * mime_content)
{
  struct mailmime_single_fields single_fields;
  struct etpan_part_header * header;
  char * content_type;
  int mime_encoding;
  char * filename;
  char * decoded_filename;
  
  header = etpan_part_header_new();
  
  mailmime_single_fields_init(&single_fields,
      mime_fields, mime_content);
  
  if (single_fields.fld_content == NULL)
    content_type = strdup("text/plain");
  else
    content_type = etpan_get_content_type_str(single_fields.fld_content);
  if (content_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_part_header_set_content_type(header, content_type);
  
  free(content_type);
  
  if (single_fields.fld_disposition_filename != NULL)
    filename = single_fields.fld_disposition_filename;
  else
    filename = single_fields.fld_content_name;
  
  if (filename != NULL)
    decoded_filename = etpan_decode_mime_header(filename);
  else
    decoded_filename = NULL;
  
  etpan_part_header_set_filename(header, decoded_filename);
  free(decoded_filename);
  
  mime_encoding = ETPAN_PART_MIME_ENCODING_OTHER;
  if (single_fields.fld_encoding != NULL) {
    switch (single_fields.fld_encoding->enc_type) {
    case MAILMIME_MECHANISM_QUOTED_PRINTABLE:
      mime_encoding = ETPAN_PART_MIME_ENCODING_QUOTEDPRINTABLE;
      break;
    case MAILMIME_MECHANISM_BASE64:
      mime_encoding = ETPAN_PART_MIME_ENCODING_BASE64;
      break;
    }
    etpan_part_header_set_mime_encoding(header, mime_encoding);
  }
  
  etpan_part_header_set_charset_encoding(header,
      single_fields.fld_content_charset);
  
  return header;
}
