#include "etpan-log.h"

#include <pthread.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/time.h>
#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#define MAX_LOG_LINE 4096
static pthread_mutex_t log_lock = PTHREAD_MUTEX_INITIALIZER;
static char log_line[MAX_LOG_LINE];
static char log_suffix[MAX_LOG_LINE];
static chash * log_filter = NULL;

void (* etpan_log_callback)(const char * str) = NULL;

static void etpan_str_log(void)
{
  fprintf(stderr, "%s\n", log_line);
}

void etpan_log_init(void)
{
  pthread_mutex_lock(&log_lock);
  if (log_filter == NULL) {
    char * env_value;
    
    env_value = getenv("ETPAN_LOG");
    if (env_value != NULL) {
      strncpy(log_line, env_value, sizeof(log_line));
      log_line[sizeof(log_line) - 1] = '\0';
    }
    else {
      * log_line = '\0';
    }
    env_value = log_line;
    
    log_filter = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
    if (log_filter != NULL) {
      chashdatum key;
      chashdatum value;
      
      key.data = "LOG";
      key.len = strlen("LOG");
      value.data = NULL;
      value.len = 0;
      chash_set(log_filter, &key, &value, NULL);

      key.data = "WARNING";
      key.len = strlen("WARNING");
      value.data = NULL;
      value.len = 0;
      chash_set(log_filter, &key, &value, NULL);
      
      while (env_value != NULL) {
        char * p;
        
        p = strchr(env_value, ' ');
        if (p != NULL) {
          * p = '\0';
          key.data = env_value;
          key.len = strlen(env_value);
          value.data = NULL;
          value.len = 0;
          chash_set(log_filter, &key, &value, NULL);
          
          env_value = p + 1;
        }
        else {
          key.data = env_value;
          key.len = strlen(env_value);
          value.data = NULL;
          value.len = 0;
          chash_set(log_filter, &key, &value, NULL);
          
          env_value = p;
        }
      }
    }
  }
  pthread_mutex_unlock(&log_lock);
}

void etpan_log_done(void)
{
  pthread_mutex_lock(&log_lock);
  if (log_filter != NULL) {
    chash_free(log_filter);
    log_filter = NULL;
  }
  pthread_mutex_unlock(&log_lock);
}

void etpan_log(char * log_id, char * filename, unsigned int line_number,
    char * format, ...)
{
  va_list argp;
  struct timeval time_info;
  int r;
  chashdatum key;
  chashdatum value;
  int show_log;
  
  etpan_log_init();
  
  show_log = 0;
  key.data = log_id;
  key.len = strlen(log_id);
  r = chash_get(log_filter, &key, &value);
  if (r == 0) {
    show_log = 1;
  }
  
   gettimeofday(&time_info, NULL);
   
   va_start(argp, format);
   pthread_mutex_lock(&log_lock);
   vsnprintf(log_suffix, sizeof(log_suffix), format, argp);
   snprintf(log_line, sizeof(log_line), "%4lu.%03u [%s] %s (%s:%u)",
       time_info.tv_sec % 3600,
       (unsigned int) (time_info.tv_usec / 1000),
       log_id,
       log_suffix, filename, line_number);
   
   if (show_log)
     etpan_str_log();
   
   if (etpan_log_callback != NULL)
     etpan_log_callback(log_line);
   
   pthread_mutex_unlock(&log_lock);
   va_end(argp);
}

#if defined(__APPLE__) && defined(__MACH__)
#include <mach/vm_types.h>

extern void thread_stack_pcs(vm_address_t *buffer,
    unsigned max, unsigned *num);

void etpan_log_stack(void)
{
  unsigned buffer[256];
  unsigned int num_frames;
  unsigned int i;
  char output[1024];
  char * current_output;
  size_t remaining;
  
  thread_stack_pcs(buffer, sizeof(buffer) / sizeof(buffer[0]), &num_frames);
  remaining = sizeof(output);
  current_output = output;
  for(i = 0 ; i < num_frames ; i ++) {
    size_t len;
    
    snprintf(current_output, remaining, "0x%x ", buffer[i]);
    len = strlen(current_output);
    remaining -= len;
    current_output += len;
    if (remaining == 0)
      break;
  }
  ETPAN_STACK_LOG(output);
}
#else
#if defined(__linux__)
#include <execinfo.h>
#include "etpan-symbols.h"
#include <pthread.h>
#define HAVE_BFD

struct etpan_symbol_table * symtable = NULL;
static pthread_mutex_t symtable_lock = PTHREAD_MUTEX_INITIALIZER;

static void symbol_init(void)
{
  pthread_mutex_lock(&symtable_lock);
  if (symtable == NULL)
    symtable = etpan_get_symtable(getpid());
  pthread_mutex_unlock(&symtable_lock);
}

static const char * log_basename(const char * basename)
{
  const char * result;
  const char * p;
  
  result = basename;
  p = result;
  
  while ((p = strchr(result, '/')) != NULL) {
    result = p + 1;
  }
  
  return result;
}

void etpan_log_stack(void)
{
  void * buffer[256];
  int num_frames;
  int i;
  char output[8192];
  char * current_output;
  size_t remaining;
  size_t len;
  
  symbol_init();
  num_frames = backtrace(buffer, sizeof(buffer) / sizeof(buffer[0]));
  
  remaining = sizeof(output);
  * output = 0;
  current_output = output;
  
#ifndef HAVE_BFD
  snprintf(current_output, remaining, "stack (use addr2line to resolve):\n");
  len = strlen(current_output);
  remaining -= len;
  current_output += len;
#else
  snprintf(current_output, remaining, "stack:\n");
  len = strlen(current_output);
  remaining -= len;
  current_output += len;
#endif
  
  for(i = 0 ; i < num_frames ; i ++) {
#ifndef HAVE_BFD
    snprintf(current_output, remaining, "%p\n", buffer[i]);
    len = strlen(current_output);
    remaining -= len;
    current_output += len;
    if (remaining == 0)
      break;
#else
    struct etpan_debug_symbol data;
    int r;
    
    r = etpan_get_symbol(symtable, buffer[i], &data);
    if (r < 0) {
      snprintf(current_output, remaining, "%p\n", buffer[i]);
      len = strlen(current_output);
      remaining -= len;
      current_output += len;
      if (remaining == 0)
        break;
    }
    else {
      const char *name;
      char address_str[32];
      
      name = data.functionname;
      if (name == NULL || *name == '\0') {
        snprintf(address_str, sizeof(address_str), "%p",
            buffer[i]);
        name = address_str;
      }
      
      if (data.filename != NULL) {
        snprintf(current_output, remaining, "%p %s (in %s) %s:%u\n",
            buffer[i], name,
            log_basename(data.libname),
            log_basename(data.filename), data.line);
      }
      else {
        snprintf(current_output, remaining, "%p %s (in %s)\n",
            buffer[i], name, log_basename(data.libname));
      }
    }
    
    len = strlen(current_output);
    remaining -= len;
    current_output += len;
    if (remaining == 0)
      break;
#endif
  }
  ETPAN_STACK_LOG(output);
}

#else

void etpan_log_stack(void)
{
  ETPAN_STACK_LOG("this feature not available");
}

#endif
#endif

void etpan_crash(void)
{
  etpan_crash_report_unsetup();
  ETPAN_STACK_LOG("crash at:");
  etpan_log_stack();
  * (int *) 0x0 = 12345;
}

static struct sigaction old_act[32];

static void crash_report(int signal_value, siginfo_t * siginfo, void * data)
{
  char * signal_name;
  
  signal_name = "Unknown";
  switch (signal_value) {
  case SIGQUIT:
    signal_name = "quit from keyboard";
    break;
  case SIGILL:
    signal_name = "illegal instruction";
    break;
  case SIGABRT:
    signal_name = "abort signal from abort()";
    break;
  case SIGFPE:
    signal_name = "floating point exception";
    break;
  case SIGSEGV:
    signal_name = "invalid memory reference";
    break;
  case SIGBUS:
    signal_name = "bus error (bad memory access)";
    break;
  case SIGSYS:
    signal_name = "bad argument to routine";
    break;
  case SIGTRAP:
    signal_name = "trace/breakpoint trap";
    break;
  case SIGXCPU:
    signal_name = "cpu time limit exceeded";
    break;
  case SIGXFSZ:
    signal_name = "file size limit exceeded";
    break;
  }
  ETPAN_WARN_LOG("signal %s (%i) was raised", signal_name, signal_value);

  etpan_crash();
  exit(1);
}

void etpan_crash_report_setup(void)
{
  struct sigaction act;
  
  memset(&act, 0, sizeof(act));
  act.sa_sigaction = crash_report;
  
  sigaction(SIGQUIT, &act, &old_act[SIGQUIT]);
  sigaction(SIGILL, &act, &old_act[SIGILL]);
  sigaction(SIGABRT, &act, &old_act[SIGABRT]);
  sigaction(SIGFPE, &act, &old_act[SIGFPE]);
  sigaction(SIGSEGV, &act, &old_act[SIGSEGV]);
  sigaction(SIGBUS, &act, &old_act[SIGBUS]);
  sigaction(SIGSYS, &act, &old_act[SIGSYS]);
  sigaction(SIGTRAP, &act, &old_act[SIGTRAP]);
  sigaction(SIGXCPU, &act, &old_act[SIGXCPU]);
  sigaction(SIGXFSZ, &act, &old_act[SIGXFSZ]);
}

void etpan_crash_report_unsetup(void)
{
  struct sigaction act;
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = SIG_DFL;
  sigaction(SIGQUIT, &act, NULL);
  sigaction(SIGILL, &act, NULL);
  sigaction(SIGABRT, &act, NULL);
  sigaction(SIGFPE, &act, NULL);
  sigaction(SIGSEGV, &act, NULL);
  sigaction(SIGBUS, &act, NULL);
  sigaction(SIGSYS, &act, NULL);
  sigaction(SIGTRAP, &act, NULL);
  sigaction(SIGXCPU, &act, NULL);
  sigaction(SIGXFSZ, &act, NULL);
}
