#include "etpan-mail-manager.h"

#include <stdlib.h>
#include <string.h>

#define ETPAN_MODULE_LOG_NAME "MAILMANAGER"

#include "etpan-log.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-error.h"
#include "etpan-time.h"
#include "etpan-thread-manager-app.h"
#include "etpan-msg-list-fetch.h"
#include "etpan-signal.h"
#include "etpan-nls.h"
#include "etpan-message.h"
#include "etpan-message-private.h"
#include "etpan-folder-indexer.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-storage-folder-order.h"
#include "etpan-message-copy.h"
#include "etpan-message-delete.h"
#include "etpan-folder-copy.h"
#include "etpan-uuid.h"
#include "etpan-utils.h"
#include "etpan-imap-sync.h"
#include "etpan-storage-imap-sync-private.h"
#include "etpan-folder-imap-sync.h"
#include "etpan-folder-imap-sync-private.h"
#include "etpan-imap-folder-sync.h"
#include "etpan-imap-folder-sync-private.h"
#include "etpan-pop-sync.h"
#include "etpan-storage-pop-sync-private.h"
#include "etpan-folder-filter.h"
#include "etpan-filter-config.h"
#include "etpan-folder-create.h"

#define LONG_INTERVAL (5 * 60) /* 5 minutes */
#define SOON_INTERVAL (0.1) /* 0.1 sec */
#define NOW_INTERVAL 0 /* 0 sec */


/* storage */

enum {
  STORAGE_STATE_IDLE,
  STORAGE_STATE_FOLDER_OP,
  STORAGE_STATE_UPDATE,
};

enum {
  STORAGE_OP_TYPE_NONE,
  STORAGE_OP_TYPE_UPDATE,
  STORAGE_OP_TYPE_FOLDER,
};

struct storage_state {
  struct etpan_mail_manager * manager;
  struct etpan_storage * storage;
  
  struct etpan_error * error;
  struct etpan_storage_folder_order * order;
  chash * folder_to_state;
  
  int stop;
  int state;
  int update_requested;
  double update_request_date;
  struct etpan_thread_op * op;
  
  int filter_running;
  int imap_sync_syncing;
  int pop_sync_syncing;
  
  struct etpan_folder_filter * inbox_filter;
};

static struct storage_state * storage_state_new(void);
static void storage_state_free(struct storage_state * state);
static void storage_cancel(struct storage_state * state);
static void storage_state_stop(struct storage_state * state);
static int storage_state_is_stopped(struct storage_state * state);
static void storage_state_set_error(struct storage_state * state,
    struct etpan_error * error);
static struct storage_state *
get_storage_state(struct etpan_mail_manager * manager,
    struct etpan_storage * storage);
static void remove_storage_state(struct etpan_mail_manager * manager,
    struct etpan_storage * storage);
static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void folder_order_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void open_storage(struct storage_state * state);
static void close_storage(struct storage_state * state);
static void storage_schedule_update(struct storage_state * state,
    double delay);
static void storage_stop_folders(struct storage_state * state);
static void storage_cleanup_stopped_folders(struct storage_state * state);
static void storage_update(struct storage_state * state);
static void update_list_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * callback_data);
static void update_finished(struct storage_state * state);
static double storage_get_next_task_delay(struct storage_state * state);
static int storage_get_next_task_type(struct storage_state * state);
static void storage_perform_next_task(struct storage_state * state);
static void storage_folder_task_done(struct storage_state * state);


/* folder */

enum {
  FOLDER_STATE_IDLE,
  FOLDER_STATE_STATUS,
  FOLDER_STATE_CHECK,
};

enum {
  FOLDER_OP_TYPE_NONE,
  FOLDER_OP_TYPE_CONTENT,
  FOLDER_OP_TYPE_STATUS,
  FOLDER_OP_TYPE_CHECK,
};

struct folder_state {
  struct etpan_mail_manager * manager;
  struct storage_state * storage_state;
  
  struct etpan_folder * folder;
  int first_filter;
  int opened_count;
  int state;
  int stop;
  
  struct etpan_error * error;
  int has_count;
  int unread_count;
  int total_count;
  
  /*
    - status
    This will be requested periodically. And sometimes forced.
  */
  int status_requested;
  double status_request_date;
  
  /*
    - check
    When flags has been changed, a check will be requested
    (all msg will be checked)
    check will be first done
    will look at flags to see whether expunge is necessary
    (when msg_list is enabled),
    followed by status
  */
  int check_requested;
  double check_request_date;
  int check_needed;
  int expunge_needed;
  chash * msg_to_check;
  
  struct etpan_msg_list_fetch * fetcher;
  struct etpan_thread_op * op;
  struct etpan_folder_indexer * indexer;
};

static struct folder_state * folder_state_new(void);
static void folder_state_free(struct folder_state * state);
static void folder_state_stop(struct folder_state * state);
static void folder_cancel(struct folder_state * state);
static int folder_is_stopped(struct folder_state * state);
static void folder_perform_next_task(struct folder_state * state);
static double folder_get_next_task_delay(struct folder_state * state);
static int folder_get_next_task_type(struct folder_state * state);
static void folder_state_set_error(struct folder_state * state,
    struct etpan_error * error);
static void remove_folder_state(struct storage_state * storage_state, struct etpan_folder * folder);
static struct folder_state * get_folder_state(struct storage_state * storage_state, struct etpan_folder * folder);

static void open_folder(struct folder_state * state);
static void indexer_state_changed(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void indexer_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void close_folder(struct folder_state * storage_state);

static void schedule_status_folder(struct folder_state * state, double delay);
static void status_folder(struct folder_state * state);
static void folder_status_callback(int cancelled,
    struct etpan_folder_status_result * result, void * callback_data);
static void fetched_with_lost(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void status_finished(struct folder_state * state);

static void schedule_check_folder(struct folder_state * state, double delay);
static void check_folder(struct folder_state * folder);
static void do_check_msg_list(struct folder_state * state);
static void check_msg_list_callback(int cancelled,
    struct etpan_folder_check_msg_list_result * result, void * callback_data);
static void do_check(struct folder_state * state);
static void check_callback(int cancelled,
    struct etpan_folder_check_result * result, void * callback_data);
static void do_expunge(struct folder_state * state);
static void expunge_callback(int cancelled,
    struct etpan_folder_expunge_result * result, void * cb_data);
static void do_status(struct folder_state * state);
static void check_finished(struct folder_state * state);


/* main */

static void account_list_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void stop_storages(struct etpan_mail_manager * manager);
static void cleanup_stopped_storages(struct etpan_mail_manager * manager);
static void disconnect_before_destroy_callback(void * cb_data);
static void update_storage(struct etpan_mail_manager * manager);
static void reschedule(struct etpan_mail_manager * manager);
static void task_done(struct etpan_mail_manager * manager);




static struct etpan_mail_manager * default_manager = NULL;

struct etpan_mail_manager * etpan_mail_manager_new(void)
{
  struct etpan_mail_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->stop = 0;
  manager->stop_callback = NULL;
  manager->stop_cb_data = NULL;
  manager->storage_to_state = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager->storage_to_state == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  manager->copy_msg_error = NULL;
  manager->copy_msg_op_list = carray_new(16);
  if (manager->copy_msg_op_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  manager->delete_msg_error = NULL;
  manager->delete_msg_op_list = carray_new(16);
  if (manager->delete_msg_op_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  manager->copy_folder_error = NULL;
  manager->copy_folder_op_list = carray_new(16);
  if (manager->copy_folder_op_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  manager->delete_msg_error = NULL;
  manager->delete_msg_op_list = carray_new(16);
  if (manager->delete_msg_op_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return manager;
}

void etpan_mail_manager_free(struct etpan_mail_manager * manager)
{
  carray_free(manager->copy_folder_op_list);
  carray_free(manager->delete_msg_op_list);
  carray_free(manager->copy_msg_op_list);
  chash_free(manager->storage_to_state);
  free(manager);
}

void etpan_mail_manager_set_default(struct etpan_mail_manager * manager)
{
  default_manager = manager;
}

struct etpan_mail_manager * etpan_mail_manager_get_default(void)
{
  return default_manager;
}

static void imap_sync_open_folder(struct etpan_folder * folder)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = etpan_folder_imap_sync_get_folder_sync(folder);
  etpan_imap_folder_set_sync_now(folder_sync);
  etpan_imap_folder_sync_open(folder_sync);
}

static void imap_sync_close_folder(struct etpan_folder * folder)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = etpan_folder_imap_sync_get_folder_sync(folder);
  etpan_imap_folder_sync_close(folder_sync);
}

void etpan_mail_manager_folder_open(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct folder_state * state;
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  struct etpan_folder_driver * driver;
  
  driver = etpan_folder_get_driver(folder);
  if (strcmp(driver->name, "imap-sync") == 0) {
    imap_sync_open_folder(folder);
  }
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  state->opened_count ++;
  if (state->opened_count == 1)
    open_folder(state);
}

void etpan_mail_manager_folder_close(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct folder_state * state;
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  struct etpan_folder_driver * driver;
  
  driver = etpan_folder_get_driver(folder);
  if (strcmp(driver->name, "imap-sync") == 0) {
    imap_sync_close_folder(folder);
  }
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  state->opened_count --;
  if (state->opened_count == 0)
    close_folder(state);
}

void etpan_mail_manager_folder_check(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  struct folder_state * state;
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  schedule_check_folder(state, SOON_INTERVAL);
}

struct etpan_folder_indexer * etpan_mail_manager_get_indexer(struct etpan_mail_manager * manager, struct etpan_folder * folder)
{
  struct folder_state * state;
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  if (state->opened_count == 0)
    ETPAN_CRASH("getting indexer on folder that was not opened");
  
  return state->indexer;
}


double etpan_mail_manager_next_task_date(struct etpan_mail_manager * manager)
{
  /* find minimum date on all storage */
  chashiter * iter;
  double min_time;
  
  min_time = etpan_get_time() + 24 * 60 * 60; /* + 1 day */
  
  /* perform all storage task */
  for(iter = chash_begin(manager->storage_to_state) ; iter != NULL ;
      iter = chash_next(manager->storage_to_state, iter)) {
    struct storage_state * state;
    chashdatum value;
    double storage_time;
    
    chash_value(iter, &value);
    state = value.data;
    storage_time = storage_get_next_task_delay(state);
    if (storage_time < min_time)
      min_time = storage_time;
  }
  ETPAN_LOCAL_LOG("next task date %g", min_time);
  
  return min_time;
}

void etpan_mail_manager_perform_next_task(struct etpan_mail_manager * manager)
{
  chashiter * iter;
  
  /* perform all storage task */
  for(iter = chash_begin(manager->storage_to_state) ; iter != NULL ;
      iter = chash_next(manager->storage_to_state, iter)) {
    struct storage_state * state;
    chashdatum value;
    
    chash_value(iter, &value);
    state = value.data;
    storage_perform_next_task(state);
  }
}

void etpan_mail_manager_storage_update(struct etpan_mail_manager * manager,
    struct etpan_storage * storage)
{
  struct storage_state * state;
  
  state = get_storage_state(manager, storage);
  if (state->state == STORAGE_STATE_UPDATE)
    return;
  
  storage_schedule_update(state, NOW_INTERVAL);
}

void etpan_mail_manager_remove_storage(struct etpan_mail_manager * manager,
    struct etpan_storage * storage)
{
  struct etpan_account * account;
  
  account = etpan_storage_get_account(storage);
  etpan_account_manager_remove_account(etpan_account_manager_get_default(),
      account);
  stop_storages(manager);
}

struct etpan_error *
etpan_mail_manager_storage_get_error(struct etpan_mail_manager * manager,
    struct etpan_storage * storage)
{
  struct storage_state * state;
  
  state = get_storage_state(manager, storage);
  return state->error;
}

struct etpan_storage_folder_order *
etpan_mail_manager_storage_get_order(struct etpan_mail_manager * manager,
    struct etpan_storage * storage)
{
  struct storage_state * state;
  
  state = get_storage_state(manager, storage);
  return state->order;
}

struct etpan_error *
etpan_mail_manager_folder_get_error(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct storage_state * storage_state;
  struct folder_state * state;
  struct etpan_storage * storage;
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  state = get_folder_state(storage_state, folder);
  return state->error;
}

void etpan_mail_manager_setup(struct etpan_mail_manager * manager)
{
  update_storage(manager);
  ETPAN_SIGNAL_ADD_HANDLER(etpan_account_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      account_list_updated, manager);
}

void etpan_mail_manager_stop(struct etpan_mail_manager * manager,
    void (* callback)(void *),
    void * user_data)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->copy_msg_op_list) ; i ++) {
    struct etpan_message_copy * op;
    
    op = carray_get(manager->copy_msg_op_list, i);
    if (op == NULL)
      continue;
    
    etpan_message_copy_cancel(op);
  }
  for(i = 0 ; i < carray_count(manager->copy_folder_op_list) ; i ++) {
    struct etpan_folder_copy * op;
    
    op = carray_get(manager->copy_folder_op_list, i);
    if (op == NULL)
      continue;
    
    etpan_folder_copy_cancel(op);
  }
  for(i = 0 ; i < carray_count(manager->delete_msg_op_list) ; i ++) {
    struct etpan_message_delete * op;
    
    op = carray_get(manager->delete_msg_op_list, i);
    if (op == NULL)
      continue;
    
    etpan_message_delete_cancel(op);
  }
  
  manager->stop = 1;
  manager->stop_callback = callback;
  manager->stop_cb_data = user_data;
  stop_storages(manager);
}

void etpan_mail_manager_unsetup(struct etpan_mail_manager * manager)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_account_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      account_list_updated, manager);
  if (chash_count(manager->storage_to_state) > 0) {
    ETPAN_WARN_LOG("storage still remains");
  }
}

int etpan_mail_manager_folder_has_count(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct folder_state * state;
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  chashdatum key;
  chashdatum value;
  int r;
  
  storage = etpan_folder_get_storage(folder);
  
  key.data = &storage;
  key.len = sizeof(storage);
  r = chash_get(manager->storage_to_state, &key, &value);
  if (r < 0)
    return 0;
  
  storage_state = get_storage_state(manager, storage);
  
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(storage_state->folder_to_state, &key, &value);
  if (r < 0)
    return 0;
  
  state = get_folder_state(storage_state, folder);
  return state->has_count;
}

unsigned int etpan_mail_manager_folder_get_total_count(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct folder_state * state;
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  return state->total_count;
}

unsigned int etpan_mail_manager_folder_get_unread_count(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct folder_state * state;
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  return state->unread_count;
}

void etpan_mail_manager_folder_update(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  struct etpan_storage * storage;
  struct storage_state * storage_state;
  struct folder_state * state;
  
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  
  state = get_folder_state(storage_state, folder);
  if (state->state == FOLDER_OP_TYPE_CONTENT)
    return;
  
  schedule_status_folder(state, NOW_INTERVAL);
}

static void account_list_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_mail_manager * manager;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  manager = user_data;
  update_storage(manager);
}

static void update_storage(struct etpan_mail_manager * manager)
{
  chashiter * iter;
  chash * account_hash;
  
  if (manager->stop)
    return;
  
  account_hash = etpan_account_manager_get_account_hash(etpan_account_manager_get_default());
  for(iter = chash_begin(account_hash) ; iter != NULL ;
      iter = chash_next(account_hash, iter)) {
    chashdatum value;
    struct etpan_account * account;
    struct etpan_storage * storage;
    
    chash_value(iter, &value);
    account = value.data;
    storage = etpan_account_get_storage(account);
    
    get_storage_state(manager, storage);
  }
  
  stop_storages(manager);
}

static void stop_storages(struct etpan_mail_manager * manager)
{
  chashiter * iter;
  chash * account_hash;
  int r;
  carray * account_list;
  unsigned int i;
  
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  account_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (account_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    chashdatum key;
    chashdatum value;
    struct etpan_storage * storage;
    struct etpan_account * account;
    
    account = carray_get(account_list, i);
    storage = etpan_account_get_storage(account);
    key.data = &storage;
    key.len = sizeof(storage);
    value.data = NULL;
    value.len = 0;
    r = chash_set(account_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(iter = chash_begin(manager->storage_to_state) ; iter != NULL ;
      iter = chash_next(manager->storage_to_state, iter)) {
    struct storage_state * state;
    chashdatum key;
    chashdatum value;
    char * name;
    int do_stop;
    
    chash_value(iter, &value);
    state = value.data;
    
    do_stop = 0;
    
    name = etpan_storage_get_id(state->storage);
    key.data = &state->storage;
    key.len = sizeof(state->storage);
    r = chash_get(account_hash, &key, &value);
    if (r < 0) {
      do_stop = 1;
    }
    
    if (manager->stop) {
      do_stop = 1;
    }
    
    if (do_stop) {
      if (!state->stop) {
        storage_state_stop(state);
      }
    }
  }
  chash_free(account_hash);
  
  cleanup_stopped_storages(manager);
}

static void cleanup_stopped_storages(struct etpan_mail_manager * manager)
{
  chashiter * iter;
  carray * account_list;
  unsigned int i;
  int r;
  chash * account_hash;
  
  /* could be improved by creating a list of pending folders */
  
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  account_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (account_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    chashdatum key;
    chashdatum value;
    struct etpan_storage * storage;
    struct etpan_account * account;
    
    account = carray_get(account_list, i);
    storage = etpan_account_get_storage(account);
    key.data = &storage;
    key.len = sizeof(storage);
    value.data = NULL;
    value.len = 0;
    r = chash_set(account_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  account_list = carray_new(chash_count(manager->storage_to_state));
  if (account_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(manager->storage_to_state) ; iter != NULL ;
      iter = chash_next(manager->storage_to_state, iter)) {
    struct storage_state * state;
    chashdatum value;
    
    chash_value(iter, &value);
    state = value.data;
    r = carray_add(account_list, state, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    struct storage_state * state;
    struct etpan_storage * storage;
    struct etpan_account * account;
    chashdatum key;
    chashdatum value;
    
    state = carray_get(account_list, i);
    
    if (!storage_state_is_stopped(state))
      continue;
    
    storage = state->storage;
    storage_cleanup_stopped_folders(state);
    ETPAN_LOCAL_LOG("remove storage %s", storage->id);
    remove_storage_state(manager, storage);
    
    key.data = &storage;
    key.len = sizeof(storage);
    r = chash_get(account_hash, &key, &value);
    if (r == 0) {
      /* account will be stopped by account manager */
      continue;
    }
    
    account = etpan_storage_get_account(storage);
    etpan_account_stop(account, disconnect_before_destroy_callback, account);
  }
  
  carray_free(account_list);
  chash_free(account_hash);
  
  if (manager->stop) {
    if (chash_count(manager->storage_to_state) == 0) {
      if (manager->stop_callback != NULL) {
        manager->stop_callback(manager->stop_cb_data);
        manager->stop_callback = NULL;
      }
    }
  }
}

static void disconnect_before_destroy_callback(void * cb_data)
{
  struct etpan_account * account;
  
  account = cb_data;
  etpan_account_unsetup(account);
  etpan_account_free(account);
}

static void reschedule(struct etpan_mail_manager * manager)
{
  ETPAN_SIGNAL_SEND(manager, ETPAN_MAIL_MANAGER_SCHEDULE_SIGNAL);
}

static void task_done(struct etpan_mail_manager * manager)
{
  cleanup_stopped_storages(manager);
  reschedule(manager);
}


/* **************************************** */
/* storage */

static struct storage_state * storage_state_new(void)
{
  struct storage_state * state;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  state->manager = NULL;
  state->storage = NULL;
  state->error = NULL;
  state->order = NULL;
  state->inbox_filter = NULL;
  state->folder_to_state = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (state->folder_to_state == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  state->stop = 0;
  state->state = STORAGE_STATE_IDLE;
  state->update_requested = 0;
  state->update_request_date = 0;
  state->op = NULL;
  state->imap_sync_syncing = 0;
  state->pop_sync_syncing = 0;
  state->filter_running = 0;
  
  return state;
}

static void storage_state_free(struct storage_state * state)
{
  chash_free(state->folder_to_state);
  free(state);
}

static int storage_state_is_stopped(struct storage_state * state)
{
  int type;
  chashiter * iter;
  
  if (!state->stop)
    return 0;
  
  ETPAN_LOCAL_LOG("stopped : %s %i %i", state->storage->id, state->pop_sync_syncing,
            state->imap_sync_syncing);
  if (state->filter_running)
    return 0;
  if (state->imap_sync_syncing)
    return 0;
  if (state->pop_sync_syncing)
    return 0;
  
  if (state->state != STORAGE_STATE_IDLE)
    return 0;
  
  for(iter = chash_begin(state->folder_to_state) ; iter != NULL ;
      iter = chash_next(state->folder_to_state, iter)) {
    chashdatum value;
    struct folder_state * folder_state;
    
    chash_value(iter, &value);
    folder_state = value.data;
    if (!folder_is_stopped(folder_state)) {
      return 0;
    }
  }
  
  type = storage_get_next_task_type(state);
  if (type != STORAGE_OP_TYPE_NONE) {
    return 0;
  }
  
  return 1;
}

static void storage_pop_sync_cancel(struct storage_state * state);
static void storage_filter_cancel(struct storage_state * state);

static void storage_state_stop(struct storage_state * state)
{
  int schedule_needed;
  chashiter * iter;
  
  state->stop = 1;
  schedule_needed = 0;
  
  storage_filter_cancel(state);
  storage_pop_sync_cancel(state);
  
  if (state->update_requested) {
    state->update_requested = 0;
    schedule_needed = 1;
  }
  
  for(iter = chash_begin(state->folder_to_state) ; iter != NULL ;
      iter = chash_next(state->folder_to_state, iter)) {
    chashdatum value;
    struct folder_state * folder_state;
    
    chash_value(iter, &value);
    folder_state = value.data;
    folder_state_stop(folder_state);
  }
  
  if (schedule_needed)
    reschedule(state->manager);
}

static void storage_state_set_error(struct storage_state * state,
    struct etpan_error * error)
{
  ETPAN_ERROR_FREE(state->error);
  if (error != NULL) {
    etpan_error_log(error);
    state->error = etpan_error_dup(error);
  }
  else
    state->error = NULL;
}

static struct storage_state *
get_storage_state(struct etpan_mail_manager * manager,
    struct etpan_storage * storage)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct storage_state * state;
  
  key.data = &storage;
  key.len = sizeof(storage);
  r = chash_get(manager->storage_to_state, &key, &value);
  if (r == 0) {
    state = value.data;
    
    return state;
  }
  
  if (manager->stop) {
    ETPAN_WARN_LOG("creation of storage state while manager is stopped");
    etpan_log_stack();
  }
  
  state = storage_state_new();
  state->manager = manager;
  state->storage = storage;
  
  value.data = state;
  value.len = 0;
  r = chash_set(manager->storage_to_state, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  open_storage(state);
  
  return state;
}

static void remove_storage_state(struct etpan_mail_manager * manager,
    struct etpan_storage * storage)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct storage_state * state;
  
  key.data = &storage;
  key.len = sizeof(storage);
  r = chash_get(manager->storage_to_state, &key, &value);
  if (r < 0)
    return;

  chash_delete(manager->storage_to_state, &key, NULL);
  state = value.data;
  close_storage(state);

  storage_state_free(state);
}

static void imap_sync_storage_content_handler(char * signal_name,
    void * sender, void * signal_data, void * user_data);

static void storage_filter_setup(struct storage_state * state)
{
  struct etpan_folder_filter * folder_filter;
  struct etpan_folder * inbox;
  struct etpan_storage_driver * driver;
  char * path;
  
  if (state->inbox_filter != NULL)
    return;
  
  inbox = NULL;
  
  driver = state->storage->driver;
  
  ETPAN_LOCAL_LOG("filter: %s", driver->name);
  
  if (strcasecmp(driver->name, "imap-sync") == 0) {
    inbox = etpan_storage_get_folder(state->storage, "INBOX");
  }
  else if (strcasecmp(driver->name, "pop-sync") == 0) {
    inbox = etpan_storage_get_folder(state->storage, "");
  }
  
  if (inbox == NULL) {
    return;
  }
  
  folder_filter = etpan_folder_filter_new();
  ETPAN_LOCAL_LOG("set default filter %s %p %p",
      etpan_storage_get_id(state->storage),
      folder_filter,
      etpan_filter_config_get_default());
  etpan_folder_filter_set_folder(folder_filter, inbox);
  path = etpan_folder_filter_get_path_for_folder(folder_filter);
  etpan_folder_filter_set_path(folder_filter, path);
  free(path);
  etpan_folder_filter_set_filter(folder_filter,
      etpan_filter_config_get_default());
  etpan_folder_filter_setup(folder_filter);
  state->inbox_filter = folder_filter;
}

static void storage_filter_unsetup(struct storage_state * state)
{
  if (state->inbox_filter == NULL)
    return;
  
  etpan_folder_filter_unref(state->inbox_filter);
  state->inbox_filter = NULL;
}

static void storage_filter_cancel(struct storage_state * state)
{
  if (state->inbox_filter == NULL)
    return;
  
  etpan_folder_filter_cancel(state->inbox_filter);
}

static void filter_step_callback(void * data);
static void filter_callback(void * data);

static void storage_filter_run(struct storage_state * state)
{
  if (state->inbox_filter == NULL)
    return;
  
  state->filter_running = 1;
  etpan_folder_filter_run(state->inbox_filter,
      filter_step_callback,
      filter_callback,
      state);
}

static void filter_step_callback(void * data)
{
  struct storage_state * state;
  
  state = data;
}

static void filter_callback(void * data)
{
  struct storage_state * state;
  
  state = data;
  state->filter_running = 0;
  
  task_done(state->manager);
}

static void open_storage(struct storage_state * state)
{
  struct etpan_storage_folder_order * folder_order;
  char * path;
  
  folder_order = etpan_storage_folder_order_new();
  path = etpan_storage_folder_order_get_path_for_storage(folder_order,
      state->storage);
  etpan_storage_folder_order_set_path(folder_order, path);
  free(path);
  etpan_storage_folder_order_set_storage(folder_order, state->storage);
  etpan_storage_folder_order_setup(folder_order);
  
  state->order = folder_order;
  
  storage_filter_setup(state);
  
  ETPAN_SIGNAL_ADD_HANDLER(state->storage ,
      ETPAN_STORAGE_FOLDERLIST_UPDATED,
      folder_list_updated_handler,
      state);
  
  ETPAN_SIGNAL_ADD_HANDLER(folder_order,
      ETPAN_STORAGE_FOLDER_ORDER_CHANGED,
      folder_order_updated_handler,
      state);
  
  ETPAN_SIGNAL_ADD_HANDLER(state->storage,
      ETPAN_STORAGE_IMAP_SYNC_CONTENT_UPDATED_ENDED_SIGNAL,
      imap_sync_storage_content_handler, state);
  
  storage_schedule_update(state, SOON_INTERVAL);
}

static void imap_sync_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void close_storage(struct storage_state * state)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(state->storage,
      ETPAN_STORAGE_IMAP_SYNC_CONTENT_UPDATED_ENDED_SIGNAL,
      imap_sync_storage_content_handler, state);
  
  ETPAN_SIGNAL_REMOVE_HANDLER(state->order,
      ETPAN_STORAGE_FOLDER_ORDER_CHANGED,
      folder_order_updated_handler,
      state);
  
  ETPAN_SIGNAL_REMOVE_HANDLER(state->storage ,
      ETPAN_STORAGE_FOLDERLIST_UPDATED,
      folder_list_updated_handler,
      state);
  
  storage_cancel(state);
  storage_filter_unsetup(state);
  etpan_storage_folder_order_unsetup(state->order);
  etpan_storage_folder_order_free(state->order);
  state->order = NULL;
}

static void imap_sync_storage_content_handler(char * signal_name,
    void * sender, void * signal_data, void * user_data)
{
  struct storage_state * state;
  
  state = user_data;
  storage_schedule_update(state, SOON_INTERVAL);
}

static chash * get_uuid_hash(struct etpan_account * account)
{
  char * path;
  FILE * f;
  char buf[PATH_MAX];
  chash * uuid_hash;
  int r;
  
  uuid_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  
  path = etpan_get_uuid_filename(etpan_account_manager_get_default(), account);
  f = fopen(path, "r");
  if (f == NULL) {
    free(path);
    return uuid_hash;
  }
  
  while (fgets(buf, sizeof(buf), f)) {
    char * p;
    chashdatum key;
    chashdatum value;
    char * uuid;
    char * location;
    
    p = strchr(buf, '\n');
    if (p == NULL)
      continue;
    
    * p = '\0';
    
    p = strchr(buf, ' ');
    if (p == NULL)
      continue;
    
    * p = '\0';
    uuid = buf;
    location = p + 1;
    
    key.data = location;
    key.len = strlen(location) + 1;
    value.data = uuid;
    value.len = strlen(uuid) + 1;
    r = chash_set(uuid_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  fclose(f);
  
  free(path);
  
  return uuid_hash;
}

static void save_uuid_hash(struct etpan_account * account)
{
  struct etpan_storage * storage;
  chash * folder_hash;
  chashiter * iter;
  char * path;
  FILE * f;
  
  path = etpan_get_uuid_filename(etpan_account_manager_get_default(), account);
  f = fopen(path, "w");
  if (f == NULL) {
    free(path);
    return;
  }
  
  storage = etpan_account_get_storage(account);
  folder_hash = etpan_storage_get_folder_list(storage);
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    char * uid;
    char * location;
    
    chash_value(iter, &value);
    folder = value.data;
    uid = etpan_folder_get_uid(folder);
    location = etpan_folder_get_location(folder);
    fprintf(f, "%s %s\n", uid, location);
  }
  
  fclose(f);
  free(path);
}

static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  chash * folder_hash;
  struct storage_state * state;
  chashiter * iter;
  chash * uuid_hash;
  int r;
  int has_new;
  
  has_new = 0;
  uuid_hash = NULL;
  state = user_data;
  folder_hash = etpan_storage_get_folder_list(state->storage);
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_folder * folder;
    char * uid;
    
    chash_value(iter, &value);
    folder = value.data;
    uid = etpan_folder_get_uid(folder);
    if (uid == NULL) {
      char * location;
      
      if (uuid_hash == NULL) {
        uuid_hash = get_uuid_hash(etpan_storage_get_account(state->storage));
      }
      
      location = etpan_folder_get_location(folder);
      key.data = location;
      key.len = strlen(location) + 1;
      r = chash_get(uuid_hash, &key, &value);
      if (r == 0) {
        uid = strdup(value.data);
      }
      else {
        uid = etpan_uuid_generate();
        has_new = 1;
      }
      
      etpan_folder_set_uid(folder, uid);
      free(uid);
    }
  }
  
  if (has_new)
    save_uuid_hash(etpan_storage_get_account(state->storage));
  
  if (uuid_hash != NULL)
    chash_free(uuid_hash);
  
  storage_filter_setup(state);
}

static void folder_order_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct storage_state * state;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  state = user_data;
  
  if (state->stop)
    return;
  
  ETPAN_SIGNAL_SEND_WITH_DATA(state->manager,
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL, state->storage);
}

static void storage_cancel(struct storage_state * state)
{
  switch (state->state) {
  case STORAGE_STATE_UPDATE:
    etpan_thread_op_cancel(state->op);
    break;
  }
}

static void storage_schedule_update(struct storage_state * state,
    double delay)
{
  double next_delay;
  
  next_delay = etpan_get_time() + delay;
  if (state->update_requested) {
    /* modify so that it happen sooner */
    if (next_delay < state->update_request_date) {
      state->update_request_date = next_delay;
      reschedule(state->manager);
    }
    return;
  }
  
  state->update_requested = 1;
  state->update_request_date = next_delay;
  
  reschedule(state->manager);
}

static void storage_stop_folders(struct storage_state * state)
{
  int r;
  chashiter * iter;
  chash * folder_hash;
  
  folder_hash = etpan_storage_get_folder_list(state->storage);
  for(iter = chash_begin(state->folder_to_state) ; iter != NULL ;
      iter = chash_next(state->folder_to_state, iter)) {
    struct folder_state * folder_state;
    chashdatum key;
    chashdatum value;
    int do_stop;
    int no_check;
    char * location;
    
    chash_value(iter, &value);
    folder_state = value.data;
    
    no_check = 0;
    do_stop = 0;
    location = etpan_folder_get_location(folder_state->folder);
    key.data = location;
    key.len = strlen(location) + 1;
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      /* folder does no more exist */
      do_stop = 1;
      no_check = 1;
    }
    else {
      if (etpan_folder_is_lost(folder_state->folder)) {
        do_stop = 1;
        no_check = 1;
      }
    }
    
    if (state->stop) {
      do_stop = 1;
    }
    
    if (do_stop) {
      if (!folder_state->stop) {
        folder_state_stop(folder_state);
        if (no_check) {
          folder_state->check_requested = 0;
        }
      }
    }
  }
  
  storage_cleanup_stopped_folders(state);
}

static void storage_cleanup_stopped_folders(struct storage_state * state)
{
  carray * folder_list;
  int r;
  unsigned int i;
  chashiter * iter;
  
  /* could be improved by creating a list of pending folders */
  
  folder_list = carray_new(chash_count(state->folder_to_state));
  if (folder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  for(iter = chash_begin(state->folder_to_state) ; iter != NULL ;
      iter = chash_next(state->folder_to_state, iter)) {
    struct folder_state * folder_state;
    chashdatum value;
    
    chash_value(iter, &value);
    folder_state = value.data;
    r = carray_add(folder_list, folder_state, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(folder_list) ; i ++) {
    struct folder_state * folder_state;
    
    folder_state = carray_get(folder_list, i);
    
    if (!folder_is_stopped(folder_state))
      continue;
    
    remove_folder_state(state, folder_state->folder);
  }
  
  carray_free(folder_list);
}

static void storage_update(struct storage_state * state)
{
  state->state = STORAGE_STATE_UPDATE;
  state->update_requested = 0;
  
  state->op = etpan_storage_fetch_folder_list(etpan_thread_manager_app_get_default(),
      state->storage, update_list_callback, state);
}

static void update_list_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * callback_data)
{
  struct storage_state * state;
  struct etpan_mail_manager * manager;
  chashiter * iter;
  chash * folder_hash;
  struct etpan_storage * storage;
  struct etpan_error * error;
  
  state = callback_data;
  state->state = STORAGE_STATE_IDLE;
  manager = state->manager;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Update of list of folders cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled update of the list of folders of storage %s information."), etpan_storage_get_id(state->storage));
    storage_state_set_error(state, result->error);
    ETPAN_ERROR_FREE(error);
    
    update_finished(state);
    return;
  }
  
  if (result->error != NULL) {
    storage_state_set_error(state, result->error);
    update_finished(state);
    return;
  }
  
  storage = state->storage;
  error = etpan_storage_update_folder_list(storage, result->folder_list);
  if (error != NULL) {
    storage_state_set_error(state, result->error);
    update_finished(state);
    return;
  }

  storage_state_set_error(state, NULL);
  
  if (!state->stop) {
    /* add new folders state */
    folder_hash = etpan_storage_get_folder_list(storage);
    for(iter = chash_begin(folder_hash) ; iter != NULL ;
        iter = chash_next(folder_hash, iter)) {
      chashdatum value;
      struct etpan_folder * folder;
      
      chash_value(iter, &value);
      folder = value.data;
      
      get_folder_state(state, folder);
    }
  }
  
  etpan_storage_remove_lost_folders(storage);
  
  manager = state->manager;
  
  if (!state->stop) {
    ETPAN_SIGNAL_SEND_WITH_DATA(manager,
        ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL, state->storage);
  }
  
  update_finished(state);
}

static void update_finished(struct storage_state * state)
{
  storage_stop_folders(state);
  
  task_done(state->manager);
  
  if (!state->stop) {
    storage_schedule_update(state, LONG_INTERVAL);
  }
}

static double storage_imap_sync_get_next_delay(struct storage_state * state)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  struct etpan_imap_sync * imap_sync;
  double default_value;
  double delay;
  
  storage = state->storage;
  driver = etpan_storage_get_driver(storage);
  default_value = etpan_get_time() + 24 * 60 * 60; /* + 1 day */
  if (strcmp(driver->name, "imap-sync") != 0)
    return default_value;
  
  imap_sync = etpan_storage_imap_sync_get_imap_sync(storage);
  if (imap_sync == NULL)
    return default_value;
  
  delay = etpan_imap_sync_next_op_date(imap_sync);
#if 0
  ETPAN_LOCAL_LOG("delay: %s %i %lg", imap_sync->id,
      etpan_imap_sync_is_syncing(imap_sync),
      delay - etpan_get_time());
#endif
  return delay;
}

static void imap_sync_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void storage_imap_sync_perform_done(struct storage_state * state);

static void storage_imap_sync_perform(struct storage_state * state)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  struct etpan_imap_sync * imap_sync;
  double date;
  
  if (state->imap_sync_syncing)
    return;
  
  if (state->stop)
    return;
  
  storage = state->storage;
  driver = etpan_storage_get_driver(storage);
  if (strcmp(driver->name, "imap-sync") != 0)
    return;
  
  imap_sync = etpan_storage_imap_sync_get_imap_sync(storage);
  if (imap_sync == NULL)
    return;
  
  date = storage_imap_sync_get_next_delay(state);
  if (date > etpan_get_time())
    return;
  
  ETPAN_SIGNAL_ADD_HANDLER(imap_sync, ETPAN_IMAP_SYNC_OP_ENDED_SIGNAL,
      imap_sync_done_handler, state);
  
  ETPAN_LOCAL_LOG("perform: %s", imap_sync->id);
  state->imap_sync_syncing = 1;
  etpan_imap_sync(etpan_thread_manager_app_get_default(), imap_sync);
}

static void imap_sync_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_imap_sync * imap_sync;
  struct storage_state * state;
  
  state = user_data;
  imap_sync = sender;
  ETPAN_SIGNAL_REMOVE_HANDLER(imap_sync, ETPAN_IMAP_SYNC_OP_ENDED_SIGNAL,
      imap_sync_done_handler, state);
  
  state->imap_sync_syncing = 0;
  storage_imap_sync_perform_done(state);
  ETPAN_LOCAL_LOG("perform done: %s", imap_sync->id);
}

static void storage_imap_sync_perform_done(struct storage_state * state)
{
  struct etpan_storage * storage;
  
  storage = state->storage;
  task_done(state->manager);
}


static double storage_pop_sync_get_next_delay(struct storage_state * state)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  struct etpan_pop_sync * pop_sync;
  double default_value;
  double delay;
  
  storage = state->storage;
  driver = etpan_storage_get_driver(storage);
  default_value = etpan_get_time() + 24 * 60 * 60; /* + 1 day */
  if (strcmp(driver->name, "pop-sync") != 0)
    return default_value;
  
  pop_sync = etpan_storage_pop_sync_get_pop_sync(storage);
  if (pop_sync == NULL)
    return default_value;

  delay = etpan_pop_sync_next_op_date(pop_sync);
  return delay;
}

static void pop_sync_callback(void * user_data);

static void storage_pop_sync_cancel(struct storage_state * state)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  struct etpan_pop_sync * pop_sync;
  
  storage = state->storage;
  driver = etpan_storage_get_driver(storage);
  if (strcmp(driver->name, "pop-sync") != 0)
    return;
  
  pop_sync = etpan_storage_pop_sync_get_pop_sync(storage);
  if (pop_sync == NULL)
    return;
  
  etpan_pop_sync_cancel(pop_sync);
  ETPAN_LOCAL_LOG("cancel pop sync !");
}

static void storage_pop_sync_perform(struct storage_state * state)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  struct etpan_pop_sync * pop_sync;
  double date;
  
  if (state->pop_sync_syncing)
    return;

  if (state->stop)
    return;
  
  storage = state->storage;
  driver = etpan_storage_get_driver(storage);
  if (strcmp(driver->name, "pop-sync") != 0)
    return;
  
  pop_sync = etpan_storage_pop_sync_get_pop_sync(storage);
  if (pop_sync == NULL)
    return;
  
  date = storage_pop_sync_get_next_delay(state);
  if (date > etpan_get_time())
    return;
  
  state->pop_sync_syncing = 1;
  etpan_pop_sync_start(pop_sync, pop_sync_callback, state);
}

static void pop_sync_callback(void * user_data)
{
  struct storage_state * state;
  
  ETPAN_LOCAL_LOG("pop sync callback !");
  state = user_data;
  state->pop_sync_syncing = 0;
  task_done(state->manager);
}

static void get_next_task(struct storage_state * state,
    struct etpan_folder ** result_folder, int * result_type,
    double * result_time);

static double storage_get_next_task_delay(struct storage_state * state)
{
  double min_time;
  double time_value;
  
  get_next_task(state, NULL, NULL, &min_time);
  time_value = storage_imap_sync_get_next_delay(state);
  if (time_value < min_time) {
    min_time = time_value;
  }
  time_value = storage_pop_sync_get_next_delay(state);
  if (time_value < min_time) {
    min_time = time_value;
  }
  
  return min_time;
}

#define MAX_PRIORITY 5

static int storage_get_next_task_type(struct storage_state * state)
{
  int type;
  
  get_next_task(state, NULL, &type, NULL);
  
  return type;
}

static void get_next_task(struct storage_state * state,
    struct etpan_folder ** result_folder, int * result_type,
    double * result_time)
{
  carray * folder_list;
  double current_time;
  struct etpan_folder * chosen_folder;
  int chosen_priority;
  struct folder_state * folder_state;
  unsigned int i;
  int r;
  double min_time;
  
  if (state->state != STORAGE_STATE_IDLE) {
    if (result_folder != NULL)
      * result_folder = NULL;
    if (result_type != NULL)
      * result_type = STORAGE_OP_TYPE_NONE;
    if (result_time != NULL)
      * result_time = etpan_get_time() + 24 * 60 * 60;
    return;
  }
  
  current_time = etpan_get_time();
  
  chosen_priority = MAX_PRIORITY;
  chosen_folder = NULL;
  min_time = etpan_get_time() + 24 * 60 * 60;
  
  if (state->update_requested) {
    chosen_priority = 4;
    chosen_folder = NULL;
    min_time = state->update_request_date;
  }
  
  /* folder op */
  /* priority for check, content, then, status */
  folder_list = etpan_storage_folder_order_get_folder_order(state->order);
  for(i = 0 ; i < carray_count(folder_list) ; i ++) {
    char * folder_name;
    struct etpan_folder * folder;
    int priority;
    int type;
    double time_value;
    chashdatum key;
    chashdatum value;
    
    folder_name = carray_get(folder_list, i);
    folder = etpan_storage_folder_order_get_folder(state->order, folder_name);
    
    key.data = &folder;
    key.len = sizeof(folder);
    r = chash_get(state->folder_to_state, &key, &value);
    if (r < 0)
      continue;
    
    folder_state = get_folder_state(state, folder);
    type = folder_get_next_task_type(folder_state);
    if (type == FOLDER_OP_TYPE_NONE)
      continue;
    
    priority = MAX_PRIORITY;
    switch (type) {
    case FOLDER_OP_TYPE_CHECK:
      priority = 0;
      break;
    case FOLDER_OP_TYPE_CONTENT:
      priority = 1;
      break;
    case FOLDER_OP_TYPE_STATUS:
      priority = 2;
      break;
    }
    
    time_value = folder_get_next_task_delay(folder_state);
#if 0
    ETPAN_LOCAL_LOG("perform %s %s %lg", state->storage->id, folder_name,
        time_value - etpan_get_time());
#endif
    if (time_value <= current_time) {
      if ((time_value <= current_time) && (priority < chosen_priority)) {
        chosen_folder = folder;
        chosen_priority = priority;
        min_time = time_value;
      }
    }
    else if (time_value < min_time) {
      chosen_folder = folder;
      chosen_priority = priority;
      min_time = time_value;
    }
  }
  
  if (chosen_folder != NULL) {
    if (result_folder != NULL)
      * result_folder = chosen_folder;
    if (result_type != NULL)
      * result_type = STORAGE_OP_TYPE_FOLDER;
    if (result_time != NULL)
      * result_time = min_time;
  }
  else if (chosen_priority == 4) {
    if (result_folder != NULL)
      * result_folder = chosen_folder;
    if (result_type != NULL)
      * result_type = STORAGE_OP_TYPE_UPDATE;
    if (result_time != NULL)
      * result_time = min_time;
  }
  else {
    if (result_folder != NULL)
      * result_folder = chosen_folder;
    if (result_type != NULL)
      * result_type = STORAGE_OP_TYPE_NONE;
    if (result_time != NULL)
      * result_time = min_time;
  }
}

static void storage_perform_next_task(struct storage_state * state)
{
  double current_time;
  struct etpan_folder * chosen_folder;
  struct folder_state * folder_state;
  double result_time;
  
  storage_imap_sync_perform(state);
  storage_pop_sync_perform(state);
  
#if 0
  ETPAN_LOCAL_LOG("perform %s", state->storage->id);
#endif

  current_time = etpan_get_time();
  get_next_task(state, &chosen_folder, NULL, &result_time);
  
  if (result_time <= current_time) {
    if (chosen_folder != NULL) {
#if 0
      ETPAN_LOCAL_LOG("perform %s %s", state->storage->id, chosen_folder->location);
#endif
      state->state = STORAGE_STATE_FOLDER_OP;
      folder_state = get_folder_state(state, chosen_folder);
      folder_perform_next_task(folder_state);
    }
    else {
#if 0
      ETPAN_LOCAL_LOG("perform %s storage update", state->storage->id);
#endif
      /* perform update */
      storage_update(state);
    }
  }
  else {
#if 0
    ETPAN_LOCAL_LOG("perform %s nothing", state->storage->id);
#endif
  }
}

static void storage_folder_task_done(struct storage_state * state)
{
  storage_cleanup_stopped_folders(state);
  
  state->state = STORAGE_STATE_IDLE;
  task_done(state->manager);
}

/* **************************************** */
/* folder */


static struct folder_state * folder_state_new(void)
{
  struct folder_state * state;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  state->manager = NULL;
  state->storage_state = NULL;
  state->folder = NULL;
  state->first_filter = 1;
  state->opened_count = 0;
  state->state = FOLDER_STATE_IDLE;
  state->stop = 0;
  state->error = NULL;
  state->has_count = 0;
  state->unread_count = 0;
  state->total_count = 0;
  state->status_requested = 0;
  state->status_request_date = 0;
  state->check_requested = 0;
  state->check_request_date = 0;
  state->check_needed = 0;
  state->msg_to_check = NULL;
  state->expunge_needed = 0;
  state->fetcher = NULL;
  state->op = NULL;
  state->indexer = NULL;
  
  return state;
}

static void folder_state_free(struct folder_state * state)
{
  if (state->folder != NULL)
    etpan_folder_unref(state->folder);
  ETPAN_ERROR_FREE(state->error);
  state->error = NULL;
  free(state);
}

static int folder_is_stopped(struct folder_state * state)
{
  if (!state->stop) {
    return 0;
  }
  
  if (state->state != FOLDER_STATE_IDLE) {
    return 0;
  }
  
  if (state->opened_count > 0) {
    return 0;
  }
  
  if (folder_get_next_task_type(state) != FOLDER_OP_TYPE_NONE) {
    return 0;
  }
  
  return 1;
}

static void folder_perform_next_task(struct folder_state * state)
{
  int type;

  type = folder_get_next_task_type(state);
  switch (type) {
  case FOLDER_OP_TYPE_CHECK:
    check_folder(state);
    break;
    
  case FOLDER_OP_TYPE_CONTENT:
  case FOLDER_OP_TYPE_STATUS:
    status_folder(state);
    break;
  }
}

static double folder_get_next_task_delay(struct folder_state * state)
{
  double min_time;
  
  min_time = etpan_get_time() + 24 * 60 * 60; /* + 1 day */
  
  if (state->state != FOLDER_STATE_IDLE) {
    return min_time;
  }
  
  if (state->status_requested) {
    if (state->status_request_date < min_time) {
      min_time = state->status_request_date;
    }
  }

  if (state->check_requested) {
    if (state->check_request_date < min_time) {
      min_time = state->check_request_date;
    }
  }
  
  return min_time;
}


static int folder_get_next_task_type(struct folder_state * state)
{
  double min_time;
  int type;
  
  type = FOLDER_OP_TYPE_NONE;
  min_time = 0;
  
  if (state->state != FOLDER_STATE_IDLE) {
    return type;
  }
  
  if (state->check_requested) {
    type = FOLDER_OP_TYPE_CHECK;
    min_time = state->check_request_date;
  }
  
  if (state->status_requested) {
    if (type != FOLDER_OP_TYPE_NONE) {
      if (state->status_request_date < min_time) {
        min_time = state->status_request_date;
        type = FOLDER_OP_TYPE_STATUS;
      }
    }
    else {
      type = FOLDER_OP_TYPE_STATUS;
    }
    if (type == FOLDER_OP_TYPE_STATUS) {
      if (state->opened_count > 0)
        type = FOLDER_OP_TYPE_CONTENT;
      else
        type = FOLDER_OP_TYPE_STATUS;
    }
  }

  return type;
}

static void folder_cancel(struct folder_state * state)
{
  switch (state->state) {
  case FOLDER_STATE_STATUS:
    if (state->fetcher != NULL) {
      etpan_msg_list_fetch_cancel(state->fetcher);
    }
    else if (state->op != NULL) {
      etpan_thread_op_cancel(state->op);
    }
    else {
      ETPAN_WARN_LOG("status with no operation");
    }
    break;
  }
}

static void folder_state_stop(struct folder_state * state)
{
  int schedule_needed;
  
  folder_cancel(state);
  
  schedule_needed = 0;
  state->stop = 1;
  if (state->status_requested) {
    state->status_requested = 0;
    schedule_needed = 1;
  }
  
  if (state->check_requested) {
    double perform_date;
    
    perform_date = etpan_get_time() + NOW_INTERVAL;
    state->check_request_date = perform_date;
    schedule_needed = 1;
  }
  
  if (schedule_needed)
    reschedule(state->manager);
}

static void folder_state_set_error(struct folder_state * state,
    struct etpan_error * error)
{
  ETPAN_ERROR_FREE(state->error);
  if (error != NULL) {
    etpan_error_log(error);
    state->error = etpan_error_dup(error);
  }
  else
    state->error = NULL;
}

static void imap_sync_folder_content_handler(char * signal_name,
    void * sender, void * signal_data, void * user_data);

static void remove_folder_state(struct storage_state * storage_state, struct etpan_folder * folder)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct folder_state * state;
  
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(storage_state->folder_to_state, &key, &value);
  if (r < 0)
    return;

  chash_delete(storage_state->folder_to_state, &key, NULL);
  state = value.data;

  ETPAN_SIGNAL_REMOVE_HANDLER(folder,
      ETPAN_FOLDER_IMAP_SYNC_CONTENT_UPDATED_ENDED_SIGNAL,
      imap_sync_folder_content_handler, state);
  
  folder_state_free(state);
}

static struct folder_state * get_folder_state(struct storage_state * storage_state, struct etpan_folder * folder)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct folder_state * state;
  
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(storage_state->folder_to_state, &key, &value);
  if (r == 0) {
    state = value.data;
    
    return state;
  }
  
  if (storage_state->manager->stop) {
    ETPAN_WARN_LOG("creation of folder state while manager is stopped");
    etpan_log_stack();
  }
  
  state = folder_state_new();
  state->manager = storage_state->manager;
  state->storage_state = storage_state;
  state->folder = folder;
  etpan_folder_ref(folder);
  
  value.data = state;
  value.len = 0;
  r = chash_set(storage_state->folder_to_state, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_SIGNAL_ADD_HANDLER(folder,
      ETPAN_FOLDER_IMAP_SYNC_CONTENT_UPDATED_ENDED_SIGNAL,
      imap_sync_folder_content_handler, state);
  
  schedule_status_folder(state, SOON_INTERVAL);
  
  return state;
}

static void imap_sync_folder_content_handler(char * signal_name,
    void * sender, void * signal_data, void * user_data)
{
  struct folder_state * state;
  
  state = user_data;
  schedule_status_folder(state, SOON_INTERVAL);
}

static void open_folder(struct folder_state * state)
{
  /* do a status when opening the folder */
  char * path;
  
  etpan_folder_ref_msg_list(state->folder);
  schedule_status_folder(state, SOON_INTERVAL);
  
  state->indexer = etpan_folder_indexer_new();
  etpan_folder_indexer_set_folder(state->indexer, state->folder);
  path = etpan_folder_indexer_get_path_for_folder(state->indexer);
  etpan_folder_indexer_set_path(state->indexer, path);
  free(path);
  etpan_folder_indexer_setup(state->indexer);
  
  ETPAN_SIGNAL_ADD_HANDLER(state->indexer,
      ETPAN_FOLDER_INDEXER_STATE_CHANGED,
      indexer_state_changed, state);
  ETPAN_SIGNAL_ADD_HANDLER(state->indexer,
      ETPAN_FOLDER_INDEXER_UPDATED_RESULT,
      indexer_updated, state);
}

static void close_folder(struct folder_state * state)
{
  folder_cancel(state);
  /* do a check after closing the folder */
  ETPAN_SIGNAL_ADD_HANDLER(state->indexer,
      ETPAN_FOLDER_INDEXER_UPDATED_RESULT,
      indexer_updated, state);
  ETPAN_SIGNAL_ADD_HANDLER(state->indexer,
      ETPAN_FOLDER_INDEXER_STATE_CHANGED,
      indexer_state_changed, state);
  etpan_folder_indexer_unsetup(state->indexer);
  etpan_folder_indexer_free(state->indexer);
  state->indexer = NULL;
  
  etpan_folder_unref_msg_list(state->folder);
  
  reschedule(state->manager);
}

static void schedule_status_folder(struct folder_state * state, double delay)
{
  double next_delay;
  
  next_delay = etpan_get_time() + delay;
  if (state->status_requested) {
    /* modify so that it happen sooner */
    if (next_delay < state->status_request_date) {
      state->status_request_date = next_delay;
      reschedule(state->manager);
    }
    return;
  }
  
  state->status_requested = 1;
  state->status_request_date = next_delay;
  
  reschedule(state->manager);
}

static int is_inbox(struct folder_state * state)
{
  struct etpan_folder * inbox;
  struct etpan_storage * storage;
  struct etpan_folder_driver * driver;
  
  driver = etpan_folder_get_driver(state->folder);
  storage = etpan_folder_get_storage(state->folder);
  inbox = NULL;
  if (strcasecmp(driver->name, "imap-sync") == 0) {
    inbox = etpan_storage_get_folder(storage, "INBOX");
  }
  else if (strcasecmp(driver->name, "pop-sync") == 0) {
    inbox = etpan_storage_get_folder(storage, "");
  }
  ETPAN_LOG("inbox: %p %p", inbox, state->folder);
  
  return (inbox == state->folder);
}

static void status_folder(struct folder_state * state)
{
  /*
    will do a status or a status, or a fetch of msg list
    number of unread will either read the status
    or use the msg list if available.
  */
  
  state->state = FOLDER_STATE_STATUS;
  state->status_requested = 0;
  
  ETPAN_LOG("status-folder %i %s", state->opened_count,
      etpan_folder_get_location(state->folder));
  
  if ((state->opened_count == 0) && (!is_inbox(state))) {
    ETPAN_LOG("status");
    state->op = etpan_folder_status(etpan_thread_manager_app_get_default(),
        state->folder, folder_status_callback, state);
  }
  else {
    ETPAN_LOG("fetch");
    state->fetcher = etpan_msg_list_fetch_new();
    if (state->storage_state->inbox_filter != NULL) {
#if 1 /* XXX - this will hide messages that were not filtered */
      if (etpan_folder_filter_get_folder(state->storage_state->inbox_filter) ==
          state->folder) {
        etpan_msg_list_fetch_set_filter(state->fetcher,
            state->storage_state->inbox_filter);
      }
#endif
    }
    etpan_msg_list_fetch_set_folder(state->fetcher, state->folder);
    etpan_msg_list_fetch_setup(state->fetcher);
    
    ETPAN_SIGNAL_ADD_HANDLER(state->fetcher,
        ETPAN_MSG_LIST_FETCH_FINISHED_WITH_LOST_SIGNAL,
        fetched_with_lost, state);
    ETPAN_SIGNAL_ADD_HANDLER(state->fetcher,
        ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL,
        fetched, state);
    
    etpan_msg_list_fetch_run(state->fetcher);
  }
}

static void folder_status_callback(int cancelled,
    struct etpan_folder_status_result * result, void * callback_data)
{
  struct folder_state * state;
  
  state = callback_data;
  state->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Update of folder information cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled update of folder %s information."), etpan_folder_get_ui_path(state->folder));
    folder_state_set_error(state, result->error);
    ETPAN_ERROR_FREE(error);
    
    status_finished(state);
    return;
  }
  
  if (result->error != NULL) {
    folder_state_set_error(state, result->error);
    status_finished(state);
    return;
  }
  
  folder_state_set_error(state, NULL);
  etpan_folder_set_count(state->folder,
      result->count, result->unseen, result->recent);
  state->has_count = 1;
  state->total_count = result->count;
  state->unread_count = result->unseen;
  status_finished(state);
}


static void fetched_with_lost(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_mail_manager * manager;
  struct folder_state * state;
  (void) sender;
  (void) signal_name;
  (void) signal_data;
  
  state = user_data;
  manager = state->manager;
  ETPAN_SIGNAL_SEND_WITH_DATA(manager,
      ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL, state->folder);
}

static void fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_mail_manager * manager;
  struct etpan_error * error;
  struct folder_state * state;
  (void) sender;
  (void) signal_name;
  (void) signal_data;
  chashiter * iter;
  unsigned int unread_count;
  unsigned int total_count;
  chash * msg_hash;
  
  state = user_data;
  manager = state->manager;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(state->fetcher,
      ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL,
      fetched, state);
  ETPAN_SIGNAL_REMOVE_HANDLER(state->fetcher,
      ETPAN_MSG_LIST_FETCH_FINISHED_WITH_LOST_SIGNAL,
      fetched_with_lost, state);
  
  state->state = FOLDER_STATE_IDLE;
  
  error = etpan_msg_list_fetch_get_error(state->fetcher);
  if (error != NULL) {
    folder_state_set_error(state, error);
    
    etpan_msg_list_fetch_unref(state->fetcher);
    state->fetcher = NULL;
    
    ETPAN_SIGNAL_SEND_WITH_DATA(manager,
        ETPAN_MAIL_MANAGER_MSG_LIST_UPDATED_SIGNAL, state->folder);
    
    status_finished(state);
    return;
  }
  
  ETPAN_LOG("fetched");
  
  if (is_inbox(state)) {
    if (/*state->first_filter*/ 1) {
      state->first_filter = 0;
      etpan_folder_filter_force_run(state->storage_state->inbox_filter);
    }
  }
  
  etpan_msg_list_fetch_unref(state->fetcher);
  state->fetcher = NULL;
  
  unread_count = 0;
  total_count = 0;
  msg_hash = etpan_folder_get_message_hash(state->folder);
  for(iter = chash_begin(msg_hash) ; iter != NULL ;
      iter = chash_next(msg_hash, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    int flags_value;
    
    chash_value(iter, &value);
    msg = value.data;
    flags = etpan_message_get_flags(msg);
    flags_value = etpan_message_flags_get_value(flags);
    if ((flags_value & ETPAN_FLAGS_DELETED) != 0)
      continue;
    
    total_count ++;
    if ((flags_value & ETPAN_FLAGS_SEEN) == 0)
      unread_count ++;
  }
  folder_state_set_error(state, NULL);
  state->has_count = 1;
  state->total_count = total_count;
  state->unread_count = unread_count;
  status_finished(state);
}

static void status_finished(struct folder_state * state)
{
  struct etpan_mail_manager * manager;
  
  manager = state->manager;
  state->state = FOLDER_STATE_IDLE;
  
  ETPAN_SIGNAL_SEND_WITH_DATA(manager,
      ETPAN_MAIL_MANAGER_STATUS_UPDATED_SIGNAL, state->folder);
  
  storage_folder_task_done(state->storage_state);
  
  if (state->stop)
    return;
  
  schedule_status_folder(state, LONG_INTERVAL);
}


static void schedule_check_folder(struct folder_state * state, double delay)
{
  double next_delay;
  
  next_delay = etpan_get_time() + delay;
  if (state->check_requested) {
    /* modify so that it happen sooner */
    if (next_delay < state->check_request_date) {
      state->check_request_date = next_delay;
      reschedule(state->manager);
    }
    return;
  }
  
  etpan_folder_ref_msg_list(state->folder);
  state->check_requested = 1;
  state->check_request_date = next_delay;
  
  reschedule(state->manager);
}

static void check_folder(struct folder_state * state)
{
  /*
    When flags has been changed, a check will be requested
    (all msg will be checked)
    check will be first done
    will look at flags to see whether expunge is necessary
    (when msg_list is enabled),
    followed by status
  */
  
  chash * msg_hash;
  int expunge_needed;
  int check_needed;
  chash * msg_to_check;
  int r;
  chashiter * iter;
  unsigned int unread_count;
  unsigned int total_count;
  
  state->state = FOLDER_STATE_CHECK;
  state->check_requested = 0;
  
  msg_to_check = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_to_check == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_hash = etpan_folder_get_message_hash(state->folder);
  
  unread_count = 0;
  total_count = 0;
  for(iter = chash_begin(msg_hash) ; iter != NULL ;
      iter = chash_next(msg_hash, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    int flags_value;
    
    chash_value(iter, &value);
    msg = value.data;
    flags = etpan_message_get_flags(msg);
    flags_value = etpan_message_flags_get_value(flags);
    if ((flags_value & ETPAN_FLAGS_DELETED) != 0)
      continue;
    
    total_count ++;
    if ((flags_value & ETPAN_FLAGS_SEEN) == 0)
      unread_count ++;
  }
  state->has_count = 1;
  state->total_count = total_count;
  state->unread_count = unread_count;
  
  ETPAN_SIGNAL_SEND_WITH_DATA(state->manager,
      ETPAN_MAIL_MANAGER_STATUS_UPDATED_SIGNAL, state->folder);
  
  expunge_needed = 0;
  check_needed = 0;
  for(iter = chash_begin(msg_hash) ; iter != NULL ;
      iter = chash_next(msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    int flags_value;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    flags = etpan_message_get_flags(msg);
    flags_value = etpan_message_flags_get_value(flags);
    if ((flags_value & ETPAN_FLAGS_DELETED) == 0) {
      expunge_needed = 1;
    }
    
    if (etpan_message_has_dirty_flags(msg)) {
      r = chash_set(msg_to_check, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      
      check_needed = 1;
    }
  }
  
  state->msg_to_check = msg_to_check;
  state->check_needed = check_needed;
  state->expunge_needed = expunge_needed;
  
  do_check_msg_list(state);
}

static void do_check_msg_list(struct folder_state * state)
{
  if (state->check_needed) {
    state->op =
      etpan_folder_check_msg_list(etpan_thread_manager_app_get_default(),
          state->folder, state->msg_to_check, check_msg_list_callback, state);
  }
  else {
    do_expunge(state);
  }
}

static void check_msg_list_callback(int cancelled,
    struct etpan_folder_check_msg_list_result * result, void * callback_data)
{
  struct folder_state * state;
  
  state = callback_data;
  
  state->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Save of folder information cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled save of folder %s information."), etpan_folder_get_ui_path(state->folder));
    
    folder_state_set_error(state, error);
    
    ETPAN_ERROR_FREE(error);
    check_finished(state);
    
    return;
  }
  
  if (result->error != NULL) {
    folder_state_set_error(state, result->error);
    check_finished(state);
    return;
  }
  
  do_check(state);
}

static void do_check(struct folder_state * state)
{
  state->op =
    etpan_folder_check(etpan_thread_manager_app_get_default(),
        state->folder, check_callback, state);
}

static void check_callback(int cancelled,
    struct etpan_folder_check_result * result, void * callback_data)
{
  struct folder_state * state;
  
  state = callback_data;
  
  state->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Save of folder information cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled save of folder %s information."), etpan_folder_get_ui_path(state->folder));
    
    folder_state_set_error(state, error);
    
    ETPAN_ERROR_FREE(error);
    check_finished(state);
    
    return;
  }
  
  if (result->error != NULL) {
    folder_state_set_error(state, result->error);
    check_finished(state);
    return;
  }
  
  do_expunge(state);
}

static void do_expunge(struct folder_state * state)
{
  if (state->expunge_needed) {
    state->op = etpan_folder_expunge(etpan_thread_manager_app_get_default(),
        state->folder, expunge_callback, state);
  }
  else {
    do_status(state);
  }
}

static void expunge_callback(int cancelled,
    struct etpan_folder_expunge_result * result, void * cb_data)
{
  struct folder_state * state;
  
  state = cb_data;
  state->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Save of folder information cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled save of folder %s information."), etpan_folder_get_ui_path(state->folder));
    folder_state_set_error(state, result->error);
    ETPAN_ERROR_FREE(error);
    
    check_finished(state);
    return;
  }
  
  if (result->error != NULL) {
    folder_state_set_error(state, result->error);
    check_finished(state);
    return;
  }
  
  folder_state_set_error(state, NULL);
  
  do_status(state);
}

static void do_status(struct folder_state * state)
{
#if 0
  schedule_status_folder(state, SOON_INTERVAL);
#endif
  
  check_finished(state);
}

static void check_finished(struct folder_state * state)
{
  chash_free(state->msg_to_check);
  state->msg_to_check = NULL;
  state->state = FOLDER_STATE_IDLE;
  etpan_folder_unref_msg_list(state->folder);
  
  storage_folder_task_done(state->storage_state);
}


static void indexer_state_changed(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct folder_state * state;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  state = user_data;
  
  if (state->stop)
    return;
  
  ETPAN_SIGNAL_SEND_WITH_DATA(state->manager,
      ETPAN_MAIL_MANAGER_INDEXER_UPDATED_SIGNAL, state->folder);
}

static void indexer_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct folder_state * state;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  state = user_data;
  
  if (state->stop)
    return;
  
  ETPAN_SIGNAL_SEND_WITH_DATA(state->manager,
      ETPAN_MAIL_MANAGER_INDEXER_UPDATED_SIGNAL, state->folder);
}

#if 0
void etpan_mail_manager_folder_append_message(struct etpan_mail_manager * manager,
    struct etpan_folder * folder, struct etpan_message * msg)
{
}

void etpan_mail_manager_folder_append_content(struct etpan_mail_manager * manager,
    struct etpan_folder * folder, char * message, size_t length)
{
}
#endif

static void copy_with_delete(struct etpan_mail_manager * manager,
    struct etpan_folder * folder, chash * msg_hash, int delete);
static void copy_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

unsigned int etpan_mail_manager_get_next_copy_messages_id(struct etpan_mail_manager * manager)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->copy_msg_op_list) ; i ++) {
    struct etpan_message_copy * msg_copy;
    
    msg_copy = carray_get(manager->copy_msg_op_list, i);
    if (msg_copy == NULL)
      return i;
  }
  
  return carray_count(manager->copy_msg_op_list);
}

unsigned int etpan_mail_manager_get_next_delete_messages_id(struct etpan_mail_manager * manager)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->delete_msg_op_list) ; i ++) {
    struct etpan_message_delete * msg_delete;
    
    msg_delete = carray_get(manager->delete_msg_op_list, i);
    if (msg_delete == NULL)
      return i;
  }
  
  return carray_count(manager->delete_msg_op_list);
}

static void delete_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_mail_manager_delete_messages(struct etpan_mail_manager * manager,
    chash * msg_hash)
{
  struct etpan_message_delete * msg_delete;
  unsigned int op_index;
  
  ETPAN_LOCAL_LOG("delete of msg");
  
  msg_delete = etpan_message_delete_new();
  etpan_message_delete_set_msglist(msg_delete, msg_hash);
  etpan_message_delete_setup(msg_delete);
  
  op_index = etpan_mail_manager_get_next_delete_messages_id(manager);
  if (op_index >= carray_count(manager->delete_msg_op_list))
    carray_add(manager->delete_msg_op_list, msg_delete, NULL);
  else
    carray_set(manager->delete_msg_op_list, op_index, msg_delete);
  
  ETPAN_SIGNAL_ADD_HANDLER(msg_delete, ETPAN_MESSAGE_DELETE_FINISHED_SIGNAL,
      delete_finished_handler, manager);
  
  etpan_message_delete_run(msg_delete);
}

static void delete_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_delete * msg_delete;
  chash * folder_hash;
  struct etpan_mail_manager * manager;
  struct etpan_error * error;
  unsigned int op_index;
  unsigned int i;
  chash * pending;
  chashiter * iter;
  int r;
  
  msg_delete = sender;
  manager = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(msg_delete, ETPAN_MESSAGE_DELETE_FINISHED_SIGNAL,
      delete_finished_handler, manager);

  error = etpan_message_delete_get_error(msg_delete);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  ETPAN_LOCAL_LOG("finished delete of msg");
  
  op_index = 0;
  for(i = 0 ; i < carray_count(manager->delete_msg_op_list) ; i ++) {
    struct etpan_message_delete * cur_msg_delete;
    
    cur_msg_delete = carray_get(manager->delete_msg_op_list, i);
    if (cur_msg_delete == msg_delete) {
      carray_set(manager->delete_msg_op_list, i, NULL);
      op_index = i;
      break;
    }
  }

  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  pending = etpan_message_delete_get_pending_for_deletion(msg_delete);
  for(iter = chash_begin(pending) ; iter != NULL ;
      iter = chash_next(pending, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;    
    struct etpan_folder * msg_folder;
    struct etpan_storage * msg_storage;
    char * folder_location;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    msg = value.data;
    msg_folder = etpan_message_get_folder(msg);
    key.data = &msg_folder;
    key.len = sizeof(msg_folder);
    value.data = msg_folder;
    value.len = 0;
    r = chash_set(folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    msg_storage = etpan_folder_get_storage(msg_folder);
    
    folder_location = etpan_storage_get_sub_folder_location(msg_storage,
        NULL, "Deleted Messages");
    folder = etpan_storage_get_folder(msg_storage, folder_location);
    free(folder_location);
    
    if (folder != NULL) {
      key.data = &folder;
      key.len = sizeof(folder);
      value.data = folder;
      value.len = 0;
      r = chash_set(folder_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum value;
    struct etpan_folder * msg_folder;
    struct etpan_storage * msg_storage;
    struct folder_state * state;
    struct storage_state * storage_state;
    
    chash_value(iter, &value);
    msg_folder = value.data;
    msg_storage = etpan_folder_get_storage(msg_folder);
    storage_state = get_storage_state(manager, msg_storage);
    state = get_folder_state(storage_state, msg_folder);
    schedule_status_folder(state, SOON_INTERVAL);
  }
  
  chash_free(folder_hash);
  
  if (error != NULL) {
    manager->delete_msg_error = etpan_error_dup(error);
  }
  
  ETPAN_SIGNAL_SEND_WITH_DATA(manager,
      ETPAN_MAIL_MANAGER_DELETE_MSG_FINISHED_SIGNAL, &op_index);
  
  ETPAN_ERROR_FREE(manager->delete_msg_error);
  manager->delete_msg_error = NULL;
  
  etpan_message_delete_unref(msg_delete);
}

void etpan_mail_manager_copy_messages(struct etpan_mail_manager * manager,
    struct etpan_folder * folder, chash * msg_hash)
{
  copy_with_delete(manager, folder, msg_hash, 0);
}

void etpan_mail_manager_move_messages(struct etpan_mail_manager * manager,
    struct etpan_folder * folder, chash * msg_hash)
{
  copy_with_delete(manager, folder, msg_hash, 1);
}

static void copy_with_delete(struct etpan_mail_manager * manager,
    struct etpan_folder * folder, chash * msg_hash, int delete)
{
  struct etpan_message_copy * msg_copy;
  unsigned int op_index;
  
  if (delete)
    ETPAN_LOCAL_LOG("move %i messages", chash_count(msg_hash));
  else 
    ETPAN_LOCAL_LOG("copy %i messages", chash_count(msg_hash));
  
  msg_copy = etpan_message_copy_new();
  etpan_message_copy_set_delete(msg_copy, delete);
  etpan_message_copy_set_msglist(msg_copy, msg_hash);
  etpan_message_copy_set_destination(msg_copy, folder);
  etpan_message_copy_setup(msg_copy);
  
  op_index = etpan_mail_manager_get_next_copy_messages_id(manager);
  if (op_index >= carray_count(manager->copy_msg_op_list))
    carray_add(manager->copy_msg_op_list, msg_copy, NULL);
  else
    carray_set(manager->copy_msg_op_list, op_index, msg_copy);
  
  ETPAN_SIGNAL_ADD_HANDLER(msg_copy, ETPAN_MESSAGE_COPY_FINISHED_SIGNAL,
      copy_finished_handler, manager);
  
  etpan_message_copy_run(msg_copy);
}

static void copy_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_copy * msg_copy;
  struct etpan_mail_manager * manager;
  struct etpan_folder * folder;
  struct folder_state * state;
  struct storage_state * storage_state;
  struct etpan_storage * storage;
  struct etpan_error * error;
  unsigned int op_index;
  unsigned int i;
  int r;
  chash * source_folder_hash;
  chash * pending;
  chashiter * iter;
  (void) signal_name;
  (void) signal_data;
  
  manager = user_data;
  msg_copy = sender;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(msg_copy, ETPAN_MESSAGE_COPY_FINISHED_SIGNAL,
      copy_finished_handler, manager);
  
  error = etpan_message_copy_get_error(msg_copy);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  ETPAN_LOCAL_LOG("finished move/copy of msg");
  
  op_index = 0;
  for(i = 0 ; i < carray_count(manager->copy_msg_op_list) ; i ++) {
    struct etpan_message_copy * cur_msg_copy;
    
    cur_msg_copy = carray_get(manager->copy_msg_op_list, i);
    if (cur_msg_copy == msg_copy) {
      carray_set(manager->copy_msg_op_list, i, NULL);
      op_index = i;
      break;
    }
  }
  
  /* update count of destination folder */
  folder = etpan_message_copy_get_destination(msg_copy);
  storage = etpan_folder_get_storage(folder);
  storage_state = get_storage_state(manager, storage);
  state = get_folder_state(storage_state, folder);
  schedule_status_folder(state, SOON_INTERVAL);
  
  source_folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (source_folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  pending = etpan_message_copy_get_pending_for_deletion(msg_copy);
  for(iter = chash_begin(pending) ; iter != NULL ;
      iter = chash_next(pending, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;    
    struct etpan_folder * msg_folder;
    
    chash_value(iter, &value);
    msg = value.data;
    msg_folder = etpan_message_get_folder(msg);
    key.data = &msg_folder;
    key.len = sizeof(msg_folder);
    value.data = msg_folder;
    value.len = 0;
    r = chash_set(source_folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  for(iter = chash_begin(source_folder_hash) ; iter != NULL ;
      iter = chash_next(source_folder_hash, iter)) {
    chashdatum value;
    struct etpan_folder * msg_folder;
    
    chash_value(iter, &value);
    msg_folder = value.data;
    state = get_folder_state(storage_state, msg_folder);
    schedule_status_folder(state, SOON_INTERVAL);
  }
  
  chash_free(source_folder_hash);
  
  if (error != NULL) {
    manager->copy_msg_error = etpan_error_dup(error);
  }
  
  ETPAN_SIGNAL_SEND_WITH_DATA(manager,
      ETPAN_MAIL_MANAGER_COPY_MSG_FINISHED_SIGNAL, &op_index);
  
  ETPAN_ERROR_FREE(manager->copy_msg_error);
  manager->copy_msg_error = NULL;
  
  etpan_message_copy_unref(msg_copy);
}

struct etpan_error * etpan_mail_manager_copy_messages_get_error(struct etpan_mail_manager * manager)
{
  return manager->copy_msg_error;
}

static void copy_folder(struct etpan_mail_manager * manager,
    struct etpan_storage * dest_storage,
    struct etpan_folder * dest_folder,
    struct etpan_folder * source_folder,
    int delete);
static void copy_folder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

unsigned int etpan_mail_manager_get_next_copy_folder_id(struct etpan_mail_manager * manager)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->copy_folder_op_list) ; i ++) {
    struct etpan_folder_copy * folder_copy;
    
    folder_copy = carray_get(manager->copy_folder_op_list, i);
    if (folder_copy == NULL)
      return i;
  }
  
  return carray_count(manager->copy_folder_op_list);
}

void etpan_mail_manager_copy_folder(struct etpan_mail_manager * manager,
    struct etpan_storage * dest_storage,
    struct etpan_folder * dest_folder,
    struct etpan_folder * source_folder)
{
  copy_folder(manager,
      dest_storage, dest_folder, source_folder, 0);
}

void etpan_mail_manager_move_folder(struct etpan_mail_manager * manager,
    struct etpan_storage * dest_storage,
    struct etpan_folder * dest_folder,
    struct etpan_folder * source_folder)
{
  copy_folder(manager,
      dest_storage, dest_folder, source_folder, 1);
}

static void copy_folder(struct etpan_mail_manager * manager,
    struct etpan_storage * dest_storage,
    struct etpan_folder * dest_folder,
    struct etpan_folder * source_folder,
    int delete)
{
  struct etpan_folder_copy * folder_copy;
  unsigned int op_index;
  
  if (delete)
    ETPAN_LOCAL_LOG("move folder");
  else
    ETPAN_LOCAL_LOG("copy folder");
  
  folder_copy = etpan_folder_copy_new();
  etpan_folder_copy_set_delete(folder_copy, delete);
  etpan_folder_copy_set_source(folder_copy, source_folder);
  etpan_folder_copy_set_dest_storage(folder_copy, dest_storage);
  etpan_folder_copy_set_dest_folder(folder_copy, dest_folder);
  etpan_folder_copy_setup(folder_copy);
  
  op_index = etpan_mail_manager_get_next_copy_folder_id(manager);
  if (op_index >= carray_count(manager->copy_folder_op_list))
    carray_add(manager->copy_folder_op_list, folder_copy, NULL);
  else
    carray_set(manager->copy_folder_op_list, op_index, folder_copy);
  
  ETPAN_SIGNAL_ADD_HANDLER(folder_copy, ETPAN_FOLDER_COPY_FINISHED_SIGNAL,
      copy_folder_finished_handler, manager);
  
  etpan_folder_copy_run(folder_copy);
  
  /* XXX - reorder should be applied at folder creation,
     if folder copy, uses mailmanager copy */
}

static void copy_folder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_copy * folder_copy;
  struct etpan_folder * dest_folder;
  struct etpan_storage * dest_storage;
#if 0
  struct folder_tree * created_tree;
  struct folder_tree * dest_tree;
#endif
  struct etpan_error * error;
  struct etpan_mail_manager * manager;
  unsigned int op_index;
  unsigned int i;
  (void) signal_name;
  (void) signal_data;
  
  manager = user_data;
  folder_copy = sender;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(folder_copy, ETPAN_FOLDER_COPY_FINISHED_SIGNAL,
      copy_folder_finished_handler, manager);
  
  error = etpan_folder_copy_get_error(folder_copy);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  dest_folder = etpan_folder_copy_get_dest_folder(folder_copy);
  dest_storage = etpan_folder_copy_get_dest_storage(folder_copy);
  
  ETPAN_LOCAL_LOG("finished move/copy of folder");
  
  op_index = 0;
  for(i = 0 ; i < carray_count(manager->copy_folder_op_list) ; i ++) {
    struct etpan_folder_copy * cur_folder_copy;
    
    cur_folder_copy = carray_get(manager->copy_folder_op_list, i);
    if (cur_folder_copy == folder_copy) {
      carray_set(manager->copy_folder_op_list, i, NULL);
      op_index = i;
      break;
    }
  }
  
  if (error != NULL) {
    manager->copy_folder_error = etpan_error_dup(error);
  }
  
  ETPAN_SIGNAL_SEND_WITH_DATA(manager,
      ETPAN_MAIL_MANAGER_COPY_FOLDER_FINISHED_SIGNAL, &op_index);
  
  ETPAN_ERROR_FREE(manager->copy_folder_error);
  manager->copy_folder_error = NULL;  
  etpan_folder_copy_unref(folder_copy);
  
#if 0
  created_folder = etpan_folder_copy_get_created_folder(folder_copy);
  if (created_folder != NULL)
    etpan_folder_ref(created_folder);
  
  etpan_folder_copy_unref(folder_copy);
  
  /* reorder */
  created_tree = NULL;
  if (created_folder != NULL) {
    created_tree = get_tree_by_ui_path(folder_list,
        etpan_folder_get_ui_path(created_folder));
    etpan_folder_unref(created_folder);
  }
  
  dest_tree = NULL;
  if (folder_list->copy_dest_path != NULL) {
    dest_tree = get_tree_by_ui_path(folder_list, folder_list->copy_dest_path);
    free(folder_list->copy_dest_path);
    folder_list->copy_dest_path = NULL;
  }
  
  if ((dest_tree != NULL) && (created_tree != NULL)) {
    reorder(folder_list, created_tree,
        folder_list->copy_dest_type, dest_tree);
  }
#endif
}

struct etpan_error * etpan_mail_manager_copy_folder_get_error(struct etpan_mail_manager * manager)
{
  return manager->copy_folder_error;
}

chash * etpan_mail_manager_get_pending_added_messages(struct etpan_mail_manager * manager, struct etpan_folder * folder)
{
  unsigned int i;
  chash * pending;
  int r;
  
  pending = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (pending == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(manager->copy_msg_op_list) ; i ++) {
    struct etpan_message_copy * copy_msg;
    chash * current_pending;
    chashiter * iter;
    struct etpan_folder * current_dest;
    
    copy_msg = carray_get(manager->copy_msg_op_list, i);
    if (copy_msg == NULL)
      continue;
    
    current_dest = etpan_message_copy_get_destination(copy_msg);
    if (current_dest != folder)
      continue;
    
    /* a pending message will appear only once */
    current_pending = etpan_message_copy_get_pending_for_copy(copy_msg);
    for(iter = chash_begin(current_pending) ; iter != NULL ;
        iter = chash_next(current_pending, iter)) {
      chashdatum key;
      chashdatum value;
      
      chash_key(iter, &key);
      chash_value(iter, &value);
      
      r = chash_set(pending, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return pending;
}

chash * etpan_mail_manager_get_pending_deleted_messages(struct etpan_mail_manager * manager)
{
  unsigned int i;
  chash * pending;
  int r;
  
  pending = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (pending == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(manager->copy_msg_op_list) ; i ++) {
    struct etpan_message_copy * copy_msg;
    chash * current_pending;
    chashiter * iter;
    
    copy_msg = carray_get(manager->copy_msg_op_list, i);
    if (copy_msg == NULL)
      continue;
    
    current_pending = etpan_message_copy_get_pending_for_deletion(copy_msg);
    for(iter = chash_begin(current_pending) ; iter != NULL ;
        iter = chash_next(current_pending, iter)) {
      chashdatum key;
      chashdatum value;
      struct etpan_message * msg;
      
      chash_key(iter, &key);
      chash_value(iter, &value);

      msg = value.data;
      
      r = chash_set(pending, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  for(i = 0 ; i < carray_count(manager->delete_msg_op_list) ; i ++) {
    struct etpan_message_delete * delete_msg;
    chash * current_pending;
    chashiter * iter;
    
    delete_msg = carray_get(manager->delete_msg_op_list, i);
    if (delete_msg == NULL)
      continue;
    
    current_pending = etpan_message_delete_get_pending_for_deletion(delete_msg);
    for(iter = chash_begin(current_pending) ; iter != NULL ;
        iter = chash_next(current_pending, iter)) {
      chashdatum key;
      chashdatum value;
      struct etpan_message * msg;
      
      chash_key(iter, &key);
      chash_value(iter, &value);
      
      msg = value.data;
      
      r = chash_set(pending, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return pending;
}

int etpan_mail_manager_operation_is_allowed(struct etpan_mail_manager * manager,
    struct etpan_folder * folder)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(manager->copy_msg_op_list) ; i ++) {
    struct etpan_message_copy * copy_msg;
    struct etpan_folder * dest;
    
    copy_msg = carray_get(manager->copy_msg_op_list, i);
    if (copy_msg == NULL)
      continue;
    
    dest = etpan_message_copy_get_destination(copy_msg);
    if (dest == folder)
      return 0;

    if (etpan_folder_is_sub_folder(dest, folder))
      return 0;
  }
  
  return 1;
}

#if 0
void etpan_mail_manager_reschedule(struct etpan_mail_manager * manager)
{
  reschedule(manager);
}
#endif
