#include "etpan-message-archive.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-signal.h"
#include "etpan-nls.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-message-copy-local.h"

static struct etpan_error * rewrite_error(struct etpan_error * error);

struct etpan_message_archive * etpan_message_archive_new(void)
{
  struct etpan_message_archive * msg_archive;
  
  msg_archive = malloc(sizeof(* msg_archive));
  if (msg_archive == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_archive->error = NULL;
  msg_archive->ref_count = 1;
  msg_archive->msg_copy_local = etpan_message_copy_local_new();
  etpan_message_copy_local_set_delete(msg_archive->msg_copy_local, 1);
  etpan_message_copy_local_set_folder_name(msg_archive->msg_copy_local,
      "Archived Messages");
  
  return msg_archive;
}

static void etpan_message_archive_free(struct etpan_message_archive * msg_archive)
{
  etpan_message_copy_local_unref(msg_archive->msg_copy_local);
  ETPAN_ERROR_FREE(msg_archive->error);
  free(msg_archive);
}

void etpan_message_archive_ref(struct etpan_message_archive * msg_archive)
{
  msg_archive->ref_count ++;
}

void etpan_message_archive_unref(struct etpan_message_archive * msg_archive)
{
  msg_archive->ref_count --;
  if (msg_archive->ref_count <= 0) {
    etpan_message_archive_free(msg_archive);
  }
}

void etpan_message_archive_set_msglist(struct etpan_message_archive * msg_archive,
    chash * msglist)
{
  etpan_message_copy_local_set_msglist(msg_archive->msg_copy_local, msglist);
}

void etpan_message_archive_setup(struct etpan_message_archive * msg_archive)
{
  etpan_message_copy_local_setup(msg_archive->msg_copy_local);
}

static void copy_local_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_message_archive_run(struct etpan_message_archive * msg_archive)
{
  ETPAN_SIGNAL_ADD_HANDLER(msg_archive->msg_copy_local,
      ETPAN_MESSAGE_COPY_LOCAL_FINISHED_SIGNAL,
      copy_local_finished_handler, msg_archive);
  
  etpan_message_copy_local_run(msg_archive->msg_copy_local);
}

static void copy_local_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_archive * msg_archive;
  struct etpan_error * error;
  
  msg_archive = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(msg_archive->msg_copy_local,
      ETPAN_MESSAGE_COPY_LOCAL_FINISHED_SIGNAL,
      copy_local_finished_handler, msg_archive);
  
  error = etpan_message_copy_local_get_error(msg_archive->msg_copy_local);
  if (error != NULL) {
    msg_archive->error = rewrite_error(error);
  }
  
  etpan_message_archive_ref(msg_archive);
  ETPAN_SIGNAL_SEND(msg_archive, ETPAN_MESSAGE_ARCHIVE_FINISHED_SIGNAL);
  
  ETPAN_ERROR_FREE(msg_archive->error);
  msg_archive->error = NULL;
  etpan_message_archive_unref(msg_archive);
}

void etpan_message_archive_cancel(struct etpan_message_archive * msg_archive)
{
  etpan_message_copy_local_cancel(msg_archive->msg_copy_local);
}

struct etpan_error *
etpan_message_archive_get_error(struct etpan_message_archive * msg_archive)
{
  return msg_archive->error;
}

static struct etpan_error * rewrite_error(struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred while archiving messages\n"
          "%s"),
      previous_long_description);
  
  return new_error;
}

chash * etpan_message_archive_get_pending_for_deletion(struct etpan_message_archive * msg_archive)
{
  return etpan_message_copy_local_get_pending_for_deletion(msg_archive->msg_copy_local);
}
