#include "etpan-message-color.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-config-types.h"
#include "etpan-tc.h"
#include "etpan-log.h"
#include "etpan-nls.h"

/* should be done by folder */

static struct etpan_message_color * default_manager = NULL;

struct etpan_message_color * etpan_message_color_get_default(void)
{
  return default_manager;
}

void etpan_message_color_set_default(struct etpan_message_color * msg_color)
{
  default_manager = msg_color;
}

struct etpan_message_color * etpan_message_color_new(void)
{
  struct etpan_message_color * msg_color;
  
  msg_color = malloc(sizeof(* msg_color));
  if (msg_color == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_color->msgid_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (msg_color->msgid_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return msg_color;
}

void etpan_message_color_free(struct etpan_message_color * msg_color)
{
  chash_free(msg_color->msgid_hash);
  free(msg_color);
}

char * etpan_message_color_get_color(struct etpan_message_color * msg_color,
    char * msgid)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = msgid;
  key.len = strlen(msgid) + 1;
  
  r = chash_get(msg_color->msgid_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

void etpan_message_color_set_color(struct etpan_message_color * msg_color,
    char * msgid, char * color)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = msgid;
  key.len = strlen(msgid) + 1;
  value.data = color;
  value.len = strlen(color) + 1;
  
  r = chash_set(msg_color->msgid_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static inline char * get_entry(tcconf_section_t * conf, char * name)
{
  int r;
  char * value;
  
  r = tcconf_getvalue(conf, name, "%s", &value);
  if (r <= 0)
    return NULL;
  
  return value;
}

struct etpan_error * etpan_message_color_read(struct etpan_message_color * msg_color,
    char * filename)
{
  tcconf_section_t * config;
  void * iter;
  
  config = tcconf_load_file(NULL, filename);
  if (config == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Address book configuration at %s could not be parsed."), filename);
    
    return error;
  }
  
  iter = NULL;
  do {
    tcconf_section_t * msg_info;
    
    msg_info = tcconf_nextsection(config, "message", &iter);
    if (msg_info != NULL) {
      char * msgid;
      char * color;
      
      msgid = get_entry(msg_info, "message-id");
      color = get_entry(msg_info, "color");
      
      if ((msgid != NULL) && (color != NULL)) {
        etpan_message_color_set_color(msg_color, msgid, color);
      }
      
      free(color);
      free(msgid);
      
      tcfree(msg_info);
    }
    else {
      break;
    }
  } while (1);
  
  tcfree(config);
  
  return NULL;
}

struct etpan_error * etpan_message_color_read_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_message_color * manager;
  
  manager = etpan_message_color_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_COLOR_FILE);
  
  return etpan_message_color_read(manager, filename);
}

struct etpan_error *
etpan_message_color_save(struct etpan_message_color * msg_color,
    char * filename)
{
  FILE * f;
  chashiter * iter;
  
  f = fopen(filename, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Configuration file could not be written"));
    etpan_error_strf_long_description(error, _("Color configuration file %s could not be written to disk."), filename);
    
    return error;
  }
  
  for(iter = chash_begin(msg_color->msgid_hash) ; iter != NULL ;
      iter = chash_next(msg_color->msgid_hash, iter)) {
    chashdatum key;
    chashdatum value;
    char * msgid;
    char * color;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msgid = key.data;
    color = value.data;
    
    fprintf(f, "message [\n");
    {
      char * quoted_msgid;
      
      quoted_msgid = etpan_quote_string(msgid);
      fprintf(f, "  message-id \"%s\"\n", quoted_msgid);
      free(quoted_msgid);
    }
    fprintf(f, "  color \"%s\"\n", color);
    fprintf(f, "]\n");
    fprintf(f, "\n");
  }
  
  fclose(f);
  
  return NULL;
}

struct etpan_error *
etpan_message_color_save_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_message_color * manager;
  
  manager = etpan_message_color_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_COLOR_FILE);
  
  return etpan_message_color_save(manager, filename);
}
