#include "etpan-message-composer.h"

#include <libgen.h>
#include <unistd.h>

#include "etpan-message-header.h"
#include "etpan-message.h"
#include "etpan-error.h"
#include "etpan-lep.h"
#include "etpan-message-fetcher.h"
#include "etpan-signal.h"
#include "etpan-text-quote.h"
#include "etpan-text-wrapper.h"
#include "etpan-address.h"
#include "etpan-part.h"
#include "etpan-part-header.h"
#include "etpan-nls.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-folder.h"
#include "etpan-outbox.h"
#include "etpan-sender.h"

#define ETPAN_MODULE_LOG_NAME "COMPOSER"
#include "etpan-log.h"

#define DEFAULT_CHARSET "utf-8"

static void set_reference_message(struct etpan_message_composer * composer,
    struct etpan_message * msg);

struct etpan_message_composer * etpan_message_composer_new(void)
{
  struct etpan_message_composer * composer;
  
  composer = malloc(sizeof(* composer));
  if (composer == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  composer->header = etpan_message_header_new();
  
  composer->attachment_list = carray_new(4);
  if (composer->attachment_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  composer->cursor = 0;
  composer->fetcher = NULL;
  composer->error = NULL;
  composer->reply_type = ETPAN_REPLY_TYPE_AUTO;
  composer->forward_type = ETPAN_FORWARD_TYPE_NORMAL;
  composer->account_id = NULL;
  composer->compose_type = ETPAN_COMPOSE_TYPE_NEW;
  composer->reference_msg = NULL;
  
  return composer;
}

static void free_item(struct etpan_attach_item * item)
{
  switch (item->type) {
  case ETPAN_ATTACH_TYPE_TEXT:
    free(item->text);
    break;
  case ETPAN_ATTACH_TYPE_FILE:
    free(item->filename);
    break;
  }
  
  free(item);
}

void etpan_message_composer_free(struct etpan_message_composer * composer)
{
  unsigned int i;
  
  set_reference_message(composer, NULL);
  free(composer->account_id);
  for(i = 0 ; i < carray_count(composer->attachment_list) ; i ++) {
    struct etpan_attach_item * item;
    
    item = carray_get(composer->attachment_list, i);
    if (item->temporary)
      unlink(item->filename);
    free_item(item);
  }
  carray_free(composer->attachment_list);
  
  etpan_message_header_free(composer->header);
  free(composer);
}

void etpan_message_composer_add_text(struct etpan_message_composer * composer,
    char * text, int length)
{
  int r;
  struct etpan_attach_item * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item->temporary = 0;
  item->type = ETPAN_ATTACH_TYPE_TEXT;
  item->text = malloc(length + 1);
  if (item->text == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  strncpy(item->text, text, length);
  item->text[length] = '\0';
  item->size = length;
  
  r = carray_add(composer->attachment_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_message_composer_preprend_text(struct etpan_message_composer * composer,
    char * text, int length)
{
  int r;
  struct etpan_attach_item * item;
  unsigned int i;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item->temporary = 0;
  item->type = ETPAN_ATTACH_TYPE_TEXT;
  item->text = malloc(length + 1);
  if (item->text == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  strncpy(item->text, text, length);
  item->text[length] = '\0';
  item->size = length;
  
  r = carray_add(composer->attachment_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = carray_count(composer->attachment_list) - 1 ; i > 0 ; i --) {
    struct etpan_attach_item * cur_item;
    
    cur_item = carray_get(composer->attachment_list, i - 1);
    carray_set(composer->attachment_list, i, cur_item);
  }
  carray_set(composer->attachment_list, 0, item);
}

void etpan_message_composer_add_file(struct etpan_message_composer * composer,
    char * filename, int temporary)
{
  int r;
  struct etpan_attach_item * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item->type = ETPAN_ATTACH_TYPE_FILE;
  item->filename = strdup(filename);
  item->temporary = temporary;
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(composer->attachment_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_message_composer_set_references(struct etpan_message_composer * composer, carray * references)
{
  struct etpan_message_header * header;
  char * replyto_msgid;
  carray * inreplyto;
  
  header = composer->header;
  
  if (carray_count(references) == 0) {
    etpan_message_header_set_references(header, references);
    etpan_message_header_set_inreplyto(header, references);
    return;
  }
  
  replyto_msgid = carray_get(references, carray_count(references) - 1);
  
  etpan_message_header_set_references(header, references);
  
  inreplyto = etpan_message_id_list_new();
  if (inreplyto == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  etpan_message_id_list_add(inreplyto, replyto_msgid);
  etpan_message_header_set_inreplyto(header, inreplyto);
  
  etpan_message_id_list_free(inreplyto);
}

void etpan_message_composer_set_message(struct etpan_message_composer * composer, struct etpan_message * msg)
{
  (void) composer;
  (void) msg;
  
  /* XXX - needs to be implemented */
  
  ETPAN_WARN_LOG("not yet implemented");
  etpan_crash();
}

struct mailmime * get_file_part(char * filename)
{
  char * name;
  char basename_buf[PATH_MAX];
  char * disposition_name;
  int encoding_type;
  struct mailmime_disposition * disposition;
  struct mailmime_mechanism * encoding;
  struct mailmime_content * content;
  char * dup_filename;
  struct mailmime * mime;
  struct mailmime_fields * mime_fields;
  int r;
  
  if (filename != NULL) {
    strncpy(basename_buf, filename, sizeof(basename_buf));
    name = basename(basename_buf);
  }
  else {
    name = NULL;
  }
  
  disposition_name = strdup(name);
  if (disposition_name == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  disposition =
    mailmime_disposition_new_with_data(MAILMIME_DISPOSITION_TYPE_ATTACHMENT,
        disposition_name, NULL, NULL, NULL, (size_t) -1);
  if (disposition == NULL) {
    free(disposition_name);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  content = mailmime_content_new_with_str("application/octet-stream");
  if (content == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  encoding_type = MAILMIME_MECHANISM_BASE64;
  encoding = mailmime_mechanism_new(encoding_type, NULL);
  if (encoding == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  mime_fields = mailmime_fields_new_with_data(encoding,
      NULL, NULL, disposition, NULL);
  if (mime_fields == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  mime = mailmime_new_empty(content, mime_fields);
  if (mime == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  dup_filename = strdup(filename);
  if (dup_filename == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = mailmime_set_body_file(mime, dup_filename);
  if (r != MAILIMF_NO_ERROR) {
    free(dup_filename);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return mime;
}

struct mailmime * get_text_part(char * text, size_t length)
{
  struct mailmime_fields * mime_fields;
  struct mailmime * mime;
  int r;
  struct mailmime_content * content;
  struct mailmime_parameter * param;
  
  mime_fields = mailmime_fields_new_encoding(MAILMIME_MECHANISM_8BIT);
  if (mime_fields == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  content = mailmime_content_new_with_str("text/plain");
  if (content == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param = mailmime_param_new_with_data("charset", DEFAULT_CHARSET);
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = clist_append(content->ct_parameters, param);
  if (r < 0) {
    mailmime_parameter_free(param);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  mime = mailmime_new_empty(content, mime_fields);
  if (mime == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = mailmime_set_body_text(mime, text, length);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return mime;
}

void etpan_message_composer_render(struct etpan_message_composer * composer,
    char ** result_text, size_t * result_size)
{
  struct mailimf_fields * fields;
  unsigned int i;
  struct mailmime * mime;
  struct mailmime * submime;
  int r;
  MMAPString * str;
  int col;
  size_t length;
  char * text;
  
  fields = etpan_header_to_lep_header(composer->header);
  
  mime = mailmime_new_message_data(NULL);
  if (mime == NULL) {
    mailimf_fields_free(fields);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  mailmime_set_imf_fields(mime, fields);
  
  for(i = 0 ; i < carray_count(composer->attachment_list) ; i ++) {
    struct etpan_attach_item * item;
    
    item = carray_get(composer->attachment_list, i);
    switch (item->type) {
    case ETPAN_ATTACH_TYPE_TEXT:
      submime = get_text_part(item->text, item->size);
      
      r = mailmime_smart_add_part(mime, submime);
      if (r != MAILIMF_NO_ERROR) {
        mailmime_free(submime);
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      break;
    case ETPAN_ATTACH_TYPE_FILE:
      submime = get_file_part(item->filename);
      
      r = mailmime_smart_add_part(mime, submime);
      if (r != MAILIMF_NO_ERROR) {
        mailmime_free(submime);
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      break;
    }
  }
  
  str = mmap_string_new("");
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  col = 0;
  
  r = mailmime_write_mem(str, &col, mime);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  text = malloc(str->len + 1);
  if (text == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  strncpy(text, str->str, str->len);
  text[str->len] = '\0';
  
  length = str->len;
  
  mmap_string_free(str);
  mailmime_free(mime);
  
  * result_text = text;
  * result_size = length;
}

struct etpan_message_header * etpan_message_composer_get_header(struct etpan_message_composer * composer)
{
  return composer->header;
}

carray * etpan_message_composer_get_part_list(struct etpan_message_composer * composer)
{
  return composer->attachment_list;
}

int etpan_attach_item_get_type(struct etpan_attach_item * item)
{
  return item->type;
}

char * etpan_attach_item_file_get_name(struct etpan_attach_item * item)
{
  return item->filename;
}

int etpan_attach_item_file_is_temporary(struct etpan_attach_item * item)
{
  return item->temporary;
}

void etpan_attach_item_file_set_temporary(struct etpan_attach_item * item,
    int temporary)
{
  item->temporary = temporary;
}

void etpan_attach_item_get_text_content(struct etpan_attach_item * item,
    char ** p_content, size_t * p_content_size)
{
  * p_content = item->text;
  * p_content_size = item->size;
}

void etpan_attach_item_set_text_content(struct etpan_attach_item * item,
    char * content, size_t content_size)
{
  free(item->text);
  item->text = strdup(content);
  item->size = content_size;
}

void etpan_message_composer_set_cursor(struct etpan_message_composer * composer, int part_index)
{
  composer->cursor = part_index;
}

int etpan_message_composer_get_cursor(struct etpan_message_composer * composer)
{
  return composer->cursor;
}


/* fetcher */


static int header_set_reply_normal(struct etpan_message_composer * composer,
    struct etpan_message_header * original_header,
    struct etpan_message_header * header)
{
  carray * original_replyto;
  carray * to;
  struct etpan_address * original_addr;
  struct etpan_address * addr;
  carray * original_from;
  
  original_replyto = etpan_message_header_get_replyto(original_header);
  if (carray_count(original_replyto) == 0)
    goto fallback;
  
  original_addr = carray_get(original_replyto, 0);
  
  to = etpan_message_header_get_to(header);
  addr = etpan_address_dup(original_addr);
  etpan_address_list_add_address(to, addr);
  
  return 1;
  
 fallback:
  original_from = etpan_message_header_get_from(original_header);
  if (carray_count(original_from) == 0)
    return 0;
  
  original_addr = carray_get(original_from, 0);
  
  to = etpan_message_header_get_to(header);
  addr = etpan_address_dup(original_addr);
  etpan_address_list_add_address(to, addr);
  
  return 1;
}

static int header_set_reply_list(struct etpan_message_composer * composer,
    struct etpan_message_header * original_header,
    struct etpan_message_header * header)
{
  struct etpan_address * listpost;
  carray * to;
  struct etpan_address * addr;
  carray * original_from;
  
  listpost = etpan_message_header_get_listpost(original_header);
  if (listpost == NULL)
    return 0;
  
  /* for mailman */
  original_from = etpan_message_header_get_from(original_header);
  if (carray_count(original_from) > 0) {
    char * mail;
    
    addr = carray_get(original_from, 0);
    mail = etpan_address_get_address(addr);
    if (mail != NULL) {
      if (strstr(mail, "-request@") != NULL)
        return 0;
    }
  }
  
  to = etpan_message_header_get_to(header);
  addr = etpan_address_dup(listpost);
  etpan_address_list_add_address(to, addr);
  
  return 1;
}

static int header_set_reply_all(struct etpan_message_composer * composer,
    struct etpan_message_header * original_header,
    struct etpan_message_header * header)
{
  chash * addr_hash;
  unsigned int i;
  carray * to;
  carray * cc;
  carray * original_to;
  carray * original_cc;
  int r;
  carray * original_replyto;
  carray * original_from;
  struct etpan_account * account;
  
  header_set_reply_normal(composer, original_header, header);
  addr_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  
  if (composer->account_id != NULL) {
    account = etpan_account_manager_get_account(etpan_account_manager_get_default(),
        composer->account_id);
    if (account != NULL) {
      chashdatum key;
      chashdatum value;
      char * email;
    
      email = etpan_account_get_mail(account);
      if (email != NULL) {
        key.data = email;
        key.len = strlen(email);
        value.data = NULL;
        value.len = 0;
        r = chash_set(addr_hash, &key, &value, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  to = etpan_message_header_get_to(header);
  for(i = 0 ; i < carray_count(to) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * email;
    struct etpan_address * addr;
    
    addr = carray_get(to, i);
    email = etpan_address_get_address(addr);
    if (email == NULL)
      continue;
    
    key.data = email;
    key.len = strlen(email);
    value.data = NULL;
    value.len = 0;
    r = chash_set(addr_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  cc = etpan_message_header_get_cc(header);
  for(i = 0 ; i < carray_count(cc) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * email;
    struct etpan_address * addr;
    
    addr = carray_get(cc, i);
    email = etpan_address_get_address(addr);
    if (email == NULL)
      continue;
    
    key.data = email;
    key.len = strlen(email);
    value.data = NULL;
    value.len = 0;
    r = chash_set(addr_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  original_to = etpan_message_header_get_to(original_header);
  for(i = 0 ; i < carray_count(original_to) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * email;
    struct etpan_address * original_addr;
    struct etpan_address * addr;
    
    original_addr = carray_get(original_to, i);
    email = etpan_address_get_address(original_addr);
    if (email == NULL)
      continue;
    
    key.data = email;
    key.len = strlen(email);
    r = chash_get(addr_hash, &key, &value);
    if (r == 0)
      continue;
    
    value.data = NULL;
    value.len = 0;
    r = chash_set(addr_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    addr = etpan_address_dup(original_addr);
    etpan_address_list_add_address(cc, addr);
  }
  
  original_cc = etpan_message_header_get_cc(original_header);
  for(i = 0 ; i < carray_count(original_cc) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * email;
    struct etpan_address * original_addr;
    struct etpan_address * addr;
    
    original_addr = carray_get(original_cc, i);
    email = etpan_address_get_address(original_addr);
    if (email == NULL)
      continue;
    
    key.data = email;
    key.len = strlen(email);
    r = chash_get(addr_hash, &key, &value);
    if (r == 0)
      continue;
    
    value.data = NULL;
    value.len = 0;
    r = chash_set(addr_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    addr = etpan_address_dup(original_addr);
    etpan_address_list_add_address(cc, addr);
  }
  
  original_replyto = etpan_message_header_get_replyto(original_header);
  for(i = 0 ; i < carray_count(original_replyto) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * email;
    struct etpan_address * original_addr;
    struct etpan_address * addr;
    
    original_addr = carray_get(original_replyto, i);
    email = etpan_address_get_address(original_addr);
    if (email == NULL)
      continue;
    
    key.data = email;
    key.len = strlen(email);
    r = chash_get(addr_hash, &key, &value);
    if (r == 0)
      continue;
    
    value.data = NULL;
    value.len = 0;
    r = chash_set(addr_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    addr = etpan_address_dup(original_addr);
    etpan_address_list_add_address(cc, addr);
  }
  
  if (carray_count(original_replyto) == 0) {
    original_from = etpan_message_header_get_from(original_header);
    for(i = 0 ; i < carray_count(original_from) ; i ++) {
      chashdatum key;
      chashdatum value;
      char * email;
      struct etpan_address * original_addr;
      struct etpan_address * addr;
    
      original_addr = carray_get(original_from, i);
      email = etpan_address_get_address(original_addr);
      if (email == NULL)
        continue;
    
      key.data = email;
      key.len = strlen(email);
      r = chash_get(addr_hash, &key, &value);
      if (r == 0)
        continue;
    
      value.data = NULL;
      value.len = 0;
      r = chash_set(addr_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      addr = etpan_address_dup(original_addr);
      etpan_address_list_add_address(cc, addr);
    }
  }
  
  chash_free(addr_hash);
  
  return 1;
}

static void header_set_reply_auto(struct etpan_message_composer * composer,
    struct etpan_message_header * original_header,
    struct etpan_message_header * header)
{
  if (header_set_reply_list(composer, original_header, header))
    return;
  
  header_set_reply_all(composer, original_header, header);
}


static void show_header_part(MMAPString * buffer,
    char * content, size_t content_length);

static void show_reply_header_part(MMAPString * buffer,
    char * content, size_t content_length);

static void reply_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_composer * composer;
  struct etpan_message_fetcher * fetcher;
  carray * part_list;
  unsigned int i;
  int cursor;
  struct etpan_message_header * header;
  struct etpan_error * error;  
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  composer = user_data;
  
  fetcher = composer->fetcher;
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, composer,
      reply_callback);
  
  composer->fetcher = NULL;
  
  error = etpan_message_fetcher_get_error(fetcher);
  if (error != NULL) {
    composer->error = etpan_error_dup(error);
    goto free_fetcher;
  }
  
  part_list = etpan_message_fetcher_get_part_list(fetcher);
  
  /* setup headers */
  header = etpan_message_composer_get_header(composer);
  if (carray_count(part_list) > 0) {
    struct etpan_part_fetch_info * info;
    struct etpan_part_header * part_header;
    struct etpan_part * part;
    int type;
    
    info = carray_get(part_list, 0);
    type = etpan_part_fetch_info_get_type(info);
    
    if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
      part = etpan_part_fetch_info_get_part(info);
      part_header = etpan_part_get_header(part);
      if (part_header != NULL) {
        char * content_type;
        
        content_type = etpan_part_header_get_content_type(part_header);
        if (strcasecmp(content_type, "message/rfc822") == 0) {
          char * content;
          size_t content_length;
          struct etpan_message_header * original_header;
          char * subject;
          carray * references;
          char * msgid;
          
          etpan_part_fetch_info_get_content(info, &content, &content_length);
          
          original_header = etpan_message_header_parse(content, content_length);
          msgid = etpan_message_header_get_msgid(original_header);
          references = etpan_message_header_get_references(original_header);
          references = etpan_message_id_list_dup(references);
          if (msgid != NULL)
            etpan_message_id_list_add(references, msgid);
          etpan_message_composer_set_references(composer, references);
            
          subject = etpan_message_header_get_subject(original_header);
          if (subject != NULL) {
            char * normalized_subject;
            char * reply_subject;
            size_t len;
              
            normalized_subject = etpan_message_header_subject_normalize(subject);
            len = strlen(normalized_subject) + 5;
            reply_subject = malloc(len);
            snprintf(reply_subject, len, "Re: %s", normalized_subject);
            etpan_message_header_set_subject(header, reply_subject);
            free(reply_subject);
            free(normalized_subject);
          }
          else {
            etpan_message_header_set_subject(header, "Re: ");
          }
          
          switch (composer->reply_type) {
          case ETPAN_REPLY_TYPE_NORMAL:
            header_set_reply_normal(composer, original_header, header);
            break;
          case ETPAN_REPLY_TYPE_LIST:
            header_set_reply_list(composer, original_header, header);
            break;
          case ETPAN_REPLY_TYPE_ALL:
            header_set_reply_all(composer, original_header, header);
            break;
          case ETPAN_REPLY_TYPE_AUTO:
            header_set_reply_auto(composer, original_header, header);
            break;
          }
        }
      }
    }
  }
  
  for(i = 0 ; i < carray_count(part_list) ; i ++) {
    struct etpan_part_fetch_info * info;
    int type;
    
    info = carray_get(part_list, i);
    type = etpan_part_fetch_info_get_type(info);
    
    if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
      char * content;
      size_t content_length;
      struct etpan_part_header * header;
      char * content_type;
      struct etpan_part * part;
      
      part = etpan_part_fetch_info_get_part(info);
      header = etpan_part_get_header(part);
      if (header == NULL) {
        ETPAN_WARN_LOG("WARNING! part has no header");
        continue;
      }
      
      content_type = etpan_part_header_get_content_type(header);
      
      etpan_part_fetch_info_get_content(info, &content, &content_length);
      
      if (strcasecmp(content_type, "message/rfc822") == 0) {
        MMAPString * buffer;
        
        buffer = mmap_string_new("");
        if (buffer == NULL) {
          ETPAN_LOG_MEMORY_ERROR;
        }
        
        show_reply_header_part(buffer, content, content_length);
        
        etpan_message_composer_add_text(composer,
            buffer->str, buffer->len);
        
        mmap_string_free(buffer);
      }
      else {
        char * quoted;
        size_t quoted_length;
        
        etpan_quote_text(content, content_length,
            &quoted, &quoted_length);
        
        etpan_message_composer_add_text(composer, quoted, quoted_length);
        
        free(quoted);
        
        etpan_message_composer_add_text(composer, "\n", 1);
      }
    }
    else if (type == ETPAN_PART_FETCH_INFO_TYPE_FILE) {
      char * filename;
      int temporary;
      
      filename = etpan_part_fetch_info_get_filename(info);
      temporary = etpan_part_fetch_info_is_temporary(info);
      etpan_part_fetch_info_set_temporary(info, 0);
      
      etpan_message_composer_add_file(composer, filename, temporary);
    }
    else if (type == ETPAN_PART_FETCH_INFO_NONE) {
      char * filename;
      struct etpan_part * part;
      struct etpan_part_header * part_header;
      
      part = etpan_part_fetch_info_get_part(info);
      filename = NULL;
      part_header = etpan_part_get_header(part);
      if (part_header != NULL) {
        filename = etpan_part_header_get_filename(part_header);
        if (filename != NULL) {
          MMAPString * buffer;
          
          buffer = mmap_string_new("");
          if (buffer == NULL) {
            ETPAN_LOG_MEMORY_ERROR;
          }
          
          if (mmap_string_append(buffer, _("\n> - attachment ")) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          
          if (mmap_string_append(buffer, filename) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          if (mmap_string_append(buffer, _("\n")) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          
          etpan_message_composer_add_text(composer,
              buffer->str, buffer->len);
          
          mmap_string_free(buffer);
        }
      }
      if (filename == NULL) {
        char * text;
        
        text = _("\n> - unnamed attachment\n");
        
        etpan_message_composer_add_text(composer, text, strlen(text));
      }
    }
  }
  
  etpan_message_composer_add_text(composer, _("\n"), 1);
  etpan_message_composer_add_text(composer, _("\n"), 1);
  
  cursor = (carray_count(etpan_message_composer_get_part_list(composer)) - 1);
  etpan_message_composer_set_cursor(composer, cursor);
  
  etpan_message_composer_add_signature(composer);
  
 free_fetcher:
  etpan_message_fetcher_unref(fetcher);
  composer->fetcher = NULL;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, composer,
      NULL);
  
  ETPAN_ERROR_FREE(composer->error);
  composer->error = NULL;
}


void etpan_message_composer_reply(struct etpan_message_composer * composer,
    struct etpan_message * msg, int reply_type)
{
  struct etpan_message_fetcher * fetcher;
  
  composer->reply_type = reply_type;
  
  composer->compose_type = ETPAN_COMPOSE_TYPE_REPLY;
  set_reference_message(composer, msg);
  
  fetcher = etpan_message_fetcher_new();
  
  etpan_message_fetcher_set_flags(fetcher, 0);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, composer,
      reply_callback);
  composer->fetcher = fetcher;
  
  etpan_message_fetcher_run(fetcher);
}

void etpan_message_composer_reply_cancel(struct etpan_message_composer * composer)
{
  if (composer->fetcher != NULL)
    etpan_message_fetcher_cancel(composer->fetcher);
}

struct etpan_error *
etpan_message_composer_reply_get_error(struct etpan_message_composer * composer)
{
  return composer->error;
}







/* fill composer window */

static void show_address_list(MMAPString * buffer,
    carray * address_list)
{
  unsigned int i;
  int first;
  
  first = 1;
  for(i = 0 ; i < carray_count(address_list) ; i ++) {
    char * name;
    char * mail;
    struct etpan_address * address;
    
    address = carray_get(address_list, i);
    name = etpan_address_get_display_name(address);
    mail = etpan_address_get_address(address);
    
    if (first)
      first = 0;
    else
      if (mmap_string_append(buffer, ", ") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    
    if (name != NULL) {
      if (mmap_string_append(buffer, name) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else if (mail != NULL) {
      if (mmap_string_append(buffer, mail) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
}

static void show_header_part(MMAPString * buffer,
    char * content, size_t content_length)
{
  carray * unparsed_header;
  unsigned int i;
  time_t date;
  char * subject;
  carray * from;
  carray * replyto;
  carray * to;
  carray * cc;
  carray * bcc;
  struct etpan_message_header * msg_header;
  
  msg_header = etpan_message_header_parse(content, content_length);
  
  from = etpan_message_header_get_from(msg_header);
  if (from != NULL) {
    if (carray_count(from) > 0) {
      if (mmap_string_append(buffer, _("From: ")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      show_address_list(buffer, from);
      mmap_string_append(buffer, _("\n"));
    }
  }
  replyto = etpan_message_header_get_replyto(msg_header);
  if (replyto != NULL) {
    if (carray_count(replyto) > 0) {
      mmap_string_append(buffer, _("Reply-To: "));
      show_address_list(buffer, replyto);
      mmap_string_append(buffer, _("\n"));
    }
  }
  subject = etpan_message_header_get_subject(msg_header);
  if (subject != NULL) {
    mmap_string_append(buffer, _("Subject: "));
    mmap_string_append(buffer, subject);
    mmap_string_append(buffer, _("\n"));
  }
        
  date = etpan_message_header_get_date(msg_header);
  if (date != (time_t) -1) {
    char * date_str;
          
    date_str = etpan_message_get_long_date_str(date);
    if (mmap_string_append(buffer, _("Date: ")) == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    if (mmap_string_append(buffer, date_str) == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    if (mmap_string_append(buffer, _("\n")) == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    free(date_str);
  }
        
  to = etpan_message_header_get_to(msg_header);
  if (to != NULL) {
    if (carray_count(to) > 0) {
      if (mmap_string_append(buffer, _("To: ")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      show_address_list(buffer, to);
      if (mmap_string_append(buffer, _("\n")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  cc = etpan_message_header_get_cc(msg_header);
  if (cc != NULL) {
    if (carray_count(cc) > 0) {
      if (mmap_string_append(buffer, _("Cc: ")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      show_address_list(buffer, cc);
      if (mmap_string_append(buffer, _("\n")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  bcc = etpan_message_header_get_bcc(msg_header);
  if (bcc != NULL) {
    if (carray_count(bcc) > 0) {
      if (mmap_string_append(buffer, _("Bcc: ")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      show_address_list(buffer, bcc);
      if (mmap_string_append(buffer, _("\n")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  etpan_message_header_free(msg_header);
  
  unparsed_header = etpan_unparsed_header_parse(content, content_length);
  
  for(i = 0 ; i < carray_count(unparsed_header) ; i ++) {
    struct etpan_unparsed_header_item * item;
    char * name;
    char * value;
        
    item = carray_get(unparsed_header, i);
    name = etpan_unparsed_header_item_get_name(item);
    value = etpan_unparsed_header_item_get_value(item);
    value = etpan_decode_mime_header(value);
        
    if (etpan_message_is_header_displayed(name)) {
      if (mmap_string_append(buffer, name) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      if (mmap_string_append(buffer, _(": ")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(buffer, value) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(buffer, _("\n")) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
        
    free(value);
  }
  
  if (mmap_string_append(buffer, _("\n")) == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_unparsed_header_free(unparsed_header);
}

static void show_reply_header_part(MMAPString * buffer,
    char * content, size_t content_length)
{
  carray * from;
  struct etpan_message_header * msg_header;
  
  msg_header = etpan_message_header_parse(content, content_length);

  from = etpan_message_header_get_from(msg_header);
  if (from != NULL) {
    if (carray_count(from) > 0) {
      char * name;
      char * mail;
      struct etpan_address * address;
      char reply_string[1024];
      char * value;
      
      address = carray_get(from, 0);
        
      name = etpan_address_get_display_name(address);
      mail = etpan_address_get_address(address);
      
      value = _("sender");
      if (name != NULL) {
        value = name;
      }
      else if (mail != NULL) {
        value = mail;
      }
      
      snprintf(reply_string, sizeof(reply_string), _("%s wrote:\n\n"), value);
      if (mmap_string_append(buffer, reply_string) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  etpan_message_header_free(msg_header);
}



/* forward */

static void forward_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_composer * composer;
  struct etpan_message_fetcher * fetcher;
  carray * part_list;
  unsigned int i;
  struct etpan_message_header * header;
  struct etpan_error * error;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  composer = user_data;
  
  fetcher = composer->fetcher;
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, composer,
      forward_callback);
  
  composer->fetcher = NULL;
  
  error = etpan_message_fetcher_get_error(fetcher);
  if (error != NULL) {
    composer->error = etpan_error_dup(error);
    goto free_fetcher;
  }
  
  part_list = etpan_message_fetcher_get_part_list(fetcher);
  
  /* setup headers */
  header = etpan_message_composer_get_header(composer);
  if (carray_count(part_list) > 0) {
    struct etpan_part_fetch_info * info;
    struct etpan_part_header * part_header;
    struct etpan_part * part;
    int type;
    
    info = carray_get(part_list, 0);
    type = etpan_part_fetch_info_get_type(info);
    
    if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
      part = etpan_part_fetch_info_get_part(info);
      part_header = etpan_part_get_header(part);
      if (part_header != NULL) {
        char * content_type;
        
        content_type = etpan_part_header_get_content_type(part_header);
        if (strcasecmp(content_type, "message/rfc822") == 0) {
          char * content;
          size_t content_length;
          struct etpan_message_header * original_header;
          char * subject;
          
          etpan_part_fetch_info_get_content(info, &content, &content_length);
          
          original_header = etpan_message_header_parse(content,
              content_length);
          subject = etpan_message_header_get_subject(original_header);
          if (subject != NULL) {
            char * normalized_subject;
            char * reply_subject;
            size_t len;
              
            normalized_subject = etpan_message_header_subject_normalize(subject);
            len = strlen(normalized_subject) + 6;
            reply_subject = malloc(len);
            snprintf(reply_subject, len, "Fwd: %s", normalized_subject);
            etpan_message_header_set_subject(header, reply_subject);
            free(reply_subject);
            free(normalized_subject);
          }
          else {
            etpan_message_header_set_subject(header, "Fwd: ");
          }
        }
      }
    }
  }
  
  for(i = 0 ; i < carray_count(part_list) ; i ++) {
    struct etpan_part_fetch_info * info;
    int type;
    
    info = carray_get(part_list, i);
    type = etpan_part_fetch_info_get_type(info);
    
    if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
      char * content;
      size_t content_length;
      struct etpan_part_header * part_header;
      char * content_type;
      struct etpan_part * part;
      
      part = etpan_part_fetch_info_get_part(info);
      part_header = etpan_part_get_header(part);
      if (part_header == NULL) {
        ETPAN_WARN_LOG("WARNING! part has no header");
        continue;
      }
      
      content_type = etpan_part_header_get_content_type(part_header);
      
      etpan_part_fetch_info_get_content(info, &content, &content_length);
      
      if (strcasecmp(content_type, "message/rfc822") == 0) {
        MMAPString * buffer;
        char * quoted;
        size_t quoted_length;
        
        buffer = mmap_string_new("");
        if (buffer == NULL) {
          ETPAN_LOG_MEMORY_ERROR;
        }
        
        show_header_part(buffer, content, content_length);
        
        etpan_quote_text(buffer->str, buffer->len,
            &quoted, &quoted_length);
        
        mmap_string_free(buffer);
        
        etpan_message_composer_add_text(composer,
            quoted, quoted_length);
        
        free(quoted);
      }
      else {
        char * quoted;
        size_t quoted_length;
        
        etpan_quote_text(content, content_length,
            &quoted, &quoted_length);
        
        etpan_message_composer_add_text(composer, quoted, quoted_length);
        
        free(quoted);
        
        etpan_message_composer_add_text(composer,
            "\n", 1);
      }
    }
    else if (type == ETPAN_PART_FETCH_INFO_TYPE_FILE) {
      char * filename;
      int temporary;
      
      filename = etpan_part_fetch_info_get_filename(info);
      temporary = etpan_part_fetch_info_is_temporary(info);
      etpan_part_fetch_info_set_temporary(info, 0);
      
      etpan_message_composer_add_file(composer, filename, temporary);
    }
    else if (type == ETPAN_PART_FETCH_INFO_NONE) {
      char * filename;
      struct etpan_part * part;
      struct etpan_part_header * part_header;
      
      part = etpan_part_fetch_info_get_part(info);
      filename = NULL;
      part_header = etpan_part_get_header(part);
      if (part_header != NULL) {
        filename = etpan_part_header_get_filename(part_header);
        if (filename != NULL) {
          MMAPString * buffer;
        
          buffer = mmap_string_new("");
          if (buffer == NULL) {
            ETPAN_LOG_MEMORY_ERROR;
          }
          
          if (mmap_string_append(buffer, _("\n> - attachment ")) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          if (mmap_string_append(buffer, filename) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          if (mmap_string_append(buffer, _("\n")) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          
          etpan_message_composer_add_text(composer,
              buffer->str, buffer->len);
          
          mmap_string_free(buffer);
        }
      }
      if (filename == NULL) {
        char * text;
        
        text = _("\n> - unnamed attachment\n");
        
        etpan_message_composer_add_text(composer, text, strlen(text));
      }
    }
  }
  
  etpan_message_composer_add_signature(composer);
  
  etpan_message_composer_set_cursor(composer, 0);
  
 free_fetcher:
  etpan_message_fetcher_unref(fetcher);
  composer->fetcher = NULL;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, composer,
      NULL);
  
  ETPAN_ERROR_FREE(composer->error);
  composer->error = NULL;
}


void etpan_message_composer_forward(struct etpan_message_composer * composer,
    struct etpan_message * msg, int forward_type)
{
  struct etpan_message_fetcher * fetcher;
  
  composer->forward_type = forward_type;
  
  composer->compose_type = ETPAN_COMPOSE_TYPE_FORWARD;
  set_reference_message(composer, msg);
  
  fetcher = etpan_message_fetcher_new();
  
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_ATTACHMENT);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, composer,
      forward_callback);
  composer->fetcher = fetcher;
  
  etpan_message_fetcher_run(fetcher);
}

void etpan_message_composer_forward_cancel(struct etpan_message_composer * composer)
{
  if (composer->fetcher != NULL)
    etpan_message_fetcher_cancel(composer->fetcher);
}

struct etpan_error *
etpan_message_composer_forward_get_error(struct etpan_message_composer * composer)
{
  return composer->error;
}

void etpan_message_composer_add_signature(struct etpan_message_composer * composer)
{
  FILE * f;
  char * home;
  char path[PATH_MAX];
  char buffer[1024];
  MMAPString * str;
  int r;
  char * converted;
  size_t converted_length;
  int is_utf8;
  
  home = getenv("HOME");
  if (home == NULL)
    goto err;
  
  snprintf(path, sizeof(path), "%s/.signature", home);

  str = mmap_string_new("-- \n");
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  f = fopen(path, "r");
  if (f == NULL)
    goto free_str;
  
  while (fgets(buffer, sizeof(buffer), f)) {
    if (mmap_string_append(str, buffer) == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = charconv_buffer("utf-8", "utf-8",
      str->str, str->len,
      &converted, &converted_length);
  if (r != MAIL_CHARCONV_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  is_utf8 = (strcmp(converted, str->str) == 0);
  charconv_buffer_free(converted);
  
  if (is_utf8) {
    etpan_message_composer_add_text(composer, str->str, str->len);
  }
  else {
    charconv_buffer("utf-8", "iso-8859-1",
        str->str, str->len,
        &converted, &converted_length);
    if (r != MAIL_CHARCONV_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_message_composer_add_text(composer, converted, converted_length);
    
    charconv_buffer_free(converted);
  }
  
  mmap_string_free(str);
  
  fclose(f);
  
  return;
  
 free_str:
  mmap_string_free(str);
 err:
  {}
}

void etpan_message_composer_set_account(struct etpan_message_composer * composer, struct etpan_account * account)
{
  free(composer->account_id);
  composer->account_id = strdup(etpan_account_get_id(account));
  if (composer->account_id == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

static void set_reference_message(struct etpan_message_composer * composer,
    struct etpan_message * msg)
{
  if (composer->reference_msg != NULL) {
    struct etpan_folder * folder;
    
    folder = etpan_message_get_folder(composer->reference_msg);
    etpan_message_unref(composer->reference_msg);
    etpan_folder_unref_msg_list(folder);
    composer->reference_msg = NULL;
  }
  
  if (msg != NULL) {
    struct etpan_folder * folder;
    
    folder = etpan_message_get_folder(msg);
    etpan_folder_ref_msg_list(folder);
    etpan_message_ref(msg);
  }
  composer->reference_msg = msg;
}

void etpan_message_composer_set_reference_message(struct etpan_message_composer * composer, struct etpan_message * msg, int compose_type)
{
  set_reference_message(composer, msg);
  composer->compose_type = compose_type;
}

void etpan_message_composer_mark_done(struct etpan_message_composer * composer)
{
  int value;
  struct etpan_message_flags * flags;
  
  if (composer->reference_msg == NULL)
    return;
  
  flags = etpan_message_get_flags(composer->reference_msg);
  flags = etpan_message_flags_dup(flags);
  value = etpan_message_flags_get_value(flags);
  
  switch (composer->compose_type) {
  case ETPAN_COMPOSE_TYPE_NEW:
    break;
  case ETPAN_COMPOSE_TYPE_REPLY:
    value |= ETPAN_FLAGS_ANSWERED;
    break;
  case ETPAN_COMPOSE_TYPE_FORWARD:
    value |= ETPAN_FLAGS_FORWARDED;
    break;
  }
  etpan_message_flags_set_value(flags, value);
  
  etpan_message_set_flags(composer->reference_msg, flags);
  etpan_message_flags_free(flags);
}

/* message send */

enum {
  OUTBOX_TYPE_NONE,
  OUTBOX_TYPE_SMTP,
  OUTBOX_TYPE_SENT_FOLDER,
  OUTBOX_TYPE_NEWS,
};

static int get_outbox_type(struct etpan_outbox * outbox)
{
  struct etpan_sender * sender;
  struct etpan_sender_driver * driver;
  
  sender = etpan_outbox_get_sender(outbox);
  if (sender == NULL)
    return OUTBOX_TYPE_NONE;
  
  driver = etpan_sender_get_driver(sender);
  if (strcasecmp(driver->name, "smtp") == 0)
    return OUTBOX_TYPE_SMTP;
  else if (strcasecmp(driver->name, "news") == 0)
    return OUTBOX_TYPE_NEWS;
  else if (strcasecmp(driver->name, "sent-folder") == 0)
    return OUTBOX_TYPE_SENT_FOLDER;
  else
    return OUTBOX_TYPE_NONE;
}

struct send_data {
  void (* callback)(struct etpan_error * error, void * cb_data);
  void * cb_data;
  int outbox_count;
  carray * error_list;
};

static void outbox_added(struct etpan_outbox * outbox, void * user_data);
static void outbox_processed(struct etpan_outbox * outbox, void * user_data);
static void message_sent(struct send_data * send_data);

void etpan_message_composer_send(struct etpan_message_composer * composer,
    void (* callback)(struct etpan_error * error, void * cb_data),
    void * cb_data)
{
  char * result_content;
  size_t result_length;
  carray * outbox_list;
  struct send_data * send_data;
  unsigned int i;
  carray * list;
  struct etpan_message_header * header;
  int has_recipient;
  int has_newsgroups;
  char * newsgroups;
  struct etpan_account * account;
  
  account = etpan_account_manager_get_account(etpan_account_manager_get_default(), composer->account_id);
  
  header = etpan_message_composer_get_header(composer);
  
  has_recipient = 0;
  list = etpan_message_header_get_to(header);
  if (carray_count(list) > 0)
    has_recipient = 1;
  list = etpan_message_header_get_cc(header);
  if (carray_count(list) > 0)
    has_recipient = 1;
  list = etpan_message_header_get_bcc(header);
  if (carray_count(list) > 0)
    has_recipient = 1;
  
  has_newsgroups = 0;
  newsgroups = etpan_message_header_get_newsgroups(header);
  if (newsgroups != NULL)
    has_newsgroups = 1;
  
  etpan_message_composer_render(composer, &result_content, &result_length);
  
  send_data = malloc(sizeof(* send_data));
  if (send_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  send_data->callback = callback;
  send_data->cb_data = cb_data;
  send_data->outbox_count = 0;
  send_data->error_list = carray_new(4);
  if (send_data->error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  outbox_list = etpan_account_get_outbox_list(account);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    int type;
    
    outbox = carray_get(outbox_list, i);
    type = get_outbox_type(outbox);
    
    switch (type) {
    case OUTBOX_TYPE_NEWS:
      if (has_newsgroups) {
        send_data->outbox_count ++;
        etpan_outbox_add(outbox, result_content, result_length,
            outbox_added, send_data);
      }
      break;
    case OUTBOX_TYPE_SMTP:
      if (has_recipient) {
        send_data->outbox_count ++;
        etpan_outbox_add(outbox, result_content, result_length,
            outbox_added, send_data);
      }
      break;
    case OUTBOX_TYPE_SENT_FOLDER:
      send_data->outbox_count ++;
      etpan_outbox_add(outbox, result_content, result_length,
          outbox_added, send_data);
      break;
    }
  }
  
  if (has_newsgroups || has_recipient) {
    etpan_message_composer_mark_done(composer);
  }
  
  free(result_content);
}

static void outbox_added(struct etpan_outbox * outbox, void * user_data)
{
  struct send_data * send_data;
  struct etpan_error * error;
  int r;
  
  send_data = user_data;
  
  error = etpan_outbox_added_get_error(outbox);
  if (error != NULL) {
    error = etpan_error_dup(error);
    r = carray_add(send_data->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    send_data->outbox_count --;
    message_sent(send_data);
    return;
  }
  
  etpan_outbox_process(outbox, outbox_processed, user_data);
}

static void outbox_processed(struct etpan_outbox * outbox, void * user_data)
{
  struct send_data * send_data;
  struct etpan_error * error;
  int r;
  
  send_data = user_data;
  
  error = etpan_outbox_processed_get_error(outbox);
  if (error != NULL) {
    error = etpan_error_dup(error);
    r = carray_add(send_data->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    send_data->outbox_count --;
    message_sent(send_data);
    return;
  }
  
  send_data->outbox_count --;
  message_sent(send_data);
}

static void message_sent(struct send_data * send_data)
{
  struct etpan_error * error;
  int r;
  unsigned int i;
  
  if (send_data->outbox_count != 0)
    return;
  
  if (carray_count(send_data->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(send_data->error_list) == 1) {
    error = carray_get(send_data->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(send_data->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(send_data->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  
  r = carray_set_size(send_data->error_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (error != NULL)
    etpan_error_log(error);
  
  send_data->callback(error, send_data->cb_data);
  
  ETPAN_ERROR_FREE(error);
  carray_free(send_data->error_list);
  free(send_data);
}
