#include "etpan-message-copy-local.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-folder.h"
#include "etpan-message.h"
#include "etpan-storage.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-folder-create.h"
#include "etpan-message-copy.h"
#include "etpan-signal.h"
#include "etpan-nls.h"

static struct etpan_error * rewrite_error(struct etpan_error * error, int delete);

struct etpan_message_copy_local * etpan_message_copy_local_new(void)
{
  struct etpan_message_copy_local * msg_copy_local;
  
  msg_copy_local = malloc(sizeof(* msg_copy_local));
  if (msg_copy_local == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_copy_local->folder_name = NULL;
  msg_copy_local->delete = 0;
  msg_copy_local->ref_count = 1;
  msg_copy_local->msg_copy_list = NULL;
  msg_copy_local->folder_create_list = NULL;
  msg_copy_local->msgtab = NULL;
  msg_copy_local->storage_hash = NULL;
  msg_copy_local->done_count = 0;
  msg_copy_local->error_list = NULL;
  msg_copy_local->error = NULL;
  msg_copy_local->pending_for_deletion = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_copy_local->pending_for_deletion == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return msg_copy_local;
}

static void etpan_message_copy_local_free(struct etpan_message_copy_local * msg_copy_local)
{
  chash_free(msg_copy_local->pending_for_deletion);
  
  ETPAN_ERROR_FREE(msg_copy_local->error);
  msg_copy_local->error = NULL;
  
  if (msg_copy_local->msgtab != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(msg_copy_local->msgtab) ; i ++) {
      struct etpan_message * msg;
      struct etpan_folder * folder;
      
      msg = carray_get(msg_copy_local->msgtab, i);
      folder = etpan_message_get_folder(msg);
      etpan_folder_unref_msg_list(folder);
      etpan_message_unref(msg);
    }
    carray_free(msg_copy_local->msgtab);
  }
  free(msg_copy_local->folder_name);
  
  free(msg_copy_local);
}

void etpan_message_copy_local_set_delete(struct etpan_message_copy_local * msg_copy_local, int delete)
{
  msg_copy_local->delete = delete;
}

void etpan_message_copy_local_set_folder_name(struct etpan_message_copy_local * msg_copy_local, char * folder_name)
{
  if (folder_name != msg_copy_local->folder_name) {
    free(msg_copy_local->folder_name);
    if (folder_name != NULL) {
      msg_copy_local->folder_name = strdup(folder_name);
      if (msg_copy_local->folder_name == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
    else
      msg_copy_local->folder_name = NULL;
  }
}

void etpan_message_copy_local_ref(struct etpan_message_copy_local * msg_copy_local)
{
  msg_copy_local->ref_count ++;
}

void etpan_message_copy_local_unref(struct etpan_message_copy_local * msg_copy_local)
{
  msg_copy_local->ref_count --;
  if (msg_copy_local->ref_count <= 0) {
    etpan_message_copy_local_free(msg_copy_local);
  }
}

void etpan_message_copy_local_set_msglist(struct etpan_message_copy_local * msg_copy_local,
    chash * msglist)
{
  int r;
  chashiter * iter;
  
  msg_copy_local->msgtab = carray_new(chash_count(msglist));
  if (msg_copy_local->msgtab == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(msglist) ; iter != NULL ;
      iter = chash_next(msglist, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    chashdatum key;
    struct etpan_folder * folder;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    r = carray_add(msg_copy_local->msgtab, msg, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    folder = etpan_message_get_folder(msg);
    etpan_message_ref(msg);
    etpan_folder_ref_msg_list(folder);
  }
}

void etpan_message_copy_local_setup(struct etpan_message_copy_local * msg_copy_local)
{
}

static void copy_storage_messages(struct etpan_message_copy_local * msg_copy_local,
    struct etpan_storage * msg_storage,
    chash * msg_hash);

static void copy_messages(struct etpan_message_copy_local * msg_copy_local)
{
  chash * storage_hash;
  chashiter * iter;
  int r;
  unsigned int i;
  
  storage_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (storage_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(msg_copy_local->msgtab) ; i ++) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    chashdatum storage_key;
    chashdatum storage_value;
    chash * storage_msg_hash;
    struct etpan_folder * folder;
    struct etpan_storage * storage;
    char * identifier;
    
    msg = carray_get(msg_copy_local->msgtab, i);
    folder = etpan_message_get_folder(msg);
    storage = etpan_folder_get_storage(folder);
    storage_key.data = &storage;
    storage_key.len = sizeof(storage);
    r = chash_get(storage_hash, &storage_key, &storage_value);
    if (r < 0) {
      storage_msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
      if (storage_msg_hash == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      storage_value.data = storage_msg_hash;
      storage_value.len = 0;
      r = chash_set(storage_hash, &storage_key, &storage_value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      storage_msg_hash = storage_value.data;
    }
    
    identifier = etpan_message_get_uid(msg);
    key.data = identifier;
    key.len = strlen(identifier) + 1;
    value.data = msg;
    value.len = 0;
    r = chash_set(storage_msg_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  msg_copy_local->msg_copy_list = carray_new(4);
  if (msg_copy_local->msg_copy_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  msg_copy_local->folder_create_list = carray_new(4);
  if (msg_copy_local->folder_create_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_copy_local->storage_hash = storage_hash;
  for(iter = chash_begin(storage_hash) ; iter != NULL ;
      iter = chash_next(storage_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_storage * storage;
    chash * storage_msg_hash;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    memcpy(&storage, key.data, sizeof(storage));
    storage_msg_hash = value.data;
    copy_storage_messages(msg_copy_local, storage, storage_msg_hash);
  }
}

static void deletion_created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

struct copy_data {
  struct etpan_message_copy_local * msg_copy_local;
  chash * msg_hash;
};

static void move_message(struct etpan_message_copy_local * msg_copy_local,
    struct etpan_folder * folder, chash * msg_hash);

static void copy_storage_messages(struct etpan_message_copy_local * msg_copy_local,
    struct etpan_storage * msg_storage,
    chash * msg_hash)
{
  char * copy_message_folder_name;
  char * folder_location;
  struct etpan_folder * folder;
  int r;
  
  copy_message_folder_name = msg_copy_local->folder_name;
  
  /* ensure the folder is created */
  folder_location = etpan_storage_get_sub_folder_location(msg_storage,
      NULL, copy_message_folder_name);
  
  folder = etpan_storage_get_folder(msg_storage, folder_location);
  free(folder_location);
  
  if (folder == NULL) {
    struct etpan_folder_create * folder_create;
    struct copy_data * data;
    
    data = malloc(sizeof(* data));
    if (data == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    data->msg_copy_local = msg_copy_local;
    data->msg_hash = msg_hash;
    
    folder_create = etpan_folder_create_new();
    etpan_folder_create_set_folder_name(folder_create,
        copy_message_folder_name);
    etpan_folder_create_set_parent_folder(folder_create, NULL);
    etpan_folder_create_set_storage(folder_create, msg_storage);
    etpan_folder_create_setup(folder_create);
    
    ETPAN_SIGNAL_ADD_HANDLER(folder_create,
        ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, deletion_created_handler, data);
    
    r = carray_add(msg_copy_local->folder_create_list, folder_create, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_folder_create_run(folder_create);
  }
  else {
    move_message(msg_copy_local, folder, msg_hash);
  }
}

static void storage_done(struct etpan_message_copy_local * msg_copy_local);

static void deletion_created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_create * folder_create;
  struct etpan_folder * folder;
  struct copy_data * data;
  struct etpan_message_copy_local * msg_copy_local;
  chash * msg_hash;
  struct etpan_error * error;
  unsigned int i;
  int r;
  (void) signal_name;
  (void) signal_data;
  
  folder_create = sender;
  data = user_data;
  msg_copy_local = data->msg_copy_local;
  msg_hash = data->msg_hash;
  free(data);
  
  ETPAN_SIGNAL_REMOVE_HANDLER(folder_create,
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, deletion_created_handler, data);
  
  for(i = 0 ; i < carray_count(msg_copy_local->folder_create_list) ; i ++) {
    struct etpan_folder_create * cur_folder_create;
    
    cur_folder_create = carray_get(msg_copy_local->folder_create_list, i);
    if (cur_folder_create == folder_create) {
      carray_set(msg_copy_local->folder_create_list, i, NULL);
      break;
    }
  }
  
  error = etpan_folder_create_get_error(folder_create);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(error, msg_copy_local->delete);
    
    r = carray_add(msg_copy_local->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_folder_create_unref(folder_create); 
    
    storage_done(msg_copy_local);
    
    return;
  }
  
  folder = etpan_folder_create_get_created_folder(folder_create);
  
  move_message(msg_copy_local, folder, msg_hash);
  
  etpan_folder_create_unref(folder_create); 
}

static void move_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void move_message(struct etpan_message_copy_local * msg_copy_local,
    struct etpan_folder * folder, chash * msg_hash)
{
  struct etpan_message_copy * msg_copy;
  int r;
  
  msg_copy = etpan_message_copy_new();
  etpan_message_copy_set_delete(msg_copy, msg_copy_local->delete);
  etpan_message_copy_set_msglist(msg_copy, msg_hash);
  etpan_message_copy_set_destination(msg_copy, folder);
  etpan_message_copy_setup(msg_copy);
  
  r = carray_add(msg_copy_local->msg_copy_list, msg_copy, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_SIGNAL_ADD_HANDLER(msg_copy, ETPAN_MESSAGE_COPY_FINISHED_SIGNAL,
      move_finished_handler, msg_copy_local);
  
  etpan_message_copy_run(msg_copy);
}

static void move_done(struct etpan_message_copy_local * msg_copy_local);

static void move_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_copy_local * msg_copy_local;
  struct etpan_message_copy * msg_copy;
  struct etpan_error * error;
  unsigned int i;
  int r;
  
  msg_copy = sender;
  msg_copy_local = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(msg_copy, ETPAN_MESSAGE_COPY_FINISHED_SIGNAL,
      move_finished_handler, msg_copy_local);
  
  for(i = 0 ; i < carray_count(msg_copy_local->msg_copy_list) ; i ++) {
    struct etpan_message_copy * cur_msg_copy;
    
    cur_msg_copy = carray_get(msg_copy_local->msg_copy_list, i);
    if (cur_msg_copy == msg_copy) {
      carray_set(msg_copy_local->msg_copy_list, i, NULL);
      break;
    }
  }
  
  error = etpan_message_copy_get_error(msg_copy);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(error, msg_copy_local->delete);
    
    r = carray_add(msg_copy_local->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_message_copy_unref(msg_copy);
    
    storage_done(msg_copy_local);
    
    return;
  }
  
  etpan_message_copy_unref(msg_copy);
  
  move_done(msg_copy_local);
}

static void move_done(struct etpan_message_copy_local * msg_copy_local)
{
  storage_done(msg_copy_local);
}

static void copy_done(struct etpan_message_copy_local * msg_copy_local);

static void storage_done(struct etpan_message_copy_local * msg_copy_local)
{
  msg_copy_local->done_count ++;
  
  if (msg_copy_local->done_count < chash_count(msg_copy_local->storage_hash)) {
    return;
  }
  
  copy_done(msg_copy_local);
}

static void copy_done(struct etpan_message_copy_local * msg_copy_local)
{
  chashiter * iter;
  struct etpan_error * error;
  unsigned int i;
  
  ETPAN_LOG("copy done : %i", msg_copy_local->done_count);
  if (carray_count(msg_copy_local->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(msg_copy_local->error_list) == 1) {
    error = carray_get(msg_copy_local->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(msg_copy_local->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(msg_copy_local->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(msg_copy_local->error_list);
  msg_copy_local->error_list = NULL;
  msg_copy_local->error = error;
  
  for(iter = chash_begin(msg_copy_local->storage_hash) ; iter != NULL ;
      iter = chash_next(msg_copy_local->storage_hash, iter)) {
    chashdatum value;
    chash * storage_msg_hash;
    
    chash_value(iter, &value);
    storage_msg_hash = value.data;
    chash_free(storage_msg_hash);
  }
  
  chash_free(msg_copy_local->storage_hash);
  msg_copy_local->storage_hash = NULL;
  
  carray_free(msg_copy_local->folder_create_list);
  msg_copy_local->folder_create_list = NULL;
  carray_free(msg_copy_local->msg_copy_list);
  msg_copy_local->msg_copy_list = NULL;
  
  ETPAN_SIGNAL_SEND(msg_copy_local, ETPAN_MESSAGE_COPY_LOCAL_FINISHED_SIGNAL);
}

void etpan_message_copy_local_run(struct etpan_message_copy_local * msg_copy_local)
{
  int r;
  unsigned int i;
  
  if (msg_copy_local->delete) {
    for(i = 0 ; i < carray_count(msg_copy_local->msgtab) ; i ++) {
      char * uid;
      struct etpan_message * msg;
      chashdatum key;
      chashdatum value;
    
      msg = carray_get(msg_copy_local->msgtab, i);
      uid = etpan_message_get_uid(msg);
    
      key.data = uid;
      key.len = strlen(uid) + 1;
      value.data = msg;
      value.len = 0;
      r = chash_set(msg_copy_local->pending_for_deletion, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  msg_copy_local->error_list = carray_new(4);
  if (msg_copy_local->error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  copy_messages(msg_copy_local);
}

void etpan_message_copy_local_cancel(struct etpan_message_copy_local * msg_copy_local)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(msg_copy_local->folder_create_list) ; i ++) {
    struct etpan_folder_create * folder_create;
    
    folder_create = carray_get(msg_copy_local->folder_create_list, i);
    if (folder_create != NULL) {
      etpan_folder_create_cancel(folder_create);
    }
  }
  
  for(i = 0 ; i < carray_count(msg_copy_local->msg_copy_list) ; i ++) {
    struct etpan_message_copy * msg_copy;
    
    msg_copy = carray_get(msg_copy_local->msg_copy_list, i);
    if (msg_copy != NULL) {
      etpan_message_copy_cancel(msg_copy);
    }
  }
}

struct etpan_error *
etpan_message_copy_local_get_error(struct etpan_message_copy_local * msg_copy_local)
{
  return msg_copy_local->error;
}

static struct etpan_error * rewrite_error(struct etpan_error * error, int delete)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  if (delete) {
    etpan_error_strf_long_description(new_error,
        _("An error occurred while moving messages\n"
            "%s"),
        previous_long_description);
  }
  else {
    etpan_error_strf_long_description(new_error,
        _("An error occurred while copying messages\n"
            "%s"),
        previous_long_description);
  }
  
  return new_error;
}

chash * etpan_message_copy_local_get_pending_for_deletion(struct etpan_message_copy_local * msg_copy_local)
{
  return msg_copy_local->pending_for_deletion;
}
