#include "etpan-message-copy.h"

#include <stdlib.h>
#include <libetpan/libetpan.h>
#include <string.h>

#include "etpan-message-fetcher.h"
#include "etpan-error.h"
#include "etpan-message.h"
#include "etpan-folder.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-log.h"
#include "etpan-nls.h"
#include "etpan-mail-manager.h"

enum {
  STATE_IDLE,
  STATE_FETCHING,
  STATE_APPENDING,
};

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    struct etpan_error * error);

static void etpan_message_copy_unsetup(struct etpan_message_copy * msg_copy);
static void etpan_message_copy_free(struct etpan_message_copy * msg_copy);

static void copy_next_message(struct etpan_message_copy * msg_copy);
static void copy_next_message_fetched_callback(char * signal_name,
    void * sender,
    void * signal_data, void * user_data);
static void copy_next_message_appended_callback(int cancelled,
    struct etpan_folder_append_msg_result * result, void * cb_data);
static void copy_finish(struct etpan_message_copy * msg_copy);

struct etpan_message_copy * etpan_message_copy_new(void)
{
  struct etpan_message_copy * msg_copy;
  
  msg_copy = malloc(sizeof(* msg_copy));
  if (msg_copy == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_copy->folder = NULL;
  msg_copy->msgid_list = NULL;
  msg_copy->msgtab = NULL;
  msg_copy->next_msg = 0;
  msg_copy->fetcher = etpan_message_fetcher_new();
  msg_copy->delete = 0;
  msg_copy->state = STATE_IDLE;
  msg_copy->op = NULL;
  msg_copy->error = NULL;
  msg_copy->error_list = NULL;
  msg_copy->ref_count = 1;
  msg_copy->msg_copied = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_copy->msg_copied == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_copy->msg_deleted = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_copy->msg_deleted == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_copy->pending_for_copy = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_copy->pending_for_copy == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_copy->pending_for_deletion = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_copy->pending_for_deletion == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return msg_copy;
}

static void etpan_message_copy_free(struct etpan_message_copy * msg_copy)
{
  unsigned int i;
  
  etpan_message_copy_unsetup(msg_copy);
  
  ETPAN_ERROR_FREE(msg_copy->error);
  msg_copy->error = NULL;
  
  for(i = 0 ; i < carray_count(msg_copy->msgtab) ; i ++) {
    struct etpan_message * msg;
    struct etpan_folder * folder;
    
    msg = carray_get(msg_copy->msgtab, i);
    folder = etpan_message_get_folder(msg);
    etpan_folder_unref_msg_list(folder);
    etpan_message_unref(msg);
  }
  carray_free(msg_copy->msgtab);
  
  for(i = 0 ; i < carray_count(msg_copy->msgid_list) ; i ++) {
    char * uid;
    
    uid = carray_get(msg_copy->msgid_list, i);
    free(uid);
  }
  carray_free(msg_copy->msgid_list);
  
  chash_free(msg_copy->pending_for_deletion);
  chash_free(msg_copy->pending_for_copy);
  chash_free(msg_copy->msg_deleted);
  chash_free(msg_copy->msg_copied);
  if (msg_copy->folder != NULL) {
    etpan_folder_unref(msg_copy->folder);
  }
  etpan_message_fetcher_unref(msg_copy->fetcher);
  free(msg_copy);
}

void etpan_message_copy_ref(struct etpan_message_copy * msg_copy)
{
  msg_copy->ref_count ++;
}

void etpan_message_copy_unref(struct etpan_message_copy * msg_copy)
{
  msg_copy->ref_count --;
  if (msg_copy->ref_count == 0)
    etpan_message_copy_free(msg_copy);
}

void etpan_message_copy_set_delete(struct etpan_message_copy * msg_copy,
    int delete)
{
  msg_copy->delete = delete;
}

int etpan_message_copy_get_delete(struct etpan_message_copy * msg_copy)
{
  return msg_copy->delete;
}

void etpan_message_copy_set_msglist(struct etpan_message_copy * msg_copy,
    chash * msglist)
{
  int r;
  chashiter * iter;
    
  msg_copy->msgid_list = carray_new(chash_count(msglist));
  if (msg_copy->msgid_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
    
  msg_copy->msgtab = carray_new(chash_count(msglist));
  if (msg_copy->msgtab == NULL)
    ETPAN_LOG_MEMORY_ERROR;
    
  for(iter = chash_begin(msglist) ; iter != NULL ;
      iter = chash_next(msglist, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    chashdatum key;
    char * uid;
    struct etpan_folder * folder;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    r = carray_add(msg_copy->msgtab, msg, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    folder = etpan_message_get_folder(msg);
    etpan_message_ref(msg);
    etpan_folder_ref_msg_list(folder);
    
    uid = key.data;
    uid = strdup(uid);
    if (uid == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = carray_add(msg_copy->msgid_list, uid, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

void etpan_message_copy_set_destination(struct etpan_message_copy * msg_copy,
    struct etpan_folder * folder)
{
  if (folder != msg_copy->folder) {
    if (msg_copy->folder != NULL)
      etpan_folder_unref(msg_copy->folder);
    msg_copy->folder = folder;
    if (folder != NULL) {
      etpan_folder_ref(folder);
    }
    else
      msg_copy->folder = NULL;
  }
}

struct etpan_folder *
etpan_message_copy_get_destination(struct etpan_message_copy * msg_copy)
{
  return msg_copy->folder;
}

void etpan_message_copy_setup(struct etpan_message_copy * msg_copy)
{
  (void) msg_copy;
}

static void etpan_message_copy_unsetup(struct etpan_message_copy * msg_copy)
{
  etpan_message_copy_cancel(msg_copy);
}

void etpan_message_copy_run(struct etpan_message_copy * msg_copy)
{
  int r;
  unsigned int i;
  
  for(i = 0 ; i < carray_count(msg_copy->msgtab) ; i ++) {
    char * uid;
    struct etpan_message * msg;
    chashdatum key;
    chashdatum value;
    
    msg = carray_get(msg_copy->msgtab, i);
    uid = carray_get(msg_copy->msgid_list, i);
    
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = msg;
    value.len = 0;
    r = chash_set(msg_copy->pending_for_copy, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (msg_copy->delete) {
      r = chash_set(msg_copy->pending_for_deletion, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  msg_copy->error_list = carray_new(4);
  if (msg_copy->error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  copy_next_message(msg_copy);
}

void etpan_message_copy_cancel(struct etpan_message_copy * msg_copy)
{
  switch (msg_copy->state) {
  case STATE_FETCHING:
    etpan_message_fetcher_cancel(msg_copy->fetcher);
    break;
  case STATE_APPENDING:
    etpan_thread_op_cancel(msg_copy->op);
    break;
  }
}

static void copy_next_message(struct etpan_message_copy * msg_copy)
{
  struct etpan_message * msg;
  
  if (msg_copy->msgtab == NULL) {
    copy_finish(msg_copy);
    return;
  }
  
  if (msg_copy->next_msg >= carray_count(msg_copy->msgtab)) {
    copy_finish(msg_copy);
    return;
  }
  
  /* fetch message */
  
  msg = carray_get(msg_copy->msgtab, msg_copy->next_msg);
  etpan_message_fetcher_set_flags(msg_copy->fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE);
  etpan_message_fetcher_set_message(msg_copy->fetcher, msg);
  etpan_message_fetcher_setup(msg_copy->fetcher);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, msg_copy->fetcher, msg_copy,
      copy_next_message_fetched_callback);
  
  msg_copy->state = STATE_FETCHING;
  etpan_message_fetcher_run(msg_copy->fetcher);
}

static void copy_next_message_fetched_callback(char * signal_name,
    void * sender,
    void * signal_data, void * user_data)
{
  int r;
  struct etpan_message_copy * msg_copy;
  carray * list;
  struct etpan_part_fetch_info * info;
  char * content;
  size_t content_length;
  struct etpan_message * msg;
  struct etpan_message_flags * flags;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  msg_copy = user_data;

  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, msg_copy->fetcher, msg_copy,
      copy_next_message_fetched_callback);
  
  error = etpan_message_fetcher_get_error(msg_copy->fetcher);
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      struct etpan_error * new_error;
    
      new_error = rewrite_error(msg_copy->folder, error);
      error = new_error;
    
      r = carray_add(msg_copy->error_list, new_error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      copy_finish(msg_copy);
      return;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(msg_copy->folder, error);
    error = new_error;
    
    r = carray_add(msg_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    msg_copy->next_msg ++;
    copy_next_message(msg_copy);
    return;
  }
  
  list = etpan_message_fetcher_get_part_list(msg_copy->fetcher);
  if (carray_count(list) == 0) {
    ETPAN_LOG("can't fetch message");
    etpan_crash();
  }
  info = carray_get(list, 0);
  etpan_part_fetch_info_get_content(info, &content, &content_length);
  
  msg = carray_get(msg_copy->msgtab, msg_copy->next_msg);
  flags = etpan_message_get_flags(msg);
  
  msg_copy->state = STATE_APPENDING;
  msg_copy->op = etpan_folder_append(etpan_thread_manager_app_get_default(),
      msg_copy->folder, content, content_length,
      flags, copy_next_message_appended_callback,
      msg_copy);
  
  /* append message */
}

static void copy_next_message_appended_callback(int cancelled,
    struct etpan_folder_append_msg_result * result, void * cb_data)
{
  struct etpan_message_copy * msg_copy;
  struct etpan_error * error;
  int r;
  chashdatum key;
  chashdatum value;
  char * uid;
  struct etpan_message * msg;
  
  msg_copy = cb_data;
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Copy of messages cancelled"));
    etpan_error_strf_long_description(error,
        _("The copy of messages to folder %s has been interrupted\n"),
        etpan_folder_get_ui_path(msg_copy->folder));
    
    r = carray_add(msg_copy->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    copy_finish(msg_copy);
    return;
  }

  if (result->error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(msg_copy->folder, result->error);
    error = new_error;
    
    r = carray_add(msg_copy->error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    msg_copy->next_msg ++;
    copy_next_message(msg_copy);
    return;
  }

  uid = carray_get(msg_copy->msgid_list, msg_copy->next_msg);
  msg = carray_get(msg_copy->msgtab, msg_copy->next_msg);
  
  key.data = uid;
  key.len = strlen(uid) + 1;
  value.data = msg;
  value.len = 0;
  r = chash_set(msg_copy->msg_copied, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  chash_delete(msg_copy->pending_for_copy, &key, NULL);
  
  if (msg_copy->delete) {
    struct etpan_message_flags * flags;
    int flags_value;
    
    flags = etpan_message_get_flags(msg);
    flags = etpan_message_flags_dup(flags);
    
    flags_value = etpan_message_flags_get_value(flags);
    flags_value |= (ETPAN_FLAGS_DELETED | ETPAN_FLAGS_SEEN);
    etpan_message_flags_set_value(flags, flags_value);
    etpan_message_set_flags(msg, flags);
    
    etpan_message_flags_free(flags);
    
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = msg;
    value.len = 0;
    r = chash_set(msg_copy->msg_deleted, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    chash_delete(msg_copy->pending_for_deletion, &key, NULL);
  }
  
  msg_copy->next_msg ++;
  copy_next_message(msg_copy);
}

static void copy_finish(struct etpan_message_copy * msg_copy)
{
  struct etpan_error * error;
  unsigned int i;
  int r;
  
  if (msg_copy->delete) {
    chash * folder_hash;
    chashiter * iter;
    
    folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
    if (folder_hash == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    /* checkpoint flags */
    for(i = 0 ; i < carray_count(msg_copy->msgtab) ; i ++) {
      struct etpan_message * msg;
      struct etpan_folder * folder;
      chashdatum key;
      chashdatum value;
      
      msg = carray_get(msg_copy->msgtab, i);
      folder = etpan_message_get_folder(msg);
      key.data = &folder;
      key.len = sizeof(folder);
      value.data = NULL;
      value.len = 0;
      r = chash_set(folder_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    
    for(iter = chash_begin(folder_hash) ; iter != NULL ;
        iter = chash_next(folder_hash, iter)) {
      chashdatum key;
      struct etpan_folder * folder;
      
      chash_key(iter, &key);
      memcpy(&folder, key.data, sizeof(folder));
      
      etpan_mail_manager_folder_check(etpan_mail_manager_get_default(),
          folder);
    }
    
    chash_free(folder_hash);
  }
  
  etpan_mail_manager_folder_update(etpan_mail_manager_get_default(),
      msg_copy->folder);
  
  if (carray_count(msg_copy->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(msg_copy->error_list) == 1) {
    error = carray_get(msg_copy->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(msg_copy->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(msg_copy->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(msg_copy->error_list);
  msg_copy->error_list = NULL;
  msg_copy->error = error;
  
  msg_copy->state = STATE_IDLE;
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_COPY_FINISHED_SIGNAL, msg_copy, NULL);
}

struct etpan_error *
etpan_message_copy_get_error(struct etpan_message_copy * msg_copy)
{
  return msg_copy->error;
}

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred while copying messages to folder %s\n"
          "%s"),
      etpan_folder_get_ui_path(folder),
      previous_long_description);
  
  return new_error;
}

chash * etpan_message_copy_get_msg_copied(struct etpan_message_copy * msg_copy)
{
  return msg_copy->msg_copied;
}

chash * etpan_message_copy_get_msg_delete(struct etpan_message_copy * msg_copy)
{
  return msg_copy->msg_deleted;
}

chash * etpan_message_copy_get_pending_for_deletion(struct etpan_message_copy * msg_copy)
{
  return msg_copy->pending_for_deletion;
}

chash * etpan_message_copy_get_pending_for_copy(struct etpan_message_copy * msg_copy)
{
  return msg_copy->pending_for_copy;
}
