#include "etpan-message-delete.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-signal.h"
#include "etpan-nls.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-message-copy-local.h"

static struct etpan_error * rewrite_error(struct etpan_error * error);

struct etpan_message_delete * etpan_message_delete_new(void)
{
  struct etpan_message_delete * msg_delete;
  
  msg_delete = malloc(sizeof(* msg_delete));
  if (msg_delete == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_delete->error = NULL;
  msg_delete->ref_count = 1;
  msg_delete->msg_copy_local = etpan_message_copy_local_new();
  etpan_message_copy_local_set_delete(msg_delete->msg_copy_local, 1);
  etpan_message_copy_local_set_folder_name(msg_delete->msg_copy_local,
      "Deleted Messages");
  
  return msg_delete;
}

static void etpan_message_delete_free(struct etpan_message_delete * msg_delete)
{
  etpan_message_copy_local_unref(msg_delete->msg_copy_local);
  ETPAN_ERROR_FREE(msg_delete->error);
  free(msg_delete);
}

void etpan_message_delete_ref(struct etpan_message_delete * msg_delete)
{
  msg_delete->ref_count ++;
}

void etpan_message_delete_unref(struct etpan_message_delete * msg_delete)
{
  msg_delete->ref_count --;
  if (msg_delete->ref_count <= 0) {
    etpan_message_delete_free(msg_delete);
  }
}

void etpan_message_delete_set_msglist(struct etpan_message_delete * msg_delete,
    chash * msglist)
{
  etpan_message_copy_local_set_msglist(msg_delete->msg_copy_local, msglist);
}

void etpan_message_delete_setup(struct etpan_message_delete * msg_delete)
{
  etpan_message_copy_local_setup(msg_delete->msg_copy_local);
}

static void copy_local_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_message_delete_run(struct etpan_message_delete * msg_delete)
{
  ETPAN_SIGNAL_ADD_HANDLER(msg_delete->msg_copy_local,
      ETPAN_MESSAGE_COPY_LOCAL_FINISHED_SIGNAL,
      copy_local_finished_handler, msg_delete);
  
  etpan_message_copy_local_run(msg_delete->msg_copy_local);
}

static void copy_local_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_delete * msg_delete;
  struct etpan_error * error;
  
  msg_delete = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(msg_delete->msg_copy_local,
      ETPAN_MESSAGE_COPY_LOCAL_FINISHED_SIGNAL,
      copy_local_finished_handler, msg_delete);
  
  ETPAN_LOG("error: %p", msg_delete->error);
  error = etpan_message_copy_local_get_error(msg_delete->msg_copy_local);
  if (error != NULL) {
    msg_delete->error = rewrite_error(error);
  }
  ETPAN_LOG("error: %p", msg_delete->error);
  
  etpan_message_delete_ref(msg_delete);
  
  ETPAN_SIGNAL_SEND(msg_delete, ETPAN_MESSAGE_DELETE_FINISHED_SIGNAL);
  
  ETPAN_LOG("error: %p", msg_delete->error);
  ETPAN_ERROR_FREE(msg_delete->error);
  msg_delete->error = NULL;
  etpan_message_delete_unref(msg_delete);
}

void etpan_message_delete_cancel(struct etpan_message_delete * msg_delete)
{
  etpan_message_copy_local_cancel(msg_delete->msg_copy_local);
}

struct etpan_error *
etpan_message_delete_get_error(struct etpan_message_delete * msg_delete)
{
  return msg_delete->error;
}

static struct etpan_error * rewrite_error(struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred while deleting messages\n"
          "%s"),
      previous_long_description);
  
  return new_error;
}

chash * etpan_message_delete_get_pending_for_deletion(struct etpan_message_delete * msg_delete)
{
  return etpan_message_copy_local_get_pending_for_deletion(msg_delete->msg_copy_local);
}
