#include "etpan-message-fetcher.h"

#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <ctype.h>

#include "etpan-error.h"
#include "etpan-part.h"
#include "etpan-message-header.h"
#include "etpan-part-header.h"
#include "etpan-message.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-utils.h"
#include "etpan-nls.h"
#include "etpan-log.h"

enum {
  ERROR_DOMAIN_FETCH,
};

static struct etpan_error * rewrite_error(struct etpan_message * msg,
    int domain,
    struct etpan_error * error);

static void etpan_message_fetcher_free(struct etpan_message_fetcher * fetcher);
static void etpan_message_fetcher_unsetup(struct etpan_message_fetcher * fetcher);

#define DEFAULT_INCOMING_CHARSET "iso-8859-1"

static struct etpan_part_fetch_info * fetch_info_new(void)
{
  struct etpan_part_fetch_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  info->part = NULL;
  info->type = ETPAN_PART_FETCH_INFO_NONE;
  info->content = NULL;
  info->content_length = 0;
  info->filename = NULL;
  info->temporary = 0;
  
  return info;
}

static void fetch_info_free(struct etpan_part_fetch_info * info)
{
  if (info->temporary)
    unlink(info->filename);
  free(info->filename);
  free(info->content);
  free(info);
}

static struct etpan_part *
fetch_info_get_part(struct etpan_part_fetch_info * info)
{
  return info->part;
}

static void fetch_info_set_part(struct etpan_part_fetch_info * info,
    struct etpan_part * part)
{
  info->part = part;
}

static void fetch_info_set_content(struct etpan_part_fetch_info * info,
    char * content, size_t content_length)
{
  info->type = ETPAN_PART_FETCH_INFO_TYPE_TEXT;
  info->content = content;
  info->content_length = content_length;
}

static void fetch_info_set_file(struct etpan_part_fetch_info * info,
    char * filename, int temporary)
{
  info->filename = strdup(filename);
  if (info->filename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  info->type = ETPAN_PART_FETCH_INFO_TYPE_FILE;
  info->temporary = temporary;
}

int etpan_part_fetch_info_get_type(struct etpan_part_fetch_info * info)
{
  return info->type;
}

void etpan_part_fetch_info_get_content(struct etpan_part_fetch_info * info,
    char ** p_content, size_t * p_content_length)
{
  * p_content = info->content;
  * p_content_length = info->content_length;
}

char * etpan_part_fetch_info_get_filename(struct etpan_part_fetch_info * info)
{
  return info->filename;
}

int etpan_part_fetch_info_is_temporary(struct etpan_part_fetch_info * info)
{
  return info->temporary;
}

void etpan_part_fetch_info_set_temporary(struct etpan_part_fetch_info * info,
    int temporary)
{
  info->temporary = temporary;
}

#define SCORE_MAX 20

static int get_score(struct etpan_part * part)
{
  char * content_type;
  struct etpan_part_header * header;
  
  header = etpan_part_get_header(part);
  if (header == NULL)
    return 0;
  
  content_type = etpan_part_header_get_content_type(header);
  
  if (strcasecmp(content_type, "multipart/x-decrypted") == 0)
    return SCORE_MAX;
  
  if (strcasecmp(content_type, "multipart/x-verified") == 0)
    return SCORE_MAX;

  if (strcasecmp(content_type, "multipart/related") == 0)
    return 6;
  
  if (strcasecmp(content_type, "multipart/mixed") == 0)
    return 6;
  
  if (strcasecmp(content_type, "text/html") == 0)
    return 5;
  
  if (strcasecmp(content_type, "text/enriched") == 0)
    return 4;
  
  if (strcasecmp(content_type, "text/plain") == 0)
    return 3;
  
  if (strncasecmp(content_type, "image/", 6) == 0)
    return 2;
  
  return 1;
}

static void collect_part(struct etpan_message_fetcher * fetcher,
    struct etpan_part * part)
{
  int r;
  carray * children;
  unsigned int i;
  char * content_type;
  struct etpan_part_header * header;
  int do_collect;
  
  do_collect = 0;
  
  header = etpan_part_get_header(part);
  if (header == NULL)
    return;
  
  content_type = etpan_part_header_get_content_type(header);
  
  if (strcasecmp(content_type, "message/rfc822") == 0)
    do_collect = 1;
  
  children = etpan_part_get_children(part);
  if (carray_count(children) == 0)
    do_collect = 1;
  
  if (do_collect) {
    struct etpan_part_fetch_info * info;
    
    info = fetch_info_new();
    if (info == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    etpan_part_ref(part);
    fetch_info_set_part(info, part);
    
    r = carray_add(fetcher->part_list, info, NULL);
    if (r < 0) {
      fetch_info_free(info);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  if ((fetcher->flags & ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE) != 0) {
    return;
  }
  
  if (strcasecmp(content_type, "multipart/alternative") == 0) {
    int prefered_score;
    struct etpan_part * prefered_part;
    
    prefered_part = NULL;
    prefered_score = 0;
    
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_part * child;
      int score;
      
      child = carray_get(children, i);
      score = get_score(child);
      if (score > prefered_score) {
        prefered_score = score;
        prefered_part = child;
      }
    }
    
    if (prefered_part != NULL) {
      collect_part(fetcher, prefered_part);
    }
  }
  else {
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_part * child;
      
      child = carray_get(children, i);
      
      collect_part(fetcher, child);
    }
  }
}

static void collect_part_list(struct etpan_message_fetcher * fetcher)
{
  unsigned int i;
  int r;
  
  fetcher->current_part = 0;
  for(i = 0 ; i < carray_count(fetcher->part_list) ; i++) {
    struct etpan_part_fetch_info * info;
    
    info = carray_get(fetcher->part_list, i);
    fetch_info_free(info);
  }
  r = carray_set_size(fetcher->part_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  collect_part(fetcher, etpan_message_get_main_part(fetcher->message));
}

static void update_message_callback(int cancelled,
    struct etpan_message_fetch_part_list_result * result,
    void * callback_data);
static void fetch_part(struct etpan_message_fetcher * fetcher);
static void fetch_part_callback(int cancelled,
    struct etpan_part_fetch_result * result,
    void * callback_data);
static void notify_part_fetched(struct etpan_message_fetcher * fetcher);
static void notify_finished(struct etpan_message_fetcher * fetcher);

static void decode_enriched(char * text, size_t text_length, char * charset,
    char ** result_text, size_t * result_text_length);
static struct etpan_error * decode_html(char * text, size_t text_length, char * charset,
    char ** result_text, size_t * result_text_length);
static void charset_decode(char * text, size_t text_length, char * charset,
    char ** result_text, size_t * result_text_length);

static int is_attachment(struct etpan_part * part);
static int is_part_fetchable(struct etpan_message_fetcher * fetcher,
    struct etpan_part * part);

static void update_message(struct etpan_message_fetcher * fetcher)
{
  struct etpan_thread_op * op;
  
  if (fetcher->fetched) {
    fetcher->error = NULL;
    notify_finished(fetcher);
    return;
  }
  
  fetcher->error = NULL;
  
  if (fetcher->message == NULL) {
    ETPAN_LOG("message was not defined");
    etpan_crash();
    return;
  }
  
  op = etpan_message_fetch_part_list(etpan_thread_manager_app_get_default(),
      fetcher->message, update_message_callback, fetcher);
  
  fetcher->part_op = op;
}

static void update_message_callback(int cancelled,
    struct etpan_message_fetch_part_list_result * result,
    void * callback_data)
{
  struct etpan_message_fetcher * fetcher;
  
  fetcher = callback_data;
  
  fetcher->part_op = NULL;
  
#if 0
  ETPAN_LOG("fetch callback %p", fetcher->message);
#endif
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Retrieval of message cancelled"));
    etpan_error_strf_long_description(error,
        _("Retrieval of the following message was interrupted:\n"
            "%s"), etpan_message_get_description(fetcher->message));
    
    fetcher->error = error;
    notify_finished(fetcher);
    return;
  }
  
  if (result->error != NULL) {
    fetcher->error = result->error;
    result->error = NULL;
    notify_finished(fetcher);
    return;
  }
  
#if 0
  ETPAN_LOG("main part : %p", result->part);
#endif
  etpan_message_set_main_part(fetcher->message,
      result->part);
  fetcher->fetched = 1;
#if 0
  etpan_part_unref(result->part);
  result->part = NULL;
#endif
  
  collect_part_list(fetcher);
  
  fetch_part(fetcher);
}

static void fetch_part(struct etpan_message_fetcher * fetcher)
{
  struct etpan_thread_op * op;
  struct etpan_part * part;
  char * content_type;
  struct etpan_part_header * header;
  int fetch_source;
  
  part = NULL;
  while (fetcher->current_part < carray_count(fetcher->part_list)) {
    struct etpan_part_fetch_info * info;
    struct etpan_part * current_part;
    
    info = carray_get(fetcher->part_list, fetcher->current_part);
    current_part = fetch_info_get_part(info);
    if (is_part_fetchable(fetcher, current_part)) {
      part = current_part;
      break;
    }
    
    fetcher->current_part ++;
  }
  
  if (part == NULL) {
    notify_finished(fetcher);
    return;
  }
  
  header = etpan_part_get_header(part);
  if (header == NULL) {
    ETPAN_LOG("WARNING! part has no header");
    
    fetcher->current_part ++;
    fetch_part(fetcher);
    
    return;
  }
  
  content_type = etpan_part_header_get_content_type(header);
  
  fetch_source = ((fetcher->flags & ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE) != 0);
  if ((strcasecmp(content_type, "message/rfc822") == 0) && (!fetch_source)) {
    op = etpan_part_fetch_header(etpan_thread_manager_app_get_default(),
        part, fetch_part_callback, fetcher);
  }
  else {
    op = etpan_part_fetch(etpan_thread_manager_app_get_default(),
        part, fetch_part_callback, fetcher);
  }
  
  fetcher->part_op = op;
}

static void fetch_part_callback(int cancelled,
    struct etpan_part_fetch_result * result,
    void * callback_data)
{
  struct etpan_message_fetcher * fetcher;
  char * decoded;
  size_t decoded_length;
  struct etpan_part * part;
  struct etpan_part_header * header;
  char * content_type;
  int encoding;
  struct etpan_part_fetch_info * info;
  int is_text;
  int is_header;
  int is_attach;
  int do_decode;
  int is_decoded;
  int is_html;
  int is_enriched;
  struct etpan_error * error;
  
  fetcher = callback_data;
  
  fetcher->part_op = NULL;
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Retrieval of message cancelled"));
    etpan_error_strf_long_description(error,
        _("Retrieval of the following message was interrupted:\n"
            "%s"), etpan_message_get_description(fetcher->message));
    
    fetcher->error = error;
    notify_finished(fetcher);
    goto stop_progress;
  }
  
  if (result->error != NULL) {
    fetcher->error = result->error;
    result->error = NULL;
    
    notify_finished(fetcher);
    goto stop_progress;
  }
  
  info = carray_get(fetcher->part_list, fetcher->current_part);
  part = fetch_info_get_part(info);
  
  header = etpan_part_get_header(part);
  if (header == NULL) {
    ETPAN_LOG("WARNING! part has no header, but this should not be reached");
    etpan_crash();
  }
  
  /* MIME decoding */
  encoding = etpan_part_header_get_mime_encoding(header);
  error = etpan_mime_decode(result->content, result->length, encoding,
      &decoded, &decoded_length);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(fetcher->message, ERROR_DOMAIN_FETCH, error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    fetcher->error = error;
    notify_finished(fetcher);
    goto stop_progress;
  }
  
  /* this is a text part anyway */
  content_type = etpan_part_header_get_content_type(header);
  
  is_attach = 0;
  if (is_attachment(part)) {
    is_attach = 1;
  }
  
  is_text = 0;
  if ((strcasecmp(content_type, "text/plain") == 0) ||
      (strcasecmp(content_type, "text/enriched") == 0) ||
      (strcasecmp(content_type, "text/html") == 0)) {
    is_text = 1;
  }
  
  is_html = 0;
  if (strcasecmp(content_type, "text/html") == 0) {
    is_html = 1;
  }

  is_enriched = 0;
  if (strcasecmp(content_type, "text/enriched") == 0) {
    is_enriched = 1;
  }
  
  is_header = 0;
  if (strcasecmp(content_type, "message/rfc822") == 0) {
    is_header = 1;
  }
  
  do_decode = ((fetcher->flags &
          ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET) != 0);
  
  is_decoded = 0;
  if ((!is_attach || do_decode) && is_text) {
    char * charset;
    char * charset_decoded;
    size_t charset_decoded_length;
    
    if (is_enriched) {
      char * decoded_enriched;
      size_t decoded_enriched_length;
      char * charset;
      
      charset = etpan_part_header_get_charset_encoding(header);
      if (charset == NULL)
        charset = "iso-8859-1";
      
      decode_enriched(decoded, decoded_length, charset,
          &decoded_enriched, &decoded_enriched_length);
      
      free(decoded);
      
      decoded = decoded_enriched;
      decoded_length = decoded_enriched_length;
      is_html = 1;
    }
    
    if (is_html) {
      char * decoded_html;
      size_t decoded_html_length;
      char * charset;
      
      charset = etpan_part_header_get_charset_encoding(header);
      if (charset == NULL)
        charset = "iso-8859-1";
      
      error = decode_html(decoded, decoded_length, charset,
          &decoded_html, &decoded_html_length);
      if (error != NULL) {
        struct etpan_error * new_error;
    
        new_error = rewrite_error(fetcher->message, ERROR_DOMAIN_FETCH, error);
        ETPAN_ERROR_FREE(error);
        error = new_error;
        
        free(decoded);
        fetcher->error = error;
        notify_finished(fetcher);
        goto stop_progress;
      }
      
      free(decoded);
      
      decoded = decoded_html;
      decoded_length = decoded_html_length;
    }
    
    /* charset decoding */
    charset = etpan_part_header_get_charset_encoding(header);
    if (charset == NULL)
      charset = DEFAULT_INCOMING_CHARSET;
    
    charset_decode(decoded, decoded_length, charset,
        &charset_decoded, &charset_decoded_length);
    
    /* replace decoded */
    free(decoded);
    decoded = charset_decoded;
    decoded_length = charset_decoded_length;
    
    is_decoded = 1;
  }
  
  if (is_decoded || is_header) {
    fetch_info_set_content(info, decoded, decoded_length);
  }
  else {
    char filename[PATH_MAX];
    char * attach_filename;
    char base_filename[PATH_MAX];
    char * base;
    FILE * f;
    
    attach_filename = etpan_part_header_get_filename(header);
    if (attach_filename != NULL) {
      strncpy(base_filename, attach_filename, sizeof(base_filename));
      base_filename[sizeof(base_filename) - 1] = '\0';
      base = basename(base_filename);
      snprintf(filename, sizeof(filename), "%s/%s",
          fetcher->temp_dir, base);
      
      f = fopen(filename, "wb");
    }
    else {
      f = etpan_get_tmp_file(filename, sizeof(filename));
    }
    
    if (f == NULL) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      etpan_error_set_short_description(error,
          _("Retrieval of message failed"));
      etpan_error_strf_long_description(error,
          _("An error occurred while fetching the following message:\n"
              "%s\n"
              "Could not write data to the following file: %s"),
          etpan_message_get_description(fetcher->message),
          filename);
      
      fetcher->error = error;
      notify_finished(fetcher);
      goto stop_progress;
    }
    
    fwrite(decoded, 1, decoded_length, f);
    fclose(f);
    
    free(decoded);
    
    fetch_info_set_file(info, filename, 1);
  }
  
  notify_part_fetched(fetcher);
  
  fetcher->current_part ++;
  
  fetch_part(fetcher);
  
 stop_progress:
  {}
}

static void notify_part_fetched(struct etpan_message_fetcher * fetcher)
{
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_PARTFETCHED_SIGNAL, fetcher,
      &fetcher->current_part);
}

static void notify_finished(struct etpan_message_fetcher * fetcher)
{
  if (fetcher->error != NULL)
    etpan_error_log(fetcher->error);
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, NULL);
}



static void charset_decode(char * text, size_t text_length, char * charset,
    char ** result_text, size_t * result_text_length)
{
  int r;
  char * dup_decoded;
  char * decoded;
  size_t decoded_length;
  
  r = charconv_buffer("utf-8", charset,
      text, text_length, &decoded, &decoded_length);
  if (r != MAIL_CHARCONV_NO_ERROR) {
    r = charconv_buffer("utf-8", "iso-8859-1",
        text, text_length, &decoded, &decoded_length);
    if (r != MAIL_CHARCONV_NO_ERROR) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  dup_decoded = malloc(decoded_length + 1);
  if (dup_decoded == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  memcpy(dup_decoded, decoded, decoded_length);
  dup_decoded[decoded_length] = '\0';
  
  charconv_buffer_free(decoded);
  
  * result_text = dup_decoded;
  * result_text_length = decoded_length;
}

static void decode_enriched(char * text, size_t text_length, char * charset,
    char ** result_text, size_t * result_text_length)
{
  char c;
  int paramct;
  int nofill;
  char token[62];
  char * p;
  size_t cur_token;
  MMAPString * str;
  char * decoded;
  (void) charset;
  
  str = mmap_string_new("");
  if (str == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  paramct = 0;
  nofill = 0;
  cur_token = 0;
  
  while (1) {
    if (cur_token >= text_length)
      break;
    
    c = text[cur_token];
    cur_token ++;
    
    if (c == '<') {
      if (cur_token >= text_length)
        break;
      c = text[cur_token];
      cur_token ++;
      
      if (c == '<') {
        if (mmap_string_append_c(str, '<') == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }
      else {
        unsigned int i;
        
        cur_token --;
        
        p = token;
        token[0] = '\0';
        for(i = 0 ; cur_token < text_length ; i++) {
          if (cur_token >= text_length)
            break;
          c = text[cur_token];
          cur_token ++;
          
          if (c == '>')
            break;
          
          if (i < sizeof(token) - 1)
            * p ++ = tolower(c);
        }
        
        * p = '\0';
        if (cur_token >= text_length)
          break;
        
        if (strcmp(token, "/param") == 0) {
          paramct --;
          if (mmap_string_append_c(str, '>') == NULL)
            ETPAN_LOG_MEMORY_ERROR;
        }
        else if (paramct > 0) {
          if (mmap_string_append_c(str, '<') == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          if (mmap_string_append(str, token) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          if (mmap_string_append_c(str, '>') == NULL)
            ETPAN_LOG_MEMORY_ERROR;
        }
        else {
          if (mmap_string_append_c(str, '<') == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          if (strcmp(token, "nofill") == 0) {
            nofill ++;
            if (mmap_string_append(str, "pre") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "/nofill") == 0) {
            nofill--;
            if (mmap_string_append(str, "/pre") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "bold") == 0) {
            if (mmap_string_append(str, "b") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "/bold") == 0) {
            if (mmap_string_append(str, "/b") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "italic") == 0) {
            if (mmap_string_append(str, "i") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "/italic") == 0) {
            if (mmap_string_append(str, "/i") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "fixed") == 0) {
            if (mmap_string_append(str, "tt") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "/fixed") == 0) {
            if (mmap_string_append(str, "/tt") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "excerpt") == 0) {
            if (mmap_string_append(str, "blockquote") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else if (strcmp(token, "/excerpt") == 0) {
            if (mmap_string_append(str, "/blockquote") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
          }
          else {
            if (mmap_string_append(str, "?") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
            if (mmap_string_append(str, token) == NULL)
              ETPAN_LOG_MEMORY_ERROR;
            if (strcmp(token, "param") == 0) {
              paramct ++;
              if (mmap_string_append_c(str, ' ') == NULL)
                ETPAN_LOG_MEMORY_ERROR;
              continue;
            }
          }
          if (mmap_string_append_c(str, '>') == NULL)
            ETPAN_LOG_MEMORY_ERROR;
        }
      }
    }
    else if (c == '>') {
      if (mmap_string_append_c(str, '>') == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else if (c == '&') {
      if (mmap_string_append_c(str, '&') == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      if(c == '\n' && nofill <= 0 && paramct <= 0) {
        unsigned int line_count;
        
        line_count = 0;
        while (1) {
          char ch;
          
          if (cur_token >= text_length)
            break;
          ch = text[cur_token];
          cur_token ++;
          
          if (ch == '\r')
            continue;
          
          if (ch != '\n')
            break;
          
          line_count ++;
        }
        if (line_count == 0) {
          if (mmap_string_append_c(str, ' ') == NULL)
            ETPAN_LOG_MEMORY_ERROR;
        }
        else {
          unsigned int i;
          
          for(i = 0 ; i < line_count ; i ++)
            if (mmap_string_append(str, "<br>\n") == NULL)
              ETPAN_LOG_MEMORY_ERROR;
        }
        
        if (cur_token >= text_length) {
          break;
        }
        
        cur_token --;
      }
      if (c != '\r') {
        if (mmap_string_append_c(str, c) == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  decoded = strdup(str->str);
  if (decoded == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  * result_text = decoded;
  * result_text_length = str->len;
  
  mmap_string_free(str);
}

static struct etpan_error *
decode_html(char * text, size_t text_length, char * charset,
    char ** result_text, size_t * result_text_length)
{
  pid_t pid;
  int input[2];
  int output[2];
  int r;
  struct etpan_error * error;
  
  r = pipe(input);
  if (r < 0) {
    goto err;
  }
  
  r = pipe(output);
  if (r < 0) {
    close(input[0]);
    close(input[1]);
    goto err;
  } 
  
  pid = fork();
  switch (pid) {
  case -1:
    close(input[0]);
    close(input[1]);
    close(output[0]);
    close(output[1]);
    goto err;
    
  case 0:
    /* child */
    {
      char command[PATH_MAX];
      
      dup2(input[0], 0);
      close(input[0]);
      close(input[1]);
      dup2(output[1], 1);
      close(output[0]);
      close(output[1]);
      /* redirect fd */
      
      snprintf(command, sizeof(command), "lynx -dump -stdin -display-charset=%s -width=10240", charset);
      
      execl("/bin/sh", "sh", "-c", command, NULL);
      exit(0);
    }
    break;
  default:
    /* parent */
    {
      int max_fds;
      char * current_input;
      size_t remaining_input;
      MMAPString * output_str;
      char buf[4096];
      char * decoded;
      int status;
      int input_open;
      
      output_str = mmap_string_new("");
      if (output_str == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      close(input[0]);
      close(output[1]);
      max_fds = input[1];
      if (output[0] > max_fds)
        max_fds = output[0];
      
      current_input = text;
      remaining_input = text_length;
      
      input_open = 1;
      while (1) {
        int r;
        fd_set readfds;
        fd_set writefds;
        
        FD_ZERO(&readfds);
        FD_ZERO(&writefds);
        if (remaining_input > 0) {
          FD_SET(input[1], &writefds);
        }
        else if (input_open) {
          close(input[1]);
          input_open = 0;
        }
        
        FD_SET(output[0], &readfds);
        r = select(max_fds + 1, &readfds, &writefds, NULL, NULL);
        if (r < 0) {
          break;
        }
        
        if (FD_ISSET(input[1], &writefds)) {
          ssize_t count;
          
          count = write(input[1], current_input, remaining_input);
          if (count < 0) {
            break;
          }
          
          current_input += count;
          remaining_input -= count;
        }
        if (FD_ISSET(output[0], &readfds)) {
          ssize_t count;
          
          count = read(output[0], buf, sizeof(buf));
          if (count < 0) {
            break;
          }
          
          if (mmap_string_append_len(output_str, buf, count) == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          
          if (count == 0) {
            break;
          }
        }
      }
      
      close(output[0]);
      if (input_open) {
        close(input[1]);
        input_open = 0;
      }
      
      wait4(pid, &status, 0, NULL);
      
      decoded = strdup(output_str->str);
      if (decoded == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      * result_text_length = output_str->len;
      * result_text = decoded;
      
      mmap_string_free(output_str);
    }
    break;
  }
  
  return NULL;
  
 err:
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_SYSTEM);
  etpan_error_set_short_description(error, _("System resource exhausted"));
  etpan_error_strf_long_description(error,
      _("System resource have been exhausted. The HTML message could not be decoded."));
  
  return error;
}

static int is_attachment(struct etpan_part * part)
{
  char * content_type;
  struct etpan_part_header * header;
  char * filename;
  
  header = etpan_part_get_header(part);
  if (header == NULL)
    return 0;
  
  filename = etpan_part_header_get_filename(header);
  
  content_type = etpan_part_header_get_content_type(header);
  if ((strcasecmp(content_type, "text/plain") == 0) ||
      (strcasecmp(content_type, "text/enriched") == 0) ||
      (strcasecmp(content_type, "text/html") == 0)) {
    if (filename == NULL)
      return 0;
    else
      return 1;
  }
  
  return 1;
}

static int is_part_fetchable(struct etpan_message_fetcher * fetcher,
    struct etpan_part * part)
{
  char * content_type;
  struct etpan_part_header * header;
  int fetch_image;
  int fetch_part;
  
  fetch_part = (fetcher->flags & ETPAN_MESSAGE_FETCHER_FLAGS_ATTACHMENT) != 0;
  fetch_image = (fetcher->flags & ETPAN_MESSAGE_FETCHER_FLAGS_IMAGE) != 0;
  if (fetch_part)
    fetch_image = 1;
  
  header = etpan_part_get_header(part);
  if (header == NULL)
    return 0;
  
  content_type = etpan_part_header_get_content_type(header);
  if ((strcasecmp(content_type, "text/plain") == 0) ||
      (strcasecmp(content_type, "text/html") == 0) ||
      (strcasecmp(content_type, "text/enriched") == 0))
    return 1;
  
  if (strncasecmp(content_type, "image/", 5) == 0)
    return fetch_image;
  
  if (strcasecmp(content_type, "message/rfc822") == 0)
    return 1;
  
  return fetch_part;
}






struct etpan_message_fetcher * etpan_message_fetcher_new(void)
{
  struct etpan_message_fetcher * fetcher;
  char tmp_dir[PATH_MAX];
  int r;
  
  fetcher = malloc(sizeof(* fetcher));
  if (fetcher == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetcher->flags = 0;
  fetcher->current_part = 0;
  fetcher->part_list = carray_new(4);
  if (fetcher->part_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetcher->part_op = NULL;
  fetcher->message = NULL;

  r = etpan_get_tmp_path(tmp_dir, sizeof(tmp_dir));
  fetcher->temp_dir = strdup(tmp_dir);
  if (fetcher->temp_dir == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetcher->error = NULL;
  fetcher->fetched = 0;
  fetcher->ref_count = 1;
  
  return fetcher;
}

static void clear_part_list(struct etpan_message_fetcher * fetcher)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(fetcher->part_list) ; i ++) {
    struct etpan_part_fetch_info * info;
    
    info = carray_get(fetcher->part_list, i);
    if (info->type == ETPAN_PART_FETCH_INFO_TYPE_FILE) {
      if (info->temporary)
        unlink(info->filename);
    }
    etpan_part_unref(info->part);
    
    fetch_info_free(info);
  }
  r = carray_set_size(fetcher->part_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (fetcher->fetched) {
    struct etpan_part * part;
    
    etpan_message_clear_part_list(etpan_thread_manager_app_get_default(),
        fetcher->message);
    fetcher->fetched = 0;
  }
}

static void etpan_message_fetcher_free(struct etpan_message_fetcher * fetcher)
{
  etpan_message_fetcher_unsetup(fetcher);
  
  ETPAN_ERROR_FREE(fetcher->error);
  fetcher->error = NULL;
  rmdir(fetcher->temp_dir);
  free(fetcher->temp_dir);
  carray_free(fetcher->part_list);
  free(fetcher);
}

void etpan_message_fetcher_ref(struct etpan_message_fetcher * fetcher)
{
  fetcher->ref_count ++;
}

void etpan_message_fetcher_unref(struct etpan_message_fetcher * fetcher)
{
  fetcher->ref_count --;
  if (fetcher->ref_count == 0)
    etpan_message_fetcher_free(fetcher);
}

void etpan_message_fetcher_setup(struct etpan_message_fetcher * fetcher)
{
  (void) fetcher;
}

static void etpan_message_fetcher_unsetup(struct etpan_message_fetcher * fetcher)
{
  etpan_message_fetcher_cancel(fetcher);
  etpan_message_fetcher_set_message(fetcher, NULL);
  clear_part_list(fetcher);
}

void etpan_message_fetcher_set_message(struct etpan_message_fetcher * fetcher,
    struct etpan_message * message)
{
  if (message != fetcher->message) {
    etpan_message_fetcher_cancel(fetcher);
    clear_part_list(fetcher);
    
    if (fetcher->message != NULL)
      etpan_message_unref(fetcher->message);
    if (message != NULL)
      etpan_message_ref(message);
    fetcher->message = message;
    fetcher->fetched = 0;
  }
}

struct etpan_message * etpan_message_fetcher_get_message(struct etpan_message_fetcher * fetcher)
{
  return fetcher->message;
}

void etpan_message_fetcher_set_flags(struct etpan_message_fetcher * fetcher,
    int flags)
{
  fetcher->flags = flags;
}

int etpan_message_fetcher_get_flags(struct etpan_message_fetcher * fetcher)
{
  return fetcher->flags;
}

carray * etpan_message_fetcher_get_part_list(struct etpan_message_fetcher * fetcher)
{
  return fetcher->part_list;
}

struct etpan_error *
etpan_message_fetcher_get_error(struct etpan_message_fetcher * fetcher)
{
  return fetcher->error;
}

void etpan_message_fetcher_run(struct etpan_message_fetcher * fetcher)
{
  update_message(fetcher);
}

void etpan_message_fetcher_cancel(struct etpan_message_fetcher * fetcher)
{
  if (fetcher->part_op != NULL)
    etpan_thread_op_cancel(fetcher->part_op);
}

struct etpan_part * etpan_part_fetch_info_get_part(struct etpan_part_fetch_info * info)
{
  return info->part;
}

int etpan_message_is_header_displayed(char * name)
{
  if ((strcasecmp(name, "From") == 0) ||
      (strcasecmp(name, "Reply-To") == 0) ||
      (strcasecmp(name, "To") == 0) ||
      (strcasecmp(name, "Cc") == 0) ||
      (strcasecmp(name, "Bcc") == 0) ||
      (strcasecmp(name, "Date") == 0) ||
      (strcasecmp(name, "Subject") == 0))
    return 0;
  
  if ((strcasecmp(name, "X-Mailer") == 0) ||
      (strcasecmp(name, "X-Newsreader") == 0) ||
      (strcasecmp(name, "User-Agent") == 0)) {
    return 1;
  }
  
  return 0;
}


static struct etpan_error * rewrite_error(struct etpan_message * msg,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while fetching the following message:\n"
            "%s\n"
            "%s"),
        etpan_message_get_description(msg),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with the following message:\n"
            "%s\n"
            "%s"),
        etpan_message_get_description(msg),
        previous_long_description);
    break;
  }
  
  return new_error;
}
