#include "etpan-message-header.h"

#include <libetpan/libetpan.h>
#include <ctype.h>

#include "etpan-error.h"
#include "etpan-address.h"
#include "etpan-lep.h"
#include "etpan-log.h"
#include "etpan-serialize.h"

struct etpan_message_header * etpan_message_header_new(void)
{
  struct etpan_message_header * header;
  
  header = malloc(sizeof(* header));
  if (header == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  header->from = etpan_address_list_new();
  header->replyto = etpan_address_list_new();
  header->to = etpan_address_list_new();
  header->cc = etpan_address_list_new();
  header->bcc = etpan_address_list_new();
  header->resent_from = etpan_address_list_new();
  header->resent_to = etpan_address_list_new();
  header->resent_cc = etpan_address_list_new();
  header->resent_bcc = etpan_address_list_new();
  
  header->subject = NULL;
  header->newsgroups = NULL;
  header->listpost = NULL;
  header->date = time(NULL);
  header->msgid = mailimf_get_message_id();
  if (header->msgid == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  header->resent_date = 0;
  header->resent_msgid = NULL;
  
  header->references = etpan_message_id_list_new();
  header->inreplyto = etpan_message_id_list_new();
  
  return header;
}


void etpan_message_header_free(struct etpan_message_header * header)
{
  etpan_message_id_list_free(header->inreplyto);
  etpan_message_id_list_free(header->references);
  free(header->resent_msgid);
  free(header->msgid);
  if (header->listpost != NULL)
    etpan_address_free(header->listpost);
  free(header->newsgroups);
  free(header->subject);
  etpan_address_list_free(header->resent_bcc);
  etpan_address_list_free(header->resent_cc);
  etpan_address_list_free(header->resent_to);
  etpan_address_list_free(header->resent_from);
  etpan_address_list_free(header->bcc);
  etpan_address_list_free(header->cc);
  etpan_address_list_free(header->to);
  etpan_address_list_free(header->replyto);
  etpan_address_list_free(header->from);
  free(header);
}

struct etpan_message_header *
etpan_message_header_parse(char * content, size_t length)
{
  int r;
  size_t cur_token;
  struct mailimf_fields * fields;
  struct etpan_message_header * header;
  
  cur_token = 0;
  r = mailimf_envelope_and_optional_fields_parse(content, length,
      &cur_token, &fields);
  if (r != MAILIMF_NO_ERROR)
    goto fallback;
  
  header = etpan_header_from_lep_header(fields);
  
  mailimf_fields_free(fields);
  
  return header;
  
 fallback:
  return etpan_message_header_new();
}

time_t etpan_message_header_get_date(struct etpan_message_header * header)
{
  return header->date;
}

void etpan_message_header_set_date(struct etpan_message_header * header,
    time_t date)
{
  header->date = date;
}

char * etpan_message_header_get_subject(struct etpan_message_header * header)
{
  return header->subject;
}

void etpan_message_header_set_subject(struct etpan_message_header * header,
    const char * subject)
{
  if (subject != header->subject) {
    free(header->subject);
    if (subject != NULL) {
      header->subject = strdup(subject);
      if (header->subject == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      header->subject = NULL;
  }
}

char * etpan_message_header_get_newsgroups(struct etpan_message_header * header)
{
  return header->newsgroups;
}

void etpan_message_header_set_newsgroups(struct etpan_message_header * header,
    const char * newsgroups)
{
  if (newsgroups != header->newsgroups) {
    free(header->newsgroups);
    if (newsgroups != NULL) {
      header->newsgroups = strdup(newsgroups);
      if (header->newsgroups == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      header->newsgroups = NULL;
  }
}

carray * etpan_message_header_get_from(struct etpan_message_header * header)
{
  return header->from;
}

carray * etpan_message_header_get_replyto(struct etpan_message_header * header)
{
  return header->replyto;
}

carray * etpan_message_header_get_to(struct etpan_message_header * header)
{
  return header->to;
}

carray * etpan_message_header_get_cc(struct etpan_message_header * header)
{
  return header->cc;
}

carray * etpan_message_header_get_bcc(struct etpan_message_header * header)
{
  return header->bcc;
}

void etpan_message_header_set_from(struct etpan_message_header * header,
    carray * from)
{
  carray * old_from;
  
  old_from = header->from;
  header->from = etpan_address_list_dup(from);
  
  etpan_address_list_free(old_from);
}

void etpan_message_header_set_replyto(struct etpan_message_header * header,
    carray * replyto)
{
  carray * old_replyto;
  
  old_replyto = header->replyto;
  header->replyto = etpan_address_list_dup(replyto);
  
  etpan_address_list_free(old_replyto);
}

void etpan_message_header_set_to(struct etpan_message_header * header,
    carray * to)
{
  carray * old_to;
  
  old_to = header->to;
  header->to = etpan_address_list_dup(to);
  
  etpan_address_list_free(old_to);
}

void etpan_message_header_set_cc(struct etpan_message_header * header,
    carray * cc)
{
  carray * old_cc;
  
  old_cc = header->cc;
  header->cc = etpan_address_list_dup(cc);
  
  etpan_address_list_free(old_cc);
}

void etpan_message_header_set_bcc(struct etpan_message_header * header,
    carray * bcc)
{
  carray * old_bcc;
  
  old_bcc = header->bcc;
  header->bcc = etpan_address_list_dup(bcc);
  
  etpan_address_list_free(old_bcc);
}

void etpan_message_header_set_references(struct etpan_message_header * header,
    carray * references)
{
  carray * old_references;
  
  old_references = header->references;
  header->references = etpan_message_id_list_dup(references);
  
  etpan_address_list_free(old_references);
}

void etpan_message_header_set_inreplyto(struct etpan_message_header * header,
    carray * inreplyto)
{
  carray * old_inreplyto;
  
  old_inreplyto = header->inreplyto;
  header->inreplyto = etpan_message_id_list_dup(inreplyto);
  
  etpan_address_list_free(old_inreplyto);
}

carray * etpan_message_header_get_references(struct etpan_message_header * header)
{
  return header->references;
}

carray * etpan_message_header_get_inreplyto(struct etpan_message_header * header)
{
  return header->inreplyto;
}

void etpan_message_header_set_msgid(struct etpan_message_header * header,
    char * msgid)
{
  if (msgid != header->msgid) {
    free(header->msgid);
    if (msgid != NULL) {
      header->msgid = strdup(msgid);
      if (header->msgid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      header->msgid = NULL;
  }
}

char * etpan_message_header_get_msgid(struct etpan_message_header * header)
{
  return header->msgid;
}


carray * etpan_message_id_list_new(void)
{
  carray * list;
  
  list = carray_new(4);
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return list;
}

void etpan_message_id_list_free(carray * msgid_list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(msgid_list) ; i ++) {
    char * item;
    
    item = carray_get(msgid_list, i);
    
    free(item);
  }
  carray_free(msgid_list);
}

void etpan_message_id_list_add(carray * msgid_list, char * msgid)
{
  char * dupmsgid;
  int r;
  
  dupmsgid = strdup(msgid);
  if (dupmsgid == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(msgid_list, dupmsgid, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

carray * etpan_message_id_list_dup(carray * msgid_list)
{
  unsigned int i;
  carray * dup_msgidlist;
  
  dup_msgidlist = etpan_message_id_list_new();
  
  for(i = 0 ; i < carray_count(msgid_list) ; i ++) {
    char * item;
    
    item = carray_get(msgid_list, i);
    
    etpan_message_id_list_add(dup_msgidlist, item);
  }
  
  return dup_msgidlist;
}

/* array of etpan_unparsed_header_item */
carray * etpan_unparsed_header_new(void)
{
  carray * list;
  
  list = carray_new(16);
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return list;
}

void etpan_unparsed_header_free(carray * header_list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(header_list) ; i ++) {
    struct etpan_unparsed_header_item * item;
    
    item = carray_get(header_list, i);
    etpan_unparsed_header_item_free(item);
  }
  
  carray_free(header_list);
}

carray * etpan_unparsed_header_parse(char * content, size_t length)
{
  int r;
  size_t cur_token;
  struct mailimf_fields * fields;
  clistiter * iter;
  carray * header_list;
  
  header_list = etpan_unparsed_header_new();
  
  cur_token = 0;
  r = mailimf_optional_fields_parse(content, length, &cur_token, &fields);
  if (r != MAILIMF_NO_ERROR)
    goto fallback;
  
  for(iter = clist_begin(fields->fld_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailimf_field * field;
    char * name;
    char * value;
    struct etpan_unparsed_header_item * item;
    
    field = clist_content(iter);
    if (field->fld_type != MAILIMF_FIELD_OPTIONAL_FIELD)
      continue;
    
    name = field->fld_data.fld_optional_field->fld_name;
    value = field->fld_data.fld_optional_field->fld_value;
    
    item = etpan_unparsed_header_item_new();
    etpan_unparsed_header_item_set_name(item, name);
    etpan_unparsed_header_item_set_value(item, value);
    etpan_unparsed_header_add(header_list, item);
    
    continue;
  }
  
  mailimf_fields_free(fields);
  
  return header_list;
  
 fallback:
  return header_list;
}

void etpan_unparsed_header_add(carray * header_list,
    struct etpan_unparsed_header_item * item)
{
  int r;
  
  r = carray_add(header_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

struct etpan_unparsed_header_item * etpan_unparsed_header_item_new(void)
{
  struct etpan_unparsed_header_item * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item->name = NULL;
  item->value = NULL;
  
  return item;
}

void etpan_unparsed_header_item_free(struct etpan_unparsed_header_item * item)
{
  free(item->value);
  free(item->name);
  free(item);
}

void etpan_unparsed_header_item_set_name(struct etpan_unparsed_header_item *
    item, char * name)
{
  if (name != item->name) {
    free(item->name);
    if (name != NULL) {
      item->name = strdup(name);
      if (item->name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      item->name = NULL;
  }
}

char *
etpan_unparsed_header_item_get_name(struct etpan_unparsed_header_item * item)
{
  return item->name;
}

void etpan_unparsed_header_item_set_value(struct etpan_unparsed_header_item *
    item, char * value)
{
  if (value != item->value) {
    free(item->value);
    if (value != NULL) {
      item->value = strdup(value);
      if (item->value == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      item->value = NULL;
  }
}

char *
etpan_unparsed_header_item_get_value(struct etpan_unparsed_header_item * item)
{
  return item->value;
}

char * etpan_unparsed_header_get_value(carray * header_list, char * name)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(header_list) ; i ++) {
    struct etpan_unparsed_header_item * item;
    char * cur_name;
    
    item = carray_get(header_list, i);
    cur_name = etpan_unparsed_header_item_get_name(item);
    if (cur_name == NULL)
      continue;
    
    if (strcasecmp(cur_name, name) == 0)
      return etpan_unparsed_header_item_get_value(item);
  }
  
  return NULL;
}

void etpan_message_header_set_from_str(struct etpan_message_header * header,
    char * str)
{
  carray * list;
  
  list = etpan_address_list_to_from_str(str);
  
  etpan_message_header_set_from(header, list);
  etpan_address_list_free(list);
}

void etpan_message_header_set_replyto_str(struct etpan_message_header * header,
    char * str)
     
{
  carray * list;
  
  list = etpan_address_list_to_from_str(str);
  
  etpan_message_header_set_replyto(header, list);
  etpan_address_list_free(list);
}

void etpan_message_header_set_to_str(struct etpan_message_header * header,
    char * str)
{
  carray * list;
  
  list = etpan_address_list_to_from_str(str);
  
  etpan_message_header_set_to(header, list);
  etpan_address_list_free(list);
}

void etpan_message_header_set_cc_str(struct etpan_message_header * header,
    char * str)
{
  carray * list;
  
  list = etpan_address_list_to_from_str(str);
  
  etpan_message_header_set_cc(header, list);
  etpan_address_list_free(list);
}

void etpan_message_header_set_bcc_str(struct etpan_message_header * header,
    char * str)
{
  carray * list;
  
  list = etpan_address_list_to_from_str(str);
  
  etpan_message_header_set_bcc(header, list);
  etpan_address_list_free(list);
}



/* normalize subject */

static inline int skip_subj_blob(char * subj, size_t * begin,
    size_t length)
{
  /* subj-blob       = "[" *BLOBCHAR "]" *WSP */
  size_t cur_token;

  cur_token = * begin;

  if (subj[cur_token] != '[')
    return 0;

  cur_token ++;

  while (1) {
    if (cur_token >= length)
      return 0;

    if (subj[cur_token] == '[')
      return 0;

    if (subj[cur_token] == ']')
      break;

    cur_token ++;
  }

  cur_token ++;

  while (1) {
    if (cur_token >= length)
      break;

    if (subj[cur_token] != ' ')
      break;

    cur_token ++;
  }

  * begin = cur_token;

  return 1;
}

static inline int skip_subj_refwd(char * subj, size_t * begin,
    size_t length)
{
  /* subj-refwd      = ("re" / ("fw" ["d"])) *WSP [subj-blob] ":" */
  size_t cur_token;
  int prefix;

  cur_token = * begin;

  prefix = 0;
  if (length >= 3) {
    if (strncasecmp(subj + cur_token, "fwd", 3) == 0) {
      cur_token += 3;
      prefix = 1;
    }
  }
  if (!prefix) {
    if (length >= 2) {
      if (strncasecmp(subj + cur_token, "fw", 2) == 0) {
	cur_token += 2;
	prefix = 1;
      }
      else if (strncasecmp(subj + cur_token, "re", 2) == 0) {
	cur_token += 2;
	prefix = 1;
      }
    }
  }

  if (!prefix)
    return 0;

  while (1) {
    if (cur_token >= length)
      break;
      
    if (subj[cur_token] != ' ')
      break;

    cur_token ++;
  }
  
  skip_subj_blob(subj, &cur_token, length);

  if (subj[cur_token] != ':')
    return 0;

  cur_token ++;

  * begin = cur_token;

  return 1;
}

static inline int skip_subj_leader(char * subj, size_t * begin,
    size_t length)
{
  size_t cur_token;

  cur_token = * begin;

  /* subj-leader     = (*subj-blob subj-refwd) / WSP */
  
  if (subj[cur_token] == ' ') {
    cur_token ++;
  }
  else {
    while (cur_token < length) {
      if (!skip_subj_blob(subj, &cur_token, length))
	break;
    }
    if (!skip_subj_refwd(subj, &cur_token, length))
      return 0;
  }

  * begin = cur_token;

  return 1;
}


static char * extract_subject(char * str)
{
  char * subj;
  char * cur;
  char * write_pos;
  size_t len;
  size_t begin;
  int do_repeat_5;
  int do_repeat_6;

  /*
    (1) Convert any RFC 2047 encoded-words in the subject to
    UTF-8.
    We work on UTF-8 string -- DVH
  */
  
  subj = strdup(str);
  if (subj == NULL)
    return NULL;
  
  len = strlen(subj);

  /*
    Convert all tabs and continuations to space.
    Convert all multiple spaces to a single space.
  */

  cur = subj;
  write_pos = subj;
  while (* cur != '\0') {
    int cont;

    switch (* cur) {
    case '\t':
    case '\r':
    case '\n':
      cont = 1;

      cur ++;
      while (* cur && cont) {
	switch (* cur) {
	case '\t':
	case '\r':
	case '\n':
	  cont = 1;
	  break;
	default:
	  cont = 0;
	  break;
	}
	cur ++;
      }
      
      * write_pos = ' ';
      write_pos ++;

      break;
      
    default:
      * write_pos = * cur;
      write_pos ++;

      cur ++;

      break;
    }
  }
  * write_pos = '\0';

  begin = 0;

  do {
    do_repeat_6 = 0;

    /*
      (2) Remove all trailing text of the subject that matches
      the subj-trailer ABNF, repeat until no more matches are
      possible.
    */

    while (len > 0) {
      int chg;

      chg = 0;

      /* subj-trailer    = "(fwd)" / WSP */
      if (subj[len - 1] == ' ') {
	subj[len - 1] = '\0';
	len --;
      }
      else {
	if (len < 5)
	  break;

	if (strncasecmp(subj + len - 5, "(fwd)", 5) != 0)
	  break;

	subj[len - 5] = '\0';
	len -= 5;
      }
    }

    do {
      size_t saved_begin;

      do_repeat_5 = 0;

      /*
	(3) Remove all prefix text of the subject that matches the
	subj-leader ABNF.
      */
    
      if (skip_subj_leader(subj, &begin, len))
	do_repeat_5 = 1;

      /*
	(4) If there is prefix text of the subject that matches the
	subj-blob ABNF, and removing that prefix leaves a non-empty
	subj-base, then remove the prefix text.
      */
    
      saved_begin = begin;
      if (skip_subj_blob(subj, &begin, len)) {
	if (begin == len) {
	  /* this will leave a empty subject base */
	  begin = saved_begin;
	}
	else
	  do_repeat_5 = 1;
      }

      /*
	(5) Repeat (3) and (4) until no matches remain.
	Note: it is possible to defer step (2) until step (6),
	but this requires checking for subj-trailer in step (4).
      */
    
    }
    while (do_repeat_5);

    /*
      (6) If the resulting text begins with the subj-fwd-hdr ABNF
      and ends with the subj-fwd-trl ABNF, remove the
      subj-fwd-hdr and subj-fwd-trl and repeat from step (2).
    */

    if (len >= 5) {
      size_t saved_begin;

      saved_begin = begin;
      if (strncasecmp(subj + begin, "[fwd:", 5) == 0) {
	begin += 5;
	
	if (subj[len - 1] != ']')
	  saved_begin = begin;
	else {
	  subj[len - 1] = '\0';
	  len --;
	  do_repeat_6 = 1;
	}
      }
    }

  }
  while (do_repeat_6);

  /*
    (7) The resulting text is the "base subject" used in
    threading.
  */

  /* convert to upper case */

  cur = subj + begin;
  write_pos = subj;

  while (* cur != '\0') {
    * write_pos = * cur;
    cur ++;
    write_pos ++;
  }
  * write_pos = '\0';

  return subj;
}

char * etpan_message_header_subject_normalize(char * subject)
{
  char * value;
  
  value = extract_subject(subject);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}

struct etpan_address *
etpan_message_header_get_listpost(struct etpan_message_header * header)
{
  return header->listpost;
}

void etpan_message_header_set_listpost(struct etpan_message_header * header,
    struct etpan_address * listpost)
{
  if (listpost != header->listpost) {
    if (header->listpost != NULL)
      etpan_address_free(header->listpost);
    if (listpost != NULL) {
      header->listpost = etpan_address_dup(listpost);
      if (header->listpost == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      header->listpost = NULL;
  }
}

carray * etpan_message_header_get_resent_from(struct etpan_message_header * header)
{
  return header->resent_from;
}

carray * etpan_message_header_get_resent_to(struct etpan_message_header * header)
{
  return header->resent_to;
}

carray * etpan_message_header_get_resent_cc(struct etpan_message_header * header)
{
  return header->resent_cc;
}

carray * etpan_message_header_get_resent_bcc(struct etpan_message_header * header)
{
  return header->resent_bcc;
}

void etpan_message_header_set_resent_from(struct etpan_message_header * header,
    carray * from)
{
  carray * old_from;
  
  old_from = header->resent_from;
  header->resent_from = etpan_address_list_dup(from);
  
  etpan_address_list_free(old_from);
}

void etpan_message_header_set_resent_to(struct etpan_message_header * header,
    carray * to)
{
  carray * old_to;
  
  old_to = header->resent_to;
  header->resent_to = etpan_address_list_dup(to);
  
  etpan_address_list_free(old_to);
}

void etpan_message_header_set_resent_cc(struct etpan_message_header * header,
    carray * cc)
{
  carray * old_cc;
  
  old_cc = header->resent_cc;
  header->resent_cc = etpan_address_list_dup(cc);
  
  etpan_address_list_free(old_cc);
}

void etpan_message_header_set_resent_bcc(struct etpan_message_header * header,
    carray * bcc)
{
  carray * old_bcc;
  
  old_bcc = header->resent_bcc;
  header->resent_bcc = etpan_address_list_dup(bcc);
  
  etpan_address_list_free(old_bcc);
}

time_t etpan_message_header_get_resent_date(struct etpan_message_header * header)
{
  return header->resent_date;
}

void etpan_message_header_set_resent_date(struct etpan_message_header * header,
    time_t date)
{
  header->resent_date = date;
}

char * etpan_message_header_get_resent_msgid(struct etpan_message_header * header)
{
  return header->resent_msgid;
}

void etpan_message_header_set_resent_msgid(struct etpan_message_header * header,
    char * resent_msgid)
{
  if (resent_msgid != header->resent_msgid) {
    free(header->resent_msgid);
    if (resent_msgid != NULL) {
      header->resent_msgid = strdup(resent_msgid);
      if (header->resent_msgid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      header->resent_msgid = NULL;
  }
}

static struct etpan_serialize_data * encode_addr(struct etpan_address * addr)
{
  struct etpan_serialize_data * data;  
  
  data = etpan_serialize_data_new_hash();
  if (addr != NULL) {
    etpan_serialize_hash_set_uint32(data, "notnull", 1);
    etpan_serialize_hash_set_str(data, "displayname",
        etpan_address_get_display_name(addr));
    etpan_serialize_hash_set_str(data, "address",
        etpan_address_get_address(addr));
  }
  else {
    etpan_serialize_hash_set_uint32(data, "notnull", 0);
  }
  
  return data;
}

struct etpan_address * decode_addr(struct etpan_serialize_data * data)
{
  struct etpan_address * addr;
  char * display_name;
  char * addr_spec;
  int notnull;
  
  notnull = etpan_serialize_hash_get_uint32(data, "notnull");
  if (notnull) {
    addr = etpan_address_new();
    display_name = etpan_serialize_hash_get_str(data, "displayname");
    addr_spec = etpan_serialize_hash_get_str(data, "address");
    etpan_address_set_display_name(addr, display_name);
    etpan_address_set_address(addr, addr_spec);
  }
  else {
    addr = NULL;
  }
  
  return addr;
}

static struct etpan_serialize_data * encode_addr_list(carray * addr_list)
{
  struct etpan_serialize_data * data;  
  unsigned int i;
  
  data = etpan_serialize_data_new_array();
  for(i = 0 ; i < carray_count(addr_list) ; i ++) {
    struct etpan_serialize_data * item;
    struct etpan_address * addr;
    
    addr = carray_get(addr_list, i);
    item = encode_addr(addr);
    etpan_serialize_array_add(data, item);
  }
  
  return data;
}

static carray * decode_addr_list(struct etpan_serialize_data * data)
{
  unsigned int count;
  unsigned int i;
  carray * addr_list;
  
  addr_list = etpan_address_list_new();
  
  count = etpan_serialize_array_count(data);
  for(i = 0 ; i < count ; i ++) {
    struct etpan_serialize_data * item;
    struct etpan_address * addr;
    
    item = etpan_serialize_array_get(data, i);
    addr = decode_addr(item);
    etpan_address_list_add_address(addr_list, addr);
  }
  
  return addr_list;
}

static struct etpan_serialize_data * encode_msgid_list(carray * msgid_list)
{
  struct etpan_serialize_data * data;  
  unsigned int i;
  
  data = etpan_serialize_data_new_array();
  for(i = 0 ; i < carray_count(msgid_list) ; i ++) {
    char * msgid;
    
    msgid = carray_get(msgid_list, i);
    etpan_serialize_array_add_str(data, msgid);
  }
  
  return data;
}

static carray * decode_msgid_list(struct etpan_serialize_data * data)
{
  unsigned int count;
  unsigned int i;
  carray * msgid_list;
  
  msgid_list = etpan_message_id_list_new();
  
  count = etpan_serialize_array_count(data);
  for(i = 0 ; i < count ; i ++) {
    char * msgid;
    
    msgid = etpan_serialize_array_get_str(data, i);
    etpan_message_id_list_add(msgid_list, msgid);
  }
  
  return msgid_list;
}

static struct etpan_serialize_data * encode_header(struct etpan_message_header * header)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * item;
  struct etpan_address * addr;
  time_t date;
  char * subject;
  char * newsgroups;
  char * resentmsgid;
  time_t resentdate;
  
  hash = etpan_serialize_data_new_hash();
  
  date = etpan_message_header_get_date(header);
  etpan_serialize_hash_set_int64(hash, "date", date);
  
  subject = etpan_message_header_get_subject(header);
  etpan_serialize_hash_set_str(hash, "subject", subject);
  
  newsgroups = etpan_message_header_get_newsgroups(header);
  etpan_serialize_hash_set_str(hash, "newsgroups", newsgroups);
  
  addr = etpan_message_header_get_listpost(header);
  item = encode_addr(addr);
  etpan_serialize_hash_set(hash, "listpost", item);
  
  item = encode_addr_list(etpan_message_header_get_from(header));
  etpan_serialize_hash_set(hash, "from", item);
  
  item = encode_addr_list(etpan_message_header_get_replyto(header));
  etpan_serialize_hash_set(hash, "replyto", item);
  
  item = encode_addr_list(etpan_message_header_get_to(header));
  etpan_serialize_hash_set(hash, "to", item);
  
  item = encode_addr_list(etpan_message_header_get_cc(header));
  etpan_serialize_hash_set(hash, "cc", item);

  item = encode_addr_list(etpan_message_header_get_bcc(header));
  etpan_serialize_hash_set(hash, "bcc", item);
  
  item = encode_msgid_list(etpan_message_header_get_references(header));
  etpan_serialize_hash_set(hash, "references", item);
  
  item = encode_msgid_list(etpan_message_header_get_inreplyto(header));
  etpan_serialize_hash_set(hash, "inreplyto", item);
  
  etpan_serialize_hash_set_str(hash, "msgid",
      etpan_message_header_get_msgid(header));
  
  resentmsgid = etpan_message_header_get_resent_msgid(header);
  etpan_serialize_hash_set_str(hash, "resentmsgid", resentmsgid);
  
  item = encode_addr_list(etpan_message_header_get_resent_from(header));
  etpan_serialize_hash_set(hash, "resentfrom", item);
  
  item = encode_addr_list(etpan_message_header_get_resent_to(header));
  etpan_serialize_hash_set(hash, "resentto", item);
  
  item = encode_addr_list(etpan_message_header_get_resent_cc(header));
  etpan_serialize_hash_set(hash, "resentcc", item);
  
  item = encode_addr_list(etpan_message_header_get_resent_bcc(header));
  etpan_serialize_hash_set(hash, "resentbcc", item);
  
  resentdate = etpan_message_header_get_date(header);
  etpan_serialize_hash_set_int64(hash, "resentdate", resentdate);
  
  return hash;
}

static struct etpan_message_header * decode_header(struct etpan_serialize_data * hash)
{
  time_t date;
  time_t resentdate;
  struct etpan_address * addr;
  carray * addr_list;
  char * resentmsgid;
  char * msgid;
  char * subject;
  carray * msgid_list;
  struct etpan_serialize_data * item;
  struct etpan_message_header * header;
  
  header = etpan_message_header_new();
  
  date = etpan_serialize_hash_get_int64(hash, "date");
  etpan_message_header_set_date(header, date);
  
  subject = etpan_serialize_hash_get_str(hash, "subject");
  etpan_message_header_set_subject(header, subject);
  
  etpan_serialize_hash_set_str(hash, "newsgroups",
      etpan_message_header_get_newsgroups(header));
  
  item = etpan_serialize_hash_get(hash, "listpost");
  addr = decode_addr(item);
  etpan_message_header_set_listpost(header, addr);
  
  item = etpan_serialize_hash_get(hash, "from");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_from(header, addr_list);
  etpan_address_list_free(addr_list);

  item = etpan_serialize_hash_get(hash, "replyto");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_replyto(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "to");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_to(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "cc");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_cc(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "bcc");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_bcc(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "references");
  msgid_list = decode_msgid_list(item);
  etpan_message_header_set_references(header, msgid_list);
  etpan_message_id_list_free(msgid_list);
  
  item = etpan_serialize_hash_get(hash, "inreplyto");
  msgid_list = decode_msgid_list(item);
  etpan_message_header_set_inreplyto(header, msgid_list);
  etpan_message_id_list_free(msgid_list);
  
  msgid = etpan_serialize_hash_get_str(hash, "msgid");
  etpan_message_header_set_msgid(header, msgid);

  resentmsgid = etpan_serialize_hash_get_str(hash, "resentmsgid");
  etpan_message_header_set_resent_msgid(header, resentmsgid);
  
  resentdate = etpan_serialize_hash_get_int64(hash, "resentdate");
  etpan_message_header_set_resent_date(header, resentdate);
  
  item = etpan_serialize_hash_get(hash, "resentfrom");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_resent_from(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "resentto");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_resent_to(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "resentcc");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_resent_cc(header, addr_list);
  etpan_address_list_free(addr_list);
  
  item = etpan_serialize_hash_get(hash, "resentbcc");
  addr_list = decode_addr_list(item);
  etpan_message_header_set_resent_bcc(header, addr_list);
  etpan_address_list_free(addr_list);
  
  return header;
}

void etpan_message_header_serialize(struct etpan_message_header * header,
    void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * hash;
  void * data;
  size_t length;
  
  hash = encode_header(header);
  
  etpan_serialize_encode(hash, &data, &length);
  
  etpan_serialize_data_free(hash);
  
  * p_data = data;
  * p_length = length;
}

struct etpan_message_header *
etpan_message_header_unserialize(void * data, size_t length)
{
  struct etpan_serialize_data * hash;
  struct etpan_message_header * header;
  
  hash = etpan_serialize_decode(data, length);
  
  header = decode_header(hash);
  
  etpan_serialize_data_free(hash);
  
  return header;
}
