#include "etpan-message-lep.h"

#include <stdlib.h>

#include "etpan-lep.h"
#include "etpan-message-header.h"
#include "etpan-address.h"
#include "etpan-error.h"
#include "etpan-part-lep.h"
#include "etpan-nls.h"
#include "etpan-folder.h"

#define ETPAN_MODULE_LOG_NAME "LEP"
#include "etpan-log.h"

enum {
  ERROR_DOMAIN_FETCH,
};

static struct etpan_error * rewrite_error(struct etpan_message * msg,
    int domain,
    struct etpan_error * error);

static struct etpan_message_header * get_header(struct etpan_message * msg);

static struct etpan_error * check(struct etpan_message * msg,
    struct etpan_message_flags * flags);

static struct etpan_error * fetch_part_list(struct etpan_message * msg,
    struct etpan_part ** p_main_part);

static void clear_part_list(struct etpan_message * msg);

static void free_data(struct etpan_message * msg);

static struct etpan_message_driver lep_driver = {
  .name = "libetpan",
  
  .get_header = get_header,
  
  .check = check,
  
  .fetch_part_list = fetch_part_list,
  .clear_part_list = clear_part_list,
  
  .free_data = free_data,
};



struct message_data {
  struct mailmessage * ep_msg;
  struct etpan_part * current_part;
  int part_ref;
};


static struct etpan_message_flags * get_flags(struct etpan_message * msg,
    struct mailmessage * ep_msg);

struct etpan_message * etpan_message_lep_new(void)
{
  struct etpan_message * msg;
  struct message_data * data;
  
  msg = etpan_message_new();
  if (msg == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->ep_msg = NULL;
  data->current_part = NULL;
  data->part_ref = 0;
  
  etpan_message_set_data(msg, data);
  etpan_message_set_driver(msg, &lep_driver);
  
  return msg;
}

void etpan_message_lep_set_msg(struct etpan_message * msg,
    struct mailmessage * ep_msg)
{
  struct message_data * data;
  struct mailmessage * old_ep_msg;
  struct etpan_message_flags * flags;
  char * uid;
  char * folder_uid;
  size_t len;
  
  mailmessage_resolve_single_fields(ep_msg);
  data = etpan_message_get_data(msg);
  
  old_ep_msg = data->ep_msg;
  
  folder_uid = etpan_folder_get_uid(msg->folder);
  len = strlen(ep_msg->msg_uid) + strlen(folder_uid) + 2;
  uid = malloc(len);
  if (uid == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  snprintf(uid, len, "%s-%s", folder_uid, ep_msg->msg_uid);
  uid[len - 1] = '\0';
  etpan_message_set_uid(msg, uid);
  free(uid);
  
  flags = get_flags(msg, ep_msg);
  
  etpan_message_set_flags(msg, flags);
  etpan_message_flags_free(flags);
  
  if (old_ep_msg != NULL) { 
    mailmessage_free(old_ep_msg);
    ETPAN_CRASH("replace old message");
  }
  
  data->ep_msg = ep_msg;
}

struct mailmessage * etpan_message_lep_get_msg(struct etpan_message * msg)
{
  struct message_data * data;
  struct mailmessage * ep_msg;
  struct mailfolder * lep_folder;
  
  data = etpan_message_get_data(msg);
  ep_msg = data->ep_msg;
  lep_folder = etpan_lep_folder_from_folder(msg->folder);
  ep_msg->msg_folder = lep_folder;
  if (lep_folder == NULL)
    ep_msg->msg_session = NULL;
  else
    ep_msg->msg_session = lep_folder->fld_session;
  
  if (ep_msg->msg_driver == imap_cached_message_driver) {
    mailmessage * ancestor_msg;
    
    ancestor_msg = ep_msg->msg_data;
    
    if ((lep_folder == NULL) || (lep_folder->fld_session == NULL)) {
      ancestor_msg->msg_session = NULL;
    }
    else {
      struct imap_cached_session_state_data * imap_cached_data;
      
      imap_cached_data = lep_folder->fld_session->sess_data;
      ancestor_msg->msg_session = imap_cached_data->imap_ancestor;
    }
  }
  
  return ep_msg;
}

static struct etpan_message_header * get_header(struct etpan_message * msg)
{
  struct etpan_message_header * header;
  struct mailmessage * ep_msg;
  
  ep_msg = etpan_message_lep_get_msg(msg);
  
  if (ep_msg->msg_fields == NULL)
    header = etpan_message_header_new();
  else
    header = etpan_header_from_lep_header(ep_msg->msg_fields);
  
  return header;
}

static struct etpan_error * check(struct etpan_message * msg,
    struct etpan_message_flags * flags)
{
  struct mail_flags * ep_flags;
  struct mailmessage * ep_msg;
  int r;
  
  ep_msg = etpan_message_lep_get_msg(msg);
  ep_flags = etpan_lep_flags_to_lep(flags);
  
  mail_flags_free(ep_msg->msg_flags);
  ep_msg->msg_flags = ep_flags;
  
  r = mailmessage_check(ep_msg);
  if (r != MAIL_NO_ERROR) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CHECK);
    etpan_error_set_short_description(error, _("Could not store flags"));
    etpan_error_strf_long_description(error,
        _("An error occurred on the following message:\n"
            "%s"
            "The flags of the message could not be stored."),
        etpan_message_get_description(msg));
    return error;
  }
  
  return NULL;
}

static struct etpan_message_flags * get_flags(struct etpan_message * msg,
    struct mailmessage * ep_msg)
{
  struct mail_flags * ep_flags;
  struct etpan_message_flags * flags;
  (void) msg;
  
  ep_flags = ep_msg->msg_flags;
  flags = etpan_lep_flags_from_lep(ep_flags);
  
  return flags;
}
 
static struct etpan_part * recursive_mime_from_lep(struct mailmime * mime,
    char * uid, struct etpan_message * msg);
 
static struct etpan_error * fetch_part_list(struct etpan_message * msg,
      struct etpan_part ** p_main_part)
{
  struct etpan_error * error;
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  if (data->current_part == NULL) {
    struct mailmessage * ep_msg;
    struct etpan_part * part;
    
    ep_msg = etpan_message_lep_get_msg(msg);
    
    ETPAN_LOCAL_LOG("fetch message body %p", ep_msg);
    
    error = etpan_fetch_bodystructure(ep_msg);
    if (error != NULL) {
      struct etpan_error * new_error;
      
      new_error = rewrite_error(msg, ERROR_DOMAIN_FETCH, error);
      ETPAN_ERROR_FREE(error);
      error = new_error;
      
      return error;
    }
    
    part = recursive_mime_from_lep(ep_msg->msg_mime, "part", msg);
    
    data->current_part = part;
    etpan_part_ref(data->current_part);
  }
  else {
    etpan_part_ref(data->current_part);
  }
  data->part_ref ++;
  {
    struct mailmessage * ep_msg;
    
    ep_msg = etpan_message_lep_get_msg(msg);
    ETPAN_LOCAL_LOG("get message part list %p %p %p %i", msg, ep_msg, data->current_part, data->part_ref);
  }
  
  * p_main_part = data->current_part;

  return NULL;
}

static void clear_part_list(struct etpan_message * msg)
{
  struct mailmessage * ep_msg;
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  data->part_ref --;
  
  ep_msg = etpan_message_lep_get_msg(msg);
  ETPAN_LOCAL_LOG("flush message %p %p %p %i", msg, ep_msg, data->current_part, data->part_ref);
  if (data->part_ref != 0)
    return;
  
  etpan_part_unref(data->current_part);
  data->current_part = NULL;
  
  if (ep_msg->msg_mime != NULL)
    etpan_flush_bodystructure(ep_msg);
}

static void free_data(struct etpan_message * msg)
{
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  
  if (data->ep_msg != NULL) {
    if (data->ep_msg->msg_mime != NULL) {
      ETPAN_WARN_LOG("WARNING ! flush message should not be necessary here");
      etpan_log_stack();
      etpan_flush_bodystructure(data->ep_msg);
    }
    mailmessage_free(data->ep_msg);
  }
  
  free(data);
}


static struct etpan_part * recursive_mime_from_lep(struct mailmime * mime,
    char * uid, struct etpan_message * msg)
{
  struct etpan_part * part;
  
  part = etpan_part_lep_new();
  if (part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_part_set_message(part, msg);
  etpan_part_lep_set_mime(part, mime);
  
  etpan_part_set_uid(part, uid);
  
  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    etpan_part_set_type(part, ETPAN_PART_TYPE_SINGLE);
    break;
    
  case MAILMIME_MULTIPLE:
    {
      clistiter * cur;
      unsigned int i;
      
      etpan_part_set_type(part, ETPAN_PART_TYPE_MULTIPLE);
      
      i = 1;
      for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
          cur != NULL ; cur = clist_next(cur)) {
        struct etpan_part * child;
        struct mailmime * lep_child;
        char * sub_uid;
        int sub_uid_size;
        
        lep_child = clist_content(cur);
        
        sub_uid_size = strlen(uid) + 20;
        sub_uid = malloc(sub_uid_size);
        if (sub_uid == NULL)
          ETPAN_LOG_MEMORY_ERROR;
        
        snprintf(sub_uid, sub_uid_size, "%s.%u", uid, i);
        i ++;
        
        child = recursive_mime_from_lep(lep_child, sub_uid, msg);
        
        etpan_part_add_child(part, child);
        
        etpan_part_unref(child);
        
        free(sub_uid);
      }
    }
    break;
    
  case MAILMIME_MESSAGE:
    {
      struct etpan_part * child;
      struct mailmime * lep_child;
      char * sub_uid;
      int sub_uid_size;
      
      etpan_part_set_type(part, ETPAN_PART_TYPE_MESSAGE);
      
      sub_uid_size = strlen(uid) + 20;
      sub_uid = malloc(sub_uid_size);
      if (uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      snprintf(sub_uid, sub_uid_size, "%s.1", uid);
      
      lep_child = mime->mm_data.mm_message.mm_msg_mime;
      child = recursive_mime_from_lep(lep_child, sub_uid, msg);
      
      etpan_part_add_child(part, child);
      
      etpan_part_unref(child);
      
      free(sub_uid);
    }
    break;
    
  default:
    ETPAN_WARN_LOG("Invalid MIME part type");
    etpan_crash();
    break;
  }
  
  return part;
}


static struct etpan_error * rewrite_error(struct etpan_message * msg,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while gettting content of the following message:\n"
            "%s"
            "%s"),
        etpan_message_get_description(msg),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with the following message:\n"
            "%s"
            "%s"),
        etpan_message_get_description(msg),
        previous_long_description);
    break;
  }
  
  return new_error;
}
