#include "etpan-message-tree-lep.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-message-lep.h"
#include "etpan-message.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-address.h"
#include "etpan-message-header.h"

#define DEFAULT_INCOMING_CHARSET "iso-8859-1"

struct etpan_message_tree * etpan_message_tree_new(void)
{
  struct etpan_message_tree * tree;
  
  tree = malloc(sizeof(* tree));
  if (tree == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tree->msg = NULL;
  tree->children = NULL;
  tree->parent = NULL;
  
  return tree;
}

void etpan_message_tree_free(struct etpan_message_tree * tree)
{
  if (tree->children != NULL)
    carray_free(tree->children);
  if (tree->msg != NULL)
    etpan_message_unref(tree->msg);
  
  free(tree);
}

void etpan_message_tree_free_recursive(struct etpan_message_tree * tree)
{
  carray * children;
  
  children = etpan_message_tree_get_children(tree);
  if (children != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(children, i);
      etpan_message_tree_free_recursive(child);
    }
  }
  
  etpan_message_tree_free(tree);
}

void etpan_message_tree_set_message(struct etpan_message_tree * tree,
    struct etpan_message * msg)
{
  if (tree->msg != NULL)
    etpan_message_unref(tree->msg);
  etpan_message_ref(msg);
  tree->msg = msg;
}

void etpan_message_tree_add_child(struct etpan_message_tree * parent,
    struct etpan_message_tree * child)
{
  int r;
  
  if (parent->children == NULL) {
    parent->children = carray_new(4);
    if (parent->children == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  child->parent = parent;
  
  r = carray_add(parent->children, child, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

carray * etpan_message_tree_get_children(struct etpan_message_tree * parent)
{
  return parent->children;
}

/* XXX - this should be skip_older */
static void recursive_build_msg_tree(struct etpan_message_tree * parent,
    chash * msg_hash,
    chash * lep_hash,
    struct mailmessage_tree * lep_msg_tree,
    int skip_first)
{
  unsigned int i;
  carray * children;
  int first;
  unsigned int start_index;
  int r;
  
  first = 1;
  
  start_index = 0;
  if (skip_first)
    start_index = 1;
  
  children = lep_msg_tree->node_children;
  for(i = start_index ; i < carray_count(children) ; i ++) {
    struct mailmessage_tree * child;
    struct mailmessage * lep_msg;
    int skip_first_child;
    struct mailmessage_tree * new_child;
    
    child = carray_get(children, i);
    
    new_child = NULL;
    skip_first_child = 0;
    lep_msg = child->node_msg;
    if (lep_msg == NULL) {
      if (carray_count(child->node_children) > 0) {
        struct mailmessage_tree * first_child;
        
        first_child = carray_get(child->node_children, 0);
        lep_msg = first_child->node_msg;
        skip_first_child = 1;
        new_child = first_child;
      }
    }
    
    if (lep_msg != NULL) {
      struct etpan_message * msg;
      chashdatum key;
      chashdatum value;
      
      msg = NULL;
      
      key.data = &lep_msg;
      key.len = sizeof(lep_msg);
      r = chash_get(lep_hash, &key, &value);
      if (r == 0)
        msg = value.data;
      
      if (msg != NULL) {
        struct etpan_message_tree * child_tree;
        
        child_tree = etpan_message_tree_new();
        etpan_message_tree_set_message(child_tree, msg);
        
        etpan_message_tree_add_child(parent, child_tree);
        
        if (skip_first_child) {
          recursive_build_msg_tree(child_tree, msg_hash,
              lep_hash,
              new_child, 0);
        }
        
        recursive_build_msg_tree(child_tree, msg_hash,
            lep_hash,
            child, skip_first_child);
      }
    }
  }
}

#if 0
static inline time_t tree_get_date(struct mailmessage_tree * tree)
{
  time_t youngest;
  int r;
  chashdatum key;
  chashdatum value;
  unsigned int i;
  
  youngest = -1;
  
  key.data = &tree;
  key.len = sizeof(tree);
  r = chash_get(threadsort_hash, &key, &value);
  if (r == 0) {
    memcpy(&youngest, value.data, sizeof(youngest));
    return youngest;
  }
  
  if (tree->node_msg != NULL)
    youngest = tree->node_date;
  else
    youngest = (time_t) -1;
  
  for(i = 0 ; i < carray_count(tree->node_children) ; i ++) {
    struct mailmessage_tree * node;
    time_t timestamp;
    
    node = carray_get(tree->node_children, i);
    timestamp = tree_get_date(node);
    if (youngest == (time_t) -1) {
      youngest = timestamp;
    }
    else if (timestamp != (time_t) -1) {
      if (timestamp > youngest)
        youngest = timestamp;
    }
  }
  
  value.data = &youngest;
  value.len = sizeof(youngest);
  chash_set(threadsort_hash, &key, &value, NULL);
  
  return youngest;
}

static inline uint32_t tree_get_index(struct mailmessage_tree * tree)
{
  if (tree->node_msg == NULL)
    return 0;

  return tree->node_msg->msg_index;
}

static inline int etpan_timecomp(struct mailmessage_tree ** ptree1,
    struct mailmessage_tree ** ptree2)
{
  time_t date1;
  time_t date2;

  date1 = tree_get_date(* ptree1);
  date2 = tree_get_date(* ptree2);
  
  if ((date1 == (time_t) -1) && (date2 == (time_t) -1)) {
    uint32_t index1;
    uint32_t index2;

    index1 = tree_get_index(* ptree1);
    index2 = tree_get_index(* ptree2);
    return (int) ((long) index1 - (long) index2);
  }
  else if (date1 == (time_t) -1) {
    return -1;
  }
  else if (date2 == (time_t) -1) {
    return 1;
  }
  
  return (int) ((long) date1 - (long) date2);
}

static inline int etpan_reverse_timecomp(struct mailmessage_tree ** ptree1,
    struct mailmessage_tree ** ptree2)
{
  return - etpan_timecomp(ptree1, ptree2);
}
#endif

static pthread_mutex_t threadsort_lock = PTHREAD_MUTEX_INITIALIZER;
static chash * threadsort_hash = NULL;

static inline time_t tree_get_date(struct etpan_message_tree * tree)
{
  time_t youngest;
  int r;
  chashdatum key;
  chashdatum value;
  unsigned int i;
  struct etpan_message * msg;
  carray * children;
  
  youngest = -1;
  
  key.data = &tree;
  key.len = sizeof(tree);
  r = chash_get(threadsort_hash, &key, &value);
  if (r == 0) {
    memcpy(&youngest, value.data, sizeof(youngest));
    return youngest;
  }
  
  youngest = (time_t) -1;
  msg = etpan_message_tree_get_message(tree);
  if (msg != NULL) {
    struct etpan_message_header * header;
    
    header = etpan_message_get_header(msg);
    if (header != NULL) {
      youngest = etpan_message_header_get_date(header);
    }
  }
  
  children = etpan_message_tree_get_children(tree);
  if (children != NULL) {
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_message_tree * child;
      time_t timestamp;
    
      child = carray_get(children, i);
      timestamp = tree_get_date(child);
      if (youngest == (time_t) -1) {
        youngest = timestamp;
      }
      else if (timestamp != (time_t) -1) {
        if (timestamp > youngest)
          youngest = timestamp;
      }
    }
  }
  
  value.data = &youngest;
  value.len = sizeof(youngest);
  r = chash_set(threadsort_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  return youngest;
}

static inline int etpan_timecomp(struct etpan_message_tree ** ptree1,
    struct etpan_message_tree ** ptree2)
{
  time_t date1;
  time_t date2;

  date1 = tree_get_date(* ptree1);
  date2 = tree_get_date(* ptree2);
  
  if ((date1 == (time_t) -1) && (date2 == (time_t) -1)) {
    return 0;
  }
  else if (date1 == (time_t) -1) {
    return -1;
  }
  else if (date2 == (time_t) -1) {
    return 1;
  }
  
  return (int) ((long) date1 - (long) date2);
}

static int etpan_reverse_timecomp(struct etpan_message_tree ** ptree1,
    struct etpan_message_tree ** ptree2)
{
  return - etpan_timecomp(ptree1, ptree2);
}

static void recursive_sort_msg_tree(struct etpan_message_tree * tree)
{
  carray * children;
  unsigned int i;
  
  children = etpan_message_tree_get_children(tree);
  if (children == NULL)
    return;
  
  qsort(carray_data(children), carray_count(children), sizeof(void *),
      (int(*)(const void *, const void *)) etpan_reverse_timecomp);
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_message_tree * child;
    
    child = carray_get(children, i);
    recursive_sort_msg_tree(child);
  }
}

static mailmessage * get_lep_msg(struct etpan_message * msg)
{
  struct mailimf_fields * fields;
  struct mailimf_field * field;
  int r;
  struct etpan_message_header * header;
  mailmessage * lep_msg;
  char * subject;
  char * msgid;
  carray * msgid_tab;
  clist * msgid_list;
  struct mailimf_in_reply_to * inreplyto_field;
  struct mailimf_references * references_field;
  unsigned int i;
  char * uid;
  
  header = etpan_message_get_header(msg);
  lep_msg = mailmessage_new();
  if (lep_msg == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  fields = mailimf_fields_new_empty();
  if (fields == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  lep_msg->msg_fields = fields;
  
  subject = etpan_message_header_get_subject(header);
  if (subject != NULL) {
    struct mailimf_subject * subject_field;
    
    subject = strdup(subject);
    if (subject == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    subject_field = mailimf_subject_new(subject);
    field = mailimf_field_new(MAILIMF_FIELD_SUBJECT,
        NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, subject_field, NULL, NULL, NULL);
    if (field == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimf_fields_add(fields, field);
    if (r != MAILIMF_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  msgid = etpan_message_header_get_msgid(header);
  if (msgid != NULL) {
    struct mailimf_message_id * msgid_field;
    
    msgid = strdup(msgid);
    if (msgid == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    msgid_field = mailimf_message_id_new(msgid);
    if (msgid_field == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    field = mailimf_field_new(MAILIMF_FIELD_MESSAGE_ID,
        NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, msgid_field,
        NULL, NULL, NULL, NULL, NULL, NULL);
    if (field == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimf_fields_add(fields, field);
    if (r != MAILIMF_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  msgid_tab = etpan_message_header_get_inreplyto(header);
  msgid_list = clist_new();
  if (msgid_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(msgid_tab) ; i ++) {
    msgid = carray_get(msgid_tab, i);
    msgid = strdup(msgid);
    if (msgid == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = clist_append(msgid_list, msgid);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  inreplyto_field = mailimf_in_reply_to_new(msgid_list);
  if (inreplyto_field == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  field = mailimf_field_new(MAILIMF_FIELD_IN_REPLY_TO,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      inreplyto_field, NULL, NULL, NULL, NULL, NULL);
  if (field == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = mailimf_fields_add(fields, field);
  if (r != MAILIMF_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  msgid_tab = etpan_message_header_get_references(header);
  msgid_list = clist_new();
  if (msgid_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(msgid_tab) ; i ++) {
    msgid = carray_get(msgid_tab, i);
    msgid = strdup(msgid);
    if (msgid == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = clist_append(msgid_list, msgid);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  references_field = mailimf_references_new(msgid_list);
  if (references_field == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  field = mailimf_field_new(MAILIMF_FIELD_REFERENCES,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, references_field, NULL, NULL, NULL, NULL);
  if (field == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = mailimf_fields_add(fields, field);
  if (r != MAILIMF_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  uid = etpan_message_get_uid(msg);
  if (uid != NULL) {
    uid = strdup(uid);
    if (uid == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  lep_msg->msg_uid = uid;
  
  return lep_msg;
}

static struct etpan_message_tree * build_msg_list(chash * msg_hash)
{
  int r;
  struct mailmessage_list * lep_msg_list;
  struct mailmessage_tree * lep_msg_tree;
  carray * msg_tab;
  chashiter * iter;
  struct etpan_message_tree * root;
  chash * lep_hash;
  
  msg_tab = carray_new(16);
  if (msg_tab == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  lep_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (lep_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(msg_hash) ; iter != NULL ;
      iter = chash_next(msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    struct mailmessage * lep_msg;
    
    chash_value(iter, &value);
    msg = value.data;
    
#if 0
    lep_msg = etpan_message_lep_get_msg(msg);
#else
    lep_msg = get_lep_msg(msg);
#endif
    r = carray_add(msg_tab, lep_msg, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    key.data = &lep_msg;
    key.len = sizeof(lep_msg);
    value.data = msg;
    value.len = 0;
    r = chash_set(lep_hash, &key, &value, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  lep_msg_list = mailmessage_list_new(msg_tab);
  if (lep_msg_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = mail_build_thread(MAIL_THREAD_REFERENCES_NO_SUBJECT,
      DEFAULT_INCOMING_CHARSET, lep_msg_list, &lep_msg_tree,
      /* etpan_reverse_timecomp */ NULL);
  if (r != MAIL_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  root = etpan_message_tree_new();
  recursive_build_msg_tree(root, msg_hash, lep_hash, lep_msg_tree, 0);

  pthread_mutex_lock(&threadsort_lock);
  /* create global hash, store cache of timestamp of each node */
  threadsort_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (threadsort_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  recursive_sort_msg_tree(root);
  chash_free(threadsort_hash);
  threadsort_hash = NULL;
  pthread_mutex_unlock(&threadsort_lock);
  
#if 0
  carray_set_size(msg_tab, 0);
#endif
  mailmessage_tree_free_recursive(lep_msg_tree);
  mailmessage_list_free(lep_msg_list);
  chash_free(lep_hash);
  
  return root;
}

struct etpan_message_tree * etpan_message_tree_generate(chash * msg_hash)
{
  return build_msg_list(msg_hash);
}

struct etpan_message * etpan_message_tree_get_message(struct etpan_message_tree * tree)
{
  return tree->msg;
}

struct etpan_message_tree * etpan_message_tree_get_parent(struct etpan_message_tree * tree)
{
  return tree->parent;
}

static void show(struct etpan_message_tree * tree)
{
  struct etpan_message * msg;
  struct etpan_message_header * header;
  char * subject;
  carray * from;
  struct etpan_address * addr;
  char * name;
  
  msg = etpan_message_tree_get_message(tree);
  if (msg == NULL)
    return;
  
  header = etpan_message_get_header(msg);
  if (header == NULL)
    return;
  
  from = etpan_message_header_get_from(header);
  addr = carray_get(from, 0);
  name = etpan_address_get_display_name(addr);
  subject = etpan_message_header_get_subject(header);
  ETPAN_LOG("tree : %s %s", name, subject);
}

static struct etpan_message_tree *
fix_tree(struct etpan_message_tree * tree)
{
  struct etpan_message_tree * new_child;
  carray * children;
  unsigned int i;
  
  children = etpan_message_tree_get_children(tree);
  if (children == NULL)
    return NULL;
  
  if (carray_count(children) == 0)
    return NULL;
  
  new_child = carray_get(children, 0);
  
  for(i = 1 ; i < carray_count(children) ; i ++) {
    struct etpan_message_tree * child;
    
    child = carray_get(children, i);
    etpan_message_tree_add_child(new_child, child);
  }
  
  carray_set_size(children, 0);
  carray_free(children);
  tree->children = NULL;
  
  return new_child;
}

/* very inefficient */
void etpan_message_tree_remove_child_list(struct etpan_message_tree * root,
    chash * msg_hash)
{
  carray * tab;
  unsigned int i;
  unsigned int current_index;
  int r;
  
  tab = etpan_message_tree_get_children(root);
  if (tab == NULL)
    return;
  
  current_index = 0;
  for(i = 0 ; i < carray_count(tab) ; i ++) {
    struct etpan_message_tree * child;
    chashdatum key;
    chashdatum value;
    
    child = carray_get(tab, i);
    
    key.data = &child;
    key.len = sizeof(child);
    
    r = chash_get(msg_hash, &key, &value);
    if (r < 0) {
      carray_set(tab, current_index, child);
      current_index ++;
      
      etpan_message_tree_remove_child_list(child, msg_hash);
    }
    else {
      struct etpan_message_tree * new_child;
      
      etpan_message_tree_remove_child_list(child, msg_hash);
      
      new_child = fix_tree(child);
      if (new_child != NULL) {
        new_child->parent = root;
        carray_set(tab, current_index, new_child);
        current_index ++;
      }
      
      /* free */
      etpan_message_tree_free_recursive(child);
    }
  }
  
  r = carray_set_size(tab, current_index);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}
