#include "etpan-message.h"
#include "etpan-message-private.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-folder-private.h"
#include "etpan-part.h"
#include "etpan-thread-manager-app.h"
#include "etpan-message-header.h"
#include "etpan-nls.h"
#include "etpan-address.h"
#include "etpan-serialize.h"

#define ETPAN_MODULE_LOG_NAME "MSG"
#include "etpan-log.h"

static struct etpan_error * folder_connect(struct etpan_folder * folder);
static void folder_disconnect(struct etpan_folder * folder);

static int msg_count = 0;

void etpan_message_print_stats(void)
{
  ETPAN_WARN_LOG("messages count : %i", msg_count);
}

struct etpan_message * etpan_message_new(void)
{
  struct etpan_message * message;
  int r;
  
  msg_count ++;
  
  message = malloc(sizeof(* message));
  if (message == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = pthread_mutex_init(&message->lock, NULL);
  if (r != 0) {
    ETPAN_LOG("failed to create mutex (message)");
    etpan_crash();
  }
  
  message->ref_count = 1;
  message->folder = NULL;
  message->uid = NULL;
  message->lost = 0;
  message->part_ref = 0;
  message->main_part = NULL;
  message->part_hash = NULL;
  message->data = NULL;
  message->driver = NULL;
  message->flags = NULL;
  message->original_flags = NULL;
  message->header = NULL;
  message->description = NULL;
  message->filtered = 1;
  
  return message;
}

void etpan_message_ref(struct etpan_message * message)
{
  pthread_mutex_lock(&message->lock);
  message->ref_count ++;
#if 0
  ETPAN_LOG("msg ref %p %i", message, message->ref_count);
  etpan_log_stack();
#endif
  pthread_mutex_unlock(&message->lock);
}

void etpan_message_unref(struct etpan_message * message)
{
  int ref_count;
  
  pthread_mutex_lock(&message->lock);
  message->ref_count --;
  ref_count = message->ref_count;
#if 0
  ETPAN_LOG("msg unref %p %i", message, message->ref_count);
  etpan_log_stack();
#endif
  pthread_mutex_unlock(&message->lock);
  if (ref_count == 0)
    etpan_message_free(message);
}

void etpan_message_free(struct etpan_message * message)
{
  msg_count --;
  
  if (message->driver->free_data != NULL)
    message->driver->free_data(message);
  
  free(message->description);
  
  if (message->header != NULL)
    etpan_message_header_free(message->header);
  if (message->original_flags != NULL)
    etpan_message_flags_free(message->original_flags);
  if (message->flags != NULL)
    etpan_message_flags_free(message->flags);
  
  if (message->part_hash != NULL)
    chash_free(message->part_hash);
  if (message->main_part != NULL)
    etpan_part_unref(message->main_part);
  
  free(message->uid);
  pthread_mutex_destroy(&message->lock);
  
  free(message);
}

void etpan_message_set_data(struct etpan_message * message, void * data)
{
  message->data = data;
}

void * etpan_message_get_data(struct etpan_message * message)
{
  return message->data;
}

void etpan_message_set_driver(struct etpan_message * message,
    struct etpan_message_driver * driver)
{
  message->driver = driver;
}

void etpan_message_set_folder(struct etpan_message * message,
    struct etpan_folder * folder)
{
  message->folder = folder;
}

struct etpan_folder * etpan_message_get_folder(struct etpan_message * message)
{
  return message->folder;
}

void etpan_message_set_uid(struct etpan_message * message,
    char * uid)
{
  if (uid != message->uid) {
    free(message->uid);
    if (uid != NULL) {
      message->uid = strdup(uid);
      if (message->uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      message->uid = NULL;
  }
}

char * etpan_message_get_uid(struct etpan_message * message)
{
  return message->uid;
}

void etpan_message_set_lost(struct etpan_message * message, int lost)
{
  message->lost = lost;
}

int etpan_message_is_lost(struct etpan_message * message)
{
  return message->lost;
}

int etpan_message_is_filtered(struct etpan_message * message)
{
  return message->filtered;
}

void etpan_message_set_filtered(struct etpan_message * message, int value)
{
  message->filtered = value;
}

/* get/set flags */

void etpan_message_set_flags(struct etpan_message * message,
    struct etpan_message_flags * flags)
{
  struct etpan_message_flags * old_flags;
  
  if (message->original_flags == NULL) {
    if (message->flags != NULL) {
      message->original_flags = etpan_message_flags_dup(message->flags);
    }
  }
  
  old_flags = message->flags;
  if (flags != NULL) {
    message->flags = etpan_message_flags_dup(flags);
  }
  else {
    message->flags = NULL;
  }
  
  if (old_flags != NULL)
    etpan_message_flags_free(old_flags);
}

struct etpan_message_flags *
etpan_message_get_flags(struct etpan_message * message)
{
  return message->flags;
}


/* fetch part list */

struct etpan_message_fetch_part_list_param {
  struct etpan_message * message;
};


static void
fetch_part_list_cleanup(struct etpan_thread_op * op)
{
  struct etpan_message_fetch_part_list_param * param;
  struct etpan_message_fetch_part_list_result * result;
  
  param = op->param;
  etpan_message_unref(param->message);
  free(op->param);
  
  result = op->result;
  if (result->part != NULL) 
    etpan_part_unref(result->part);
  ETPAN_ERROR_FREE(result->error);
  free(op->result);
}

static void
fetch_part_list_run(struct etpan_thread_op * op)
{
  struct etpan_message_fetch_part_list_param * param;
  struct etpan_message_fetch_part_list_result * result;
  struct etpan_error * error;
  struct etpan_part * main_part;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->message->folder);
  if (error == NULL) {
    if (param->message->driver->fetch_part_list != NULL) {
      error = param->message->driver->fetch_part_list(param->message,
          &main_part);
      if (error == NULL) {
        result->part = main_part;
      }
    }
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("fetch-part-list is not implemented for %s"),
          param->message->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_message_fetch_part_list(struct etpan_thread_manager_app * manager,
    struct etpan_message * msg,
    void (* callback)(int, struct etpan_message_fetch_part_list_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_message_fetch_part_list_param * param;
  struct etpan_message_fetch_part_list_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->message = msg;
  etpan_message_ref(param->message);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  result->part = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_part_list_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = fetch_part_list_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager,
      msg->folder->storage, op);
  
  return op;
}

static void store_part_hash(struct etpan_message * msg,
    struct etpan_part * part)
{
  char * uid;
  chashdatum key;
  chashdatum value;
  int r;
  unsigned int i;
  
  uid = etpan_part_get_uid(part);
  key.data = uid;
  key.len = strlen(uid + 1);
  value.data = part;
  value.len = 0;
  r = chash_set(msg->part_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(part->children) ; i ++) {
    struct etpan_part * subpart;
    
    subpart = carray_get(part->children, i);
    store_part_hash(msg, subpart);
  }
}

void etpan_message_set_main_part(struct etpan_message * msg,
    struct etpan_part * main_part)
{
  if (msg->main_part == main_part) {
    msg->part_ref ++;
    return;
  }
  
  msg->part_ref ++;
  
  msg->part_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg->part_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  store_part_hash(msg, main_part);
  etpan_part_ref(main_part);
  msg->main_part = main_part;
}

struct etpan_message_clear_part_list_param {
  struct etpan_message * message;
};

static void
clear_part_list_cleanup(struct etpan_thread_op * op)
{
  struct etpan_message_clear_part_list_param * param;
  
  if (op->param != NULL) {
    param = op->param;
    etpan_message_unref(param->message);
    
    free(op->param);
  }
}

static void
clear_part_list_run(struct etpan_thread_op * op)
{
  struct etpan_message_clear_part_list_param * param;
  struct etpan_message * msg;
  
  param = op->param;
  
  msg = param->message;
  
  if (msg->driver->clear_part_list != NULL)
    msg->driver->clear_part_list(msg);
}


struct etpan_thread_op *
etpan_message_clear_part_list(struct etpan_thread_manager_app * manager,
    struct etpan_message * msg)
{
  struct etpan_thread_op * op;
  struct etpan_message_clear_part_list_param * param;
  
  if (msg->part_ref == 0) {
    ETPAN_WARN_LOG("WARNING! clean part list called to much times");
    return NULL;
  }
  
  msg->part_ref --;
  
  if (msg->part_ref == 0) {
    if (msg->part_hash != NULL) {
      chash_free(msg->part_hash);
      msg->part_hash = NULL;
    }
  
    etpan_part_unref(msg->main_part);
    msg->main_part = NULL;
  }
  
  etpan_message_ref(msg);
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  param->message = msg;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = clear_part_list_run;
  op->callback = NULL;
  op->callback_data = NULL;
  op->cleanup = clear_part_list_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager,
      msg->folder->storage, op);
  
  return op;
}

struct etpan_part *
etpan_message_get_main_part(struct etpan_message * msg)
{
  return msg->main_part;
}

struct etpan_part *
etpan_message_get_part(struct etpan_message * msg, char * uid)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  if (msg->part_hash == NULL)
    return NULL;
  
  key.data = uid;
  key.len = strlen(uid + 1);
  
  r = chash_get(msg->part_hash, &key, &value);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value.data;
}

struct etpan_message_header *
etpan_message_get_header(struct etpan_message * msg)
{
  if (msg->header == NULL) {
    struct etpan_message_header * header;
    
    if (msg->driver->get_header == NULL)
      return NULL;
    
    header = msg->driver->get_header(msg);
    msg->header = header;
  }
  
  return msg->header;
}

void etpan_message_set_header(struct etpan_message * msg,
    struct etpan_message_header * header)
{
  if (msg->header != NULL) {
    etpan_message_header_free(msg->header);
  }
  
  msg->header = header;
}

/* flags */

struct etpan_message_flags * etpan_message_flags_new(void)
{
  struct etpan_message_flags * flags;
  
  flags = malloc(sizeof(* flags));
  if (flags == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  flags->value = 0;
  flags->ext = carray_new(4);
  if (flags->ext == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return flags;
}

void etpan_message_flags_free(struct etpan_message_flags * flags)
{
  unsigned int i;
    
  for(i = 0 ; i < carray_count(flags->ext) ; i ++) {
    char * str;
    
    str = carray_get(flags->ext, i);
    free(str);
  }
  
  carray_free(flags->ext);
  
  free(flags);
}

void etpan_message_flags_set_value(struct etpan_message_flags * flags,
    int value)
{
  flags->value = value;
}

int etpan_message_flags_get_value(struct etpan_message_flags * flags)
{
  return flags->value;
}

struct etpan_message_flags *
etpan_message_flags_dup(struct etpan_message_flags * flags)
{
  struct etpan_message_flags * new_flags;
  int value;
  unsigned int i;
  
  new_flags = etpan_message_flags_new();
  value = etpan_message_flags_get_value(flags);
  etpan_message_flags_set_value(new_flags, value);
  
  for(i = 0 ; i < carray_count(flags->ext) ; i ++) {
    char * str;
    
    str = carray_get(flags->ext, i);
    etpan_message_flags_add_ext(new_flags, str);
  }
  
  return new_flags;
}

int etpan_message_flags_has_ext(struct etpan_message_flags * flags,
    char * ext)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(flags->ext) ; i ++) {
    char * str;
    
    str = carray_get(flags->ext, i);
    if (strcmp(str, ext) == 0)
      return 1;
  }
  
  return 0;
}

void etpan_message_flags_add_ext(struct etpan_message_flags * flags,
    char * ext)
{
  char * str;
  int r;
  
  if (etpan_message_flags_has_ext(flags, ext))
    return;
  
  str = strdup(ext);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(flags->ext, str, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_message_flags_remove_ext(struct etpan_message_flags * flags,
    char * ext)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(flags->ext) ; i ++) {
    char * str;
    
    str = carray_get(flags->ext, i);
    if (strcmp(str, ext) == 0) {
      carray_delete(flags->ext, i);
      break;
    }
  }
}

carray * etpan_message_flags_get_ext(struct etpan_message_flags * flags)
{
  return flags->ext;
}

void etpan_message_flags_serialize(struct etpan_message_flags * flags,
    void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * hash;
  carray * ext;
  int flags_value;
  unsigned int i;
  struct etpan_serialize_data * serialized_ext;
  size_t length;
  void * data;
  
  hash = etpan_serialize_data_new_hash();
  flags_value = etpan_message_flags_get_value(flags);
  etpan_serialize_hash_set_int32(hash, "value", flags_value);
  ext = etpan_message_flags_get_ext(flags);
  serialized_ext = etpan_serialize_data_new_array();
  for(i = 0 ; i < carray_count(ext) ; i ++) {
    char * ext_item;
    
    ext_item = carray_get(ext, i);
    etpan_serialize_array_add_str(serialized_ext, ext_item);
  }
  etpan_serialize_hash_set(hash, "ext", serialized_ext);
  
  etpan_serialize_encode(hash, &data, &length);
  
  etpan_serialize_data_free(hash);
  
  * p_data = data;
  * p_length = length;
}

struct etpan_message_flags *
etpan_message_flags_unserialize(void * data, size_t length)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * serialized_ext;
  int flags_value;
  unsigned int count;
  unsigned int i;
  struct etpan_message_flags * flags;
  
  flags = etpan_message_flags_new();
  
  hash =  etpan_serialize_decode(data, length);
  
  flags_value = etpan_serialize_hash_get_int32(hash, "value");
  etpan_message_flags_set_value(flags, flags_value);
  
  serialized_ext = etpan_serialize_hash_get(hash, "ext");
  
  count = etpan_serialize_array_count(serialized_ext);
  for(i = 0 ; i < count ; i ++) {
    char * ext_item;
    
    ext_item = etpan_serialize_array_get_str(serialized_ext, i);
    etpan_message_flags_add_ext(flags, ext_item);
  }
  
  etpan_serialize_data_free(hash);
  
  return flags;
}

static struct etpan_error * folder_connect(struct etpan_folder * folder)
{
  return etpan_folder_connect_nt(folder);
}

static void folder_disconnect(struct etpan_folder * folder)
{
  etpan_folder_disconnect_nt(folder);
}

char * etpan_message_compute_description_string(struct etpan_message_header * msg_header)
{
  char description[1024];
  char * subject;
  char * sender;
  carray * from;
  time_t date;
  char * result;
  char * date_str;
  
  from = etpan_message_header_get_from(msg_header);
  sender = NULL;
  if (from != NULL) {
    if (carray_count(from) > 0) {
      struct etpan_address * addr;
      char * display_name;
      char * email;
      
      addr = carray_get(from, 0);
      
      display_name = etpan_address_get_display_name(addr);
      email = etpan_address_get_address(addr);
      if (display_name != NULL)
        sender = display_name;
      else if (email != NULL)
        sender = email;
    }
  }
  if (sender == NULL)
    sender = "";
  
  subject = etpan_message_header_get_subject(msg_header);
  if (subject == NULL)
    subject = "";
  
  date = etpan_message_header_get_date(msg_header);
  
  date_str = NULL;
  if (date != (time_t) -1) {
    date_str = etpan_message_get_long_date_str(date);
  }
  else {
    date_str = strdup("");
    if (date_str == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  snprintf(description, sizeof(description),
      _("From: %s\nSubject: %s\nDate: %s"),
      sender, subject, date_str);
  free(date_str);
  
  result = strdup(description);
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return result;
}

void etpan_message_compute_description(struct etpan_message * msg)
{
  struct etpan_message_header * msg_header;
  
  msg_header = etpan_message_get_header(msg);
  msg->description = etpan_message_compute_description_string(msg_header);
}

char * etpan_message_get_description(struct etpan_message * msg)
{
  return msg->description;
}

#ifndef WRONG
#define WRONG	(-1)
#endif /* !defined WRONG */

static int tmcomp(struct tm * atmp, struct tm * btmp)
{
  register int	result;
  
  if ((result = (atmp->tm_year - btmp->tm_year)) == 0 &&
      (result = (atmp->tm_mon - btmp->tm_mon)) == 0 &&
      (result = (atmp->tm_mday - btmp->tm_mday)) == 0 &&
      (result = (atmp->tm_hour - btmp->tm_hour)) == 0 &&
      (result = (atmp->tm_min - btmp->tm_min)) == 0)
    result = atmp->tm_sec - btmp->tm_sec;
  return result;
}

static time_t mkgmtime(struct tm * tmp)
{
  register int			dir;
  register int			bits;
  register int			saved_seconds;
  time_t				t;
  struct tm			yourtm, mytm;
  
  yourtm = *tmp;
  saved_seconds = yourtm.tm_sec;
  yourtm.tm_sec = 0;
  /*
  ** Calculate the number of magnitude bits in a time_t
  ** (this works regardless of whether time_t is
  ** signed or unsigned, though lint complains if unsigned).
  */
  for (bits = 0, t = 1; t > 0; ++bits, t <<= 1)
    ;
  /*
  ** If time_t is signed, then 0 is the median value,
  ** if time_t is unsigned, then 1 << bits is median.
  */
  t = (t < 0) ? 0 : ((time_t) 1 << bits);
  for ( ; ; ) {
    gmtime_r(&t, &mytm);
    dir = tmcomp(&mytm, &yourtm);
    if (dir != 0) {
      if (bits-- < 0)
	return WRONG;
      if (bits < 0)
	--t;
      else if (dir > 0)
	t -= (time_t) 1 << bits;
      else	t += (time_t) 1 << bits;
      continue;
    }
    break;
  }
  t += saved_seconds;
  return t;
}

static int get_age(time_t timestamp, time_t current_time)
{
  time_t today;
  struct tm today_date_time;
  struct tm date_time;
  time_t today_gm;
  time_t time_gm;
  
  today = current_time;
  
  if (today < timestamp)
    return -1;
  
  if (localtime_r(&today, &today_date_time) == NULL)
    return -1;
  today_date_time.tm_hour = 0;
  today_date_time.tm_min = 0;
  today_date_time.tm_sec = 0;
  
  if (localtime_r(&timestamp, &date_time) == NULL)
    return -1;
  date_time.tm_hour = 0;
  date_time.tm_min = 0;
  date_time.tm_sec = 0;
  
  today_gm = mkgmtime(&today_date_time);
  time_gm = mkgmtime(&date_time);
  
  return (today_gm - time_gm) / (24 * 60 * 60);
}

char * etpan_message_get_short_date_str(time_t timestamp, time_t current_time)
{
  struct tm date_time;
  char date_str[256];
  int age;
  char * day_name;
  char * day_name_list[7];
  char * str;
  
  day_name_list[0] = _("Sunday");
  day_name_list[1] = _("Monday");
  day_name_list[2] = _("Tuesday");
  day_name_list[3] = _("Wednesday");
  day_name_list[4] = _("Thursday");
  day_name_list[5] = _("Friday");
  day_name_list[6] = _("Saturday");
  
  date_str[0] = '\0';
  if (localtime_r(&timestamp, &date_time) == NULL)
    goto err;
  
  age = get_age(timestamp, current_time);
  
  switch (age) {
  case 0:
    strftime(date_str, sizeof(date_str), _("%H:%M"), &date_time);
    break;
    
  case 1:
    snprintf(date_str, sizeof(date_str), _("Yesterday"));
    break;
    
  case 2:
  case 3:
  case 4:
  case 5:
  case 6:
    day_name = day_name_list[date_time.tm_wday];
    snprintf(date_str, sizeof(date_str), "%s", day_name);
    break;
    
  default:
    {
      struct tm today_date_time;
      int show_year;
      
      show_year = 0;
      if (localtime_r(&current_time, &today_date_time) != NULL) {
        if (localtime_r(&timestamp, &date_time) != NULL) {
          if (today_date_time.tm_year != date_time.tm_year)
            show_year = 1;
        }
      }
      
      if (show_year)
        strftime(date_str, sizeof(date_str), _("%d/%m/%Y"), &date_time);
      else
        strftime(date_str, sizeof(date_str), _("%d/%m"), &date_time);
    }
  }
  
 err:
  str = strdup(date_str);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return str;
}
    
char * etpan_message_get_long_date_str(time_t timestamp)
{
  struct tm date_time;
  char date_str[256];
  char * str;
  
  date_str[0] = '\0';
  if (localtime_r(&timestamp, &date_time) != NULL) {
    strftime(date_str, sizeof(date_str), "%d/%m/%Y %H:%M", &date_time);
  }
  
  str = strdup(date_str);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return str;
}

void etpan_message_set_clean(struct etpan_message * msg)
{
  etpan_message_flags_free(msg->original_flags);
  msg->original_flags = NULL;
}

int etpan_message_has_dirty_flags(struct etpan_message * msg)
{
  int original_value;
  int value;
  carray * original_ext;
  carray * ext;
  unsigned int i;
  
  if (msg->original_flags == NULL)
    return 0;
  
  original_value = etpan_message_flags_get_value(msg->original_flags);
  value = etpan_message_flags_get_value(msg->flags);
  
  if (value != original_value)
    return 1;
  
  original_ext = etpan_message_flags_get_ext(msg->original_flags);
  for(i = 0 ; i < carray_count(original_ext) ; i ++) {
    char * ext_value;
    
    ext_value = carray_get(original_ext, i);
    if (!etpan_message_flags_has_ext(msg->flags, ext_value))
      return 1;
  }

  ext = etpan_message_flags_get_ext(msg->original_flags);
  for(i = 0 ; i < carray_count(ext) ; i ++) {
    char * ext_value;
    
    ext_value = carray_get(ext, i);
    if (!etpan_message_flags_has_ext(msg->original_flags, ext_value))
      return 1;
  }
  
  return 0;
}

struct etpan_error * etpan_message_check(struct etpan_message * msg,
    struct etpan_message_flags * flags)
{
  if (msg->driver->check == NULL) {
    struct etpan_error * error;
    char long_description[1024];
    
    snprintf(long_description, sizeof(long_description),
        _("check is not implemented for %s"),
        msg->driver->name);
    
    error = etpan_error_internal(long_description);
    
    return error;
  }
  
  return msg->driver->check(msg, flags);
}

int etpan_message_flags_equal(struct etpan_message_flags * flags,
    struct etpan_message_flags * other)
{
  carray * ext;
  carray * other_ext;
  unsigned int i;
  
  if (etpan_message_flags_get_value(flags) != etpan_message_flags_get_value(other))
    return 0;
  
  ext = etpan_message_flags_get_ext(flags);
  other_ext = etpan_message_flags_get_ext(flags);
  if (carray_count(ext) != carray_count(other_ext))
    return 0;
  
  for(i = 0 ; i < carray_count(ext) ; i ++) {
    char * str;
    
    str = carray_get(ext, i);
    if (!etpan_message_flags_has_ext(other, str))
      return 0;
  }
  
  return 1;
}

