#include "etpan-msg-list-fetch.h"

#include <stdlib.h>

#include "etpan-folder.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-thread-manager-app.h"
#include "etpan-nls.h"
#include "etpan-signal.h"
#include "etpan-folder-filter.h"
#include "etpan-message.h"

static void unsetup(struct etpan_msg_list_fetch * fetcher);
static struct etpan_error *
rewrite_error(struct etpan_msg_list_fetch * fetcher,
    struct etpan_error * error);

struct etpan_msg_list_fetch * etpan_msg_list_fetch_new(void)
{
  struct etpan_msg_list_fetch * fetcher;
  
  fetcher = malloc(sizeof(* fetcher));
  if (fetcher == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetcher->ref_count = 1;
  fetcher->folder = NULL;
  fetcher->op = NULL;
  fetcher->error = NULL;
  fetcher->filter = NULL;
  
  return fetcher;
}

static void etpan_msg_list_fetch_free(struct etpan_msg_list_fetch * fetcher)
{
  unsetup(fetcher);
  
  if (fetcher->filter != NULL) {
    etpan_folder_filter_unref(fetcher->filter);
  }
  if (fetcher->folder != NULL) {
    etpan_folder_unref(fetcher->folder);
  }
  
  ETPAN_ERROR_FREE(fetcher->error);
  free(fetcher);
}

void etpan_msg_list_fetch_ref(struct etpan_msg_list_fetch * fetcher)
{
  fetcher->ref_count ++;
}

void etpan_msg_list_fetch_unref(struct etpan_msg_list_fetch * fetcher)
{
  fetcher->ref_count --;
  if (fetcher->ref_count == 0)
    etpan_msg_list_fetch_free(fetcher);
}

static void unsetup(struct etpan_msg_list_fetch * fetcher)
{
  etpan_msg_list_fetch_cancel(fetcher);
}

void etpan_msg_list_fetch_setup(struct etpan_msg_list_fetch * fetcher)
{
  (void) fetcher;
}

void etpan_msg_list_fetch_set_folder(struct etpan_msg_list_fetch * fetcher,
    struct etpan_folder * folder)
{
  if (folder != fetcher->folder) {
    if (fetcher->folder != NULL)
      etpan_folder_unref(fetcher->folder);
    fetcher->folder = folder;
    if (folder != NULL) {
      etpan_folder_ref(folder);
    }
    else
      fetcher->folder = NULL;
  }
}

struct etpan_folder *
etpan_msg_list_fetch_get_folder(struct etpan_msg_list_fetch * fetcher)
{
  return fetcher->folder;
}

void etpan_msg_list_fetch_set_filter(struct etpan_msg_list_fetch * fetcher,
    struct etpan_folder_filter * filter)
{
  if (filter != fetcher->filter) {
    if (fetcher->filter != NULL)
      etpan_folder_filter_unref(fetcher->filter);
    fetcher->filter = filter;
    if (filter != NULL) {
      etpan_folder_filter_ref(filter);
    }
    else
      fetcher->filter = NULL;
  }
}

struct etpan_folder_filter * etpan_msg_list_fetch_get_filter(struct etpan_msg_list_fetch * fetcher)
{
  return fetcher->filter;
}

void etpan_msg_list_fetch_cancel(struct etpan_msg_list_fetch * fetcher)
{
  if (fetcher->op != NULL)
    etpan_thread_op_cancel(fetcher->op);
}

struct etpan_error *
etpan_msg_list_fetch_get_error(struct etpan_msg_list_fetch * fetcher)
{
  return fetcher->error;
}

static void status(struct etpan_msg_list_fetch * fetcher);
static void got_status(int cancelled,
    struct etpan_folder_status_result * result, void * cb_data);
static void got_msg_list_callback(int cancelled,
    struct etpan_folder_fetch_msg_list_result * result,
    void * callback_data);
static void fetch_msg_list(struct etpan_msg_list_fetch * fetcher);

void etpan_msg_list_fetch_run(struct etpan_msg_list_fetch * fetcher)
{
  if (fetcher->op != NULL) {
    ETPAN_CRASH("already fetching the list of msg for folder %s", etpan_folder_get_ui_path(fetcher->folder));
  }
  
  status(fetcher);
}

static void status(struct etpan_msg_list_fetch * fetcher)
{
  struct etpan_thread_op * op;
  
  op = etpan_folder_status(etpan_thread_manager_app_get_default(),
      fetcher->folder, got_status, fetcher);
  
  fetcher->op = op;
}

static void got_status(int cancelled,
    struct etpan_folder_status_result * result, void * cb_data)
{
  struct etpan_msg_list_fetch * fetcher;
  
  fetcher = cb_data;
  fetcher->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    etpan_folder_unref_msg_list(fetcher->folder);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Update of message list cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled the update of the message list of %s."), etpan_folder_get_ui_path(fetcher->folder));
    
    fetcher->error = error;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL, fetcher, NULL);
    
    return;
  }
  
  if (result->error != NULL) {
    struct etpan_error * error;
    
    etpan_folder_unref_msg_list(fetcher->folder);
    
    error = rewrite_error(fetcher, result->error);
    fetcher->error = error;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL, fetcher, NULL);
    
    return;
  }
  
  etpan_folder_set_count(fetcher->folder,
      result->count, result->unseen, result->recent);
  
  fetch_msg_list(fetcher);
}


static void fetch_msg_list(struct etpan_msg_list_fetch * fetcher)
{
  struct etpan_thread_op * op;
  
  etpan_folder_ref_msg_list(fetcher->folder);
  
  op = etpan_folder_fetch_msg_list(etpan_thread_manager_app_get_default(),
      fetcher->folder, got_msg_list_callback, fetcher);
  
  fetcher->op = op;
}

struct msg_list_data {
  struct etpan_msg_list_fetch * fetcher;
  chash * msg_hash;
};

static void got_filtered_msg_list_callback(int cancelled,
    struct etpan_folder_filter_filtered_msg_list_result * result,
    void * callback_data);

static void got_msg_list_callback(int cancelled,
    struct etpan_folder_fetch_msg_list_result * result,
    void * callback_data)
{
  struct etpan_msg_list_fetch * fetcher;
  struct msg_list_data * msg_list_data;
  
  fetcher = callback_data;

  fetcher->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    etpan_folder_unref_msg_list(fetcher->folder);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Update of message list cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled the update of the message list of %s."), etpan_folder_get_ui_path(fetcher->folder));
    
    fetcher->error = error;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL, fetcher, NULL);
    
    return;
  }
  
  if (result->error != NULL) {
    struct etpan_error * error;
    
    etpan_folder_unref_msg_list(fetcher->folder);
    
    error = rewrite_error(fetcher, result->error);
    fetcher->error = error;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL, fetcher, NULL);
    
    return;
  }
  
  msg_list_data = malloc(sizeof(* msg_list_data));
  if (msg_list_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_list_data->fetcher = fetcher;
  msg_list_data->msg_hash = result->current_message_list;
  
  result->current_message_list = NULL;
  
  /* fetch information to know whether messages were filtered or not */
  if (fetcher->filter != NULL) {
    ETPAN_LOCAL_LOG("has filter");
    etpan_folder_filter_fetch_filtered_message_list(etpan_thread_manager_app_get_default(),
        fetcher->filter,
        got_filtered_msg_list_callback, msg_list_data);
  }
  else {
    ETPAN_LOCAL_LOG("has no filter");
    got_filtered_msg_list_callback(0, NULL, msg_list_data);
  }
}

static void got_filtered_msg_list_callback(int cancelled,
    struct etpan_folder_filter_filtered_msg_list_result * result,
    void * callback_data)
{
  struct msg_list_data * msg_list_data;
  struct etpan_msg_list_fetch * fetcher;
  chashiter * iter;
  int r;
  struct etpan_folder * folder;
  
  msg_list_data = callback_data;
  
  fetcher = msg_list_data->fetcher;
  
  etpan_folder_update_msg_list(fetcher->folder,
      msg_list_data->msg_hash);
  ETPAN_LOCAL_LOG("got msg list %u", chash_count(msg_list_data->msg_hash));
  etpan_log_stack();
  
  /* free message hash */
  for(iter = chash_begin(msg_list_data->msg_hash) ; iter != NULL ;
      iter = chash_next(msg_list_data->msg_hash, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    
    chash_value(iter, &value);
    msg = value.data;
    etpan_message_unref(msg);
  }
  chash_free(msg_list_data->msg_hash);
  free(msg_list_data);
  
  /* update filtered flag */
  if (result != NULL) {
    chash * msg_hash;
    
    ETPAN_LOCAL_LOG("filtered %u", chash_count(result->msg_hash));
    msg_hash = etpan_folder_get_message_hash(fetcher->folder);
    for(iter = chash_begin(msg_hash) ; iter != NULL ;
        iter = chash_next(msg_hash, iter)) {
      chashdatum key;
      chashdatum value;
      struct etpan_message * msg;
      int filtered;
      
      chash_key(iter, &key);
      r = chash_get(result->msg_hash, &key, &value);
      if (r < 0) {
        filtered = 0;
      }
      else {
        filtered = 1;
      }
      
      chash_value(iter, &value);
      msg = value.data;
      etpan_message_set_filtered(msg, filtered);
    }
  }
  
  /* notify this so that the view can remove the message
     that are no more available */
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MSG_LIST_FETCH_FINISHED_WITH_LOST_SIGNAL, fetcher, NULL);
  
  etpan_folder_remove_lost_messages(fetcher->folder);
  
  /* fetcher may be freed after signal send */
  folder = fetcher->folder;
  etpan_folder_ref(folder);
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL, fetcher, NULL);
  /* fetcher is no more reliable from here */
  
  etpan_folder_unref_msg_list(folder);
  etpan_folder_unref(folder);
}

static struct etpan_error *
rewrite_error(struct etpan_msg_list_fetch * fetcher,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred while getting the content of the folder %s.\n"
          "%s"),
      etpan_folder_get_ui_path(fetcher->folder),
      previous_long_description);
  
  return new_error;
}
