#include "etpan-msg-list-view.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-message-tree-lep.h"
#include "etpan-thread-manager-app.h"
#include "etpan-message.h"
#include "etpan-folder.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-nls.h"
#include "etpan-signal.h"

/* XXX - avoid notification when there is no changes */

static void cancel(struct etpan_msg_list_view * view);

static void remove_orphans(struct etpan_message_tree * tree,
    chash * folders_for_orphans);

struct build_param {
  chash * msg_hash;
  chash * folders_for_orphans;
};

struct build_result {
  struct etpan_message_tree * tree;
};

static void
build_cleanup(struct etpan_thread_op * op)
{
  if (op->param) {
    struct build_param * param;
    chashiter * iter;
    chash * msglist;
    
    param = op->param;
    
    msglist = param->msg_hash;
    for(iter = chash_begin(msglist) ; iter != NULL ;
        iter = chash_next(msglist, iter)) {
      chashdatum value;
      struct etpan_message * message;
      
      chash_value(iter, &value);
      message = value.data;
      etpan_message_unref(message);
    }
    chash_free(msglist);
    chash_free(param->folders_for_orphans);
    
    free(op->param);
  }
  
  if (op->result != NULL) {
    struct build_result * result;
    
    result = op->result;
    if (result->tree != NULL) {
      etpan_message_tree_free_recursive(result->tree);
    }
  }
  free(op->result);
}

static void
build_run(struct etpan_thread_op * op)
{
  struct build_param * param;
  struct build_result * result; 
  
  param = op->param;
  result = op->result;
  
  ETPAN_LOG("build %u", chash_count(param->msg_hash));
  result->tree = etpan_message_tree_generate(param->msg_hash);
  remove_orphans(result->tree, param->folders_for_orphans);
}

static struct etpan_thread_op *
build(struct etpan_thread_manager_app * manager,
    chash * current_msg_hash,
    chash * folders_for_orphans,
    void (* callback)(int, struct build_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct build_param * param;
  struct build_result * result;
  int r;
  chashiter * iter;
  chash * msg_hash;
  
  msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(current_msg_hash) ; iter != NULL ;
      iter = chash_next(current_msg_hash, iter)) {
    chashdatum value;
    chashdatum key;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    flags = etpan_message_get_flags(msg);
    if (flags != NULL) {
      int flag_value;
      
      flag_value = etpan_message_flags_get_value(flags);
      if ((flag_value & ETPAN_FLAGS_DELETED) != 0)
        continue;
    }
    
    etpan_message_ref(msg);
    
    r = chash_set(msg_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->msg_hash = msg_hash;
  param->folders_for_orphans = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  for(iter = chash_begin(folders_for_orphans) ; iter != NULL ;
      iter = chash_next(folders_for_orphans, iter)) {
    chashdatum key;
    chashdatum value;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    r = chash_set(param->folders_for_orphans, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  result->tree = NULL;
  
  op = etpan_thread_op_new();
  
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = build_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = build_cleanup;
  
  etpan_thread_manager_app_misc_schedule(manager, op);
  
  return op;
}


struct free_msg_param {
  struct etpan_message_tree * msg_tree;
};

static void
free_msg_cleanup(struct etpan_thread_op * op)
{
  if (op->param) {
    struct free_msg_param * param;
    
    param = op->param;
    free(op->param);
  }
}

static void
free_msg_run(struct etpan_thread_op * op)
{
  struct free_msg_param * param;
  
  param = op->param;
  
  etpan_message_tree_free_recursive(param->msg_tree);
}

static struct etpan_thread_op *
free_msg(struct etpan_thread_manager_app * manager,
    struct etpan_message_tree * msg_tree,
    void (* callback)(int, void *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct free_msg_param * param;
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->msg_tree = msg_tree;
  
  op = etpan_thread_op_new();
  
  op->param = param;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = free_msg_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = free_msg_cleanup;
  
  etpan_thread_manager_app_misc_schedule(manager, op);
  
  return op;
}


struct etpan_msg_list_view *
etpan_msg_list_view_new(void)
{
  struct etpan_msg_list_view * view;
  
  view = malloc(sizeof(* view));
  if (view == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  view->msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (view->msg_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  view->folders_for_orphans = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (view->folders_for_orphans == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  view->schedule_build = 0;
  view->building_tree = 0;
  view->op = NULL;
  
  view->msg_tree = NULL;
  view->new_messages = NULL;
  view->error = NULL;
  view->changed = 0;
  
  return view;
}

void etpan_msg_list_view_free(struct etpan_msg_list_view * view)
{
  ETPAN_ERROR_FREE(view->error);
  view->error = NULL;
  chash_free(view->folders_for_orphans);
  chash_free(view->msg_hash);
  free(view);
}

void etpan_msg_list_view_setup(struct etpan_msg_list_view * view)
{
  (void) view;
}

void etpan_msg_list_view_unsetup(struct etpan_msg_list_view * view)
{
  cancel(view);
  etpan_msg_list_view_clear_messages(view);
}

void etpan_msg_list_view_clear_messages(struct etpan_msg_list_view * view)
{
  chashiter * iter;
  
  if (chash_count(view->msg_hash) == 0)
    return;
  
  for(iter = chash_begin(view->msg_hash) ; iter != NULL ; iter = chash_next(view->msg_hash, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    msg = value.data;
    folder = etpan_message_get_folder(msg);
    if (folder != NULL) {
      etpan_folder_unref_msg_list(folder);
    }
    etpan_message_unref(msg);
  }
  chash_clear(view->msg_hash);
  
  if (view->msg_tree != NULL) {
    free_msg(etpan_thread_manager_app_get_default(),
        view->msg_tree, NULL, NULL);
    view->msg_tree = NULL;
  }
  
  view->changed = 1;
}

void etpan_msg_list_view_add_messages(struct etpan_msg_list_view * view,
    chash * msg_hash)
{
  chashiter * iter;
  int r;
  
  for(iter = chash_begin(msg_hash) ; iter != NULL ; iter = chash_next(msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    
    flags = etpan_message_get_flags(msg);
    if (flags != NULL) {
      int flag_value;
      
      flag_value = etpan_message_flags_get_value(flags);
      if ((flag_value & ETPAN_FLAGS_DELETED) != 0)
        continue;
    }
    
    r = chash_get(view->msg_hash, &key, &value);
    if (r == 0) {
      struct etpan_message * current_msg;
      
      chash_value(iter, &value);
      current_msg = value.data;
      if (msg != current_msg) {
        ETPAN_CRASH("WARNING ! a message with same UID is not the same");
      }
      
      if (etpan_message_is_lost(msg)) {
        struct etpan_folder * folder;
        
        folder = etpan_message_get_folder(msg);
        if (folder != NULL) {
          etpan_folder_unref_msg_list(folder);
        }
        etpan_message_unref(msg);
        
        /* XXX - should not be deleted while iterating */
        chash_delete(view->msg_hash, &key, NULL);
        view->changed = 1;
      }
    }
    else {
      struct etpan_folder * folder;
      
      if (!etpan_message_is_lost(msg)) {
        etpan_message_ref(msg);
        folder = etpan_message_get_folder(msg);
        if (folder != NULL) {
          etpan_folder_ref_msg_list(folder);
        }
      
        value.data = msg;
        value.len = 0;
        r = chash_set(view->msg_hash, &key, &value, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        view->changed = 1;
      }
    }
  }
}

void etpan_msg_list_view_remove_messages(struct etpan_msg_list_view * view, carray * msg_list)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(msg_list) ; i ++) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    char * uid;
    struct etpan_folder * folder;
    
    uid = carray_get(msg_list, i);
    key.data = uid;
    key.len = strlen(uid) + 1;
    r = chash_get(view->msg_hash, &key, &value);
    if (r < 0)
      continue;
    
    msg = value.data;
    folder = etpan_message_get_folder(msg);
    if (folder != NULL) {
      etpan_folder_unref_msg_list(folder);
    }
    etpan_message_unref(msg);
    
    chash_delete(view->msg_hash, &key, NULL);
    view->changed = 1;
  }
  
  if (view->changed)
    ETPAN_LOG("remove msgs");
}

chash * etpan_msg_list_view_get_msg_hash(struct etpan_msg_list_view * view)
{
  return view->msg_hash;
}


struct etpan_message_tree * etpan_msg_list_view_get_tree(struct etpan_msg_list_view * view)
{
  return view->msg_tree;
}

static void build_callback(int cancelled, struct build_result * result,
    void * data);

static void diff_tree(struct etpan_msg_list_view * view,
    struct etpan_message_tree * old_tree,
    struct etpan_message_tree * new_tree,
    chash * new_messages,
    carray * removed_messages_list);

static void recursive_update_parent(struct etpan_message_tree * msg_tree);

static carray * get_new_message(chash * existing,
    struct etpan_message_tree * tree);

static carray * diff_message(struct etpan_message_tree * old_tree,
    struct etpan_message_tree * new_tree);

void etpan_msg_list_view_reset(struct etpan_msg_list_view * view)
{
  view->changed = 1;
}

void etpan_msg_list_view_update(struct etpan_msg_list_view * view)
{
  if (!view->changed) {
    return;
  }
  
  if (view->building_tree) {
    view->schedule_build = 1;
    return;
  }
  
  cancel(view);
  
  view->changed = 0;
  view->building_tree = 1;
  view->op = build(etpan_thread_manager_app_get_default(),
      view->msg_hash, view->folders_for_orphans,
      build_callback, view);
}

static void build_callback(int cancelled, struct build_result * result,
    void * data)
{
  struct etpan_msg_list_view * view;
  struct etpan_message_tree * msg_tree;
  carray * new_messages;
  
  view = data;
  view->building_tree = 0;
  view->op = NULL;
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Update of message list cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled the update of the message list."));
    
    view->error = error;
    
    return;
  }
  
  new_messages = diff_message(view->msg_tree, result->tree);
  
  diff_tree(view, view->msg_tree, result->tree, NULL, NULL);
  
  msg_tree = result->tree;
  result->tree = NULL;
  
  if (view->msg_tree != NULL) {
    free_msg(etpan_thread_manager_app_get_default(),
        view->msg_tree, NULL, NULL);
  }
  view->msg_tree = msg_tree;
  
  recursive_update_parent(view->msg_tree);
  
  /* signal */
  
  ETPAN_LOG("new messages : %p", new_messages);
  view->new_messages = new_messages;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MSG_LIST_VIEW_UPDATED_SIGNAL, view, NULL);
  
  carray_free(new_messages);
  view->new_messages = NULL;
  
  if (view->schedule_build) {
    view->schedule_build = 0;
    etpan_msg_list_view_update(view);
  }
}



static void diff_tree(struct etpan_msg_list_view * view,
    struct etpan_message_tree * old_tree,
    struct etpan_message_tree * new_tree,
    chash * new_messages,
    carray * removed_messages_list)
{
  carray * old_children;
  carray * new_children;
  unsigned int i;
  chash * old_msg_hash;
  chash * new_msg_hash;
  int r;
  
  old_children = NULL;
  new_children = NULL;
  
  if (old_tree != NULL)
    old_children = etpan_message_tree_get_children(old_tree);
  if (new_tree != NULL)
    new_children = etpan_message_tree_get_children(new_tree);
  
  old_msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (old_msg_hash == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  new_msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (new_msg_hash == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (old_children != NULL) {
    for(i = 0 ; i < carray_count(old_children) ; i ++) {
      struct etpan_message_tree * child;
      struct etpan_message * msg;
      chashdatum key;
      chashdatum value;
      char * uid;
      
      child = carray_get(old_children, i);
      msg = etpan_message_tree_get_message(child);
      
      uid = etpan_message_get_uid(msg);
      key.data = uid;
      key.len = strlen(uid) + 1;
      value.data = &i;
      value.len = sizeof(i);
      r = chash_set(old_msg_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  if (new_children != NULL) {
    for(i = 0 ; i < carray_count(new_children) ; i ++) {
      struct etpan_message_tree * child;
      struct etpan_message * msg;
      char * uid;
      chashdatum key;
      chashdatum value;
      
      child = carray_get(new_children, i);
      msg = etpan_message_tree_get_message(child);
      uid = etpan_message_get_uid(msg);
      key.data = uid;
      key.len = strlen(uid) + 1;
      
      if (old_children != NULL) {
        r = chash_get(old_msg_hash, &key, &value);
      }
      else {
        r = -1;
      }
      if (r == 0) {
        unsigned int old_index;
        struct etpan_message_tree * old_child;
        struct etpan_message_tree tmp_child;
        
        memcpy(&old_index, value.data, sizeof(old_index));
        old_child = carray_get(old_children, old_index);
        
        diff_tree(view, old_child, child,
            new_messages, removed_messages_list);
        
        memcpy(&tmp_child, old_child, sizeof(tmp_child));
        memcpy(old_child, child, sizeof(tmp_child));
        memcpy(child, &tmp_child, sizeof(tmp_child));
        
        carray_set(new_children, i, old_child);
        carray_set(old_children, old_index, child);
      }
      else if (new_messages != NULL) {
        value.data = msg;
        value.len = 0;
        
        r = chash_set(new_messages, &key, &value, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        diff_tree(view, NULL, child,
            new_messages, NULL);
      }
      
      if (new_msg_hash != NULL) {
        value.data = msg;
        value.len = 0;
        r = chash_set(new_msg_hash, &key, &value, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  if (removed_messages_list != NULL) {
    if (old_children != NULL) {
      for(i = 0 ; i < carray_count(old_children) ; i ++) {
        struct etpan_message_tree * child;
        struct etpan_message * msg;
        chashdatum key;
        chashdatum value;
        char * uid;
        int r;
      
        child = carray_get(old_children, i);
        msg = etpan_message_tree_get_message(child);
      
        uid = etpan_message_get_uid(msg);
        key.data = uid;
        key.len = strlen(uid) + 1;
        if (new_children != NULL) {
          r = chash_get(new_msg_hash, &key, &value);
        }
        else {
          r = -1;
        }
        if (r < 0) {
          r = carray_add(removed_messages_list, strdup(uid), NULL);
          if (r < 0)
            ETPAN_LOG_MEMORY_ERROR;
          diff_tree(view, child, NULL,
              NULL, removed_messages_list);
        }
      }
    }
  }
  
  chash_free(new_msg_hash);
  chash_free(old_msg_hash);
}

static void recursive_update_parent(struct etpan_message_tree * msg_tree)
{
  carray * list;
  
  list = etpan_message_tree_get_children(msg_tree);
  if (list != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(list, i);
      child->parent = msg_tree;
      recursive_update_parent(child);
    }
  }
}

/* unread messages list */
static carray * get_new_message(chash * existing,
    struct etpan_message_tree * tree)
{
  struct etpan_message * msg;
  carray * list;
  int r;
  carray * new_message_list;
  int is_new;
  
  new_message_list = carray_new(16);
  if (new_message_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  is_new = 0;
  list = etpan_message_tree_get_children(tree);
  msg = etpan_message_tree_get_message(tree);
  if (msg != NULL) {
    char * uid;
    chashdatum key;
    chashdatum value;
        
    is_new = 1;
    uid = etpan_message_get_uid(msg);
    key.data = uid;
    key.len = strlen(uid) + 1;
    r = chash_get(existing, &key, &value);
    if (r < 0) {
      struct etpan_message_flags * flags;
      
      flags = etpan_message_get_flags(msg);
      if (flags != NULL) {
        int flag_value;
        
        flag_value = etpan_message_flags_get_value(flags);
        if ((flag_value & ETPAN_FLAGS_SEEN) == 0) {
          r = carray_add(new_message_list, tree, NULL);
          if (r < 0)
            ETPAN_LOG_MEMORY_ERROR;
        }
      }
    }
    else {
      is_new = 0;
    }
  }
  
  if (list != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_message_tree * child;
      carray * child_message_list;
      unsigned int child_index;
      
      child = carray_get(list, i);
      if (is_new) {
        struct etpan_message * child_msg;
        
        child_msg = etpan_message_tree_get_message(child);
        if (child_msg != NULL) {
          char * uid;
          chashdatum key;
          chashdatum value;
          
          uid = etpan_message_get_uid(child_msg);
          key.data = uid;
          key.len = strlen(uid) + 1;
          r = chash_get(existing, &key, &value);
          if (r == 0) {
            chash_delete(existing, &key, NULL);
          }
        }
      }
      
      child_message_list = get_new_message(existing, child);
      
      if (carray_count(child_message_list) != 0) {
        for(child_index =  0 ;
            child_index < carray_count(child_message_list) ;
            child_index ++) {
          struct etpan_message_tree * child_elt;
          
          child_elt = carray_get(child_message_list, child_index);
          
          r = carray_add(new_message_list, child_elt, NULL);
          if (r < 0)
            ETPAN_LOG_MEMORY_ERROR;
        }
      }
      
      carray_free(child_message_list);
    }
  }
  
  return new_message_list;
}

static void mark_messages(chash * existing, struct etpan_message_tree * tree)
{
  struct etpan_message * msg;
  carray * list;
  int r;
  
  if (tree == NULL)
    return;
  
  list = etpan_message_tree_get_children(tree);
  msg = etpan_message_tree_get_message(tree);
  if (msg != NULL) {
    char * uid;
    chashdatum key;
    chashdatum value;
    
    uid = etpan_message_get_uid(msg);
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = tree;
    value.len = 0;
    r = chash_set(existing, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (list != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(list, i);
      mark_messages(existing, child);
    }
  }
}

static carray * diff_message(struct etpan_message_tree * old_tree,
    struct etpan_message_tree * new_tree)
{
  chash * existing;
  carray * new_message_list;
  
  existing = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (existing == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mark_messages(existing, old_tree);
  
  new_message_list = get_new_message(existing, new_tree);
  
  chash_free(existing);
  
  return new_message_list;
}


void etpan_msg_list_view_remove_folder(struct etpan_msg_list_view * view, struct etpan_folder * folder)
{
  chashiter * iter;
  carray * keys_to_remove;
  unsigned int i;
  
  keys_to_remove = carray_new(16);
  if (keys_to_remove == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(view->msg_hash) ; iter != NULL ;
      iter = chash_next(view->msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_folder * msg_folder;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    msg_folder = etpan_message_get_folder(msg);
    if (msg_folder == folder) {
      char * uid;
      
      uid = strdup(key.data);
      if (uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  etpan_msg_list_view_remove_messages(view, keys_to_remove);
  
  for(i = 0 ; i < carray_count(keys_to_remove) ; i ++) {
    char * uid;
    
    uid = carray_get(keys_to_remove, i);
    free(uid);
  }
  carray_free(keys_to_remove);
}

static void cancel(struct etpan_msg_list_view * view)
{
  if (view->op != NULL) {
    etpan_thread_op_cancel(view->op);
  }
}

carray * etpan_msg_list_view_get_new_messages(struct etpan_msg_list_view * view)
{
  return view->new_messages;
}

static void remove_root(struct etpan_message_tree * root,
    chash * msg_hash)
{
  carray * tab;
  unsigned int i;
  unsigned int current_index;
  int r;
  
  tab = etpan_message_tree_get_children(root);
  if (tab == NULL)
    return;
  
  current_index = 0;
  for(i = 0 ; i < carray_count(tab) ; i ++) {
    struct etpan_message_tree * child;
    chashdatum key;
    chashdatum value;
    
    child = carray_get(tab, i);
    
    key.data = &child;
    key.len = sizeof(child);
    
    r = chash_get(msg_hash, &key, &value);
    if (r < 0) {
      carray_set(tab, current_index, child);
      current_index ++;
    }
    else {
      /* free */
      etpan_message_tree_free_recursive(child);
    }
  }
  
  r = carray_set_size(tab, current_index);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static int thread_belongs_to_folders(struct etpan_message_tree * tree,
    chash * folders)
{
  carray * children;
  unsigned int i;
  struct etpan_message * msg;
  struct etpan_folder * folder;
  chashdatum key;
  chashdatum value;
  int r;
  
  msg = etpan_message_tree_get_message(tree);
  folder = etpan_message_get_folder(msg);
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(folders, &key, &value);
  if (r < 0)
    return 0;
  
  children = etpan_message_tree_get_children(tree);
  if (children != NULL) {
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(children, i);
      
      if (!thread_belongs_to_folders(child, folders))
        return 0;
    }
  }
  
  return 1;
}

static void remove_orphans(struct etpan_message_tree * tree,
    chash * folders_for_orphans)
{
  carray * children;
  unsigned int i;
  int r;
  chash * msg_hash;
  
  msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  children = etpan_message_tree_get_children(tree);
  if (children == NULL)
    return;
  
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_message_tree * child;
    chashdatum key;
    chashdatum value;
    
    child = carray_get(children, i);
    if (!thread_belongs_to_folders(child, folders_for_orphans))
      continue;
    
    key.data = &child;
    key.len = sizeof(child);
    value.data = child;
    value.len = 0;
    r = chash_set(msg_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  remove_root(tree, msg_hash);
  chash_free(msg_hash);
}

void etpan_msg_list_view_add_folder_for_orphans(struct etpan_msg_list_view * view, struct etpan_folder * folder)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = &folder;
  key.len = sizeof(folder);
  value.data = folder;
  value.len = 0;
  r = chash_set(view->folders_for_orphans, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_msg_list_view_remove_folder_for_orphans(struct etpan_msg_list_view * view, struct etpan_folder * folder)
{
  chashdatum key;
  
  key.data = &folder;
  key.len = sizeof(folder);
  chash_delete(view->folders_for_orphans, &key, NULL);
}
