#include "etpan-outbox.h"

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#include "etpan-storage-maildir.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-message.h"
#include "etpan-error.h"
#include "etpan-thread-manager-app.h"
#include "etpan-account-manager.h"
#include "etpan-utils.h"
#include "etpan-sender.h"
#include "etpan-part.h"
#include "etpan-signal.h"
#include "etpan-folder-create.h"
#include "etpan-account.h"
#include "etpan-nls.h"
#include "etpan-message-fetcher.h"
#include "etpan-msg-list-fetch.h"
#include "etpan-uuid.h"

#define ETPAN_MODULE_LOG_NAME "OUTBOX"
#include "etpan-log.h"

static struct etpan_error * rewrite_error(struct etpan_outbox * outbox,
    struct etpan_error * error);

static void etpan_outbox_cancel(struct etpan_outbox * outbox);

struct etpan_outbox * etpan_outbox_new(void)
{
  struct etpan_outbox * outbox;
  
  outbox = malloc(sizeof(* outbox));
  if (outbox == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  outbox->id = NULL;
  outbox->storage = NULL;
  outbox->folder = NULL;
  outbox->path = NULL;
  outbox->sender = NULL;
  outbox->account = NULL;
  outbox->current_op = NULL;
  outbox->sending = 0;
  
  outbox->stop_remaining = 0;
  outbox->stop_cb_data = NULL;
  outbox->stop_callback = NULL;
  outbox->process_stop_callback = NULL;
  outbox->process_stop_cb_data = NULL;
  
  outbox->process_cb_data = NULL;
  outbox->process_callback = NULL;
  outbox->process_msg_list = NULL;
  outbox->process_index = 0;
  outbox->process_error = NULL;
  outbox->process_message_content = NULL;
  outbox->process_message_length = 0;
  outbox->process_fetcher = NULL;
  outbox->process_folder_create = NULL;
  outbox->process_msg_list_fetch = NULL;
  outbox->process_error_list = NULL;
  
  outbox->append_op_list = carray_new(4);
  if (outbox->append_op_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  return outbox;
}

void etpan_outbox_free(struct etpan_outbox * outbox)
{
  carray_free(outbox->append_op_list);
  /* outbox->folder is not referenced */
  if (outbox->storage != NULL)
    etpan_storage_free(outbox->storage);
  if (outbox->sender != NULL)
    etpan_sender_free(outbox->sender);
  free(outbox->path);
  free(outbox->id);
  free(outbox);
}

static void set_sender_id(struct etpan_sender * sender, char * id)
{
  if (sender != NULL)
    return etpan_sender_set_id(sender, id);
}

void etpan_outbox_set_id(struct etpan_outbox * outbox, char * id)
{
  if (id != outbox->id) {
    free(outbox->id);
    if (id != NULL) {
      outbox->id = strdup(id);
      if (outbox->id == NULL)
        ETPAN_LOG_MEMORY_ERROR;
   }
    else
      outbox->id = NULL;
    
    if (outbox->sender != NULL)
      set_sender_id(outbox->sender, id);
  }
}

char * etpan_outbox_get_id(struct etpan_outbox * outbox)
{
  return outbox->id;
}

void etpan_outbox_set_path(struct etpan_outbox * outbox, char * path)
{
  if (path != outbox->path) {
    free(outbox->path);
    if (path != NULL) {
      outbox->path = strdup(path);
      if (outbox->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      outbox->path = NULL;
  }
}

char * etpan_outbox_get_path(struct etpan_outbox * outbox)
{
  return outbox->path;
}

void etpan_outbox_set_sender(struct etpan_outbox * outbox,
    struct etpan_sender * sender)
{
  set_sender_id(sender, outbox->id);
  outbox->sender = sender;
}

struct etpan_sender * etpan_outbox_get_sender(struct etpan_outbox * outbox)
{
  return outbox->sender;
}

static void setup_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data);

struct etpan_error * etpan_outbox_setup(struct etpan_outbox * outbox)
{
  struct etpan_thread_op * op;
  char path[PATH_MAX];
  char * cache_path;
  char * flags_path;
  struct etpan_account_manager * manager;
  char storage_id[PATH_MAX];
  struct etpan_error * error;
  
  ETPAN_LOCAL_LOG("outbox: setup %s", etpan_outbox_get_id(outbox));
  
  etpan_make_dir(outbox->path);
  
  snprintf(path, sizeof(path), "%s/.queue/cur", outbox->path);
  etpan_make_dir(path);
  snprintf(path, sizeof(path), "%s/.queue/tmp", outbox->path);
  etpan_make_dir(path);
  snprintf(path, sizeof(path), "%s/.queue/new", outbox->path);
  etpan_make_dir(path);
  
  outbox->storage = etpan_storage_maildir_new();
  
  if (outbox->id != NULL)
    snprintf(storage_id, sizeof(storage_id), "outbox-%s", outbox->id);
  else
    snprintf(storage_id, sizeof(storage_id), "outbox");
  etpan_storage_set_id(outbox->storage, storage_id);
  
  etpan_storage_maildir_set_path(outbox->storage, outbox->path);
  
  manager = etpan_account_manager_get_default();
  
  cache_path = etpan_get_outbox_cache_dir(manager, outbox);
  if (cache_path != NULL)
    etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(outbox->storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_outbox_flags_dir(manager, outbox);
  if (flags_path != NULL)
    etpan_make_dir(flags_path);
  etpan_storage_set_flags_path(outbox->storage, flags_path);
  free(flags_path);
  
  if (outbox->sender != NULL) {
    error = etpan_sender_setup(outbox->sender);
    if (error != NULL) {
      goto free_storage;
    }
  }
  
  error = etpan_storage_setup(outbox->storage);
  if (error != NULL) {
    goto unsetup_sender;
  }
  
  ETPAN_LOCAL_LOG("outbox: fetch folder list");
  op = etpan_storage_fetch_folder_list(etpan_thread_manager_app_get_default(),
    outbox->storage, setup_callback, outbox);
  
  outbox->setup_done = 0;
  do {
    etpan_thread_manager_app_loop(etpan_thread_manager_app_get_default());
  }
  while (!outbox->setup_done);
  
  error = outbox->setup_error;
  outbox->setup_error = NULL;
  if (error != NULL)
    goto unsetup_storage;
  
  return NULL;
  
 unsetup_storage:
  etpan_storage_unsetup(outbox->storage);
 unsetup_sender:
  if (outbox->sender != NULL)
    etpan_sender_unsetup(outbox->sender);
 free_storage:
  etpan_storage_free(outbox->storage);
  outbox->storage = NULL;
  return error;
}

static void setup_callback(int cancelled,
    struct etpan_storage_fetch_folder_list_result * result,
    void * cb_data)
{
  struct etpan_outbox * outbox;
  struct etpan_folder * folder;
  struct etpan_error * error;
  char * uuid;
  (void) cancelled;
  
  outbox = cb_data;
  
  ETPAN_LOCAL_LOG("outbox: callback %i", result->error);
  
  error = etpan_storage_update_folder_list(outbox->storage,
      result->folder_list);
  if (error != NULL) {
    ETPAN_ERROR_FREE(error);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error,
        _("Mail sending could not be initialized"));
    etpan_error_strf_long_description(error,
        _("Outbox folder could not be found."));
    
    outbox->setup_error = error;
    outbox->setup_done = 1;
    return;
  }
  
  folder = etpan_storage_get_folder(outbox->storage, "queue");
  if (folder == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error,
        _("Mail sending could not be initialized"));
    etpan_error_strf_long_description(error,
        _("Outbox folder could not be found."));
    
    outbox->setup_error = error;
    outbox->setup_done = 1;
    return;
  }
  
  uuid = etpan_uuid_generate();
  etpan_folder_set_uid(folder, uuid);
  free(uuid);
  ETPAN_LOCAL_LOG("outbox setup done %s", etpan_outbox_get_id(outbox));
  
  outbox->setup_error = NULL;
  outbox->folder = folder;
  outbox->setup_done = 1;
}


static void storage_disconnect_callback(int cancelled,
    void * dummy, void * cb_data);
static void process_stop_callback(struct etpan_outbox * outbox,
    void * cb_data);
static void stop_calling_callback(struct etpan_outbox * outbox);

void etpan_outbox_stop(struct etpan_outbox * outbox,
    void (* callback)(struct etpan_outbox * outbox, void * cb_data),
    void * cb_data)
{
  if (outbox->stop_remaining) {
    ETPAN_LOCAL_LOG("outbox stop called twice");
    etpan_crash();
  }
  
  ETPAN_LOCAL_LOG("outbox stop");
  outbox->stop_cb_data = cb_data;
  outbox->stop_callback = callback;
  
  outbox->stop_remaining = 0;
  
  etpan_outbox_cancel(outbox);
  if (outbox->sending) {
    outbox->process_stop_callback = process_stop_callback;
    outbox->process_stop_cb_data = NULL;
    outbox->stop_remaining ++;
  }
  
  outbox->stop_remaining ++;
  etpan_storage_disconnect(etpan_thread_manager_app_get_default(),
      outbox->storage, storage_disconnect_callback, outbox);
}

static void storage_disconnect_callback(int cancelled,
    void * dummy, void * cb_data)
{
  struct etpan_outbox * outbox;
  (void) cancelled;
  (void) dummy;
  
  ETPAN_LOCAL_LOG("outbox storage disconnected");
  outbox = cb_data;
  stop_calling_callback(outbox);
}

static void process_stop_callback(struct etpan_outbox * outbox,
    void * cb_data)
{
  (void) cb_data;
  
  stop_calling_callback(outbox);
}

static void stop_calling_callback(struct etpan_outbox * outbox)
{
  outbox->stop_remaining --;
  if (outbox->stop_remaining > 0)
    return;
  
  if (outbox->stop_callback != NULL)
    outbox->stop_callback(outbox, outbox->stop_cb_data);
  outbox->stop_callback = NULL;
  outbox->stop_cb_data = NULL;
}


void etpan_outbox_unsetup(struct etpan_outbox * outbox)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(outbox->append_op_list) ; i ++) {
    struct etpan_thread_op * op;
    
    op = carray_get(outbox->append_op_list, i);
    etpan_thread_op_cancel(op);
  }
  
  if (outbox->sender != NULL)
    etpan_sender_unsetup(outbox->sender);
  
  etpan_storage_unsetup(outbox->storage);
}

static void outbox_added_callback(int cancelled,
    struct etpan_folder_append_msg_result * result, void * cb_data);

struct outbox_add_cb_data {
  struct etpan_outbox * outbox;
  void (* callback)(struct etpan_outbox * outbox, void * user_data);
  void * user_data;
  struct etpan_thread_op * op;
};

void etpan_outbox_add(struct etpan_outbox * outbox,
    char * message, size_t length,
    void (* callback)(struct etpan_outbox * outbox, void * user_data),
    void * user_data)
{
  struct etpan_thread_op * op;
  struct etpan_message_flags * flags;
  struct outbox_add_cb_data * data;
  int r;
  
  if (outbox->folder == NULL) {
    ETPAN_WARN_LOG("could not queue message");
    etpan_crash();
  }
  
  flags = etpan_message_flags_new();
  etpan_message_flags_set_value(flags, ETPAN_FLAGS_SEEN);
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->outbox = outbox;
  data->callback = callback;
  data->user_data = user_data;
  op = etpan_folder_append(etpan_thread_manager_app_get_default(),
      outbox->folder, message, length,
      flags, outbox_added_callback, data);
  data->op = op;
  
  etpan_message_flags_free(flags);
  
  r = carray_add(outbox->append_op_list, op, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void outbox_added_callback(int cancelled,
    struct etpan_folder_append_msg_result * result, void * cb_data)
{
  struct outbox_add_cb_data * data;
  struct etpan_outbox * outbox;
  unsigned int i;
  struct etpan_error * error;
  
  data = cb_data;
  outbox = data->outbox;
  
  for(i = 0 ; i < carray_count(outbox->append_op_list) ; i ++) {
    struct etpan_thread_op * op;
    
    op = carray_get(outbox->append_op_list, i);
    if (op == data->op) {
      carray_delete(outbox->append_op_list, i);
      break;
    }
  }
  
  error = NULL;
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Sending of queued messages cancelled"));
    etpan_error_strf_long_description(error,
        _("Send of message that are in outbox of account %s has been cancelled."), etpan_outbox_get_id(outbox));
  }
  else if (result->error != NULL) {
    error = result->error;
    result->error = NULL;
  }
  
  outbox->added_error = error;
  
  data->callback(data->outbox, data->user_data);
  
  ETPAN_ERROR_FREE(outbox->added_error);
  outbox->added_error = NULL;
  
  free(data);
}

struct etpan_error * etpan_outbox_added_get_error(struct etpan_outbox * outbox)
{
  return outbox->added_error;
}

static void fetch_msg_list(struct etpan_outbox * outbox);
static void msg_list_fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void process_msg_list(struct etpan_outbox * outbox);
static void process_current_msg(struct etpan_outbox * outbox);
static void fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void message_sent_callback(int cancelled,
    struct etpan_sender_send_message_result * result,
    void * data);
#if 0
static void store_created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void store_append(struct etpan_outbox * outbox,
    struct etpan_folder * folder);
static void store_current_message(struct etpan_outbox * outbox);
static void store_appended_handler(int cancelled,
    struct etpan_folder_append_msg_result * result,
    void * user_data);
#endif
static void process_current_msg_done(struct etpan_outbox * outbox,
    int finished);
static void finished_msg_list(struct etpan_outbox * outbox);
static void process_end(struct etpan_outbox * outbox);
static void process_end_check_msglist(struct etpan_outbox * outbox);
static void check_msglist_callback(int cancelled,
    struct etpan_folder_check_msg_list_result * result, void * data);
static void check_folder_callback(int cancelled,
    struct etpan_folder_check_result * result, void * data);
static void expunged_callback(int cancelled,
    struct etpan_folder_expunge_result * result,
    void * data);
static void process_cleanup(struct etpan_outbox * outbox);

void etpan_outbox_process(struct etpan_outbox * outbox,
    void (* callback)(struct etpan_outbox * outbox, void * user_data),
    void * user_data)
{
  int r;
  
  if (outbox->sending) {
    ETPAN_WARN_LOG("already processing");
    etpan_crash();
  }
  
  outbox->process_cb_data = user_data;
  outbox->process_callback = callback;
  outbox->process_msg_list = NULL;
  outbox->process_index = 0;
  outbox->process_error = NULL;
  outbox->process_message_content = NULL;
  outbox->process_message_length = 0;
  outbox->process_error_list = carray_new(4);
  
  if (outbox->folder == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error,
        _("Sending of queued messages failed"));
    etpan_error_strf_long_description(error,
        _("Send of message that are in outbox of account %s. There is no queue folder."), etpan_outbox_get_id(outbox));
    
    r = carray_add(outbox->process_error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    process_end(outbox);
    return;
  }
  
  outbox->sending = 1;
  fetch_msg_list(outbox);
}

static void fetch_msg_list(struct etpan_outbox * outbox)
{
  outbox->process_msg_list_fetch = etpan_msg_list_fetch_new();
  etpan_msg_list_fetch_set_folder(outbox->process_msg_list_fetch,
      outbox->folder);
  etpan_msg_list_fetch_setup(outbox->process_msg_list_fetch);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL,
      outbox->process_msg_list_fetch, outbox,
      msg_list_fetched_handler);
  
  etpan_msg_list_fetch_run(outbox->process_msg_list_fetch);
}

static void msg_list_fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_outbox * outbox;
  struct etpan_error * error;
  int r;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  outbox = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MSG_LIST_FETCH_FINISHED_SIGNAL,
      outbox->process_msg_list_fetch, outbox,
      msg_list_fetched_handler);
  
  error = etpan_msg_list_fetch_get_error(outbox->process_msg_list_fetch);
  
  if (error != NULL) {
    if (etpan_error_is_cancelled(error)) {
      etpan_msg_list_fetch_unref(outbox->process_msg_list_fetch);
      outbox->process_msg_list_fetch = NULL;
      
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CANCELLED);
      etpan_error_set_short_description(error,
          _("Sending of queued messages cancelled"));
      etpan_error_strf_long_description(error,
          _("Send of message that are in outbox of account %s has been cancelled."), etpan_outbox_get_id(outbox));
    
      r = carray_add(outbox->process_error_list, error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      process_end(outbox);
    }
  }

  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(outbox, error);
    
    r = carray_add(outbox->process_error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_msg_list_fetch_unref(outbox->process_msg_list_fetch);
    outbox->process_msg_list_fetch = NULL;
    process_end(outbox);
    return;
  }
  
  etpan_folder_ref_msg_list(outbox->folder);
  
  etpan_msg_list_fetch_unref(outbox->process_msg_list_fetch);
  outbox->process_msg_list_fetch = NULL;
  
  process_msg_list(outbox);
}

static void process_msg_list(struct etpan_outbox * outbox)
{
  chash * msghash;
  int r;
  chashiter * iter;
  
  outbox->process_msg_list = carray_new(16);
  if (outbox->process_msg_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  msghash = etpan_folder_get_message_hash(outbox->folder);
  ETPAN_LOCAL_LOG("got message list %i", chash_count(msghash));
  for(iter = chash_begin(msghash) ; iter != NULL ;
      iter = chash_next(msghash, iter)) {
    chashdatum value;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    
    chash_value(iter, &value);
    msg = value.data;
    
    flags = etpan_message_get_flags(msg);
    
    if (flags != NULL) {
      int value;
      
      value = etpan_message_flags_get_value(flags);
      if ((value & ETPAN_FLAGS_DELETED) == 0) {
        etpan_message_ref(msg);
        r = carray_add(outbox->process_msg_list, msg, NULL);
        if (r < 0) {
          ETPAN_LOG_MEMORY_ERROR;
        }
      }
    }
  }
  
  outbox->process_index = 0;
  process_current_msg(outbox);
}

static void process_current_msg(struct etpan_outbox * outbox)
{
  struct etpan_message * msg;
  struct etpan_message_fetcher * fetcher;
  
  if (outbox->process_index >= carray_count(outbox->process_msg_list)) {
    finished_msg_list(outbox);
    return;
  }
  
  msg = carray_get(outbox->process_msg_list, outbox->process_index);
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE);
  etpan_message_fetcher_set_message(fetcher, msg);
  
  etpan_message_fetcher_setup(fetcher);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, outbox,
    fetched_handler);
  
  outbox->process_fetcher = fetcher;
  etpan_message_fetcher_run(fetcher);
}


static void fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_error * error;
  struct etpan_part_fetch_info * info;
  char * content;
  size_t content_length;
  int r;
  struct etpan_outbox * outbox;
  carray * part_list;
  int type;
  struct etpan_thread_op * op;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  outbox = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, outbox->process_fetcher, outbox,
    fetched_handler);
  
  error = etpan_message_fetcher_get_error(outbox->process_fetcher);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = etpan_error_dup(error);
    r = carray_add(outbox->process_error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (etpan_error_is_cancelled(new_error)) {
      process_current_msg_done(outbox, 1);
    }
    else {
      process_current_msg_done(outbox, 0);
    }
    return;
  }
  
  part_list = etpan_message_fetcher_get_part_list(outbox->process_fetcher);
  if (carray_count(part_list) != 1) {
    ETPAN_WARN_LOG("source fetch result does not have exactly one part");
    etpan_crash();
  }
  
  info = carray_get(part_list, 0);
  type = etpan_part_fetch_info_get_type(info);
  if (type != ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
    ETPAN_WARN_LOG("source fetch result is not a text part");
    etpan_crash();
  }
  
  etpan_part_fetch_info_get_content(info, &content, &content_length);
  
  outbox->process_message_content = content;
  outbox->process_message_length = content_length;
  
  /* send message */
  op = etpan_sender_send_message(etpan_thread_manager_app_get_default(),
      outbox->sender, content, content_length,
      message_sent_callback, outbox);
  outbox->current_op = op;
}

static void message_sent_callback(int cancelled,
    struct etpan_sender_send_message_result * result,
    void * data)
{
  struct etpan_outbox * outbox;
  struct etpan_message_flags * flags;
  struct etpan_message * msg;
  int value;
  int r;
  
  outbox = data;
  outbox->current_op = NULL;
  
  ETPAN_LOCAL_LOG("sending message ok");
  
  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Sending of queued messages cancelled"));
    etpan_error_strf_long_description(error,
        _("Send of message that are in outbox of account %s has been cancelled."), etpan_outbox_get_id(outbox));
    
    r = carray_add(outbox->process_error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    process_current_msg_done(outbox, 1);
    return;
  }
  
  if (result->error != NULL) {
    r = carray_add(outbox->process_error_list, result->error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    result->error = NULL;
    
    process_current_msg_done(outbox, 0);
    return;
  }
  
  msg = carray_get(outbox->process_msg_list, outbox->process_index);
  ETPAN_LOCAL_LOG("deleting outbox msg %s", etpan_message_get_description(msg));
  
  /* remove from list */
  flags = etpan_message_get_flags(msg);
  flags = etpan_message_flags_dup(flags);
  
  value = etpan_message_flags_get_value(flags);
  value |= (ETPAN_FLAGS_DELETED | ETPAN_FLAGS_SEEN);
  etpan_message_flags_set_value(flags, value);
  etpan_message_set_flags(msg, flags);
  
  etpan_message_flags_free(flags);
  
  /* store message */
#if 0
  store_current_message(outbox);
#endif
  process_current_msg_done(outbox, 0);
}

#if 0
static void store_current_message(struct etpan_outbox * outbox)
{
  struct etpan_account * account;
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  char * folder_location;
  
  /* ensure "Sent Messages" folder is created */
  account = etpan_outbox_get_account(outbox);
  storage = etpan_account_get_storage(account);
  folder_location = etpan_storage_get_sub_folder_location(storage,
      NULL, "Sent Messages");
  
  folder = etpan_storage_get_folder(storage, folder_location);
  free(folder_location);
  
  if (folder == NULL) {
    struct etpan_folder_create * folder_create;
    
    folder_create = etpan_folder_create_new();
    etpan_folder_create_set_folder_name(folder_create, "Sent Messages");
    etpan_folder_create_set_parent_folder(folder_create, NULL);
    etpan_folder_create_set_storage(folder_create, storage);
    etpan_folder_create_setup(folder_create);
    
    etpan_signal_add_handler(etpan_signal_manager_get_default(),
        ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, outbox,
        store_created_handler);
    
    etpan_folder_create_run(folder_create);
    outbox->process_folder_create = folder_create;
  }
  else {
    store_append(outbox, folder);
  }
}

static void store_created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_outbox * outbox;
  struct etpan_folder_create * folder_create;
  struct etpan_folder * folder;
  struct etpan_error * error;
  int r;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  outbox = user_data;
  folder_create = outbox->process_folder_create;
  outbox->process_folder_create = NULL;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, outbox,
      store_created_handler);
  
  error = etpan_folder_create_get_error(folder_create);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = etpan_error_dup(error);
    r = carray_add(outbox->process_error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_folder_create_unref(folder_create); 
    
    process_current_msg_done(outbox, 1);
    return;
  }
  
  folder = etpan_folder_create_get_created_folder(folder_create);
  
  store_append(outbox, folder);
  
  etpan_folder_create_unref(folder_create); 
}

static void store_append(struct etpan_outbox * outbox,
    struct etpan_folder * folder)
{
  struct etpan_message_flags * flags;
  struct etpan_thread_op * op;
  
  flags = etpan_message_flags_new();
  etpan_message_flags_set_value(flags, ETPAN_FLAGS_SEEN);
  
  op = etpan_folder_append(etpan_thread_manager_app_get_default(),
      folder,
      outbox->process_message_content,
      outbox->process_message_length,
      flags, store_appended_handler,
      outbox);
  outbox->current_op = op;
  
  etpan_message_flags_free(flags);
}

static void store_appended_handler(int cancelled,
    struct etpan_folder_append_msg_result * result,
    void * user_data)
{
  struct etpan_outbox * outbox;
  int r;
  
  outbox = user_data;
  outbox->current_op = NULL;

  if (cancelled) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Sending of queued messages cancelled"));
    etpan_error_strf_long_description(error,
        _("Send of message that are in outbox of account %s has been cancelled."), etpan_outbox_get_id(outbox));
    
    r = carray_add(outbox->process_error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    process_current_msg_done(outbox, 1);
    return;
  }
  
  if (result->error != NULL) {
    r = carray_add(outbox->process_error_list, result->error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    result->error = NULL;
    
    process_current_msg_done(outbox, 0);
    return;
  }
  
  process_current_msg_done(outbox, 0);
}
#endif

static void process_current_msg_done(struct etpan_outbox * outbox,
    int finished)
{
  if (outbox->process_fetcher != NULL) {
    outbox->process_message_content = NULL;
    etpan_message_fetcher_unref(outbox->process_fetcher);
    outbox->process_fetcher = NULL;
  }
  
  if (!finished) {
    outbox->process_index ++;
    process_current_msg(outbox);
  }
  else {
    finished_msg_list(outbox);
  }
}

static void finished_msg_list(struct etpan_outbox * outbox)
{
  process_end(outbox);
}

static void process_end(struct etpan_outbox * outbox)
{
  etpan_sender_disconnect(etpan_thread_manager_app_get_default(),
      outbox->sender,
      NULL, NULL);
  
  process_end_check_msglist(outbox);
}

static void process_end_check_msglist(struct etpan_outbox * outbox)
{
  chash * msg_hash_copy;
  chashiter * iter;
  chash * msg_hash;
  struct etpan_thread_op * check_msg_op;
  int r;
  struct etpan_folder * folder;
  
  folder = outbox->folder;
  
  msg_hash_copy = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msg_hash_copy == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_hash = etpan_folder_get_message_hash(folder);
  for(iter = chash_begin(msg_hash) ; iter != NULL ; 
      iter = chash_next(msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    r = chash_set(msg_hash_copy, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  check_msg_op =
    etpan_folder_check_msg_list(etpan_thread_manager_app_get_default(),
        folder, msg_hash_copy, check_msglist_callback, outbox);
  
  chash_free(msg_hash_copy);
}

static void check_msglist_callback(int cancelled,
    struct etpan_folder_check_msg_list_result * result, void * data)
{
  struct etpan_thread_op * op;
  struct etpan_outbox * outbox;
  int r;
  (void) cancelled;
  
  outbox = data;
  
  if (result->error != NULL) {
    r = carray_add(outbox->process_error_list, result->error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    result->error = NULL;
  }
  
  op = etpan_folder_check(etpan_thread_manager_app_get_default(),
      outbox->folder, check_folder_callback, outbox);
}

static void check_folder_callback(int cancelled,
    struct etpan_folder_check_result * result, void * data)
{
  struct etpan_outbox * outbox;
  struct etpan_thread_op * op;
  int r;
  (void) cancelled;
  
  outbox = data;
  
  if (result->error != NULL) {
    r = carray_add(outbox->process_error_list, result->error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    result->error = NULL;
  }
  
  op = etpan_folder_expunge(etpan_thread_manager_app_get_default(),
      outbox->folder, expunged_callback, outbox);
}

static void expunged_callback(int cancelled,
    struct etpan_folder_expunge_result * result,
    void * data)
{
  struct etpan_outbox * outbox;
  (void) cancelled;
  int r;
  
  outbox = data;
  
  if (result->error != NULL) {
    r = carray_add(outbox->process_error_list, result->error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    result->error = NULL;
  }
  
  ETPAN_LOCAL_LOG("finished sending message");
  
  process_cleanup(outbox);
}

static void process_cleanup(struct etpan_outbox * outbox)
{
  struct etpan_error * error;
  unsigned int i;
  
  if (outbox->process_msg_list != NULL) {
    for(i = 0 ; i < carray_count(outbox->process_msg_list) ; i ++) {
      struct etpan_message * msg;
      
      msg = carray_get(outbox->process_msg_list, i);
      etpan_message_unref(msg);
    }
    carray_free(outbox->process_msg_list);
  }
  
  if (carray_count(outbox->process_error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(outbox->process_error_list) == 1) {
    error = carray_get(outbox->process_error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(outbox->process_error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(outbox->process_error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(outbox->process_error_list);
  outbox->process_error_list = NULL;
  outbox->process_error = error;
  etpan_folder_unref_msg_list(outbox->folder);
  
  if (outbox->process_callback != NULL)
    outbox->process_callback(outbox, outbox->process_cb_data);
  outbox->process_callback = NULL;
  outbox->process_cb_data = NULL;
  ETPAN_ERROR_FREE(outbox->process_error);
  outbox->process_error = NULL;
  
  if (outbox->process_stop_callback != NULL)
    outbox->process_stop_callback(outbox, outbox->process_stop_cb_data);
  outbox->process_stop_callback = NULL;
  outbox->process_stop_cb_data = NULL;
  
  outbox->sending = 0;
}

struct etpan_error * etpan_outbox_processed_get_error(struct etpan_outbox * outbox)
{
  return outbox->process_error;
}

static void etpan_outbox_cancel(struct etpan_outbox * outbox)
{
  if (outbox->sending) {
    if (outbox->current_op != NULL) {
      etpan_thread_op_cancel(outbox->current_op);
    }
    else if (outbox->process_msg_list_fetch != NULL) {
      etpan_msg_list_fetch_cancel(outbox->process_msg_list_fetch);
    }
    else if (outbox->process_folder_create != NULL) {
      etpan_folder_create_cancel(outbox->process_folder_create);
    }
    else if (outbox->process_fetcher != NULL) {
      etpan_message_fetcher_cancel(outbox->process_fetcher);
    }
  }
}

void etpan_outbox_set_account(struct etpan_outbox * outbox,
    struct etpan_account * account)
{
  outbox->account = account;
}

struct etpan_account * etpan_outbox_get_account(struct etpan_outbox * outbox)
{
  return outbox->account;
}

static struct etpan_error * rewrite_error(struct etpan_outbox * outbox,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  etpan_error_strf_long_description(new_error,
      _("An error occurred while sending messages from %s.\n"
          "%s"),
      etpan_outbox_get_id(outbox),
      previous_long_description);
  
  return new_error;
}
