#include "etpan-part-lep.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-lep.h"
#include "etpan-message-lep.h"
#include "etpan-part-header.h"
#include "etpan-nls.h"

#define ETPAN_MODULE_LOG_NAME "LEP"
#include "etpan-log.h"

enum {
  ERROR_DOMAIN_FETCH,
};

static struct etpan_error * translate_error(struct etpan_part * part,
    int domain, int error_code);

static struct etpan_part_header * get_header(struct etpan_part * part);

static struct etpan_error * fetch(struct etpan_part * part,
    char ** p_content, size_t * p_length);

static struct etpan_error * fetch_header(struct etpan_part * part,
    char ** p_content, size_t * p_length);

static struct etpan_error * fetch_mime_header(struct etpan_part * part,
    char ** p_content, size_t * p_length);

static void free_data(struct etpan_part * part);

static struct etpan_part_driver lep_driver = {
  .name = "libetpan",
  
  .get_header = get_header,
  
  .fetch = fetch,
  .fetch_header = fetch_header,
  .fetch_mime_header = fetch_mime_header,
  
  .free_data = free_data,
};

struct part_data {
  struct mailmime * ep_mime;
};

struct etpan_part * etpan_part_lep_new(void)
{
  struct etpan_part * part;
  struct part_data * data;
  
  part = etpan_part_new();
  if (part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->ep_mime = NULL;
  
  etpan_part_set_data(part, data);
  etpan_part_set_driver(part, &lep_driver);
  
  return part;
}

void etpan_part_lep_set_mime(struct etpan_part * part,
    struct mailmime * ep_mime)
{
  struct part_data * data;
  
  data = etpan_part_get_data(part);
  data->ep_mime = ep_mime;
}

struct mailmime * etpan_part_lep_get_mime(struct etpan_part * part)
{
  struct part_data * data;
  
  data = etpan_part_get_data(part);
  return data->ep_mime;
}

typedef int (* lep_fetcher)(struct mailprivacy * privacy,
    mailmessage * msg_info,
    struct mailmime * mime,
    char ** result, size_t * result_len);

static struct etpan_error *
fetch_with_fetcher(struct etpan_part * part, lep_fetcher fetcher,
    char ** p_content, size_t * p_length)
{
  struct mailmime * lep_mime;
  struct mailmessage * lep_msg;
  struct etpan_message * msg;
  int r;
  char * content;
  size_t content_len;
  char * dup_content;
  
  msg = etpan_part_get_message(part);
  lep_msg = etpan_message_lep_get_msg(msg);
  if (lep_msg == NULL) {
    ETPAN_WARN_LOG("message does not have low-level message");
    etpan_crash();
  }
  
  lep_mime = etpan_part_lep_get_mime(part);
  if (lep_mime == NULL) {
    ETPAN_WARN_LOG("mime part does not have low-level mime part");
    etpan_crash();
  }
  
  r = fetcher(etpan_lep_privacy(),
      lep_msg, lep_mime, &content, &content_len);
  if (r != MAIL_NO_ERROR) {
    struct etpan_error * error;
    
    error = translate_error(part, ERROR_DOMAIN_FETCH, r);
    
    return error;
  }
  
  dup_content = malloc(content_len + 1);
  if (dup_content == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  memcpy(dup_content, content, content_len);
  dup_content[content_len] = 0;
  
  mailprivacy_msg_fetch_result_free(etpan_lep_privacy(),
      lep_msg, content);
  
  * p_content = dup_content;
  * p_length = content_len;
  
  return NULL;
}


static struct etpan_error * fetch(struct etpan_part * part,
    char ** p_content, size_t * p_length)
{
  return fetch_with_fetcher(part, mailprivacy_msg_fetch_section,
      p_content, p_length);
}

static struct etpan_error * fetch_header(struct etpan_part * part,
    char ** p_content, size_t * p_length)
{
  return fetch_with_fetcher(part, mailprivacy_msg_fetch_section_header,
      p_content, p_length);
}

static struct etpan_error * fetch_mime_header(struct etpan_part * part,
    char ** p_content, size_t * p_length)
{
  return fetch_with_fetcher(part, mailprivacy_msg_fetch_section_mime,
      p_content, p_length);
}

static void free_data(struct etpan_part * part)
{
  struct part_data * data;
  struct etpan_message * msg;
  struct mailmessage * ep_msg;
  
  data = etpan_part_get_data(part);
  
  msg = etpan_part_get_message(part);
  ep_msg = etpan_message_lep_get_msg(msg);
  {
    struct mailmime * ep_mime;
    ep_mime = etpan_part_lep_get_mime(part);
    ETPAN_LOCAL_LOG("free ep_mime : %p %p", part, ep_mime);
  }
  
  free(data);
}

static struct etpan_part_header * get_header(struct etpan_part * part)
{
  struct mailmime * ep_mime;
  struct etpan_part_header * header;
  
  ep_mime = etpan_part_lep_get_mime(part);
  if (ep_mime == NULL) {
    ETPAN_WARN_LOG("mime part does not have low-level mime part");
    etpan_crash();
  }
  
  header = etpan_mime_header_from_lep(ep_mime->mm_mime_fields,
    ep_mime->mm_content_type);
  
  return header;
}


static struct etpan_error * translate_error(struct etpan_part * part,
    int domain, int error_code)
{
  struct etpan_error * error;
  struct etpan_message * msg;
  
  error = etpan_error_new();
  msg  = etpan_part_get_message(part);
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH:
    switch (error_code) {
    case MAIL_ERROR_STREAM:
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("An error occurred while getting data of the following message:"
              "%s"
              "The connection closed unexpectedly."),
          etpan_message_get_description(msg));
      break;

    case MAIL_ERROR_FETCH:
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Message data could not be retrieved");
      etpan_error_strf_long_description(error,
          _("An error occurred while getting data of the following message:"
              "%s"
              "The message was not found."),
          etpan_message_get_description(msg));
      break;
      
    default:
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("An error occurred while getting data of the following message:"
              "%s"
              "An unexpected error (libetpan: %i) occurred."),
          etpan_message_get_description(msg),
          error_code);
      break;
    }
    break;
    
  default:
    etpan_error_set_code(error, ERROR_INTERNAL);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An unexpected error (libetpan: %i) occurred while working with the following message:"
            "%s"),
        error_code,
        etpan_message_get_description(msg));
    break;
  }
  
  return error;
}
