#include "etpan-part.h"

#include <stdlib.h>
#include <string.h>

#define ETPAN_MODULE_LOG_NAME "PART"

#include "etpan-error.h"
#include "etpan-thread-manager-app.h"
#include "etpan-message.h"
#include "etpan-folder.h"
#include "etpan-folder-private.h"
#include "etpan-storage.h"
#include "etpan-part-header.h"
#include "etpan-nls.h"
#include "etpan-log.h"

static struct etpan_error * folder_connect(struct etpan_folder * folder);
static struct etpan_error * folder_disconnect(struct etpan_folder * folder);

struct etpan_part * etpan_part_new(void)
{
  struct etpan_part * part;
  int r;
  
  part = malloc(sizeof(* part));
  if (part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = pthread_mutex_init(&part->lock, NULL);
  if (r != 0) {
    ETPAN_CRASH("failed to create mutex (part)");
  }
  
  part->ref_count = 1;
  part->message = NULL;
  part->header = NULL;
  part->type = ETPAN_PART_TYPE_SINGLE;
  part->uid = NULL;
  part->children = carray_new(4);
  if (part->children == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  part->data = NULL;
  part->driver = NULL;
  
  return part;
}

void etpan_part_ref(struct etpan_part * part)
{
  pthread_mutex_lock(&part->lock);
  part->ref_count ++;
  ETPAN_LOCAL_LOG("part %p ref %i", part, part->ref_count);
#if 0
  etpan_log_stack();
#endif
  pthread_mutex_unlock(&part->lock);
}

void etpan_part_unref(struct etpan_part * part)
{
  int ref_count;
  
  pthread_mutex_lock(&part->lock);
  part->ref_count --;
  ref_count = part->ref_count;
  ETPAN_LOCAL_LOG("part %p unref %i", part, part->ref_count);
#if 0
  etpan_log_stack();
#endif
  pthread_mutex_unlock(&part->lock);
  
  if (ref_count == 0)
    etpan_part_free(part);
}

void etpan_part_set_type(struct etpan_part * part, int type)
{
  part->type = type;
}

int etpan_part_get_type(struct etpan_part * part)
{
  return part->type;
}

struct etpan_part * etpan_part_new_single(void)
{
  struct etpan_part * part;
  
  part = etpan_part_new();
  etpan_part_set_type(part, ETPAN_PART_TYPE_SINGLE);
  
  return part;
}

struct etpan_part * etpan_part_new_multiple(void)
{
  struct etpan_part * part;
  
  part = etpan_part_new();
  etpan_part_set_type(part, ETPAN_PART_TYPE_MULTIPLE);
  
  return part;
}

struct etpan_part * etpan_part_new_message(void)
{
  struct etpan_part * part;
  
  part = etpan_part_new();
  etpan_part_set_type(part, ETPAN_PART_TYPE_MESSAGE);
  
  return part;
}

void etpan_part_free(struct etpan_part * part)
{
  unsigned int i;
  
  if (part->driver->free_data != NULL)
    part->driver->free_data(part);
  
  if (part->header != NULL)
    etpan_part_header_free(part->header);
  
  for(i = 0 ; i < carray_count(part->children) ; i ++) {
    struct etpan_part * subpart;
    
    subpart = carray_get(part->children, i);
    etpan_part_unref(subpart);
  }
  carray_free(part->children);
  
  free(part->uid);
  pthread_mutex_destroy(&part->lock);
  
  free(part);
}

void etpan_part_set_data(struct etpan_part * part, void * data)
{
  part->data = data;
}

void * etpan_part_get_data(struct etpan_part * part)
{
  return part->data;
}

void etpan_part_set_driver(struct etpan_part * part,
    struct etpan_part_driver * driver)
{
  part->driver = driver;
}

void etpan_part_set_message(struct etpan_part * part,
    struct etpan_message * message)
{
  part->message = message;
}

struct etpan_message * etpan_part_get_message(struct etpan_part * part)
{
  return part->message;
}

struct etpan_part_header * etpan_part_get_header(struct etpan_part * part)
{
  if (part->header == NULL) {
    struct etpan_part_header * header;
    
    if (part->driver->get_header == NULL)
      return NULL;
    
    header = part->driver->get_header(part);
    part->header = header;
  }
  
  return part->header;
}

void etpan_part_set_header(struct etpan_part * part,
    struct etpan_part_header * header)
{
  if (part->header != NULL) {
    etpan_part_header_free(part->header);
  }
  
  part->header = header;
}

carray * etpan_part_get_children(struct etpan_part * part)
{
  return part->children;
}

void etpan_part_add_child(struct etpan_part * parent,
    struct etpan_part * child)
{
  int r;
  
  etpan_part_ref(child);
  r = carray_add(parent->children, child, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

char * etpan_part_get_uid(struct etpan_part * part)
{
  return part->uid;
}

void etpan_part_set_uid(struct etpan_part * part, char * uid)
{
  if (uid != part->uid) {
    free(part->uid);
    if (uid != NULL) {
      part->uid = strdup(uid);
      if (part->uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      part->uid = NULL;
  }
}

/* fetch part */

struct etpan_part_fetch_param {
  struct etpan_error * (* fetch)(struct etpan_part * part,
      char ** p_content,
      size_t * p_length);
  
  struct etpan_part * part;
};

static void
fetch_cleanup(struct etpan_thread_op * op)
{
  struct etpan_part_fetch_param * param;
  struct etpan_part_fetch_result * result;
  
  param = op->param;
  etpan_message_unref(param->part->message);
  etpan_part_unref(param->part);
  free(op->param);
  
  result = op->result;
  free(result->content);
  free(op->result);
}

static void
fetch_run(struct etpan_thread_op * op)
{
  struct etpan_part_fetch_param * param;
  struct etpan_part_fetch_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = folder_connect(param->part->message->folder);
  if (error == NULL) {
    if (param->fetch != NULL) {
      char * content;
      size_t length;
      
      error = param->fetch(param->part, &content, &length);
      if (error == NULL) {
        result->content = content;
        result->length = length;
      }
    }
    else {
      ETPAN_CRASH("fetcher not provided");
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_part_fetch(struct etpan_thread_manager_app * manager,
    struct etpan_part * part,
    void (* callback)(int, struct etpan_part_fetch_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_part_fetch_param * param;
  struct etpan_part_fetch_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->part = part;
  etpan_message_ref(param->part->message);
  etpan_part_ref(param->part);
  param->fetch = part->driver->fetch;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result->content = NULL;
  result->length = 0;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = fetch_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager,
      part->message->folder->storage, op);
  
  return op;
}

struct etpan_thread_op *
etpan_part_fetch_header(struct etpan_thread_manager_app * manager,
    struct etpan_part * part,
    void (* callback)(int, struct etpan_part_fetch_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_part_fetch_param * param;
  struct etpan_part_fetch_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->part = part;
  etpan_message_ref(param->part->message);
  etpan_part_ref(param->part);
  param->fetch = part->driver->fetch_header;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result->content = NULL;
  result->length = 0;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = fetch_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager,
      part->message->folder->storage, op);
  
  return op;
}

struct etpan_thread_op *
etpan_part_fetch_mime_header(struct etpan_thread_manager_app * manager,
    struct etpan_part * part,
    void (* callback)(int, struct etpan_part_fetch_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_part_fetch_param * param;
  struct etpan_part_fetch_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->part = part;
  etpan_message_ref(param->part->message);
  etpan_part_ref(param->part);
  param->fetch = part->driver->fetch_mime_header;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result->content = NULL;
  result->length = 0;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = fetch_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager,
      part->message->folder->storage, op);
  
  return op;
}

struct etpan_error *
etpan_mime_decode(char * text, size_t text_length, int encoding,
    char ** result_text, size_t * result_text_length)
{
  size_t cur_token;
  char * dup_decoded;
  char * decoded;
  size_t decoded_length;
  int lep_encoding;
  int r;
  
  lep_encoding = MAILMIME_MECHANISM_8BIT;
  
  switch (encoding) {
  case ETPAN_PART_MIME_ENCODING_BASE64:
    lep_encoding = MAILMIME_MECHANISM_BASE64;
    break;
  case ETPAN_PART_MIME_ENCODING_QUOTEDPRINTABLE:
    lep_encoding = MAILMIME_MECHANISM_QUOTED_PRINTABLE;
    break;
  }
  
  cur_token = 0;
  r = mailmime_part_parse(text, text_length, &cur_token,
      lep_encoding, &decoded, &decoded_length);
  if (r != MAILIMF_NO_ERROR) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Could not analyze message"));
    etpan_error_strf_long_description(error, _("The message is invalid."));
    
    return error;
  }
  
  dup_decoded = malloc(decoded_length + 1);
  if (dup_decoded == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  memcpy(dup_decoded, decoded, decoded_length);
  dup_decoded[decoded_length] = '\0';
  
  mailmime_decoded_part_free(decoded);
  
  * result_text = dup_decoded;
  * result_text_length = decoded_length;
  
  return NULL;
}

static struct etpan_error * folder_connect(struct etpan_folder * folder)
{
  return etpan_folder_connect_nt(folder);
}

static struct etpan_error * folder_disconnect(struct etpan_folder * folder)
{
  etpan_folder_disconnect_nt(folder);
}
