#include "etpan-sender-sent-folder.h"

#include <libetpan/libetpan.h>
#include <libetpan/mailsem.h>
#include <stdlib.h>

#include "etpan-sender.h"
#include "etpan-thread-manager-app.h"
#include "etpan-folder-create.h"
#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-error.h"
#include "etpan-nls.h"
#include "etpan-log.h"
#include "etpan-message.h"
#include "etpan-signal.h"

static struct etpan_error * send_message(struct etpan_sender * sender,
    char * message, size_t size);
static void free_data(struct etpan_sender * sender);

struct sender_data {
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  struct mailsem * sem;
};

static struct etpan_sender_driver sent_folder_driver = {
  .name = "sent-folder",
  .setup = NULL,
  .free_data = free_data,
  .send_message = send_message,
  .connect = NULL,
  .disconnect = NULL,
};

struct append_cb_data {
  struct mailsem * sem;
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  char * message;
  size_t size;
  struct etpan_sender * sender;
  struct etpan_error * error;
  struct etpan_folder_create * folder_create;
  struct etpan_thread_op * op;
};

static void send_message_main_thread(void * data);
static void created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void append_message(void * data);
static void append_callback(int cancelled,
    struct etpan_folder_append_msg_result * result,
    void * cb_data);

static struct etpan_error * send_message(struct etpan_sender * sender,
    char * message, size_t size)
{
  struct sender_data * data;
  struct append_cb_data a_cb_data;
  
  data = etpan_sender_get_data(sender);
  
  a_cb_data.sem = data->sem;
  a_cb_data.storage = data->storage;
  a_cb_data.folder = NULL;
  a_cb_data.message = message;
  a_cb_data.size = size;
  a_cb_data.sender = sender;
  a_cb_data.error = NULL;
  
  etpan_thread_manager_app_run_in_main_thread(etpan_thread_manager_app_get_default(), send_message_main_thread, &a_cb_data, 1);
  
  mailsem_down(data->sem);
  
  return a_cb_data.error;
}

static void send_message_main_thread(void * data)
{
  struct append_cb_data * a_cb_data;
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  char * folder_location;
  
  a_cb_data = data;
  
  /* ensure "Sent Messages" folder is created */
  storage = a_cb_data->storage;
  folder_location = etpan_storage_get_sub_folder_location(storage,
      NULL, "Sent Messages");
  
  folder = etpan_storage_get_folder(storage, folder_location);
  free(folder_location);
  
  if (folder == NULL) {
    struct etpan_folder_create * folder_create;
    
    folder_create = etpan_folder_create_new();
    etpan_folder_create_set_folder_name(folder_create, "Sent Messages");
    etpan_folder_create_set_parent_folder(folder_create, NULL);
    etpan_folder_create_set_storage(folder_create, storage);
    etpan_folder_create_setup(folder_create);
    
    etpan_signal_add_handler(etpan_signal_manager_get_default(),
        ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, a_cb_data,
        created_handler);
    
    etpan_folder_create_run(folder_create);
    a_cb_data->folder_create = folder_create;
  }
  else {
    a_cb_data->folder = folder;
    append_message(a_cb_data);
  }
}

static void created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct append_cb_data * a_cb_data;
  struct etpan_folder * folder;
  struct etpan_error * error;
  
  a_cb_data = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL,
      a_cb_data->folder_create, a_cb_data,
      created_handler);
  
  error = etpan_folder_create_get_error(a_cb_data->folder_create);
  if (error != NULL) {
    error = etpan_error_dup(error);
    a_cb_data->error = error;
    
    mailsem_up(a_cb_data->sem);
    return;
  }
  
  folder = etpan_folder_create_get_created_folder(a_cb_data->folder_create);
  a_cb_data->folder = folder;
  etpan_folder_create_unref(a_cb_data->folder_create);
  a_cb_data->folder_create = NULL;
  
  append_message(a_cb_data);
}

static void append_message(void * data)
{
  struct append_cb_data * a_cb_data;
  struct etpan_thread_op * op;
  struct etpan_message_flags * flags;
  
  a_cb_data = data;
  
  flags = etpan_message_flags_new();
  etpan_message_flags_set_value(flags, ETPAN_FLAGS_SEEN);
  op = etpan_folder_append(etpan_thread_manager_app_get_default(),
      a_cb_data->folder, a_cb_data->message, a_cb_data->size,
      flags, append_callback, a_cb_data);
  etpan_message_flags_free(flags);
  a_cb_data->op = op;
}

static void append_callback(int cancelled,
    struct etpan_folder_append_msg_result * result,
    void * cb_data)
{
  struct sender_data * data;
  struct append_cb_data * a_cb_data;
  struct etpan_sender * sender;
  
  a_cb_data = cb_data;
  a_cb_data->op = NULL;
  sender = a_cb_data->sender;
  
  data = etpan_sender_get_data(sender);

  if (cancelled) {
    struct etpan_error * error;

    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Message was stored in Sent Messages folder"));
    etpan_error_strf_long_description(error,
        _("Store of the sent message %s was interrupted."),
        etpan_sender_get_id(a_cb_data->sender));
    
    a_cb_data->error = error;
  }
  else {
    a_cb_data->error = result->error;
    result->error = NULL;
  }
  
  mailsem_up(data->sem);
}

static void free_data(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  
  mailsem_free(data->sem);
  
  free(data);
}

struct etpan_sender * etpan_sender_sent_folder_new(void)
{
  struct etpan_sender * sender;
  struct sender_data * data;
  
  sender = etpan_sender_new();
  if (sender == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->sem = mailsem_new();
  if (data->sem == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->storage = NULL;
  
  etpan_sender_set_data(sender, data);
  etpan_sender_set_driver(sender, &sent_folder_driver);
  
  return sender;
}

void etpan_sender_sent_folder_set_storage(struct etpan_sender * sender,
    struct etpan_storage * storage)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  data->storage = storage;
}
