#include "etpan-sender-smtp.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <libetpan/libetpan.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "etpan-sender.h"
#include "etpan-error.h"
#include "etpan-lep.h"
#include "etpan-connection-types.h"
#include "etpan-log.h"
#include "etpan-nls.h"

static struct etpan_error * smtp_connect(struct etpan_sender * sender);
static void smtp_disconnect(struct etpan_sender * sender);
static struct etpan_error * send_message(struct etpan_sender * sender,
    char * message, size_t size);
static void free_data(struct etpan_sender * sender);
static struct etpan_error * setup(struct etpan_sender * sender);

struct sender_data {
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;

  char * threaded_hostname;
  int threaded_port;
  int threaded_connection_type;
  int threaded_auth_type;
  char * threaded_username;
  char * threaded_password;
  
  int connected;
  mailsmtp * smtp;
};

static struct etpan_sender_driver smtp_driver = {
  .name = "smtp",
  .setup = setup,
  .free_data = free_data,
  .send_message = send_message,
  .connect = smtp_connect,
  .disconnect = smtp_disconnect,
};

static void smtp_disconnect(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  
  if (!data->connected)
    return;
  
  mailsmtp_quit(data->smtp);
  
  mailsmtp_free(data->smtp);
  data->smtp = NULL;
  data->connected = 0;
}

static int fill_remote_ip_port(mailstream * stream, char * remote_ip_port,
                               size_t remote_ip_port_len)
{
  mailstream_low * low;
  int fd;
  struct sockaddr_in name;
  socklen_t namelen;
  char remote_ip_port_buf[128];
  int r;
  
  low = mailstream_get_low(stream);
  fd = mailstream_low_get_fd(low);
  
  namelen = sizeof(name);
  r = getpeername(fd, (struct sockaddr *) &name, &namelen);
  if (r < 0)
    return -1;
  
  if (inet_ntop(AF_INET, &name.sin_addr, remote_ip_port_buf,
          sizeof(remote_ip_port_buf)))
    return -1;
  
  snprintf(remote_ip_port, remote_ip_port_len, "%s;%i",
      remote_ip_port_buf, ntohs(name.sin_port));
  
  return 0;
}

static int fill_local_ip_port(mailstream * stream, char * local_ip_port,
                               size_t local_ip_port_len)
{
  mailstream_low * low;
  int fd;
  struct sockaddr_in name;
  socklen_t namelen;
  char local_ip_port_buf[128];
  int r;
  
  low = mailstream_get_low(stream);
  fd = mailstream_low_get_fd(low);
  
  namelen = sizeof(name);
  r = getpeername(fd, (struct sockaddr *) &name, &namelen);
  if (r < 0)
    return -1;
  
  if (inet_ntop(AF_INET, &name.sin_addr, local_ip_port_buf,
          sizeof(local_ip_port_buf)))
    return -1;
  
  snprintf(local_ip_port, local_ip_port_len, "%s;%i",
      local_ip_port_buf, ntohs(name.sin_port));
  
  return 0;
}

static struct etpan_error * smtp_connect(struct etpan_sender * sender)
{
  struct sender_data * data;
  mailsmtp * smtp;
  int r;
  struct etpan_error * error;
  
  data = etpan_sender_get_data(sender);
  
  if (data->connected) {
    r = mailsmtp_noop(data->smtp);
    if (r == MAILSMTP_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAILSMTP_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, _("Connection closed"));
      etpan_error_strf_long_description(error, _("Connection was closed while connecting to server %s:%i. Retry later."), data->threaded_hostname, data->threaded_port);
      goto err;
    }
    else if (r != MAILSMTP_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while connecting to server %s:%i"), r, data->threaded_hostname, data->threaded_port);
      goto err;
    }
    
    return NULL;
  }
  
  smtp = mailsmtp_new(0, NULL);
  if (smtp == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailsmtp_socket_connect(smtp,
      data->threaded_hostname, data->threaded_port);
  if (r == MAILSMTP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILSMTP_ERROR_CONNECTION_REFUSED) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection refuse"));
    etpan_error_strf_long_description(error, _("Connection was not established to server %s:%i. Check hostname and port."), data->threaded_hostname, data->threaded_port);
    goto free;
  }
  else if (r == MAILSMTP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while connecting to server %s:%i. Retry later."), data->threaded_hostname, data->threaded_port);
    goto free;
  }
  else if (r != MAILSMTP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while connecting to server %s:%i."), r, data->threaded_hostname, data->threaded_port);
    goto free;
  }
  
  r = mailsmtp_init(smtp);
  if (r == MAILSMTP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILSMTP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while connecting to server %s:%i. Retry later."), data->threaded_hostname, data->threaded_port);
    goto disconnect;
  }
  else if (r != MAILSMTP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while connecting to server %s:%i."), r, data->threaded_hostname, data->threaded_port);
    goto disconnect;
  }
  
  if (data->threaded_auth_type != ETPAN_AUTH_NONE) {
    char local_ip_port_buf[128];
    char remote_ip_port_buf[128];
    char * local_ip_port;
    char * remote_ip_port;
    
    r = fill_local_ip_port(smtp->stream, local_ip_port_buf,
        sizeof(local_ip_port_buf));
    if (r < 0)
      local_ip_port = NULL;
    else
      local_ip_port = local_ip_port_buf;
    
    r = fill_remote_ip_port(smtp->stream, remote_ip_port_buf,
        sizeof(remote_ip_port_buf));
    if (r < 0)
      remote_ip_port = NULL;
    else
      remote_ip_port = remote_ip_port_buf;
    
    if (local_ip_port == NULL)
      remote_ip_port = NULL;
    if (remote_ip_port == NULL)
      local_ip_port = NULL;
    
    r = mailesmtp_auth_sasl(smtp, "PLAIN",
        data->threaded_hostname,
        local_ip_port, remote_ip_port,
        data->threaded_username, data->threaded_username,
        data->threaded_password, data->threaded_hostname);
    if (r == MAILSMTP_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAILSMTP_ERROR_AUTH_LOGIN) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, _("Authentication failure"));
      etpan_error_strf_long_description(error, _("Authentication failed while connecting to server %s:%i. Check your login and password."), data->threaded_hostname, data->threaded_port);
      goto disconnect;
    }
    else if (r == MAILSMTP_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, _("Connection closed"));
      etpan_error_strf_long_description(error, _("Connection was closed while connecting to server %s:%i. Retry later."), data->threaded_hostname, data->threaded_port);
      goto disconnect;
    }
    else if (r != MAILSMTP_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while connecting to server %s:%i."), r, data->threaded_hostname, data->threaded_port);
      goto disconnect;
    }
  }
  
  data->smtp = smtp;
  data->connected = 1;
  
  return NULL;
  
 disconnect:
  mailsmtp_quit(smtp);
 free:
  mailsmtp_free(smtp);
 err:
  return error;
}

#define MAX_FROM 1024

static int get_from(char * result_from, size_t result_from_max,
    struct mailimf_fields * fields)
{
  struct mailimf_single_fields single_fields;
  struct mailimf_mailbox * mb_from;
  char * from;
  char from_value[MAX_FROM];
  
  etpan_single_resent_fields_init(&single_fields, fields);
  if ((single_fields.fld_to != NULL) || (single_fields.fld_cc != NULL) ||
      (single_fields.fld_bcc != NULL) || (single_fields.fld_from != NULL)) {
    /* resent fields */
    
    /* do nothing */
  }
  else {
    mailimf_single_fields_init(&single_fields, fields);
  }
  
  mb_from = NULL;
  if (single_fields.fld_sender != NULL) {
    mb_from = single_fields.fld_sender->snd_mb;
  }
  else if (single_fields.fld_from != NULL) {
    clistiter * iter;
    
    iter = clist_begin(single_fields.fld_from->frm_mb_list->mb_list);
    if (iter != NULL)
      mb_from = clist_content(iter);
  }
  
  if (mb_from == NULL) {
    char hostname[MAX_FROM];
    int r;
    char * user;
    
    /* must build dummy sender */
    user = getenv("USER");
    if (user == NULL) {
      goto err;
    }
    
    r = gethostname(hostname, sizeof(hostname));
    if (r < 0) {
      goto err;
    }
    
    snprintf(from_value, sizeof(from_value), "%s@%s", user, hostname);
    from = from_value;
  }
  else {
    from = mb_from->mb_addr_spec;
  }
  
  strncpy(result_from, from, result_from_max);
  result_from[result_from_max - 1] = '\0';
  
  return 0;
  
 err:
  return -1;
}


static struct mailimf_mailbox_list *
get_recipient_list(struct mailimf_fields * fields)
{
  struct mailimf_single_fields single_fields;
  struct mailimf_mailbox_list * to_list;
  struct mailimf_mailbox_list * cc_list;
  struct mailimf_mailbox_list * bcc_list;
  struct mailimf_mailbox_list * recipient_list;
  
  etpan_single_resent_fields_init(&single_fields, fields);
  if ((single_fields.fld_to != NULL) || (single_fields.fld_cc != NULL) ||
      (single_fields.fld_bcc != NULL) || (single_fields.fld_from != NULL)) {
    /* resent fields */
    
    /* do nothing */
  }
  else {
    mailimf_single_fields_init(&single_fields, fields);
  }
  
  /* build recipient list */
  
  to_list = NULL;
  if (single_fields.fld_to != NULL) {
    to_list = etpan_address_to_mailbox_list(single_fields.fld_to->to_addr_list);
  }
  
  cc_list = NULL;
  if (single_fields.fld_cc != NULL) {
    cc_list = etpan_address_to_mailbox_list(single_fields.fld_cc->cc_addr_list);
  }
  
  bcc_list = NULL;
  if (single_fields.fld_bcc != NULL) {
    bcc_list = etpan_address_to_mailbox_list(single_fields.fld_bcc->bcc_addr_list);
  }
  
  recipient_list = mailimf_mailbox_list_new_empty();
  if (recipient_list == NULL) {
    if (bcc_list != NULL)
      mailimf_mailbox_list_free(bcc_list);
    if (cc_list != NULL)
      mailimf_mailbox_list_free(cc_list);
    if (to_list != NULL)
      mailimf_mailbox_list_free(to_list);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (to_list != NULL) {
    etpan_append_mailbox_list(recipient_list, to_list);
  }

  if (cc_list != NULL) {
    etpan_append_mailbox_list(recipient_list, cc_list);
  }

  if (bcc_list != NULL) {
    etpan_append_mailbox_list(recipient_list, bcc_list);
  }
  
  if (bcc_list != NULL)
    mailimf_mailbox_list_free(bcc_list);
  if (cc_list != NULL)
    mailimf_mailbox_list_free(cc_list);
  if (to_list != NULL)
    mailimf_mailbox_list_free(to_list);
  
  return recipient_list;
}

static struct etpan_error * send_message(struct etpan_sender * sender,
    char * message, size_t size)
{
  size_t cur_token;
  struct mailimf_fields * fields;
  int r;
  char from[1024];
  struct sender_data * data;
  clist * smtp_recipient_list;
  clistiter * cur;
  struct mailimf_mailbox_list * recipient_list;
  struct etpan_error * error;
  
  data = etpan_sender_get_data(sender);
  
  cur_token = 0;
  
  r = mailimf_fields_parse(message, size, &cur_token,
      &fields);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = get_from(from, sizeof(from), fields);
  if (r < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_NO_FROM);
    etpan_error_set_short_description(error, _("No sender"));
    etpan_error_set_long_description(error, _("The message has no sender."));
    goto free_fields;
  }
  
  recipient_list = get_recipient_list(fields);
  if (recipient_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (clist_count(recipient_list->mb_list) == 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_NO_RECIPIENT);
    etpan_error_set_short_description(error, _("No recipient"));
    etpan_error_set_long_description(error, _("The message has no recipient."));
    goto free_recipient;
  }
  
  smtp_recipient_list = smtp_address_list_new();
  if (smtp_recipient_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(cur = clist_begin(recipient_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    
    mb = clist_content(cur);
    
    r = smtp_address_list_add(smtp_recipient_list, mb->mb_addr_spec);
    if (r != MAILSMTP_NO_ERROR) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  r = mailsmtp_send(data->smtp, from, smtp_recipient_list,
      message, size);
  mailsmtp_reset(data->smtp);
  
  if (r == MAILSMTP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILSMTP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while sending message from account %s. Retry later."), etpan_sender_get_id(sender));
    goto free_smtp_list;
  }
  else if (r != MAILSMTP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_SEND);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while sending message from account %s."), r, etpan_sender_get_id(sender));
    goto free_smtp_list;
  }
  
  smtp_address_list_free(smtp_recipient_list);
  mailimf_mailbox_list_free(recipient_list);
  mailimf_fields_free(fields);
  
  return NULL;
  
 free_smtp_list:
  smtp_address_list_free(smtp_recipient_list);
 free_recipient:
  mailimf_mailbox_list_free(recipient_list);
 free_fields:
  mailimf_fields_free(fields);
  return error;
}

static void free_data(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  smtp_disconnect(sender);
  
  data = etpan_sender_get_data(sender);
  
  free(data->threaded_password);
  free(data->threaded_username);
  free(data->threaded_hostname);
  
  free(data->username);
  free(data->password);
  free(data->hostname);
  
  free(data);
}

static struct etpan_error * setup(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  
  free(data->threaded_hostname);
  if (data->hostname != NULL) {
    data->threaded_hostname = strdup(data->hostname);
    if (data->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data->threaded_port = data->port;
  data->threaded_connection_type = data->connection_type;
  data->threaded_auth_type = data->auth_type;
  
  free(data->threaded_username);
  if (data->username != NULL) {
    data->threaded_username = strdup(data->username);
    if (data->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  free(data->threaded_password);
  if (data->password != NULL) {
    data->threaded_password = strdup(data->password);
    if (data->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return NULL;
}

struct etpan_sender * etpan_sender_smtp_new(void)
{
  struct etpan_sender * sender;
  struct sender_data * data;
  
  sender = etpan_sender_new();
  if (sender == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->hostname = NULL;
  data->port = 0;
  data->connection_type = ETPAN_CONNECTION_PLAIN;
  data->auth_type = ETPAN_AUTH_NONE;
  data->username = NULL;
  data->password = NULL;
  data->threaded_hostname = NULL;
  data->threaded_port = 0;
  data->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  data->threaded_auth_type = ETPAN_AUTH_NONE;
  data->threaded_username = NULL;
  data->threaded_password = NULL;
  
  data->connected = 0;
  data->smtp = NULL;
  
  etpan_sender_set_data(sender, data);
  etpan_sender_set_driver(sender, &smtp_driver);
  
  return sender;
}

void etpan_sender_smtp_set_hostname(struct etpan_sender * sender,
    char * hostname)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  if (hostname != data->hostname) {
    free(data->hostname);
    if (hostname != NULL) {
      data->hostname = strdup(hostname);
      if (data->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->hostname = NULL;
  }
}

char * etpan_sender_smtp_get_hostname(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  return data->hostname;
}

void etpan_sender_smtp_set_port(struct etpan_sender * sender, int port)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  data->port = port;
}

int etpan_sender_smtp_get_port(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  return data->port;
}

void etpan_sender_smtp_set_connection_type(struct etpan_sender * sender,
    int connection_type)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  data->connection_type = connection_type;
}

int etpan_sender_smtp_get_connection_type(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  return data->connection_type;
}

void etpan_sender_smtp_set_auth_type(struct etpan_sender * sender,
    int auth_type)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  data->auth_type = auth_type;
}

int etpan_sender_smtp_get_auth_type(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  return data->auth_type;
}

void etpan_sender_smtp_set_username(struct etpan_sender * sender,
    char * username)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  if (username != data->username) {
    free(data->username);
    if (username != NULL) {
      data->username = strdup(username);
      if (data->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->username = NULL;
  }
}

char * etpan_sender_smtp_get_username(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  return data->username;
}

void etpan_sender_smtp_set_password(struct etpan_sender * sender,
    char * password)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  if (password != data->password) {
    free(data->password);
    if (password != NULL) {
      data->password = strdup(password);
      if (data->password == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->password = NULL;
  }
}

char * etpan_sender_smtp_get_password(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  return data->password;
}
