#include "etpan-sender.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "etpan-error.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-utils.h"
#include "etpan-log.h"
#include "etpan-nls.h"

static struct etpan_error * sender_connect(struct etpan_sender * sender);
static void sender_disconnect(struct etpan_sender * sender);

struct etpan_sender * etpan_sender_new(void)
{
  struct etpan_sender * sender;
  
  sender = malloc(sizeof(* sender));
  if (sender == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  sender->id = NULL;
  sender->ref_count = 1;
  sender->account = NULL;
  sender->driver = NULL;
  sender->data = NULL;
  
  return sender;
}

void etpan_sender_free(struct etpan_sender * sender)
{
  if (sender->driver != NULL) {
    if (sender->driver->free_data != NULL)
      sender->driver->free_data(sender);
  }
  
  free(sender->id);
  free(sender);
}

void etpan_sender_set_id(struct etpan_sender * sender, char * id)
{
  if (id != sender->id) {
    free(sender->id);
    if (id != NULL) {
      sender->id = strdup(id);
      if (sender->id == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      sender->id = NULL;
  }
}

char * etpan_sender_get_id(struct etpan_sender * sender)
{
  return sender->id;
}

void etpan_sender_set_account(struct etpan_sender * sender,
    struct etpan_account * account)
{
  sender->account = account;
}

struct etpan_account * etpan_sender_get_account(struct etpan_sender * sender)
{
  return sender->account;
}

struct etpan_error *
etpan_sender_setup(struct etpan_sender * sender)
{
  if (sender->driver->setup != NULL) {
    struct etpan_error * error;
    
    error = sender->driver->setup(sender);
    if (error != NULL)
      return error;
  }
  
  return NULL;
}

void etpan_sender_unsetup(struct etpan_sender * sender)
{
  etpan_thread_manager_app_unbind_sender_thread(etpan_thread_manager_app_get_default(), sender);
}

void etpan_sender_set_data(struct etpan_sender * sender, void * data)
{
  sender->data = data;
}

void * etpan_sender_get_data(struct etpan_sender * sender)
{
  return sender->data;
}

void etpan_sender_set_driver(struct etpan_sender * sender,
    struct etpan_sender_driver * driver)
{
  sender->driver = driver;
}

struct etpan_sender_driver *
etpan_sender_get_driver(struct etpan_sender * sender)
{
  return sender->driver;
}


struct etpan_sender_send_message_param {
  struct etpan_sender * sender;
  char * filename;
};

static void
send_message_cleanup(struct etpan_thread_op * op)
{
  struct etpan_sender_send_message_param * param;
  
  param = op->param;
  unlink(param->filename);
  free(param->filename);
  free(op->param);
  op->param = NULL;
  
  if (op->result != NULL) {
    struct etpan_sender_send_message_result * result;
    
    result = op->result;
    
    free(op->result);
    op->result = NULL;
  }
}

static void
send_message_run(struct etpan_thread_op * op)
{
  struct etpan_sender_send_message_param * param;
  struct etpan_sender_send_message_result * result;
  struct etpan_error * error;
  int fd;
  char * message;
  size_t size;
  struct stat stat_buf;
  int r;
  
  param = op->param;
  result = op->result;

  if (param->filename[0] == '\0') {
    char long_description[1024];
  
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Message could not be sent"));
    snprintf(long_description, sizeof(long_description),
        _("The message could not be sent from account %s\n"
            "Temporary file could not be written."),
        etpan_sender_get_id(param->sender));
    
    error = etpan_error_internal(long_description);
    goto err;
  }
  
  if (param->sender->driver->send_message == NULL) {
    char long_description[1024];
    
    snprintf(long_description, sizeof(long_description),
        _("send-message is not implemented for %s"),
        param->sender->driver->name);
    
    error = etpan_error_internal(long_description);
    
    goto err;
  }
  
  error = sender_connect(param->sender);
  if (error != NULL) {
    goto err;
  }
  
  ETPAN_LOG("sending message %s", param->filename);
  fd = open(param->filename, O_RDONLY);
  if (fd < 0) {
    goto file_error;
  }
  
  r = fstat(fd, &stat_buf);
  if (r < 0) {
    goto file_error;
  }
  
  size = stat_buf.st_size;
  message = mmap(NULL, size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (message == (void *) MAP_FAILED) {
    goto file_error;
  }
  
  error = param->sender->driver->send_message(param->sender, message, size);
  
  munmap(message, size);
  close(fd);
  
  ETPAN_LOG("message sent %s", param->filename);
  result->error = error;
  
  return;
  
 file_error:
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_FILE);
  etpan_error_set_short_description(error, _("Message could not be sent"));
  etpan_error_strf_long_description(error, _("The message could not be sent from account %s\n"
                                        "Data from file %s could not be read."),
      etpan_sender_get_id(param->sender),
      param->filename);
  if (fd != -1)
    close(fd);
 err:
  result->error = error;
}

struct etpan_thread_op *
etpan_sender_send_message(struct etpan_thread_manager_app * manager,
    struct etpan_sender * sender, char * message, size_t size,
    void (* callback)(int, struct etpan_sender_send_message_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_sender_send_message_param * param;
  struct etpan_sender_send_message_result * result;
  int r;
  char path[PATH_MAX];
  
  r = etpan_get_tmp_filename(path, sizeof(path));
  if (r < 0) {
    * path = '\0'; 
  }
  else {
    r = etpan_write_file(path, message, size);
    if (r < 0) {
      unlink(path);
      * path = '\0'; 
    }
  }
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->filename = strdup(path);
  if (param->filename == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->sender = sender;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = send_message_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = send_message_cleanup;
  
  etpan_thread_manager_app_sender_schedule(manager, sender, op);
  
  return op;
}


/* disconnect */

struct etpan_sender_disconnect_param {
  struct etpan_sender * sender;
};

static void
disconnect_cleanup(struct etpan_thread_op * op)
{
  struct etpan_sender_disconnect_param * param;
  
  param = op->param;
  free(param);
  op->param = NULL;
  
  free(op->result);
  op->result = NULL;
}

static void
disconnect_run(struct etpan_thread_op * op)
{
  struct etpan_sender_disconnect_param * param;
  
  param = op->param;
  
  sender_disconnect(param->sender);
}

struct etpan_thread_op *
etpan_sender_disconnect(struct etpan_thread_manager_app * manager,
    struct etpan_sender * sender,
    void (* callback)(int, void * /* dummy argument */,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_sender_disconnect_param * param;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->sender = sender;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = disconnect_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = disconnect_cleanup;
  
  etpan_thread_manager_app_sender_schedule(manager, sender, op);
  
  return op;
}


static struct etpan_error * sender_connect(struct etpan_sender * sender)
{
  struct etpan_error * error;
  
  /* try to connect */
  if (sender->driver->connect != NULL)
    error = sender->driver->connect(sender);
  else
    error = NULL;
  
  /* disconnect everything if it failed */
  if (error != NULL) {
    ETPAN_LOG("reconnection of %s", etpan_sender_get_id(sender));
    
    sender_disconnect(sender);
    
    /* retry connection */
    if (sender->driver->connect != NULL)
      error = sender->driver->connect(sender);
    else
      error = NULL;
  }
  
  return error;
}

static void sender_disconnect(struct etpan_sender * sender)
{
  if (sender->driver->disconnect != NULL)
    sender->driver->disconnect(sender);
}
