#include "etpan-serialize.h"

#include <string.h>
#include <stdlib.h>
#include <libetpan/libetpan.h>

#include "etpan-address.h"
#include "etpan-log.h"

static void etpan_serialize_encode_mmapstr(MMAPString * str, struct etpan_serialize_data * data);
static struct etpan_serialize_data * etpan_serialize_decode_mmapstr(MMAPString * str,
    size_t * p_offset);

struct etpan_serialize_data * etpan_serialize_data_new_int32(int32_t value)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_INT32;
  data->data.data_int32 = value;
  
  return data;
}

struct etpan_serialize_data * etpan_serialize_data_new_int64(int64_t value)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_INT64;
  data->data.data_int64 = value;
  
  return data;
}

struct etpan_serialize_data * etpan_serialize_data_new_uint32(uint32_t value)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_UINT32;
  data->data.data_uint32 = value;
  
  return data;
}

struct etpan_serialize_data * etpan_serialize_data_new_uint64(uint64_t value)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_UINT64;
  data->data.data_uint64 = value;
  
  return data;
}

struct etpan_serialize_data * etpan_serialize_data_new_str(char * value)
{
#if 0
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_STRING;
  if (value != NULL) {
    data->data.data_str.value = strdup(value);
    if (data->data.data_str == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    data->data.data_str.length = strlen(value);
  }
  else {
    data->data.data_str.value = NULL;
    data->data.data_str.length = 0;
  }
  
  return data;
#endif
  if (value != NULL)
    return etpan_serialize_data_new_sized_str(value, strlen(value));
  else
    return etpan_serialize_data_new_sized_str(value, 0);
}

struct etpan_serialize_data * etpan_serialize_data_new_sized_str(char * value, size_t length)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_STRING;
  if (value != NULL) {
    data->data.data_str.value = malloc(length + 1);
    if (data->data.data_str.value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    memcpy(data->data.data_str.value, value, length);
    data->data.data_str.value[length] = '\0';
    data->data.data_str.length = length;
  }
  else {
    data->data.data_str.value = NULL;
    data->data.data_str.length = 0;
  }
  
  return data;
}

struct etpan_serialize_data * etpan_serialize_data_new_array(void)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_ARRAY;
  data->data.data_array = carray_new(4);
  if (data->data.data_array == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return data;
}

struct etpan_serialize_data * etpan_serialize_data_new_hash(void)
{
  struct etpan_serialize_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->type = ETPAN_SERIALIZE_HASH;
  data->data.data_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (data->data.data_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return data;
}


void etpan_serialize_hash_set(struct etpan_serialize_data * data,
    char * key, struct etpan_serialize_data * item)
{
  chashdatum key_datum;
  chashdatum value;
  int r;
  
  key_datum.data = key;
  key_datum.len = strlen(key) + 1;
  value.data = item;
  value.len = 0;
  
  r = chash_set(data->data.data_hash, &key_datum, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_serialize_hash_set_int32(struct etpan_serialize_data * data,
    char * key, int32_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_int32(value);
  etpan_serialize_hash_set(data, key, item);
}

void etpan_serialize_hash_set_int64(struct etpan_serialize_data * data,
    char * key, int64_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_int64(value);
  etpan_serialize_hash_set(data, key, item);
}

void etpan_serialize_hash_set_uint32(struct etpan_serialize_data * data,
    char * key, uint32_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_uint32(value);
  etpan_serialize_hash_set(data, key, item);
}

void etpan_serialize_hash_set_uint64(struct etpan_serialize_data * data,
    char * key, uint64_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_uint64(value);
  etpan_serialize_hash_set(data, key, item);
}

void etpan_serialize_hash_set_str(struct etpan_serialize_data * data,
    char * key, char * value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_str(value);
  etpan_serialize_hash_set(data, key, item);
}

void etpan_serialize_hash_set_sized_str(struct etpan_serialize_data * data,
    char * key, char * value, size_t length)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_sized_str(value, length);
  etpan_serialize_hash_set(data, key, item);
}

void etpan_serialize_array_add(struct etpan_serialize_data * data,
    struct etpan_serialize_data * item)
{
  int r;
  
  r = carray_add(data->data.data_array, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_serialize_array_add_int32(struct etpan_serialize_data * data,
    int32_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_int32(value);
  etpan_serialize_array_add(data, item);
}

void etpan_serialize_array_add_int64(struct etpan_serialize_data * data,
    int64_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_int64(value);
  etpan_serialize_array_add(data, item);
}

void etpan_serialize_array_add_uint32(struct etpan_serialize_data * data,
    uint32_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_uint32(value);
  etpan_serialize_array_add(data, item);
}

void etpan_serialize_array_add_uint64(struct etpan_serialize_data * data,
    uint64_t value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_uint64(value);
  etpan_serialize_array_add(data, item);
}

void etpan_serialize_array_add_str(struct etpan_serialize_data * data,
    char * value)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_str(value);
  etpan_serialize_array_add(data, item);
}

void etpan_serialize_array_add_sized_str(struct etpan_serialize_data * data,
    char * value, size_t length)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_data_new_sized_str(value, length);
  etpan_serialize_array_add(data, item);
}

int32_t etpan_serialize_data_get_int32(struct etpan_serialize_data * data)
{
  return data->data.data_int32;
}

int64_t etpan_serialize_data_get_int64(struct etpan_serialize_data * data)
{
  return data->data.data_int64;
}

uint32_t etpan_serialize_data_get_uint32(struct etpan_serialize_data * data)
{
  return data->data.data_uint32;
}

uint64_t etpan_serialize_data_get_uint64(struct etpan_serialize_data * data)
{
  return data->data.data_uint64;
}

char * etpan_serialize_data_get_str(struct etpan_serialize_data * data)
{
  return data->data.data_str.value;
}

size_t etpan_serialize_data_get_str_length(struct etpan_serialize_data * data)
{
  return data->data.data_str.length;
}

void etpan_serialize_data_get_sized_str(struct etpan_serialize_data * data,
    char ** p_value, size_t * p_length)
{
  if (p_value != NULL)
    * p_value = data->data.data_str.value;
  if (p_length != NULL)
    * p_length = data->data.data_str.length;
}

struct etpan_serialize_data * etpan_serialize_array_get(struct etpan_serialize_data * data, unsigned int i)
{
  return carray_get(data->data.data_array, i);
}

struct etpan_serialize_data * etpan_serialize_hash_get(struct etpan_serialize_data * data, char * key)
{
  chashdatum key_datum;
  chashdatum value;
  int r;
  
  key_datum.data = key;
  key_datum.len = strlen(key) + 1;
  
  r = chash_get(data->data.data_hash, &key_datum, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

int32_t etpan_serialize_hash_get_int32(struct etpan_serialize_data * data, char * key)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_hash_get(data, key);
  return etpan_serialize_data_get_int32(item);
}

int64_t etpan_serialize_hash_get_int64(struct etpan_serialize_data * data, char * key)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_hash_get(data, key);
  return etpan_serialize_data_get_int64(item);
}

uint32_t etpan_serialize_hash_get_uint32(struct etpan_serialize_data * data, char * key)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_hash_get(data, key);
  return etpan_serialize_data_get_uint32(item);
}

uint64_t etpan_serialize_hash_get_uint64(struct etpan_serialize_data * data, char * key)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_hash_get(data, key);
  return etpan_serialize_data_get_uint64(item);
}

char * etpan_serialize_hash_get_str(struct etpan_serialize_data * data, char * key)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_hash_get(data, key);
  return etpan_serialize_data_get_str(item);
}

void etpan_serialize_hash_get_sized_str(struct etpan_serialize_data * data,
    char * key, char ** p_value, size_t * p_length)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_hash_get(data, key);
  etpan_serialize_data_get_sized_str(item, p_value, p_length);
}

unsigned int etpan_serialize_array_count(struct etpan_serialize_data * data)
{
  return carray_count(data->data.data_array);
}

int32_t etpan_serialize_array_get_int32(struct etpan_serialize_data * data,
    unsigned int i)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_array_get(data, i);
  return etpan_serialize_data_get_int32(item);
}

int64_t etpan_serialize_array_get_int64(struct etpan_serialize_data * data,
    unsigned int i)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_array_get(data, i);
  return etpan_serialize_data_get_int32(item);
}

uint32_t etpan_serialize_array_get_uint32(struct etpan_serialize_data * data,
    unsigned int i)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_array_get(data, i);
  return etpan_serialize_data_get_uint32(item);
}

uint64_t etpan_serialize_array_get_uint64(struct etpan_serialize_data * data,
    unsigned int i)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_array_get(data, i);
  return etpan_serialize_data_get_uint64(item);
}


char * etpan_serialize_array_get_str(struct etpan_serialize_data * data,
    unsigned int i)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_array_get(data, i);
  return etpan_serialize_data_get_str(item);
}

void etpan_serialize_array_get_sized_str(struct etpan_serialize_data * data,
    unsigned int i, char ** p_value, size_t * p_length)
{
  struct etpan_serialize_data * item;
  
  item = etpan_serialize_array_get(data, i);
  etpan_serialize_data_get_sized_str(item, p_value, p_length);
}

static void put_uint32(MMAPString * str, uint32_t value)
{
  unsigned char tab[4];
  unsigned int i;
  
  for(i = 0 ; i < 4 ; i ++) {
    tab[i] = value & 0xff;
    value = value >> 8;
  }
  
  if (mmap_string_append_len(str, (char *) tab, 4) == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

static void put_uint64(MMAPString * str, uint64_t value)
{
  unsigned char tab[8];
  unsigned int i;
  
  for(i = 0 ; i < 8 ; i ++) {
    tab[i] = value & 0xff;
    value = value >> 8;
  }
  
  if (mmap_string_append_len(str, (char *) tab, 8) == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

static void put_sized_str(MMAPString * str, char * value, size_t length)
{
  if (value == NULL) {
    put_uint32(str, 0);
    return;
  }
  
  put_uint32(str, 1);
  put_uint32(str, length);
  mmap_string_append_len(str, value, length);
}

static void encode_array(MMAPString * str, carray * tab)
{
  unsigned int i;
  unsigned int len;
  
  len = carray_count(tab);
  put_uint32(str, len);
  for(i = 0 ; i < len ; i ++) {
    struct etpan_serialize_data * item;
    
    item = carray_get(tab, i);
    etpan_serialize_encode_mmapstr(str, item);
  }
}

static void encode_hash(MMAPString * str, chash * hash)
{
  chashiter * iter;
  unsigned int len;
  
  len = chash_count(hash);
  put_uint32(str, len);
  
  for(iter = chash_begin(hash) ; iter != NULL ;
      iter = chash_next(hash, iter)) {
    chashdatum key;
    chashdatum value;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    
    put_sized_str(str, key.data, key.len);
    etpan_serialize_encode_mmapstr(str, value.data);
  }
}

static void etpan_serialize_encode_mmapstr(MMAPString * str, struct etpan_serialize_data * data)
{
  put_uint32(str, data->type);
  switch (data->type) {
  case ETPAN_SERIALIZE_INT32:
    put_uint32(str, (int32_t) data->data.data_int32);
    break;
  case ETPAN_SERIALIZE_INT64:
    put_uint64(str, (uint64_t) data->data.data_int64);
    break;
  case ETPAN_SERIALIZE_UINT32:
    put_uint32(str, data->data.data_uint32);
    break;
  case ETPAN_SERIALIZE_UINT64:
    put_uint64(str, data->data.data_uint64);
    break;
  case ETPAN_SERIALIZE_STRING:
    put_sized_str(str, data->data.data_str.value, data->data.data_str.length);
    break;
  case ETPAN_SERIALIZE_ARRAY:
    encode_array(str, data->data.data_array);
    break;
  case ETPAN_SERIALIZE_HASH:
    encode_hash(str, data->data.data_hash);
    break;
  }
}

static uint32_t get_uint32(MMAPString * str, size_t * p_offset)
{
  unsigned char tab[4];
  unsigned int shift_value;
  uint32_t value;
  unsigned int i;
  
  value = 0;
  shift_value = 0;
  memcpy(tab, str->str + * p_offset, 4);
  shift_value = 0;
  for(i = 0 ; i < 4 ; i ++) {
    value += tab[i] << shift_value;
    shift_value += 8;
  }
  
  * p_offset += 4;
  
  return value;
}

static uint64_t get_uint64(MMAPString * str, size_t * p_offset)
{
  unsigned char tab[8];
  unsigned int shift_value;
  uint32_t value;
  unsigned int i;
  
  value = 0;
  shift_value = 0;
  memcpy(tab, str->str + * p_offset, 4);
  shift_value = 0;
  for(i = 0 ; i < 8 ; i ++) {
    value += ((uint64_t) tab[i]) << shift_value;
    shift_value += 8;
  }
  
  * p_offset += 8;
  
  return value;
}

static void get_sized_str(MMAPString * str, size_t * p_offset,
    char ** p_value, size_t * p_length)
{
  uint32_t not_null;
  uint32_t len;
  char * value;
  
#if 0
  if (str->len == 1712)
  {
    unsigned int i;
    for(i = * p_offset ; i < str->len ; i ++) {
      fprintf(stderr, "%02x", (unsigned char) str->str[i]);
    }
    fprintf(stderr, "\n");
  }
#endif
  
  not_null = get_uint32(str, p_offset);
  if (!not_null) {
    if (p_value != NULL)
      * p_value = NULL;
    if (p_length != NULL)
      * p_length = 0;
    return;
  }
  
  len = get_uint32(str, p_offset);
  if (p_length != NULL)
    * p_length = len;
  
  if (p_value != NULL) {
    value = malloc(len + 1);
    if (value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  
#if 0
    ETPAN_LOG("unserialize str len: %i %i/%i %p %p", not_null, len + * p_offset, str->len, value, str->str);
#endif
    value[len] = '\0';
    memcpy(value, str->str + * p_offset, len);
    * p_value = value;
  }
  * p_offset += len;
}

static struct etpan_serialize_data * get_array(MMAPString * str,
    size_t * p_offset)
{
  uint32_t count;
  unsigned int i;
  struct etpan_serialize_data * data;
  
  data = etpan_serialize_data_new_array();
  
  count = get_uint32(str, p_offset);
  for(i = 0 ; i < count ; i ++) {
    struct etpan_serialize_data * item;
    
    item = etpan_serialize_decode_mmapstr(str, p_offset);
    etpan_serialize_array_add(data, item);
  }
  
  return data;
}

static struct etpan_serialize_data * get_hash(MMAPString * str,
    size_t * p_offset)
{
  struct etpan_serialize_data * data;
  uint32_t count;
  unsigned int i;
  
  data = etpan_serialize_data_new_hash();
  count = get_uint32(str, p_offset);
  for(i = 0 ; i < count ; i ++) {
    struct etpan_serialize_data * item;
    char * key;
    
    key = NULL;
    get_sized_str(str, p_offset, &key, NULL);
#if 0
    ETPAN_LOG("hash: %s", key);
#endif
    item = etpan_serialize_decode_mmapstr(str, p_offset);
    etpan_serialize_hash_set(data, key, item);
    free(key);
  }
  
  return data;
}

static struct etpan_serialize_data * etpan_serialize_decode_mmapstr(MMAPString * str,
    size_t * p_offset)
{
  size_t type;
  
  type = get_uint32(str, p_offset);
  switch (type) {
  case ETPAN_SERIALIZE_INT32:
    {
      int32_t value;
      
#if 0
      ETPAN_LOG("decode int32");
#endif
      value = get_uint32(str, p_offset);
      return etpan_serialize_data_new_int32(value);
    }
    break;
  case ETPAN_SERIALIZE_INT64:
    {
      int64_t value;
      
#if 0
      ETPAN_LOG("decode int64");
#endif
      value = get_uint64(str, p_offset);
      return etpan_serialize_data_new_int64(value);
    }
    break;
  case ETPAN_SERIALIZE_UINT32:
    {
      uint32_t value;
      
#if 0
      ETPAN_LOG("decode uint32");
#endif
      value = get_uint32(str, p_offset);
      return etpan_serialize_data_new_uint32(value);
    }
    break;
  case ETPAN_SERIALIZE_UINT64:
    {
      uint64_t value;
      
#if 0
      ETPAN_LOG("decode uint64");
#endif
      value = get_uint32(str, p_offset);
      return etpan_serialize_data_new_uint64(value);
    }
    break;
  case ETPAN_SERIALIZE_STRING:
    {
      char * value;
      size_t length;
      struct etpan_serialize_data * data;
      
#if 0
      ETPAN_LOG("decode string");
#endif
      get_sized_str(str, p_offset, &value, &length);
      data = etpan_serialize_data_new_sized_str(value, length);
      free(value);
      
      return data;
    }
    break;
  case ETPAN_SERIALIZE_ARRAY:
#if 0
    ETPAN_LOG("decode array");
#endif
    return get_array(str, p_offset);
    break;
  case ETPAN_SERIALIZE_HASH:
#if 0
    ETPAN_LOG("decode hash");
#endif
    return get_hash(str, p_offset);
    break;
  }
  
  return NULL;
}

void etpan_serialize_data_free(struct etpan_serialize_data * data)
{
  switch (data->type) {
  case ETPAN_SERIALIZE_INT32:
  case ETPAN_SERIALIZE_INT64:
  case ETPAN_SERIALIZE_UINT32:
  case ETPAN_SERIALIZE_UINT64:
    break;
  case ETPAN_SERIALIZE_STRING:
    free(data->data.data_str.value);
    break;
  case ETPAN_SERIALIZE_ARRAY:
    {
      unsigned int i;
      
      for(i = 0 ; i < carray_count(data->data.data_array) ; i ++) {
        struct etpan_serialize_data * item;
        
        item = carray_get(data->data.data_array, i);
        etpan_serialize_data_free(item);
      }
      carray_free(data->data.data_array);
    }
    break;
  case ETPAN_SERIALIZE_HASH:
    {
      chashiter * iter;
      
      for(iter = chash_begin(data->data.data_hash) ; iter != NULL ;
          iter = chash_next(data->data.data_hash, iter)) {
        chashdatum value;
        struct etpan_serialize_data * item;
        
        chash_value(iter, &value);
        item = value.data;
        
        etpan_serialize_data_free(item);
      }
      chash_free(data->data.data_hash);
    }
    break;
  }
  free(data);
}

void etpan_serialize_encode(struct etpan_serialize_data * sdata,
    void ** p_data, size_t * p_length)
{
  MMAPString * str;
  void * data;
  size_t length;
  
  str = mmap_string_new("");
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_serialize_encode_mmapstr(str, sdata);
  
  length = str->len;
  data = malloc(length);
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  memcpy(data, str->str, length);
  
  mmap_string_free(str);
  
  * p_data = data;
  * p_length = length;
}

struct etpan_serialize_data * etpan_serialize_decode(void * data, size_t size)
{
  MMAPString * str;
  size_t offset;
  struct etpan_serialize_data * sdata;
  
  str = mmap_string_new_len(data, size);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
#if 0
  {
    unsigned int i;
    for(i = 0 ; i < str->len ; i ++) {
      fprintf(stderr, "%02x", str->str[i]);
    }
    fprintf(stderr, "\n");
  }
#endif
  
  offset = 0;
  sdata =  etpan_serialize_decode_mmapstr(str, &offset);
  mmap_string_free(str);
  
  return sdata;
}

struct etpan_serialize_data *
etpan_serialize_data_dup(struct etpan_serialize_data * data)
{
  void * serialized;
  size_t length;
  struct etpan_serialize_data * result;
  
  /* XXX - should be improved */
  etpan_serialize_encode(data, &serialized, &length);
  result = etpan_serialize_decode(serialized, length);
  free(serialized);
  
  return result;
}

int etpan_serialize_data_get_type(struct etpan_serialize_data * data)
{
  return data->type;
}
