#include "etpan-signal.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"

#define ETPAN_MODULE_LOG_NAME "SIGNAL"
#include "etpan-log.h"

static struct etpan_signal_handler * handler_new(void * user_data,
    void (* callback)(char *, void *, void *, void *))
{
  struct etpan_signal_handler * handler;
  
  handler = malloc(sizeof(* handler));
  if (handler == NULL)
    return NULL;
  
  handler->user_data = user_data;
  handler->callback = callback;
  
  return handler;
}

static void handler_free(struct etpan_signal_handler * handler)
{
  free(handler);
}

static struct etpan_signal_handler *
handler_dup(struct etpan_signal_handler * handler)
{
  return handler_new(handler->user_data, handler->callback);
}

static int handler_match(struct etpan_signal_handler * handler,
    void * user_data,
    void (* callback)(char *, void *, void *, void *))
{
  return (handler->user_data == user_data) &&
    (handler->callback == callback);
}

static void handler_send(struct etpan_signal_handler * handler,
    char * signal_name, void * sender, void * signal_data)
{
  handler->callback(signal_name, sender, signal_data, handler->user_data);
}

static struct etpan_signal_handler_list * handler_list_new(void)
{
  struct etpan_signal_handler_list * handler_list;
  
  handler_list = malloc(sizeof(* handler_list));
  if (handler_list == NULL)
    goto err;
  
  handler_list->data_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (handler_list == NULL)
    goto free;
  
  return handler_list;
  
 free:
  free(handler_list);
 err:
  return NULL;
}

static void handler_list_free(struct etpan_signal_handler_list * handler_list)
{
  chashiter * iter;
  
  for(iter = chash_begin(handler_list->data_hash) ; iter != NULL ;
      iter = chash_next(handler_list->data_hash, iter)) {
    chashdatum value;
    carray * callback_list;
    unsigned int i;
    
    chash_value(iter, &value);
    callback_list = value.data;
    
    for(i = 0 ; i < carray_count(callback_list) ; i ++) {
      struct etpan_signal_handler * handler;
      
      handler = carray_get(callback_list, i);
      handler_free(handler);
    }
    carray_free(callback_list);
  }
  chash_free(handler_list->data_hash);
  free(handler_list);
}

static inline int
handler_list_add_callback_list(struct etpan_signal_handler_list * handler_list,
    void * sender, carray * callback_list)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = &sender;
  key.len = sizeof(sender);
  value.data = callback_list;
  value.len = 0;
  
  r = chash_set(handler_list->data_hash, &key, &value, NULL);
  if (r < 0)
    return -1;
  
  return 0;
}

static inline void
handler_list_remove_callback_list(struct etpan_signal_handler_list * handler_list,
    void * sender)
{
  chashdatum key;
  
  key.data = &sender;
  key.len = sizeof(sender);
  
  chash_delete(handler_list->data_hash, &key, NULL);
}
  
static inline carray *
handler_list_get_callback_list(struct etpan_signal_handler_list * handler_list,
    void * sender)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = &sender;
  key.len = sizeof(sender);
  
  r = chash_get(handler_list->data_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

static int handler_list_add_callback(struct etpan_signal_handler_list * handler_list,
    void * sender,
    void * user_data,
    void (* callback)(char *, void *, void *, void *))
{
  carray * callback_list;
  struct etpan_signal_handler * handler;
  int r;
  
  callback_list = handler_list_get_callback_list(handler_list, sender);
  if (callback_list == NULL) {
    callback_list = carray_new(4);
    if (callback_list == NULL)
      goto err;
    
    r = handler_list_add_callback_list(handler_list, sender, callback_list);
    if (r < 0)
      goto free_callback_list;
  }
  
  handler = handler_new(user_data, callback);
  if (handler == NULL)
    goto remove_callback_list;
  
  r = carray_add(callback_list, handler, NULL);
  if (r < 0)
    goto free_handler;
  
  return 0;
  
 free_handler:
  handler_free(handler);
 remove_callback_list:
  if (carray_count(callback_list) == 0)
    handler_list_remove_callback_list(handler_list, sender);
 free_callback_list:
  if (carray_count(callback_list) == 0)
    carray_free(callback_list);
 err:
  return -1;
}

static void handler_list_remove_callback(struct etpan_signal_handler_list * handler_list,
    void * sender,
    void * user_data,
    void (* callback)(char *, void *, void *, void *))
{
  carray * callback_list;
  unsigned int i;
  
  callback_list = handler_list_get_callback_list(handler_list, sender);
  if (callback_list == NULL)
    return;
  
  for(i = 0 ; i < carray_count(callback_list) ; i ++) {
    struct etpan_signal_handler * handler;
    
    handler = carray_get(callback_list, i);
    
    if (handler_match(handler, user_data, callback)) {
      handler_free(handler);
      carray_delete_slow(callback_list, i);
      break;
    }
  }
  
  if (carray_count(callback_list) == 0) {
    handler_list_remove_callback_list(handler_list, sender);
    carray_free(callback_list);
  }
}

static inline void handler_list_send_data(struct etpan_signal_handler_list * handler_list,
    void * sender,
    char * signal_name, void * original_sender, void * signal_data)
{
  carray * callback_list;
  unsigned int i;
  carray * callback_list_dup;
  int r;
  
  callback_list = handler_list_get_callback_list(handler_list, sender);
  if (callback_list == NULL)
    return;
  
  callback_list_dup = carray_new(carray_count(callback_list));
  if (callback_list_dup == NULL)
    return;
  
  for(i = 0 ; i < carray_count(callback_list) ; i ++) {
    struct etpan_signal_handler * handler;
    
    handler = carray_get(callback_list, i);
    handler = handler_dup(handler);
    if (handler == NULL) {
      goto free_callback_list;
    }
    
    r = carray_add(callback_list_dup, handler, NULL);
    if (r < 0) {
      handler_free(handler);
      goto free_callback_list;
    }
  }
  
  for(i = 0 ; i < carray_count(callback_list_dup) ; i ++) {
    struct etpan_signal_handler * handler;
    
    handler = carray_get(callback_list_dup, i);
    
    handler_send(handler, signal_name, original_sender, signal_data);
  }
  
 free_callback_list:
  for(i = 0 ; i < carray_count(callback_list_dup) ; i ++) {
    struct etpan_signal_handler * handler;
    
    handler = carray_get(callback_list_dup, i);
    handler_free(handler);
  }
  carray_free(callback_list_dup);
}

static void handler_list_send(struct etpan_signal_handler_list * handler_list,
    char * signal_name, void * sender, void * signal_data)
{
  if (sender != NULL)
    handler_list_send_data(handler_list, NULL, signal_name,
        sender, signal_data);
  handler_list_send_data(handler_list, sender, signal_name,
      sender, signal_data);
}

struct etpan_signal_manager * etpan_signal_manager_new(void)
{
  struct etpan_signal_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->signal_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  
  return manager;
}

void etpan_signal_manager_free(struct etpan_signal_manager * manager)
{
  chashiter * iter;
  
  for(iter = chash_begin(manager->signal_hash) ; iter != NULL ;
      iter = chash_next(manager->signal_hash, iter)) {
    chashdatum value;
    struct etpan_signal_handler_list * handler_list_info;
    
    chash_value(iter, &value);
    handler_list_info = value.data;
    handler_list_free(handler_list_info);
  }
  chash_free(manager->signal_hash);
  
  free(manager);
}

static inline int add_handler_list(struct etpan_signal_manager * manager,
    char * signal_name,
    struct etpan_signal_handler_list * handler_list)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = signal_name;
  key.len = strlen(signal_name);
  value.data = handler_list;
  value.len = 0;
  
  r = chash_set(manager->signal_hash, &key, &value, NULL);
  if (r < 0)
    return -1;
  
  return 0;
}

static inline void remove_handler_list(struct etpan_signal_manager * manager,
    char * signal_name)
{
  chashdatum key;
  
  key.data = signal_name;
  key.len = strlen(signal_name);
  
  chash_delete(manager->signal_hash, &key, NULL);
}

static inline struct etpan_signal_handler_list *
get_handler_list(struct etpan_signal_manager * manager,
            char * signal_name)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = signal_name;
  key.len = strlen(signal_name);
  
  r = chash_get(manager->signal_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  ETPAN_LOCAL_LOG("signal handler_list : %s %p", signal_name, value.data);
  
  return value.data;
}

void etpan_signal_add_handler(struct etpan_signal_manager * manager,
    char * signal_name, void * sender, void  * user_data,
    void (* signal_handler)(char * signal_name, void * sender,
        void * signal_data, void * user_data))
{
  struct etpan_signal_handler_list * handler_list;
  int r;
  
  handler_list = get_handler_list(manager, signal_name);
  if (handler_list == NULL) {
    handler_list = handler_list_new();
    if (handler_list == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = add_handler_list(manager, signal_name, handler_list);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  ETPAN_LOCAL_LOG("add handler for user_data %p", user_data);
  r = handler_list_add_callback(handler_list, sender,
      user_data, signal_handler);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_signal_remove_handler(struct etpan_signal_manager * manager,
    char * signal_name, void * sender, void  * user_data,
    void (* signal_handler)(char * signal_name, void * sender,
        void * signal_data, void * user_data))
{
  struct etpan_signal_handler_list * handler_list;
  
  ETPAN_LOCAL_LOG("find handler list for user_data %s", signal_name);
  handler_list = get_handler_list(manager, signal_name);
  if (handler_list == NULL)
    return;
  
  ETPAN_LOCAL_LOG("remove handler for user_data %p", user_data);
  handler_list_remove_callback(handler_list, sender,
      user_data, signal_handler);
  
  if (chash_count(handler_list->data_hash) == 0)
    remove_handler_list(manager, signal_name);
  if (chash_count(handler_list->data_hash) == 0)
    handler_list_free(handler_list);
}

void etpan_signal_send(struct etpan_signal_manager * manager,
    char * signal_name, void * sender, void * signal_data)
{
  struct etpan_signal_handler_list * handler_list;
  
  handler_list = get_handler_list(manager, signal_name);
  if (handler_list == NULL)
    return;
  
  handler_list_send(handler_list, signal_name, sender, signal_data);
}

static struct etpan_signal_manager * default_manager = NULL;

struct etpan_signal_manager * etpan_signal_manager_get_default(void)
{
  return default_manager;
}

void etpan_signal_manager_set_default(struct etpan_signal_manager * manager)
{
  default_manager = manager;
}
