#include "etpan-sqldb.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "etpan-log.h"
#include "etpan-error.h"

static int prepare_statement(struct etpan_sqldb * db, char * sql,
    sqlite3_stmt ** p_statement)
{
  int r;
  sqlite3_stmt * statement;
  
  r = sqlite3_prepare(db->db, sql, -1, &statement, 0);
  if (r != SQLITE_OK) {
    sqlite3_finalize(statement);
    return -1;
  }
  
  * p_statement = statement;
  
  return 0;
}

static int prepare_get(struct etpan_sqldb * db)
{
  char sql[4096];
  unsigned int i;
  int r;
  
  db->get_statement = calloc(carray_count(db->column_list),
      sizeof(* db->get_statement));
  if (db->get_statement == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(db->column_list) ; i ++) {
    char * name;
    
    name = carray_get(db->column_list, i);
    snprintf(sql, sizeof(sql), "SELECT %s FROM kvstore WHERE key = ?", name);
    r = prepare_statement(db, sql, &db->get_statement[i]);
    if (r < 0)
      return -1;
  }
  
  return 0;
}

static int prepare_set(struct etpan_sqldb * db)
{
  char sql[4096];
  unsigned int i;
  int r;
  
  db->set_statement = calloc(carray_count(db->column_list),
      sizeof(* db->set_statement));
  if (db->set_statement == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(db->column_list) ; i ++) {
    char * name;
    
    name = carray_get(db->column_list, i);
    snprintf(sql, sizeof(sql), "INSERT OR ABORT INTO kvstore (key, %s) VALUES (?, ?)", name);
    r = prepare_statement(db, sql, &db->set_statement[i]);
    if (r < 0) {
      return -1;
    }
  }
  
  return 0;
}

static int prepare_update(struct etpan_sqldb * db)
{
  char sql[4096];
  unsigned int i;
  int r;
  
  db->update_statement = calloc(carray_count(db->column_list),
      sizeof(* db->update_statement));
  if (db->get_statement == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(db->column_list) ; i ++) {
    char * name;
    
    name = carray_get(db->column_list, i);
    snprintf(sql, sizeof(sql), "UPDATE kvstore SET %s = ? WHERE key = ?", name);
    r = prepare_statement(db, sql, &db->update_statement[i]);
    if (r < 0)
      return -1;
  }
  
  return 0;
}

static void unprepare(struct etpan_sqldb * db);

static int prepare(struct etpan_sqldb * db)
{
  int r;
  char sql[4096];
  
  r = prepare_set(db);
  if (r < 0)
    goto err;
  
  r = prepare_get(db);
  if (r < 0)
    goto err;
  
  r = prepare_update(db);
  if (r < 0)
    goto err;
  
  snprintf(sql, sizeof(sql), "SELECT key FROM kvstore");
  r = prepare_statement(db, sql, &db->get_keys_statement);
  if (r < 0)
    return r;
  
  snprintf(sql, sizeof(sql), "DELETE FROM kvstore WHERE key = ?");
  r = prepare_statement(db, sql, &db->delete_statement);
  if (r < 0)
    return r;
  
  return 0;
  
 err:
  unprepare(db);
  return -1;
}

static void unprepare(struct etpan_sqldb * db)
{
  unsigned int i;
  
  if (db->get_statement != NULL) {
    for(i = 0 ; i < carray_count(db->column_list) ; i ++) {
      if (db->get_statement[i] != NULL)
        sqlite3_finalize(db->get_statement[i]);
    }
    free(db->get_statement);
  }
  
  if (db->set_statement != NULL) {
    for(i = 0 ; i < carray_count(db->column_list) ; i ++) {
      if (db->set_statement[i] != NULL)
        sqlite3_finalize(db->set_statement[i]);
    }
    free(db->set_statement);
  }
  
  if (db->update_statement != NULL) {
    for(i = 0 ; i < carray_count(db->column_list) ; i ++) {
      if (db->update_statement[i] != NULL)
        sqlite3_finalize(db->update_statement[i]);
    }
    free(db->update_statement);
  }
  
  if (db->delete_statement != NULL) {
    sqlite3_finalize(db->delete_statement);
    db->delete_statement = NULL;
  }
  if (db->get_keys_statement != NULL) {
    sqlite3_finalize(db->get_keys_statement);
    db->get_keys_statement = NULL;
  }
}

static int execute(struct etpan_sqldb * db, char * sql)
{
  int r;
  sqlite3_stmt * statement;
  
  pthread_mutex_lock(&db->lock);
  r = sqlite3_prepare(db->db, sql, -1, &statement, 0);
  if (r != SQLITE_OK) {
    ETPAN_WARN_LOG("could not prepare %s", sql);
    sqlite3_finalize(statement);
    goto unlock;
  }
  
  r = sqlite3_step(statement);
  sqlite3_finalize(statement);
  if (r != SQLITE_DONE) {
    ETPAN_WARN_LOG("could not run %s", sql);
    goto unlock;
  }
  pthread_mutex_unlock(&db->lock);
  
  return 0;
  
 unlock:
  pthread_mutex_unlock(&db->lock);
  return -1;
}

struct etpan_sqldb * etpan_sqldb_new(const char * filename,
    carray * column_list)
{
  struct etpan_sqldb * db;
  int r;
  unsigned int i;
  
  db = malloc(sizeof(* db));
  if (db == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  db->filename = strdup(filename);
  if (db->filename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  db->db = NULL;
  db->get_statement = NULL;
  db->set_statement = NULL;
  db->update_statement = NULL;
  db->delete_statement = NULL;
  db->get_keys_statement = NULL;
  db->column_list = carray_new(carray_count(column_list));
  if (db->column_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  db->name_to_index = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (db->name_to_index == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(column_list) ; i ++) {
    char * name;
    chashdatum key;
    chashdatum value;
    
    name = carray_get(column_list, i);
    name = strdup(name);
    if (name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = carray_add(db->column_list, name, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    key.data = name;
    key.len = strlen(name) + 1;
    value.data = &i;
    value.len = sizeof(i);
    r = chash_set(db->name_to_index, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = pthread_mutex_init(&db->lock, NULL);
  if (r != 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  return db;
}

void etpan_sqldb_free(struct etpan_sqldb * sqldb)
{
  unsigned int i;
  
  pthread_mutex_destroy(&sqldb->lock);
  
  chash_free(sqldb->name_to_index);
  for(i = 0 ; i < carray_count(sqldb->column_list) ; i ++) {
    char * name;
    
    name = carray_get(sqldb->column_list, i);
    free(name);
  }
  carray_free(sqldb->column_list);
  free(sqldb->filename);
  free(sqldb);
}

static int try_open(struct etpan_sqldb * sqldb)
{
  int r;
  int file_exist;
  struct stat stat_buf;
  
  file_exist = 1;
  r = stat(sqldb->filename, &stat_buf);
  if (r < 0)
    file_exist = 0;
  
  r = sqlite3_open(sqldb->filename, &sqldb->db);
  if (r != SQLITE_OK)
    return ERROR_FILE;
  
  if (!file_exist) {
    MMAPString * str;
    unsigned int i;
    
    str = mmap_string_new("");
    if (str == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    if (mmap_string_append(str, "CREATE TABLE kvstore (key TEXT UNIQUE PRIMARY KEY") == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    for(i = 0 ; i < carray_count(sqldb->column_list) ; i ++) {
      char * name;
      
      name = carray_get(sqldb->column_list, i);
      if (mmap_string_append(str, ", ") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, name) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      if (mmap_string_append(str, " BLOB") == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    if (mmap_string_append(str, ")") == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    execute(sqldb, str->str);
    
    mmap_string_free(str);
  }
  
  r = prepare(sqldb);
  if (r < 0) {
    sqlite3_close(sqldb->db);
    return -1;
  }
  
  return 0;
}

int etpan_sqldb_open(struct etpan_sqldb * sqldb)
{
  int r;
  
  r = try_open(sqldb);
  if (r < 0) {
    unlink(sqldb->filename);
    r = try_open(sqldb);
    if (r < 0)
      return r;
  }
  
  return 0;
}

void etpan_sqldb_close(struct etpan_sqldb * sqldb)
{
  unprepare(sqldb);
  
  sqlite3_close(sqldb->db);
}

int etpan_sqldb_get(struct etpan_sqldb * sqldb,
    char * uid, char * column,
    void ** p_data, size_t * p_len)
{
  sqlite3_stmt * statement;
  unsigned int i;
  int r;
  chashdatum key;
  chashdatum value_index;
  int data_size;
  void * data;
  
  key.data = column;
  key.len = strlen(column) + 1;
  r = chash_get(sqldb->name_to_index, &key, &value_index);
  if (r < 0)
    goto err;
  memcpy(&i, value_index.data, sizeof(i));
  statement = sqldb->get_statement[i];
  
  pthread_mutex_lock(&sqldb->lock);
  sqlite3_bind_text(statement, 1, uid, -1, SQLITE_STATIC);
  r = sqlite3_step(statement);
  if (r != SQLITE_ROW) {
    sqlite3_reset(statement);
    goto unlock;
  }
  
  data_size = sqlite3_column_bytes(statement, 0);
  if (data_size == 0) {
    sqlite3_reset(statement);
    goto unlock;
  }
  
  data = malloc(data_size);
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  memcpy(data, sqlite3_column_blob(statement, 0), data_size);
  
  * p_data = data;
  * p_len = data_size;
  
  sqlite3_reset(statement);
  pthread_mutex_unlock(&sqldb->lock);
  
  return 0;
  
 unlock:
  pthread_mutex_unlock(&sqldb->lock);
 err:
  return -1;
}

int etpan_sqldb_set(struct etpan_sqldb * sqldb,
    char * uid, char * column,
    void * data, size_t len)
{
  sqlite3_stmt * statement;
  unsigned int i;
  int r;
  chashdatum key;
  chashdatum value_index;
  
  key.data = column;
  key.len = strlen(column) + 1;
  r = chash_get(sqldb->name_to_index, &key, &value_index);
  if (r < 0) {
    goto err;
  }
  memcpy(&i, value_index.data, sizeof(i));
  statement = sqldb->set_statement[i];
  
  pthread_mutex_lock(&sqldb->lock);
  sqlite3_bind_text(statement, 1, uid, -1, SQLITE_STATIC);
  sqlite3_bind_blob(statement, 2, data, len, SQLITE_STATIC);
  r = sqlite3_step(statement);
  if (r != SQLITE_DONE) {
    sqlite3_reset(statement);
    
    statement = sqldb->update_statement[i];
    sqlite3_bind_blob(statement, 1, data, len, SQLITE_STATIC);
    sqlite3_bind_text(statement, 2, uid, -1, SQLITE_STATIC);
    r = sqlite3_step(statement);
    if (r != SQLITE_DONE) {
      ETPAN_WARN_LOG("error storing %s", uid);
      sqlite3_reset(statement);
      goto unlock;
    }
  }
  
  sqlite3_reset(statement);
  pthread_mutex_unlock(&sqldb->lock);
  
  return 0;
  
 unlock:
  pthread_mutex_unlock(&sqldb->lock);
 err:
  return -1;
}

int etpan_sqldb_delete(struct etpan_sqldb * sqldb, char * uid)
{
  int r;
  sqlite3_stmt * statement;
  
  statement = sqldb->delete_statement;
  
  pthread_mutex_lock(&sqldb->lock);
  sqlite3_bind_text(statement, 1, uid, -1, SQLITE_STATIC);
  r = sqlite3_step(statement);
  sqlite3_reset(statement);
  pthread_mutex_unlock(&sqldb->lock);
  
  if (r != SQLITE_DONE) {
    ETPAN_WARN_LOG("error deleting %s", uid);
    return -1;
  }
  
  return 0;
}

carray * etpan_sqldb_get_keys(struct etpan_sqldb * sqldb)
{
  carray * keys;
  int r;
  sqlite3_stmt * statement;
  
  keys = carray_new(16);
  
  statement = sqldb->get_keys_statement;
  pthread_mutex_lock(&sqldb->lock);
  r = sqlite3_step(statement);
  while (r == SQLITE_ROW) {
    char * key;
    
    key = (char *) sqlite3_column_text(statement, 0);
    key = strdup(key);
    if (key == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = carray_add(keys, key, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = sqlite3_step(statement);
  }
  
  sqlite3_reset(statement);
  pthread_mutex_unlock(&sqldb->lock);
  
  if (r != SQLITE_DONE) {
    ETPAN_WARN_LOG("get keys failed");
  }
  
  return keys;
}

void etpan_sqldb_keys_free(carray * keys)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(keys) ; i ++) {
    char * key;
    
    key = carray_get(keys, i);
    free(key);
  }
  carray_free(keys);
}

int etpan_sqldb_reset(struct etpan_sqldb * sqldb)
{
  return execute(sqldb, "DELETE FROM kvstore");
}

int etpan_sqldb_begin_transaction(struct etpan_sqldb * sqldb)
{
  return execute(sqldb, "BEGIN");
}

int etpan_sqldb_end_transaction(struct etpan_sqldb * sqldb)
{
  return execute(sqldb, "COMMIT");
}
