#include "etpan-storage-folder-order.h"

#include <stdlib.h>
#include <string.h>
#include <libetpan/libetpan.h>

#include "etpan-folder-tree.h"
#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-signal.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-utils.h"
#include "etpan-account-manager.h"
#include "etpan-nls.h"

static void folder_list_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void remove_recursively(struct etpan_folder_tree * tree,
    chash * folder_to_remove_hash);
static int cmp_folder_name(const void * a, const void * b);
static struct etpan_folder_tree *
create_recursively(struct etpan_folder_tree * root,
    char * folder_name);
static char * get_parent_folder_name(char * folder_name);
static void setup_folder_order(struct etpan_storage_folder_order * folder_order,
    chash * folder_hash);
static void set_order_from_tree(carray * folder_order,
    chash * folder_hash,
    struct etpan_folder_tree * tree,
    chash * current_folder_hash);
static void reorder(struct etpan_storage_folder_order * folder_order,
    struct etpan_folder_tree * tree,
    carray * list, chash * folder_hash);
static void main_reorder(struct etpan_storage_folder_order * folder_order,
    carray * list);
static struct etpan_error *
save(struct etpan_storage_folder_order * folder_order);
static struct etpan_error *
load(struct etpan_storage_folder_order * folder_order);

struct etpan_storage_folder_order * etpan_storage_folder_order_new(void)
{
  struct etpan_storage_folder_order * folder_order;
  
  folder_order = malloc(sizeof(* folder_order));
  if (folder_order == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_order->path = NULL;
  folder_order->loaded = 0;
  folder_order->tree = etpan_folder_tree_new();
  if (folder_order->tree == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  folder_order->storage = NULL;
  folder_order->folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_order->folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_order->folder_order = carray_new(16);
  if (folder_order->folder_order == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return folder_order;
}

void etpan_storage_folder_order_free(struct etpan_storage_folder_order * folder_order)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(folder_order->folder_order) ; i ++) {
    free(carray_get(folder_order->folder_order, i));
  }
  carray_free(folder_order->folder_order);
  chash_free(folder_order->folder_hash);
  etpan_folder_tree_free_recursive(folder_order->tree);
  free(folder_order->path);
  free(folder_order);
}

void etpan_storage_folder_order_set_storage(struct etpan_storage_folder_order * folder_order, struct etpan_storage * storage)
{
  folder_order->storage = storage;
}

struct etpan_storage * etpan_storage_folder_order_get_storage(struct etpan_storage_folder_order * folder_order)
{
  return folder_order->storage;
}

void etpan_storage_folder_order_setup(struct etpan_storage_folder_order * folder_order)
{
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_STORAGE_FOLDERLIST_UPDATED, folder_order->storage, folder_order,
    folder_list_updated);
}

void etpan_storage_folder_order_unsetup(struct etpan_storage_folder_order * folder_order)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
    ETPAN_STORAGE_FOLDERLIST_UPDATED, folder_order->storage, folder_order,
    folder_list_updated);
}

void etpan_storage_folder_order_set_folder_order(struct etpan_storage_folder_order * folder_order, carray * list)
{
  struct etpan_error * error;
  
  main_reorder(folder_order, list);
  
  error = save(folder_order);
  ETPAN_ERROR_IGNORE(error);
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_STORAGE_FOLDER_ORDER_CHANGED,
      folder_order, NULL);
}

carray * etpan_storage_folder_order_get_folder_order(struct etpan_storage_folder_order * folder_order)
{
  return folder_order->folder_order;
}

static void folder_list_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_storage_folder_order * folder_order;
  chash * folder_hash;
  chash * original_folder_hash;
  carray * new_folder_list;
  chash * folder_to_remove_hash;
  chashiter * iter;
  int r;
  unsigned int i;
  void ** data;
  struct etpan_error * error;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  folder_order = user_data;
  
  new_folder_list = carray_new(16);
  if (new_folder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  folder_to_remove_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_to_remove_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  original_folder_hash = etpan_storage_get_folder_list(folder_order->storage);
  for(iter = chash_begin(original_folder_hash) ; iter != NULL ;
      iter = chash_next(original_folder_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_folder * folder;
    char * ui_path;
    char * folder_name;
    
    chash_value(iter, &value);
    folder = value.data;
    ui_path = etpan_folder_get_ui_path(folder);
    key.data = ui_path;
    key.len = strlen(ui_path) + 1;
    value.data = folder;
    value.len = 0;
    /* folder hash */
    r = chash_set(folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    folder_name = strdup(ui_path);
    if (folder_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    while (1) {
      char * parent_name;
      
      parent_name = get_parent_folder_name(folder_name);
      free(folder_name);
      folder_name = parent_name;
      if (parent_name == NULL)
        break;
      
      key.data = parent_name;
      key.len = strlen(parent_name) + 1;
      r = chash_get(folder_hash, &key, &value);
      if (r < 0) {
        value.data = NULL;
        value.len = 0;
        r = chash_set(folder_hash, &key, &value, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
    
    /* new folder list */
    r = chash_get(folder_order->folder_hash, &key, &value);
    if (r < 0) {
      r = carray_add(new_folder_list, folder, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  /* removed folders */
  for(iter = chash_begin(folder_order->folder_hash) ; iter != NULL ;
      iter = chash_next(folder_order->folder_hash, iter)) {
    chashdatum key;
    chashdatum value;
    
    chash_key(iter, &key);
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      value.data = NULL;
      value.len = 0;
      ETPAN_LOG("removed key %s", key.data);
      r = chash_set(folder_to_remove_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  /* remove old */
  remove_recursively(folder_order->tree, folder_to_remove_hash);
  
  /* add new */
  data = carray_data(new_folder_list);
  qsort(data, carray_count(new_folder_list),
      sizeof(* data), cmp_folder_name);
  for(i = 0 ; i < carray_count(new_folder_list) ; i ++) {
    struct etpan_folder * folder;
    char * folder_name;
    
    folder = carray_get(new_folder_list, i);
    folder_name = etpan_folder_get_ui_path(folder);
    /* add new folder */
    create_recursively(folder_order->tree, folder_name);
  }
  
  /* reset folder tree */
  setup_folder_order(folder_order, folder_hash);
  
  error = load(folder_order);
  ETPAN_ERROR_IGNORE(error);
  
  if ((chash_count(folder_to_remove_hash) > 0) || (carray_count(new_folder_list))) {
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_STORAGE_FOLDER_ORDER_CHANGED,
        folder_order, NULL);
  }
  
  chash_free(folder_hash);
  chash_free(folder_to_remove_hash);
  carray_free(new_folder_list);
}

static void remove_recursively(struct etpan_folder_tree * tree,
    chash * folder_to_remove_hash)
{
  carray * children;
  int r;
  unsigned int i;
  unsigned int count;
  
  children = etpan_folder_tree_get_children(tree);
  
  count = 0;
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_folder_tree * child;
    char * ui_path;
    
    child = carray_get(children, i);
    /* remove children first */
    remove_recursively(child, folder_to_remove_hash);
    
    ui_path = etpan_folder_tree_get_ui_path(child);
    if (ui_path != NULL) {
      chashdatum key;
      chashdatum value;
      
      key.data = ui_path;
      key.len = strlen(ui_path) + 1;
      
      r = chash_get(folder_to_remove_hash, &key, &value);
      if (r == 0) {
        etpan_folder_tree_free(child);
        continue;
      }
    }
    
    carray_set(children, count, child);
    count ++;
  }
  r = carray_set_size(children, count);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static int cmp_folder_name(const void * a, const void * b)
{
  struct etpan_folder * const * s_a;
  struct etpan_folder * const * s_b;
  struct etpan_folder * f_a;
  struct etpan_folder * f_b;
  
  s_a = a;
  s_b = b;
  f_a = * s_a;
  f_b = * s_b;
  
  return strcmp(etpan_folder_get_ui_path(f_a),
      etpan_folder_get_ui_path(f_b));
}

static struct etpan_folder_tree *
create_recursively(struct etpan_folder_tree * root,
    char * folder_name)
{
  char * parent_folder_name;
  struct etpan_folder_tree * parent_folder_tree;
  struct etpan_folder_tree * folder_tree;
  
  folder_tree = etpan_folder_tree_get_node(root,
      folder_name);
  if (folder_tree != NULL) {
    /* replace existant */
    return folder_tree;
  }
  
  parent_folder_name = get_parent_folder_name(folder_name);
  
  if (parent_folder_name == NULL)
    parent_folder_tree = root;
  else {
    parent_folder_tree = create_recursively(root, parent_folder_name);
    free(parent_folder_name);
  }
  
  folder_tree = etpan_folder_tree_new();
  etpan_folder_tree_set_ui_path(folder_tree, folder_name);
  
  etpan_folder_tree_add_child(parent_folder_tree, folder_tree);
  etpan_folder_tree_set_parent(folder_tree, parent_folder_tree);
#if 0
  if (folder != NULL)
    etpan_folder_tree_set_folder(folder_tree, folder);
#endif
  
  return folder_tree;
}

static char * get_parent_folder_name(char * folder_name)
{
  char * p;
  char * dup_folder_name;
  
  dup_folder_name = strdup(folder_name);
  if (dup_folder_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  p = strrchr(dup_folder_name, '/');
  if (p == NULL)
    return NULL;
  
  * p = '\0';
  
  return dup_folder_name;
}

static void setup_folder_order(struct etpan_storage_folder_order * folder_order,
    chash * folder_hash)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(folder_order->folder_order) ; i ++) {
    free(carray_get(folder_order->folder_order, i));
  }
  carray_set_size(folder_order->folder_order, 0);
  chash_clear(folder_order->folder_hash);
  set_order_from_tree(folder_order->folder_order,
      folder_order->folder_hash,
      folder_order->tree,
      folder_hash);
}

static void set_order_from_tree(carray * folder_order,
    chash * folder_hash,
    struct etpan_folder_tree * tree,
    chash * current_folder_hash)
{
  unsigned int i;
  struct etpan_folder * folder;
  carray * children;
  char * ui_path;
  chashdatum key;
  chashdatum value;
  char * folder_name;
  int r;
  
  ui_path = etpan_folder_tree_get_ui_path(tree);
  if (ui_path != NULL) {
    folder_name = strdup(ui_path);
    r = carray_add(folder_order, folder_name, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    key.data = folder_name;
    key.len = strlen(folder_name) + 1;
    value.data = tree;
    value.len = 0;
    r = chash_set(folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = chash_get(current_folder_hash, &key, &value);
    if (r == 0) {
      folder = value.data;
      etpan_folder_tree_set_folder(tree, folder);
    }
  }
  
  children = etpan_folder_tree_get_children(tree);
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_folder_tree * child;
    
    child = carray_get(children, i);
    set_order_from_tree(folder_order, folder_hash, child, current_folder_hash);
  }
  ui_path = etpan_folder_tree_get_ui_path(tree);
}

static void reorder(struct etpan_storage_folder_order * folder_order,
    struct etpan_folder_tree * tree,
    carray * list, chash * folder_hash)
{
  carray * children;
  chash * children_hash;
  unsigned int i;
  int has_all_children;
  int r;
  
  has_all_children = 1;
  children = etpan_folder_tree_get_children(tree);
  children_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (children_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (carray_count(children) == 0)
    goto free;
  
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_folder_tree * child;
    char * ui_path;
    chashdatum key;
    chashdatum value;
    
    child = carray_get(children, i);
    ui_path = etpan_folder_tree_get_ui_path(child);
    key.data = ui_path;
    key.len = strlen(ui_path) + 1;
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      has_all_children = 0;
      break;
    }
    
    value.data = child;
    value.len = 0;
    r = chash_set(children_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_folder_tree * child;
    
    child = carray_get(children, i);
    reorder(folder_order, child, list, folder_hash);
  }
  
  if (!has_all_children)
    goto free;
  
  r = carray_set_size(children, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    char * folder_name;
    chashdatum key;
    chashdatum value;
    struct etpan_folder_tree * child;
    
    folder_name = carray_get(list, i);
    key.data = folder_name;
    key.len = strlen(folder_name) + 1;
    r = chash_get(children_hash, &key, &value);
    if (r < 0) {
      continue;
    }
    child = value.data;
    r = carray_add(children, child, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
 free:
  chash_free(children_hash);
}

static void main_reorder(struct etpan_storage_folder_order * folder_order,
    carray * list)
{
  chash * folder_hash;
  unsigned int i;
  chashiter * iter;
  int r;
  
  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    char * folder_name;
    chashdatum key;
    chashdatum value;
    
    folder_name = carray_get(list, i);
    key.data = folder_name;
    key.len = strlen(folder_name) + 1;
    value.data = NULL;
    value.len = 0;
    r = chash_set(folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  reorder(folder_order, folder_order->tree, list, folder_hash);
  chash_free(folder_hash);
  
  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(iter = chash_begin(folder_order->folder_hash) ; iter != NULL ;
      iter = chash_next(folder_order->folder_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_folder_tree * tree;
    struct etpan_folder * folder;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    tree = value.data;
    folder = etpan_folder_tree_get_folder(tree);
    value.data = folder;
    value.len = 0;
    r = chash_set(folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  setup_folder_order(folder_order, folder_hash);
  
  chash_free(folder_hash);
}

struct etpan_folder * etpan_storage_folder_order_get_folder(struct etpan_storage_folder_order * folder_order, char * ui_path)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_folder_tree * tree;
  
  key.data = ui_path;
  key.len = strlen(ui_path) + 1;
  r = chash_get(folder_order->folder_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  tree = value.data;
  
  return etpan_folder_tree_get_folder(tree);
}


void etpan_storage_folder_order_set_path(struct etpan_storage_folder_order * folder_order,
    char * path)
{
  if (path != folder_order->path) {
    free(folder_order->path);
    if (path != NULL) {
      folder_order->path = strdup(path);
      if (folder_order->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder_order->path = NULL;
  }
}

char * etpan_storage_folder_order_get_path(struct etpan_storage_folder_order * folder_order)
{
  return folder_order->path;
}

char * etpan_storage_folder_order_get_path_for_storage(struct etpan_storage_folder_order * folder_order, struct etpan_storage * storage)
{
  struct etpan_account * account;
  char * order_dir;
  (void) folder_order;
  
  account = etpan_storage_get_account(storage);
  order_dir = etpan_get_order_dir(etpan_account_manager_get_default(),
      account);
  
  return order_dir;
}

static struct etpan_error * save(struct etpan_storage_folder_order * folder_order)
{
  FILE * f;
  unsigned int i;
  
  f = fopen(folder_order->path, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Configuration file could not be written"));
    etpan_error_strf_long_description(error, _("Address book configuration file %s could not be written to disk."), folder_order->path);
    
    return error;
  }
  
  for(i = 0 ; i < carray_count(folder_order->folder_order) ; i ++) {
    char * folder_name;
    
    folder_name = carray_get(folder_order->folder_order, i);
    fprintf(f, "%s\n", folder_name);
  }
  
  fclose(f);
  
  return NULL;
}

static struct etpan_error *
load(struct etpan_storage_folder_order * folder_order)
{
  FILE * f;
  carray * list;
  char buf[PATH_MAX];
  unsigned int i;
  int r;
  
  if (folder_order->loaded)
    return NULL;
  
  if (!etpan_storage_has_folder_list(folder_order->storage))
    return NULL;
  
  f = fopen(folder_order->path, "r");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("File not found"));
    etpan_error_strf_long_description(error, _("Folder order configuration file %s could not be found."), folder_order->path);
    
    return error;
  }
  
  list = carray_new(16);
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  while (fgets(buf, sizeof(buf), f)) {
    char * folder_name;
    size_t len;
    
    len = strlen(buf);
    if (len == 0)
      continue;
    
    if (buf[len - 1] == '\n')
      buf[len - 1] = '\0';
    
    folder_name = strdup(buf);
    if (folder_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = carray_add(list, folder_name, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  fclose(f);
  
  main_reorder(folder_order, list);
  
  for(i = 0 ; i < carray_count(list) ; i ++) {
    free(carray_get(list, i));
  }
  carray_free(list);
  
  folder_order->loaded = 1;
  
  return NULL;
}
