#include "etpan-storage.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-folder.h"
#include "etpan-folder-private.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-utils.h"
#include "etpan-signal.h"
#include "etpan-error.h"
#include "etpan-nls.h"

#define ETPAN_MODULE_LOG_NAME "STORAGE"
#include "etpan-log.h"

static struct etpan_error * storage_connect(struct etpan_storage * storage);
static void storage_disconnect(struct etpan_storage * storage);

struct etpan_storage * etpan_storage_new(void)
{
  struct etpan_storage * storage;
  
  storage = malloc(sizeof(* storage));
  if (storage == NULL)
    goto err;
  
  storage->ref_count = 1;
  storage->account = NULL;
  storage->id = NULL;
  storage->lost = 0;
  storage->cache_path = NULL;
  storage->threaded_cache_path = NULL;
  storage->flags_path = NULL;
  storage->threaded_flags_path = NULL;
  storage->folder_list_fetched = 0;
  storage->folder_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (storage->folder_list == NULL)
    goto free;
  
  storage->connected_folder_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (storage->folder_list == NULL)
    goto free_folder_list;
  
  storage->data = NULL;
  storage->driver = NULL;
  
  return storage;
  
 free_folder_list:
  chash_free(storage->folder_list);
 free:
  free(storage);
 err:
  return NULL;
}

void etpan_storage_free(struct etpan_storage * storage)
{
  ETPAN_LOCAL_LOG("storage free");
  
  etpan_storage_clear_folder_list(storage);
  
  if (storage->driver != NULL) {
    if (storage->driver->free_data != NULL)
      storage->driver->free_data(storage);
  }
  
  chash_free(storage->connected_folder_list);
  chash_free(storage->folder_list);
  
  free(storage->cache_path);
  free(storage->threaded_cache_path);
  free(storage->flags_path);
  free(storage->threaded_flags_path);
  free(storage->id);
  free(storage);
}

void etpan_storage_set_data(struct etpan_storage * storage, void * data)
{
  storage->data = data;
}

void * etpan_storage_get_data(struct etpan_storage * storage)
{
  return storage->data;
}

void etpan_storage_set_driver(struct etpan_storage * storage,
    struct etpan_storage_driver * driver)
{
  storage->driver = driver;
}

struct etpan_storage_driver *
etpan_storage_get_driver(struct etpan_storage * storage)
{
  return storage->driver;
}

void etpan_storage_set_account(struct etpan_storage * storage,
    struct etpan_account * account)
{
  storage->account = account;
}

struct etpan_account *
etpan_storage_get_account(struct etpan_storage * storage)
{
  return storage->account;
}

static void update_folder_ui_path(struct etpan_storage * storage);

void etpan_storage_set_id(struct etpan_storage * storage, char * id)
{
  if (id != storage->id) {
    free(storage->id);
    if (id != NULL) {
      storage->id = strdup(id);
      if (storage->id == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
    else
      storage->id = NULL;
  }
  
  update_folder_ui_path(storage);
}

char * etpan_storage_get_id(struct etpan_storage * storage)
{
  return storage->id;
}

/* get folder list */

struct etpan_storage_fetch_folder_list_param {
  struct etpan_storage * storage;
};

static void
fetch_folder_list_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_fetch_folder_list_result * result;
  chashiter * iter;
  
  free(op->param);
  op->param = NULL;
  
  result = op->result;
    
  for(iter = chash_begin(result->folder_list) ; iter != NULL ;
      iter = chash_next(result->folder_list, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
      
    chash_value(iter, &value);
    folder = value.data;
    etpan_folder_unref(folder);
  }
  chash_free(result->folder_list);
  ETPAN_ERROR_FREE(result->error);
  result->error = NULL;
  
  free(op->result);
  op->result = NULL;
}

static void
fetch_folder_list_run(struct etpan_thread_op * op)
{
  struct etpan_storage_fetch_folder_list_param * param;
  struct etpan_storage_fetch_folder_list_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = storage_connect(param->storage);
  if (error == NULL) {
    if (param->storage->driver->fetch_folder_list != NULL)
      error = param->storage->driver->fetch_folder_list(param->storage,
          result->folder_list);
    else {
      error = etpan_error_internal_strf(_("fetch-folder-list is not implemented for %s"),
          param->storage->driver->name);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_storage_fetch_folder_list(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    void (* callback)(int, struct etpan_storage_fetch_folder_list_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_fetch_folder_list_param * param;
  struct etpan_storage_fetch_folder_list_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  result->error = NULL;
  
  result->folder_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (result->folder_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->storage = storage;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_folder_list_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = fetch_folder_list_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}

/* create */

struct etpan_storage_create_folder_param {
  struct etpan_storage * storage;
  char * location;
};

static void
create_folder_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_create_folder_param * param;
  struct etpan_storage_create_folder_result * result;
  
  param = op->param;
  free(param->location);
  free(param);
  op->param = NULL;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  result->error = NULL;
  free(result);
  op->result = NULL;
}

static void
create_folder_run(struct etpan_thread_op * op)
{
  struct etpan_storage_create_folder_param * param;
  struct etpan_storage_create_folder_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = storage_connect(param->storage);
  if (error == NULL) {
    if (param->storage->driver->create_folder != NULL)
      error = param->storage->driver->create_folder(param->storage,
          param->location);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("create-folder is not implemented for %s"),
          param->storage->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_storage_create_folder(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    char * location,
    void (* callback)(int, struct etpan_storage_create_folder_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_create_folder_param * param;
  struct etpan_storage_create_folder_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->storage = storage;
  param->location = strdup(location);
  if (param->location == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = create_folder_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = create_folder_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}

/* delete */

struct etpan_storage_delete_folder_param {
  struct etpan_storage * storage;
  char * location;
};

static void
delete_folder_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_delete_folder_param * param;
  struct etpan_storage_delete_folder_result * result;
  
  param = op->param;
  free(param->location);
  free(param);
  op->param = NULL;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  result->error = NULL;
  free(result);
  op->result = NULL;
}

static void
delete_folder_run(struct etpan_thread_op * op)
{
  struct etpan_storage_delete_folder_param * param;
  struct etpan_storage_delete_folder_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = storage_connect(param->storage);
  if (error == NULL) {
    if (param->storage->driver->delete_folder != NULL)
      error = param->storage->driver->delete_folder(param->storage,
          param->location);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("delete-folder is not implemented for %s"),
          param->storage->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_storage_delete_folder(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    char * location,
    void (* callback)(int, struct etpan_storage_delete_folder_result *,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_delete_folder_param * param;
  struct etpan_storage_delete_folder_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->storage = storage;
  param->location = strdup(location);
  if (param->location == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = delete_folder_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = delete_folder_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}

/* rename */

struct etpan_storage_rename_folder_param {
  struct etpan_storage * storage;
  char * location;
  char * new_location;
};

static void
rename_folder_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_rename_folder_param * param;
  struct etpan_storage_rename_folder_result * result;
  
  param = op->param;
  free(param->new_location);
  free(param->location);
  free(param);
  op->param = NULL;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  result->error = NULL;
  free(result);
  op->result = NULL;
}

static void
rename_folder_run(struct etpan_thread_op * op)
{
  struct etpan_storage_rename_folder_param * param;
  struct etpan_storage_rename_folder_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = storage_connect(param->storage);
  if (error == NULL) {
    if (param->storage->driver->rename_folder != NULL)
      error = param->storage->driver->rename_folder(param->storage,
          param->location, param->new_location);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("rename-folder is not implemented for %s"),
          param->storage->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_storage_rename_folder(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    char * location, char * new_location,
    void (* callback)(int, struct etpan_storage_rename_folder_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_rename_folder_param * param;
  struct etpan_storage_rename_folder_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->storage = storage;
  param->location = strdup(location);
  if (param->location == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  param->new_location = strdup(new_location);
  if (param->new_location == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = rename_folder_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = rename_folder_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}

/* disconnect */

struct etpan_storage_disconnect_param {
  struct etpan_storage * storage;
};

static void
disconnect_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_disconnect_param * param;
  
  param = op->param;
  free(param);
  op->param = NULL;
}

static void
disconnect_run(struct etpan_thread_op * op)
{
  struct etpan_storage_disconnect_param * param;
  
  param = op->param;

  storage_disconnect(param->storage);
}

struct etpan_thread_op *
etpan_storage_disconnect(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    void (* callback)(int, void * /* dummy argument */,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_disconnect_param * param;
  struct disconnect_data * disconnect_data;
  
  ETPAN_LOCAL_LOG("disconnect %s", etpan_storage_get_id(storage));
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->storage = storage;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = disconnect_run;
  op->callback = callback;
  op->callback_data = cb_data;
  op->cleanup = disconnect_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}

struct etpan_folder * etpan_storage_get_folder(struct etpan_storage * storage,
    char * name)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_get(storage->folder_list, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

int etpan_storage_has_folder_list(struct etpan_storage * storage)
{
  return storage->folder_list_fetched;
}

chash * etpan_storage_get_folder_list(struct etpan_storage * storage)
{
  return storage->folder_list;
}

struct etpan_error *
etpan_storage_update_folder_list(struct etpan_storage * storage,
    chash * folder_list)
{
  int r;
  chashiter * iter;
  struct etpan_error * error;
  carray * error_list;
  unsigned int i;
  
  error_list = carray_new(4);
  if (error_list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(iter = chash_begin(folder_list) ; iter != NULL ;
      iter = chash_next(folder_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_key(iter, &key);
    
    r = chash_get(storage->folder_list, &key, &value);
    if (r == 0)
      continue;
    
    chash_value(iter, &value);
    folder = value.data;
    
    error = etpan_folder_setup(folder);
    if (error != NULL) {
      r = carray_add(error_list, error, NULL);
      if (r < 0) {
        etpan_error_free(error);
        ETPAN_LOG_MEMORY_ERROR;
      }
      continue;
    }
    
    etpan_folder_ref(folder);
    r = chash_set(storage->folder_list, &key, &value, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  for(iter = chash_begin(storage->folder_list) ; iter != NULL ;
      iter = chash_next(storage->folder_list, iter)) {
    chashdatum key;
    chashdatum value;
    char * location;
    struct etpan_folder * folder;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    
    location = key.data;
    folder = value.data;
    
    r = chash_get(folder_list, &key, &value);
    if (r == 0)
      continue;
    
    etpan_folder_set_lost(folder, 1);
  }
  
  storage->folder_list_fetched = 1;
  ETPAN_SIGNAL_SEND(storage, ETPAN_STORAGE_FOLDERLIST_UPDATED);
  
  if (carray_count(error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(error_list) == 1) {
    error = carray_get(error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(error_list);
  
  return error;
}

void etpan_storage_remove_lost_folders(struct etpan_storage * storage)
{
  chashiter * iter;
  carray * folder_to_remove;
  unsigned int i;
  int r;
  
  folder_to_remove = carray_new(16);
  if (folder_to_remove == NULL) {
    /* fail */
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(iter = chash_begin(storage->folder_list) ; iter != NULL ;
      iter = chash_next(storage->folder_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_folder * folder;
    char * location;
    
    chash_key(iter, &key);
    location = key.data;
    
    chash_value(iter, &value);
    folder = value.data;

    if (!etpan_folder_is_lost(folder))
      continue;
    
    etpan_folder_unref(folder);
    
    r = carray_add(folder_to_remove, location, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  for(i = 0 ; i < carray_count(folder_to_remove) ; i ++) {
    char * location;
    chashdatum key;
    
    location = carray_get(folder_to_remove, i);
    
    key.data = location;
    key.len = strlen(location) + 1;
    chash_delete(storage->folder_list, &key, NULL);
  }
  
  carray_free(folder_to_remove);
  
  ETPAN_SIGNAL_SEND(storage, ETPAN_STORAGE_FOLDERLIST_UPDATED);
}

void etpan_storage_clear_folder_list(struct etpan_storage * storage)
{
  chashiter * iter;
  
  storage->folder_list_fetched --;
  
  for(iter = chash_begin(storage->folder_list) ; iter != NULL ;
      iter = chash_next(storage->folder_list, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_folder * folder;
    char * location;
    
    chash_key(iter, &key);
    location = key.data;
    
    chash_value(iter, &value);
    folder = value.data;
    
    etpan_folder_unref(folder);
  }
  
  chash_clear(storage->folder_list);
  
  storage->folder_list_fetched = 0;
  ETPAN_SIGNAL_SEND(storage, ETPAN_STORAGE_FOLDERLIST_UPDATED);
}

void etpan_storage_remove_folder(struct etpan_storage * storage,
    struct etpan_folder * folder)
{
  chashdatum key;
  
  key.data = folder->location;
  key.len = strlen(folder->location) + 1;
  
  chash_delete(storage->folder_list, &key, NULL);
}

struct etpan_error *
etpan_storage_setup(struct etpan_storage * storage)
{
  struct etpan_error * error;
  
  free(storage->threaded_cache_path);
  if (storage->cache_path != NULL) {
    etpan_make_dir(storage->cache_path);
    /* ignore result */
    storage->threaded_cache_path = strdup(storage->cache_path);
    if (storage->threaded_cache_path == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  free(storage->threaded_flags_path);
  if (storage->flags_path != NULL) {
    etpan_make_dir(storage->flags_path);
    /* ignore result */
    storage->threaded_flags_path = strdup(storage->flags_path);
    if (storage->threaded_flags_path == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  if (storage->driver->setup != NULL) {
    error = storage->driver->setup(storage);
    if (error != NULL) {
      goto err;
    }
  }
  
  return NULL;
  
 err:
  free(storage->threaded_flags_path);
  storage->threaded_flags_path = NULL;
  free(storage->threaded_cache_path);
  storage->threaded_cache_path = NULL;
  return error;
}

void etpan_storage_unsetup(struct etpan_storage * storage)
{
  etpan_thread_manager_app_unbind_storage_thread(etpan_thread_manager_app_get_default(), storage);
}

static void update_folder_ui_path(struct etpan_storage * storage)
{
  chash * folder_hash;
  char * storage_id;
  chashiter * iter;
  
  storage_id = etpan_storage_get_id(storage);
  
  folder_hash = etpan_storage_get_folder_list(storage);
  for(iter = chash_begin(folder_hash) ; iter != NULL ;
      iter = chash_next(folder_hash, iter)) {
    chashdatum value;
    char * ui_path;
    char buffer[PATH_MAX];
    char * p;
    char new_ui_path[PATH_MAX];
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    
    folder = value.data;    
    ui_path = etpan_folder_get_ui_path(folder);
    strncpy(buffer, ui_path, sizeof(buffer));
    buffer[sizeof(buffer) - 1] = '\0';
    p = strchr(buffer, '/');
    if (p != NULL) {
      snprintf(new_ui_path, sizeof(new_ui_path), "%s%s", storage_id, p);
      etpan_folder_set_ui_path(folder, new_ui_path);
    }
  }
}

void etpan_storage_mark_connected_folder(struct etpan_storage * storage,
    struct etpan_folder * folder, int connected)
{
  chashdatum key;
  chashdatum value;
  
  key.data = &folder;
  key.len = sizeof(folder);
  value.data = folder;
  value.len = 0;
  
  if (connected)
    chash_set(storage->connected_folder_list, &key, &value, NULL);
  else
    chash_delete(storage->connected_folder_list, &key, NULL);
}

static struct etpan_error * storage_connect(struct etpan_storage * storage)
{
  struct etpan_error * error;
  
  /* try to connect */
  if (storage->driver->connect != NULL)
    error = storage->driver->connect(storage);
  else
    error = NULL;
  
  if (error != NULL) {
    /* disconnect everything if it failed */
    if (etpan_error_has_code(error, ERROR_STREAM)) {
      ETPAN_ERROR_FREE(error);
      
      ETPAN_LOCAL_LOG("reconnection of %s", etpan_storage_get_id(storage));
    
      storage_disconnect(storage);
    
      /* retry connection */
      if (storage->driver->connect != NULL)
	error = storage->driver->connect(storage);
      else
	error = NULL;
    }
  }
  
  return error;
}

static void storage_disconnect(struct etpan_storage * storage)
{
  chash * folder_list;
  chashiter * iter;
  carray * folder_array;
  unsigned int i;
  int r;
  
  /* force disconnection of every subfolders */
  
  folder_array = carray_new(4);
  if (folder_array == NULL) {
    /* error */
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  ETPAN_LOCAL_LOG("storage : %s", storage->id);
  folder_list = storage->connected_folder_list;
  for(iter = chash_begin(folder_list) ; iter != NULL ;
      iter = chash_next(folder_list, iter)) {
    chashdatum value;
    struct etpan_folder * folder;
    
    chash_value(iter, &value);
    folder = value.data;
    
    r = carray_add(folder_array, folder, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(folder_array) ; i ++) {
    struct etpan_folder * folder;
    
    folder = carray_get(folder_array, i);
    ETPAN_LOCAL_LOG("disconnect folder of storage %s", etpan_folder_get_ui_path(folder));
    
    etpan_folder_disconnect_nt(folder);
  }
  
  carray_free(folder_array);
  
  if (storage->driver != NULL) {
    if (storage->driver->disconnect != NULL)
      storage->driver->disconnect(storage);
  }
}

struct etpan_error * etpan_storage_connect_nt(struct etpan_storage * storage)
{
  return storage_connect(storage);
}

void etpan_storage_disconnect_nt(struct etpan_storage * storage)
{
  storage_disconnect(storage);
}


void etpan_storage_set_cache_path(struct etpan_storage * storage,
    char * cache_path)
{
  if (cache_path != storage->cache_path) {
    free(storage->cache_path);
    if (cache_path != NULL) {
      storage->cache_path = strdup(cache_path);
      if (storage->cache_path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      storage->cache_path = NULL;
  }
}

char * etpan_storage_get_cache_path(struct etpan_storage * storage)
{
  return storage->cache_path;
}

void etpan_storage_set_flags_path(struct etpan_storage * storage,
    char * flags_path)
{
  if (flags_path != storage->flags_path) {
    free(storage->flags_path);
    if (flags_path != NULL) {
      storage->flags_path = strdup(flags_path);
      if (storage->flags_path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      storage->flags_path = NULL;
  }
}

char * etpan_storage_get_flags_path(struct etpan_storage * storage)
{
  return storage->flags_path;
}

char * etpan_storage_get_threaded_cache_path(struct etpan_storage *
    storage)
{
  return storage->threaded_cache_path;
}

char * etpan_storage_get_threaded_flags_path(struct etpan_storage *
    storage)
{
  return storage->threaded_flags_path;
}

char * etpan_storage_get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name)
{
  if (storage->driver->get_sub_folder_location != NULL)
    return storage->driver->get_sub_folder_location(storage,
        parent, name);
  else
    return NULL;
}

int etpan_storage_allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent)
{
  if (storage->driver->allows_sub_folders != NULL)
    return storage->driver->allows_sub_folders(storage, parent);
  else
    return 0;
}
