#include "etpan-thread-manager-app.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-thread-manager.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-storage.h"

struct etpan_thread_manager_app * etpan_thread_manager_app_new(void)
{
  struct etpan_thread_manager_app * manager_app;
  
  manager_app = malloc(sizeof(* manager_app));
  if (manager_app == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager_app->manager = etpan_thread_manager_new();
  if (manager_app->manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager_app->storage_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager_app->storage_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager_app->sender_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager_app->storage_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager_app->abook_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager_app->storage_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager_app->stop_callback = NULL;
  manager_app->stop_cb_data = NULL;
  manager_app->stop_called = 0;
  
  return manager_app;
}

void etpan_thread_manager_app_free(struct etpan_thread_manager_app *
    manager_app)
{
  chash_free(manager_app->abook_hash);
  chash_free(manager_app->sender_hash);
  chash_free(manager_app->storage_hash);
  etpan_thread_manager_free(manager_app->manager);
  
  free(manager_app);
}

/* start */

void etpan_thread_manager_app_start(struct etpan_thread_manager_app *
    manager_app)
{
  etpan_thread_manager_start(manager_app->manager);
}

static void stop_callback(struct etpan_thread_manager_app * manager_app)
{
  if (etpan_thread_manager_app_is_stopped(manager_app)) {
    if (manager_app->stop_callback != NULL) {
      manager_app->stop_callback(manager_app->stop_cb_data);
    }
    manager_app->stop_callback = NULL;
    manager_app->stop_cb_data = NULL;
    manager_app->stop_called = 0;
  }
}

void etpan_thread_manager_app_stop(struct etpan_thread_manager_app *
    manager_app, void (* callback)(void *), void * cb_data)
{
  {
    chashiter * iter;
    
    for(iter = chash_begin(manager_app->storage_hash) ; iter != NULL ;
        iter = chash_next(manager_app->storage_hash, iter)) {
      chashdatum key;
      struct etpan_storage * storage;
      
      chash_key(iter, &key);
      memcpy(&storage, key.data, sizeof(storage));
      ETPAN_LOG("storage bound %s", etpan_storage_get_id(storage));
    }
  }
  
  if (manager_app->stop_called) {
    ETPAN_LOG("thread manager stop already called");
    etpan_crash();
  }
  
  manager_app->stop_called = 1;
  manager_app->stop_callback = callback;
  manager_app->stop_cb_data = cb_data;
  etpan_thread_manager_stop(manager_app->manager);
  
  stop_callback(manager_app);
}

int etpan_thread_manager_app_is_stopped(struct etpan_thread_manager_app *
    manager_app)
{
  return etpan_thread_manager_is_stopped(manager_app->manager);
}

void etpan_thread_manager_app_join(struct etpan_thread_manager_app *
    manager_app)
{
  etpan_thread_manager_join(manager_app->manager);
}

int etpan_thread_manager_app_get_fd(struct etpan_thread_manager_app *
    manager_app)
{
  return etpan_thread_manager_get_fd(manager_app->manager);
}

void etpan_thread_manager_app_loop(struct etpan_thread_manager_app *
    manager_app)
{
  etpan_thread_manager_loop(manager_app->manager);
  
  stop_callback(manager_app);
}


/* particular threads */

static struct etpan_thread *
etpan_thread_manager_app_get_misc_thread(struct etpan_thread_manager_app *
    manager_app)
{
  return etpan_thread_manager_get_thread(manager_app->manager);
}

struct etpan_thread *
etpan_thread_manager_app_get_storage_thread(struct etpan_thread_manager_app *
    manager_app, struct etpan_storage * storage)
{
  chashdatum key;
  chashdatum value;
  struct etpan_thread * thread;
  int r;
  
  key.data = &storage;
  key.len = sizeof(storage);
  
  r = chash_get(manager_app->storage_hash, &key, &value);
  if (r < 0) {
    thread = etpan_thread_manager_app_get_misc_thread(manager_app);
    
    value.data = thread;
    value.len = 0;
    r = chash_set(manager_app->storage_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_thread_bind(thread);
  }
  else {
    thread = value.data;
  }
  
  return thread;
}

struct etpan_thread *
etpan_thread_manager_app_get_sender_thread(struct etpan_thread_manager_app *
    manager_app, struct etpan_sender * sender)
{
  chashdatum key;
  chashdatum value;
  struct etpan_thread * thread;
  int r;
  
  key.data = &sender;
  key.len = sizeof(sender);
  
  r = chash_get(manager_app->sender_hash, &key, &value);
  if (r < 0) {
    thread = etpan_thread_manager_app_get_misc_thread(manager_app);
    
    value.data = thread;
    value.len = 0;
    r = chash_set(manager_app->sender_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_thread_bind(thread);
  }
  else {
    thread = value.data;
  }
  
  return thread;
}

struct etpan_thread *
etpan_thread_manager_app_get_abook_thread(struct etpan_thread_manager_app *
    manager_app, struct etpan_abook * abook)
{
  chashdatum key;
  chashdatum value;
  struct etpan_thread * thread;
  int r;
  
  key.data = &abook;
  key.len = sizeof(abook);
  
  r = chash_get(manager_app->abook_hash, &key, &value);
  if (r < 0) {
    thread = etpan_thread_manager_app_get_misc_thread(manager_app);
    
    value.data = thread;
    value.len = 0;
    r = chash_set(manager_app->abook_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    etpan_thread_bind(thread);
  }
  else {
    thread = value.data;
  }
  
  return thread;
}

void etpan_thread_manager_app_unbind_storage_thread(struct etpan_thread_manager_app *
    manager_app, struct etpan_storage * storage)
{
  chashdatum key;
  chashdatum value;
  struct etpan_thread * thread;
  int r;
  
  key.data = &storage;
  key.len = sizeof(storage);
  
  r = chash_get(manager_app->storage_hash, &key, &value);
  if (r < 0)
    return;
  
  chash_delete(manager_app->storage_hash, &key, NULL);
  thread = value.data;
  etpan_thread_unbind(thread);
}

void etpan_thread_manager_app_unbind_sender_thread(struct etpan_thread_manager_app *
    manager_app, struct etpan_sender * sender)
{
  chashdatum key;
  chashdatum value;
  struct etpan_thread * thread;
  int r;
  
  key.data = &sender;
  key.len = sizeof(sender);
  
  r = chash_get(manager_app->sender_hash, &key, &value);
  if (r < 0)
    return;
  
  chash_delete(manager_app->sender_hash, &key, NULL);
  thread = value.data;
  etpan_thread_unbind(thread);
}

void etpan_thread_manager_app_unbind_abook_thread(struct etpan_thread_manager_app *
    manager_app, struct etpan_abook * abook)
{
  chashdatum key;
  chashdatum value;
  struct etpan_thread * thread;
  int r;
  
  key.data = &abook;
  key.len = sizeof(abook);
  
  r = chash_get(manager_app->abook_hash, &key, &value);
  if (r < 0)
    return;
  
  chash_delete(manager_app->abook_hash, &key, NULL);
  thread = value.data;
  etpan_thread_unbind(thread);
}

/* manager op */

void etpan_thread_manager_app_misc_schedule(struct etpan_thread_manager_app *
    manager_app, struct etpan_thread_op * op)
{
  struct etpan_thread * thread;
  
  thread = etpan_thread_manager_app_get_misc_thread(manager_app);
  etpan_thread_op_schedule(thread, op);
}

void etpan_thread_manager_app_storage_schedule(struct etpan_thread_manager_app *
    manager_app, struct etpan_storage * storage,
    struct etpan_thread_op * op)
{
  struct etpan_thread * thread;
  
  thread = etpan_thread_manager_app_get_storage_thread(manager_app, storage);
  etpan_thread_op_schedule(thread, op);
}

void etpan_thread_manager_app_sender_schedule(struct etpan_thread_manager_app *
    manager_app, struct etpan_sender * sender,
    struct etpan_thread_op * op)
{
  struct etpan_thread * thread;
  
  thread = etpan_thread_manager_app_get_sender_thread(manager_app, sender);
  etpan_thread_op_schedule(thread, op);
}

void etpan_thread_manager_app_abook_schedule(struct etpan_thread_manager_app *
    manager_app, struct etpan_abook * abook,
    struct etpan_thread_op * op)
{
  struct etpan_thread * thread;
  
  thread = etpan_thread_manager_app_get_abook_thread(manager_app, abook);
  etpan_thread_op_schedule(thread, op);
}

static struct etpan_thread_manager_app * default_manager = NULL;

void etpan_thread_manager_app_set_default(struct etpan_thread_manager_app *
    manager_app)
{
  default_manager = manager_app;
}

struct etpan_thread_manager_app * etpan_thread_manager_app_get_default(void)
{
  return default_manager;
}

void etpan_thread_manager_app_run_in_main_thread(struct etpan_thread_manager_app * manager_app, void (* f)(void *), void * data, int wait)
{
  etpan_thread_manager_run_in_main_thread(manager_app->manager, f, data, wait);
}
