/**
 * uri.c: set of generic URI related routines 
 *
 * Reference: RFC 2396
 *
 * See Copyright for the status of this software.
 *
 * Daniel.Veillard@w3.org
 * adapted by DINH Viet Hoa for etpanX
 */

/*
Copyright  1998 World Wide Web Consortium, (Massachusetts Institute of
Technology, Institut National de Recherche en Informatique et en
Automatique, Keio University). All Rights Reserved. 

This W3C software is being provided by the copyright holders under the
following license. By obtaining, using and/or copying this software,
you agree that you have read, understood, and will comply with the following
terms and conditions: 

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee or royalty is hereby
granted, provided that you include the following on ALL copies of the
software and documentation or portions thereof, including modifications,
that you make: 

    1.A link or URL to the original W3C source. 
    2.Any pre-existing intellectual property disclaimers. If none exist,
      then a notice of the form: "Copyright  World Wide Web Consortium,
      (Massachusetts Institute of Technology, Institut National de Recherche
      en Informatique et en Automatique, Keio University). All Rights Reserved." 

When space permits, inclusion of the full text of this NOTICE should be
provided. In addition, credit shall be attributed to the copyright holders
for any software, documents, or other items or products that you create
pursuant to the implementation of the contents of this document, or any
portion thereof. 

THIS SOFTWARE AND DOCUMENTATION IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS
MAKE NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT
NOT LIMITED TO, WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR
PURPOSE OR THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE
ANY THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS. 

COPYRIGHT HOLDERS WILL NOT BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL OR
CONSEQUENTIAL DAMAGES ARISING OUT OF ANY USE OF THE SOFTWARE OR DOCUMENTATION. 

The name and trademarks of copyright holders may NOT be used in advertising
or publicity pertaining to the software without specific, written prior
permission. Title to copyright in this software and any associated
documentation will at all times remain with copyright holders. 
*/

/*
  See http://www.w3c.org for the original source
*/

#include "etpan-uri.h"
#include "etpan-error.h"
#include "etpan-log.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>


/************************************************************************
 *									*
 *		Macros to differenciate various character type		*
 *			directly extracted from RFC 2396		*
 *									*
 ************************************************************************/

/*
 * alpha    = lowalpha | upalpha
 */
#define IS_ALPHA(x) (IS_LOWALPHA(x) || IS_UPALPHA(x))


 /*
  * lowalpha = "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" |
  *            "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" |
  *            "u" | "v" | "w" | "x" | "y" | "z"
  */

#define IS_LOWALPHA(x) (((x) >= 'a') && ((x) <= 'z'))

 /*
  * upalpha = "A" | "B" | "C" | "D" | "E" | "F" | "G" | "H" | "I" | "J" |
  *           "K" | "L" | "M" | "N" | "O" | "P" | "Q" | "R" | "S" | "T" |
  *           "U" | "V" | "W" | "X" | "Y" | "Z"
  */
#define IS_UPALPHA(x) (((x) >= 'A') && ((x) <= 'Z'))

 /*
  * digit = "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9"
  */

#define IS_DIGIT(x) (((x) >= '0') && ((x) <= '9'))

 /*
  * alphanum = alpha | digit
  */

#define IS_ALPHANUM(x) (IS_ALPHA(x) || IS_DIGIT(x))

 /*
  * hex = digit | "A" | "B" | "C" | "D" | "E" | "F" |
  *               "a" | "b" | "c" | "d" | "e" | "f"
  */

#define IS_HEX(x) ((IS_DIGIT(x)) || (((x) >= 'a') && ((x) <= 'f')) || \
	    (((x) >= 'A') && ((x) <= 'F')))

 /*
  * mark = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")"
  */

#define IS_MARK(x) (((x) == '-') || ((x) == '_') || ((x) == '.') ||	\
    ((x) == '!') || ((x) == '~') || ((x) == '*') || ((x) == '\'') ||	\
    ((x) == '(') || ((x) == ')'))


 /*
  * reserved = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" | "$" | ","
  */

#define IS_RESERVED(x) (((x) == ';') || ((x) == '/') || ((x) == '?') ||	\
        ((x) == ':') || ((x) == '@') || ((x) == '&') || ((x) == '=') ||	\
	((x) == '+') || ((x) == '$') || ((x) == ','))

 /*
  * unreserved = alphanum | mark
  */

#define IS_UNRESERVED(x) (IS_ALPHANUM(x) || IS_MARK(x))

 /*
  * escaped = "%" hex hex
  */

#define IS_ESCAPED(p) ((*(p) == '%') && (IS_HEX((p)[1])) &&		\
	    (IS_HEX((p)[2])))

 /*
  * uric_no_slash = unreserved | escaped | ";" | "?" | ":" | "@" |
  *                        "&" | "=" | "+" | "$" | ","
  */
#define IS_URIC_NO_SLASH(p) ((IS_UNRESERVED(*(p))) || (IS_ESCAPED(p)) ||\
	        ((*(p) == ';')) || ((*(p) == '?')) || ((*(p) == ':')) ||\
	        ((*(p) == '@')) || ((*(p) == '&')) || ((*(p) == '=')) ||\
	        ((*(p) == '+')) || ((*(p) == '$')) || ((*(p) == ',')))

 /*
  * pchar = unreserved | escaped | ":" | "@" | "&" | "=" | "+" | "$" | ","
  */
#define IS_PCHAR(p) ((IS_UNRESERVED(*(p))) || (IS_ESCAPED(p)) ||	\
	        ((*(p) == ':')) || ((*(p) == '@')) || ((*(p) == '&')) ||\
	        ((*(p) == '=')) || ((*(p) == '+')) || ((*(p) == '$')) ||\
	        ((*(p) == ',')))

 /*
  * rel_segment   = 1*( unreserved | escaped |
  *                 ";" | "@" | "&" | "=" | "+" | "$" | "," )
  */

#define IS_SEGMENT(p) ((IS_UNRESERVED(*(p))) || (IS_ESCAPED(p)) ||	\
          ((*(p) == ';')) || ((*(p) == '@')) || ((*(p) == '&')) ||	\
	  ((*(p) == '=')) || ((*(p) == '+')) || ((*(p) == '$')) ||	\
	  ((*(p) == ',')))

 /*
  * scheme = alpha *( alpha | digit | "+" | "-" | "." )
  */

#define IS_SCHEME(x) ((IS_ALPHA(x)) || (IS_DIGIT(x)) ||			\
	              ((x) == '+') || ((x) == '-') || ((x) == '.'))

 /*
  * reg_name = 1*( unreserved | escaped | "$" | "," |
  *                ";" | ":" | "@" | "&" | "=" | "+" )
  */

#define IS_REG_NAME(p) ((IS_UNRESERVED(*(p))) || (IS_ESCAPED(p)) ||	\
       ((*(p) == '$')) || ((*(p) == ',')) || ((*(p) == ';')) ||		\
       ((*(p) == ':')) || ((*(p) == '@')) || ((*(p) == '&')) ||		\
       ((*(p) == '=')) || ((*(p) == '+')))

 /*
  * userinfo = *( unreserved | escaped | ";" | ":" | "&" | "=" |
  *                      "+" | "$" | "," )
  */
#define IS_USERINFO(p) ((IS_UNRESERVED(*(p))) || (IS_ESCAPED(p)) ||	\
       ((*(p) == ';')) || ((*(p) == ':')) || ((*(p) == '&')) ||		\
       ((*(p) == '=')) || ((*(p) == '+')) || ((*(p) == '$')) ||		\
       ((*(p) == ',')))

 /*
  * uric = reserved | unreserved | escaped
  */

#define IS_URIC(p) ((IS_UNRESERVED(*(p))) || (IS_ESCAPED(p)) ||		\
	            (IS_RESERVED(*(p))))

 /*
  * Skip to next pointer char, handle escaped sequences
  */

#define NEXT(p) ((*p == '%')? p += 3 : p++)

 /*
  * Productions from the spec.
  *
  *    authority     = server | reg_name
  *    reg_name      = 1*( unreserved | escaped | "$" | "," |
  *                        ";" | ":" | "@" | "&" | "=" | "+" )
  *
  * path          = [ abs_path | opaque_part ]
  */

 /************************************************************************
  *									*
  *			Generic URI structure functions			*
  *									*
  ************************************************************************/

 /**
  * xmlCreateURI:
  *
  * Simply creates an empty xmlURI
  *
  * Returns the new structure or NULL in case of error
  */
struct etpan_uri * etpan_uri_new(void)
{
  struct etpan_uri * ret;
  
  ret = malloc(sizeof(* ret));
  if (ret == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  memset(ret, 0, sizeof(* ret));
  return(ret);
}

/**
 * xmlSaveUri:
 * @uri:  pointer to an xmlURI
 *
 * Save the URI as an escaped string
 *
 * Returns a new string (to be deallocated by caller)
 */
static char * uri_render(struct etpan_uri * uri);

char * etpan_uri_render(struct etpan_uri * uri)
{
  char * value;
  
  value = uri_render(uri);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}

static char * uri_render(struct etpan_uri * uri)
{
  char * ret = NULL;
  const char * p;
  int len;
  int max;

  if (uri == NULL)
    return(NULL);

  max = 80;
  ret = malloc(max + 1);
  if (ret == NULL) {
    return(NULL);
  }
  len = 0;

  if (uri->scheme != NULL) {
    p = uri->scheme;
    while (*p != 0) {
      if (len >= max) {
        char * new_ret;
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(ret);
          return(NULL);
        }
        ret = new_ret;
      }
      ret[len++] = *p++;
    }
    if (len >= max) {
      char * new_ret;
      
      max *= 2;
      new_ret = realloc(ret, max + 1);
      if (new_ret == NULL) {
        free(ret);
        return(NULL);
      }
      ret = new_ret;
    }
    ret[len++] = ':';
  }
  if (uri->opaque != NULL) {
    p = uri->opaque;
    while (*p != 0) {
      if (len + 3 >= max) {
        char * new_ret;
        
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(new_ret);
          return(NULL);
        }
        ret = new_ret;
      }
      if ((IS_UNRESERVED(*(p))) ||
          ((*(p) == ';')) || ((*(p) == '?')) || ((*(p) == ':')) ||
          ((*(p) == '@')) || ((*(p) == '&')) || ((*(p) == '=')) ||
          ((*(p) == '+')) || ((*(p) == '$')) || ((*(p) == ',')))
        ret[len++] = *p++;
      else {
        int val = *(unsigned char *)p++;
        int hi = val / 0x10, lo = val % 0x10;
        ret[len++] = '%';
        ret[len++] = hi + (hi > 9? 'A'-10 : '0');
        ret[len++] = lo + (lo > 9? 'A'-10 : '0');
      }
    }
    if (len >= max) {
      char * new_ret;
      
      max *= 2;
      new_ret = realloc(ret, max + 1);
      if (new_ret == NULL) {
        free(ret);
        return(NULL);
      }
      new_ret = ret;
    }
    ret[len++] = 0;
  } else {
    if (uri->server != NULL) {
      if (len + 3 >= max) {
        char * new_ret;
        
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(ret);
          return(NULL);
        }
        ret = new_ret;
      }
      ret[len++] = '/';
      ret[len++] = '/';
      if (uri->user != NULL) {
        p = uri->user;
        while (*p != 0) {
          if (len + 3 >= max) {
            char * new_ret;
            
            max *= 2;
            new_ret = realloc(ret, max + 1);
            if (new_ret == NULL) {
              free(ret);
              return(NULL);
            }
            ret = new_ret;
          }
          if ((IS_UNRESERVED(*(p))) ||
              ((*(p) == ';')) || ((*(p) == ':')) ||
              ((*(p) == '&')) || ((*(p) == '=')) ||
              ((*(p) == '+')) || ((*(p) == '$')) ||
              ((*(p) == ',')))
            ret[len++] = *p++;
          else {
            int val = *(unsigned char *)p++;
            int hi = val / 0x10, lo = val % 0x10;
            ret[len++] = '%';
            ret[len++] = hi + (hi > 9? 'A'-10 : '0');
            ret[len++] = lo + (lo > 9? 'A'-10 : '0');
          }
        }
        if (len + 3 >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        ret[len++] = '@';
      }
      p = uri->server;
      while (*p != 0) {
        if (len >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        ret[len++] = *p++;
      }
      if (uri->port > 0) {
        if (len + 10 >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        len += sprintf((char *) &ret[len], ":%d", uri->port);
      }
    } else if (uri->authority != NULL) {
      if (len + 3 >= max) {
        char * new_ret;
        
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(ret);
          return(NULL);
        }
        ret = new_ret;
      }
      ret[len++] = '/';
      ret[len++] = '/';
      p = uri->authority;
      while (*p != 0) {
        if (len + 3 >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        if ((IS_UNRESERVED(*(p))) ||
            ((*(p) == '$')) || ((*(p) == ',')) || ((*(p) == ';')) ||
            ((*(p) == ':')) || ((*(p) == '@')) || ((*(p) == '&')) ||
            ((*(p) == '=')) || ((*(p) == '+')))
          ret[len++] = *p++;
        else {
          int val = *(unsigned char *)p++;
          int hi = val / 0x10, lo = val % 0x10;
          ret[len++] = '%';
          ret[len++] = hi + (hi > 9? 'A'-10 : '0');
          ret[len++] = lo + (lo > 9? 'A'-10 : '0');
        }
      }
    } else if (uri->scheme != NULL) {
      if (len + 3 >= max) {
        char * new_ret;
        
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(ret);
          return(NULL);
        }
        ret = new_ret;
      }
      ret[len++] = '/';
      ret[len++] = '/';
    }
    if (uri->path != NULL) {
      p = uri->path;
      while (*p != 0) {
        if (len + 3 >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        if ((IS_UNRESERVED(*(p))) || ((*(p) == '/')) ||
            ((*(p) == ';')) || ((*(p) == '@')) || ((*(p) == '&')) ||
            ((*(p) == '=')) || ((*(p) == '+')) || ((*(p) == '$')) ||
            ((*(p) == ',')))
          ret[len++] = *p++;
        else {
          int val = *(unsigned char *)p++;
          int hi = val / 0x10, lo = val % 0x10;
          ret[len++] = '%';
          ret[len++] = hi + (hi > 9? 'A'-10 : '0');
          ret[len++] = lo + (lo > 9? 'A'-10 : '0');
        }
      }
    }
    if (uri->query != NULL) {
      if (len + 3 >= max) {
        char * new_ret;
        
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(ret);
          return(NULL);
        }
        ret = new_ret;
      }
      ret[len++] = '?';
      p = uri->query;
      while (*p != 0) {
        if (len + 3 >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        if ((IS_UNRESERVED(*(p))) || (IS_RESERVED(*(p)))) 
          ret[len++] = *p++;
        else {
          int val = *(unsigned char *)p++;
          int hi = val / 0x10, lo = val % 0x10;
          ret[len++] = '%';
          ret[len++] = hi + (hi > 9? 'A'-10 : '0');
          ret[len++] = lo + (lo > 9? 'A'-10 : '0');
        }
      }
    }
    if (uri->fragment != NULL) {
      if (len + 3 >= max) {
        char * new_ret;
        
        max *= 2;
        new_ret = realloc(ret, max + 1);
        if (new_ret == NULL) {
          free(ret);
          return(NULL);
        }
        ret = new_ret;
      }
      ret[len++] = '#';
      p = uri->fragment;
      while (*p != 0) {
        if (len + 3 >= max) {
          char * new_ret;
          
          max *= 2;
          new_ret = realloc(ret, max + 1);
          if (new_ret == NULL) {
            free(ret);
            return(NULL);
          }
          ret = new_ret;
        }
        if ((IS_UNRESERVED(*(p))) || (IS_RESERVED(*(p)))) 
          ret[len++] = *p++;
        else {
          int val = *(unsigned char *)p++;
          int hi = val / 0x10, lo = val % 0x10;
          ret[len++] = '%';
          ret[len++] = hi + (hi > 9? 'A'-10 : '0');
          ret[len++] = lo + (lo > 9? 'A'-10 : '0');
        }
      }
    }
    if (len >= max) {
      char * new_ret;
      
      max *= 2;
      new_ret = realloc(ret, max + 1);
      if (new_ret == NULL) {
        free(ret);
        return(NULL);
      }
      ret = new_ret;
    }
    ret[len++] = 0;
  }
  return(ret);
}

/**
 * xmlPrintURI:
 * @stream:  a FILE* for the output
 * @uri:  pointer to an xmlURI
 *
 * Prints the URI in the stream @steam.
 */
void etpan_uri_print(FILE * stream, struct etpan_uri * uri)
{
  char * out;

  out = etpan_uri_render(uri);
  if (out != NULL) {
    fprintf(stream, "%s", out);
    free(out);
  }
}

/**
 * xmlCleanURI:
 * @uri:  pointer to an xmlURI
 *
 * Make sure the xmlURI struct is free of content
 */
void etpan_uri_reset(struct etpan_uri * uri)
{
  if (uri == NULL) return;

  if (uri->scheme != NULL) free(uri->scheme);
  uri->scheme = NULL;
  if (uri->server != NULL) free(uri->server);
  uri->server = NULL;
  if (uri->user != NULL) free(uri->user);
  uri->user = NULL;
  if (uri->path != NULL) free(uri->path);
  uri->path = NULL;
  if (uri->fragment != NULL) free(uri->fragment);
  uri->fragment = NULL;
  if (uri->opaque != NULL) free(uri->opaque);
  uri->opaque = NULL;
  if (uri->authority != NULL) free(uri->authority);
  uri->authority = NULL;
  if (uri->query != NULL) free(uri->query);
  uri->query = NULL;
}

/**
 * xmlFreeURI:
 * @uri:  pointer to an xmlURI
 *
 * Free up the xmlURI struct
 */
void etpan_uri_free(struct etpan_uri * uri)
{
  if (uri == NULL) return;

  if (uri->scheme != NULL) free(uri->scheme);
  if (uri->server != NULL) free(uri->server);
  if (uri->user != NULL) free(uri->user);
  if (uri->path != NULL) free(uri->path);
  if (uri->fragment != NULL) free(uri->fragment);
  if (uri->opaque != NULL) free(uri->opaque);
  if (uri->authority != NULL) free(uri->authority);
  if (uri->query != NULL) free(uri->query);
  memset(uri, -1, sizeof(* uri));
  free(uri);
}

/************************************************************************
 *									*
 *			Helper functions				*
 *									*
 ************************************************************************/

/**
 * xmlNormalizeURIPath:
 * @path:  pointer to the path string
 *
 * applies the 5 normalization steps to a path string
 * Normalization occurs directly on the string, no new allocation is done
 *
 * Returns 0 or an error code
 */
static int uri_normalize_path(char * path);

void etpan_uri_normalize_path(char * path)
{
  if (uri_normalize_path(path) < 0) {
    ETPAN_LOG("normalize failed %s", path);
    etpan_crash();
  }
}

static int uri_normalize_path(char * path)
{
  int cur, out;

  if (path == NULL)
    return(-1);
  cur = 0;
  out = 0;
  while ((path[cur] != 0) && (path[cur] != '/')) cur++;
  if (path[cur] == 0)
    return(0);

  /* we are positionned at the beginning of the first segment */
  cur++;
  out = cur;

  /*
   * Analyze each segment in sequence.
   */
  while (path[cur] != 0) {
    /*
     * c) All occurrences of "./", where "." is a complete path segment,
     *    are removed from the buffer string.
     */
    if ((path[cur] == '.') && (path[cur + 1] == '/')) {
      cur += 2;
      continue;
    }

    /*
     * d) If the buffer string ends with "." as a complete path segment,
     *    that "." is removed.
     */
    if ((path[cur] == '.') && (path[cur + 1] == 0)) {
      path[out] = 0;
      break;
    }

    /* read the segment */
    while ((path[cur] != 0) && (path[cur] != '/')) {
      path[out++] = path[cur++];
    }
    path[out++] = path[cur];
    if (path[cur] != 0) {
      cur++;
    }
  }

  cur = 0;
  out = 0;
  while ((path[cur] != 0) && (path[cur] != '/')) cur++;
  if (path[cur] == 0)
    return(0);
  /* we are positionned at the beginning of the first segment */
  cur++;
  out = cur;
  /*
   * Analyze each segment in sequence.
   */
  while (path[cur] != 0) {
    /*
     * e) All occurrences of "<segment>/../", where <segment> is a
     *    complete path segment not equal to "..", are removed from the
     *    buffer string.  Removal of these path segments is performed
     *    iteratively, removing the leftmost matching pattern on each
     *    iteration, until no matching pattern remains.
     */
    if ((cur > 1) && (out > 1) &&
        (path[cur] == '/') && (path[cur + 1] == '.') &&
        (path[cur + 2] == '.') && (path[cur + 3] == '/') &&
        ((path[out] != '.') || (path[out - 1] != '.') ||
            (path[out - 2] != '/'))) {
      cur += 3;
      out --;
      while ((out > 0) && (path[out] != '/')) { out --; }
      path[out] = 0;
      continue;
    }

    /*
     * f) If the buffer string ends with "<segment>/..", where <segment>
     *    is a complete path segment not equal to "..", that
     *    "<segment>/.." is removed.
     */
    if ((path[cur] == '/') && (path[cur + 1] == '.') &&
        (path[cur + 2] == '.') && (path[cur + 3] == 0) &&
        ((path[out] != '.') || (path[out - 1] != '.') ||
            (path[out - 2] != '/'))) {
      cur += 4;
      out --;
      while ((out > 0) && (path[out - 1] != '/')) { out --; }
      path[out] = 0;
      continue;
    }
        
    path[out++] = path[cur++]; /* / or 0 */
  }
  path[out] = 0;

  /*
   * g) If the resulting buffer string still begins with one or more
   *    complete path segments of "..", then the reference is 
   *    considered to be in error. Implementations may handle this
   *    error by retaining these components in the resolved path (i.e.,
   *    treating them as part of the final URI), by removing them from
   *    the resolved path (i.e., discarding relative levels above the
   *    root), or by avoiding traversal of the reference.
   *
   * We discard them from the final path.
   */
  cur = 0;
  while ((path[cur] == '/') && (path[cur + 1] == '.') &&
      (path[cur + 2] == '.'))
    cur += 3;
  if (cur != 0) {
    out = 0;
    while (path[cur] != 0) path[out++] = path[cur++];
    path[out] = 0;
  }
  return(0);
}

/**
 * xmlURIUnescapeString:
 * @str:  the string to unescape
 * @len:   the lenght in bytes to unescape (or <= 0 to indicate full string)
 * @target:  optionnal destination buffer
 *
 * Unescaping routine, does not do validity checks !
 * Output is direct unsigned char translation of %XX values (no encoding)
 *
 * Returns an copy of the string, but unescaped
 */
static char *
xmlURIUnescapeString(const char *str, int len, char *target) {
  char *ret, *out;
  const char *in;

  if (str == NULL)
    return(NULL);
  if (len <= 0) len = strlen(str);
  if (len <= 0) return(NULL);

  if (target == NULL) {
    ret = malloc(len + 1);
    if (ret == NULL) {
      return(NULL);
    }
  } else
    ret = target;
  in = str;
  out = ret;
  while(len > 0) {
    if (*in == '%') {
      in++;
      if ((*in >= '0') && (*in <= '9')) 
        *out = (*in - '0');
      else if ((*in >= 'a') && (*in <= 'f'))
        *out = (*in - 'a') + 10;
      else if ((*in >= 'A') && (*in <= 'F'))
        *out = (*in - 'A') + 10;
      in++;
      if ((*in >= '0') && (*in <= '9')) 
        *out = *out * 16 + (*in - '0');
      else if ((*in >= 'a') && (*in <= 'f'))
        *out = *out * 16 + (*in - 'a') + 10;
      else if ((*in >= 'A') && (*in <= 'F'))
        *out = *out * 16 + (*in - 'A') + 10;
      in++;
      len -= 3;
      out++;
    } else {
      *out++ = *in++;
      len--;
    }
  }
  *out = 0;
  return(ret);
}

char * etpan_uri_unescape(const char * str)
{
  char * value;
  
  value = xmlURIUnescapeString(str, strlen(str), NULL);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}

/**
 * xmlURIEscape:
 * @str:  the string of the URI to escape
 *
 * Escaping routine, does not do validity checks !
 * It will try to escape the chars needing this, but this is heuristic
 * based it's impossible to be sure.
 *
 * Returns an copy of the string, but escaped
 */
static char * uri_escape(const char * str);

char * etpan_uri_escape(const char * str)
{
  char * value;
  
  value = uri_escape(str);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}

static char * uri_escape(const char * str)
{
  char * ret;
  const char *in;
  unsigned int len, out;

  if (str == NULL)
    return(NULL);
  len = strlen(str);
  if (len <= 0) return(NULL);

  len += 20;
  ret = malloc(len);
  if (ret == NULL) {
    return(NULL);
  }
  in = (const char *) str;
  out = 0;
  while(*in != 0) {
    if (len - out <= 3) {
      char * new_ret;
      
      len += 20;
      new_ret = realloc(ret, len);
      if (new_ret == NULL) {
        free(ret);
        return(NULL);
      }
    }
    if ((!IS_UNRESERVED(*in)) && (*in != ':') && (*in != '/') &&
        (*in != '?') && (*in != '#')) {
      unsigned char val;
      ret[out++] = '%';
      val = *in >> 4;
      if (val <= 9)
        ret[out++] = '0' + val;
      else
        ret[out++] = 'A' + val - 0xA;
      val = *in & 0xF;
      if (val <= 9)
        ret[out++] = '0' + val;
      else
        ret[out++] = 'A' + val - 0xA;
      in++;
    } else {
      ret[out++] = *in++;
    }
  }
  ret[out] = 0;
  return(ret);
}

/************************************************************************
 *									*
 *			Escaped URI parsing				*
 *									*
 ************************************************************************/

/**
 * xmlParseURIFragment:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an URI fragment string and fills in the appropriate fields
 * of the @uri structure.
 * 
 * fragment = *uric
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIFragment(struct etpan_uri * uri, const char ** str) {
  const char *cur = *str;

  if (str == NULL) return(-1);

  while (IS_URIC(cur)) NEXT(cur);
  if (uri != NULL) {
    if (uri->fragment != NULL) free(uri->fragment);
    uri->fragment = xmlURIUnescapeString(*str, cur - *str, NULL);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIQuery:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse the query part of an URI
 * 
 * query = *uric
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIQuery(struct etpan_uri * uri, const char **str) {
  const char *cur = *str;

  if (str == NULL) return(-1);

  while (IS_URIC(cur)) NEXT(cur);
  if (uri != NULL) {
    if (uri->query != NULL) free(uri->query);
    uri->query = xmlURIUnescapeString(*str, cur - *str, NULL);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIScheme:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an URI scheme
 * 
 * scheme = alpha *( alpha | digit | "+" | "-" | "." )
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIScheme(struct etpan_uri * uri, const char **str) {
  const char *cur;

  if (str == NULL)
    return(-1);
    
  cur = *str;
  if (!IS_ALPHA(*cur))
    return(2);
  cur++;
  while (IS_SCHEME(*cur)) cur++;
  if (uri != NULL) {
    if (uri->scheme != NULL) free(uri->scheme);
    /* !!! strndup */
    uri->scheme = xmlURIUnescapeString(*str, cur - *str, NULL);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIOpaquePart:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an URI opaque part
 * 
 * opaque_part = uric_no_slash *uric
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIOpaquePart(struct etpan_uri * uri, const char **str) {
  const char *cur;

  if (str == NULL)
    return(-1);
    
  cur = *str;
  if (!IS_URIC_NO_SLASH(cur)) {
    return(3);
  }
  NEXT(cur);
  while (IS_URIC(cur)) NEXT(cur);
  if (uri != NULL) {
    if (uri->opaque != NULL) free(uri->opaque);
    uri->opaque = xmlURIUnescapeString(*str, cur - *str, NULL);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIServer:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse a server subpart of an URI, it's a finer grain analysis
 * of the authority part.
 * 
 * server        = [ [ userinfo "@" ] hostport ]
 * userinfo      = *( unreserved | escaped |
 *                       ";" | ":" | "&" | "=" | "+" | "$" | "," )
 * hostport      = host [ ":" port ]
 * host          = hostname | IPv4address
 * hostname      = *( domainlabel "." ) toplabel [ "." ]
 * domainlabel   = alphanum | alphanum *( alphanum | "-" ) alphanum
 * toplabel      = alpha | alpha *( alphanum | "-" ) alphanum
 * IPv4address   = 1*digit "." 1*digit "." 1*digit "." 1*digit
 * port          = *digit
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIServer(struct etpan_uri * uri, const char **str) {
  const char *cur;
  const char *host, *tmp;

  if (str == NULL)
    return(-1);
    
  cur = *str;

  /*
   * is there an userinfo ?
   */
  while (IS_USERINFO(cur)) NEXT(cur);
  if (*cur == '@') {
    if (uri != NULL) {
      if (uri->user != NULL) free(uri->user);
      uri->user = xmlURIUnescapeString(*str, cur - *str, NULL);
    }
    cur++;
  } else {
    if (uri != NULL) {
      if (uri->user != NULL) free(uri->user);
      uri->user = NULL;
    }
    cur = *str;
  }
  /*
   * This can be empty in the case where there is no server
   */
  host = cur;
  if (*cur == '/') {
    if (uri != NULL) {
      if (uri->authority != NULL) free(uri->authority);
      uri->authority = NULL;
      if (uri->server != NULL) free(uri->server);
      uri->server = NULL;
      uri->port = 0;
    }
    return(0);
  }
  /*
   * host part of hostport can derive either an IPV4 address
   * or an unresolved name. Check the IP first, it easier to detect
   * errors if wrong one
   */
  if (IS_DIGIT(*cur)) {
    while(IS_DIGIT(*cur)) cur++;
    if (*cur != '.')
      goto host_name;
    cur++;
    if (!IS_DIGIT(*cur))
      goto host_name;
    while(IS_DIGIT(*cur)) cur++;
    if (*cur != '.')
      goto host_name;
    cur++;
    if (!IS_DIGIT(*cur))
      goto host_name;
    while(IS_DIGIT(*cur)) cur++;
    if (*cur != '.')
      goto host_name;
    cur++;
    if (!IS_DIGIT(*cur))
      goto host_name;
    while(IS_DIGIT(*cur)) cur++;
    if (uri != NULL) {
      if (uri->authority != NULL) free(uri->authority);
      uri->authority = NULL;
      if (uri->server != NULL) free(uri->server);
      uri->server = xmlURIUnescapeString(host, cur - host, NULL);
    }
    goto host_done;
  }
 host_name:
  /*
   * the hostname production as-is is a parser nightmare.
   * simplify it to 
   * hostname = *( domainlabel "." ) domainlabel [ "." ]
   * and just make sure the last label starts with a non numeric char.
   */
  if (!IS_ALPHANUM(*cur))
    return(6);
  while (IS_ALPHANUM(*cur)) {
    while ((IS_ALPHANUM(*cur)) || (*cur == '-')) cur++;
    if (*cur == '.')
      cur++;
  }
  tmp = cur;
  tmp--;
  while (IS_ALPHANUM(*tmp) && (*tmp != '.') && (tmp >= host)) tmp--;
  tmp++;
  if (!IS_ALPHA(*tmp))
    return(7);
  if (uri != NULL) {
    if (uri->authority != NULL) free(uri->authority);
    uri->authority = NULL;
    if (uri->server != NULL) free(uri->server);
    uri->server = xmlURIUnescapeString(host, cur - host, NULL);
  }

 host_done:

  /*
   * finish by checking for a port presence.
   */
  if (*cur == ':') {
    cur++;
    if (IS_DIGIT(*cur)) {
      if (uri != NULL)
        uri->port = 0;
      while (IS_DIGIT(*cur)) {
        if (uri != NULL)
          uri->port = uri->port * 10 + (*cur - '0');
        cur++;
      }
    }
  }
  *str = cur;
  return(0);
}	

/**
 * xmlParseURIRelSegment:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an URI relative segment
 * 
 * rel_segment = 1*( unreserved | escaped | ";" | "@" | "&" | "=" |
 *                          "+" | "$" | "," )
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIRelSegment(struct etpan_uri * uri, const char **str) {
  const char *cur;

  if (str == NULL)
    return(-1);
    
  cur = *str;
  if (!IS_SEGMENT(cur)) {
    return(3);
  }
  NEXT(cur);
  while (IS_SEGMENT(cur)) NEXT(cur);
  if (uri != NULL) {
    if (uri->path != NULL) free(uri->path);
    uri->path = xmlURIUnescapeString(*str, cur - *str, NULL);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIPathSegments:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 * @slash:  should we add a leading slash
 *
 * Parse an URI set of path segments
 * 
 * path_segments = segment *( "/" segment )
 * segment       = *pchar *( ";" param )
 * param         = *pchar
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIPathSegments(struct etpan_uri * uri, const char **str, int slash) {
  const char *cur;

  if (str == NULL)
    return(-1);
    
  cur = *str;

  do {
    while (IS_PCHAR(cur)) NEXT(cur);
    if (*cur == ';') {
      cur++;
      while (IS_PCHAR(cur)) NEXT(cur);
    }
    if (*cur != '/') break;
    cur++;
  } while (1);
  if (uri != NULL) {
    int len, len2 = 0;
    char *path;

    /*
     * Concat the set of path segments to the current path
     */
    len = cur - *str;
    if (slash)
      len++;

    if (uri->path != NULL) {
      len2 = strlen(uri->path);
      len += len2;
    }
    path = malloc(len + 1);
    if (path == NULL) {
      *str = cur;
      return(-1);
    }
    if (uri->path != NULL)
      memcpy(path, uri->path, len2);
    if (slash) {
      path[len2] = '/';
      len2++;
    }
    path[len2] = 0;
    if (cur - *str > 0)
      xmlURIUnescapeString(*str, cur - *str, &path[len2]);
    if (uri->path != NULL)
      free(uri->path);
    uri->path = path;
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIAuthority:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse the authority part of an URI.
 * 
 * authority = server | reg_name
 * server    = [ [ userinfo "@" ] hostport ]
 * reg_name  = 1*( unreserved | escaped | "$" | "," | ";" | ":" |
 *                        "@" | "&" | "=" | "+" )
 *
 * Note : this is completely ambiguous since reg_name is allowed to
 *        use the full set of chars in use by server:
 *
 *        3.2.1. Registry-based Naming Authority
 *
 *        The structure of a registry-based naming authority is specific
 *        to the URI scheme, but constrained to the allowed characters
 *        for an authority component.
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIAuthority(struct etpan_uri * uri, const char **str) {
  const char *cur;
  int ret;

  if (str == NULL)
    return(-1);
    
  cur = *str;

  /*
   * try first to parse it as a server string.
   */
  ret = xmlParseURIServer(uri, str);
  if (ret == 0)
    return(0);

  /*
   * failed, fallback to reg_name
   */
  if (!IS_REG_NAME(cur)) {
    return(5);
  }
  NEXT(cur);
  while (IS_REG_NAME(cur)) NEXT(cur);
  if (uri != NULL) {
    if (uri->server != NULL) free(uri->server);
    uri->server = NULL;
    if (uri->user != NULL) free(uri->user);
    uri->user = NULL;
    if (uri->authority != NULL) free(uri->authority);
    uri->authority = xmlURIUnescapeString(*str, cur - *str, NULL);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseURIHierPart:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an URI hirarchical part
 * 
 * hier_part = ( net_path | abs_path ) [ "?" query ]
 * abs_path = "/"  path_segments
 * net_path = "//" authority [ abs_path ]
 *
 * Returns 0 or the error code
 */
static int
xmlParseURIHierPart(struct etpan_uri * uri, const char **str) {
  int ret;
  const char *cur;

  if (str == NULL)
    return(-1);
    
  cur = *str;

  if ((cur[0] == '/') && (cur[1] == '/')) {
    cur += 2;
    ret = xmlParseURIAuthority(uri, &cur);
    if (ret != 0)
      return(ret);
    if (cur[0] == '/') {
      cur++;
      ret = xmlParseURIPathSegments(uri, &cur, 1);
    }
  } else if (cur[0] == '/') {
    cur++;
    ret = xmlParseURIPathSegments(uri, &cur, 1);
  } else {
    return(4);
  }
  if (ret != 0)
    return(ret);
  if (*cur == '?') {
    cur++;
    ret = xmlParseURIQuery(uri, &cur);
    if (ret != 0)
      return(ret);
  }
  *str = cur;
  return(0);
}

/**
 * xmlParseAbsoluteURI:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an URI reference string and fills in the appropriate fields
 * of the @uri structure
 * 
 * absoluteURI   = scheme ":" ( hier_part | opaque_part )
 *
 * Returns 0 or the error code
 */
static int
xmlParseAbsoluteURI(struct etpan_uri * uri, const char **str) {
  int ret;

  if (str == NULL)
    return(-1);
    
  ret = xmlParseURIScheme(uri, str);
  if (ret != 0) return(ret);
  if (**str != ':')
    return(1);
  (*str)++;
  if (**str == '/')
    return(xmlParseURIHierPart(uri, str));
  return(xmlParseURIOpaquePart(uri, str));
}

/**
 * xmlParseRelativeURI:
 * @uri:  pointer to an URI structure
 * @str:  pointer to the string to analyze
 *
 * Parse an relative URI string and fills in the appropriate fields
 * of the @uri structure
 * 
 * relativeURI = ( net_path | abs_path | rel_path ) [ "?" query ]
 * abs_path = "/"  path_segments
 * net_path = "//" authority [ abs_path ]
 * rel_path = rel_segment [ abs_path ]
 *
 * Returns 0 or the error code
 */
static int
xmlParseRelativeURI(struct etpan_uri * uri, const char **str) {
  int ret = 0;
  const char *cur;

  if (str == NULL)
    return(-1);
    
  cur = *str;
  if ((cur[0] == '/') && (cur[1] == '/')) {
    cur += 2;
    ret = xmlParseURIAuthority(uri, &cur);
    if (ret != 0)
      return(ret);
    if (cur[0] == '/') {
      cur++;
      ret = xmlParseURIPathSegments(uri, &cur, 1);
    }
  } else if (cur[0] == '/') {
    cur++;
    ret = xmlParseURIPathSegments(uri, &cur, 1);
  } else if (cur[0] != '#' && cur[0] != '?') {
    ret = xmlParseURIRelSegment(uri, &cur);
    if (ret != 0)
      return(ret);
    if (cur[0] == '/') {
      cur++;
      ret = xmlParseURIPathSegments(uri, &cur, 1);
    }
  }
  if (ret != 0)
    return(ret);
  if (*cur == '?') {
    cur++;
    ret = xmlParseURIQuery(uri, &cur);
    if (ret != 0)
      return(ret);
  }
  *str = cur;
  return(ret);
}

/**
 * xmlParseURIReference:
 * @uri:  pointer to an URI structure
 * @str:  the string to analyze
 *
 * Parse an URI reference string and fills in the appropriate fields
 * of the @uri structure
 * 
 * URI-reference = [ absoluteURI | relativeURI ] [ "#" fragment ]
 *
 * Returns 0 or the error code
 */
static int xmlParseURIReference(struct etpan_uri * uri, const char * str)
{
  int ret;
  const char *tmp = str;

  if (str == NULL)
    return(-1);
  etpan_uri_reset(uri);

  /*
   * Try first to parse aboslute refs, then fallback to relative if
   * it fails.
   */
  ret = xmlParseAbsoluteURI(uri, &str);
  if (ret != 0) {
    etpan_uri_reset(uri);
    str = tmp;
    ret = xmlParseRelativeURI(uri, &str);
  }
  if (ret != 0) {
    etpan_uri_reset(uri);
    return(ret);
  }

  if (*str == '#') {
    str++;
    ret = xmlParseURIFragment(uri, &str);
    if (ret != 0) return(ret);
  }
  if (*str != 0) {
    etpan_uri_reset(uri);
    return(1);
  }
  return(0);
}

int etpan_uri_parse_reference(struct etpan_uri * uri, const char * str)
{
  int r;
  
  r = xmlParseURIReference(uri, str);
  if (r == 0)
    return 0;
  
  return -1;
}

/**
 * xmlParseURI:
 * @str:  the URI string to analyze
 *
 * Parse an URI 
 * 
 * URI-reference = [ absoluteURI | relativeURI ] [ "#" fragment ]
 *
 * Returns a newly build xmlURIPtr or NULL in case of error
 */
struct etpan_uri * etpan_uri_parse(const char * str)
{
  struct etpan_uri * uri;
  int ret;

  if (str == NULL)
    return(NULL);
  uri = etpan_uri_new();
  if (uri != NULL) {
    ret = etpan_uri_parse_reference(uri, str);
    if (ret < 0) {
      etpan_uri_free(uri);
      return(NULL);
    }
  }
  return(uri);
}

/************************************************************************
 *									*
 *			Public functions				*
 *									*
 ************************************************************************/

/**
 * xmlBuildURI:
 * @URI:  the URI instance found in the document
 * @base:  the base value
 *
 * Computes he final URI of the reference done by checking that
 * the given URI is valid, and building the final URI using the
 * base URI. This is processed according to section 5.2 of the 
 * RFC 2396
 *
 * 5.2. Resolving Relative References to Absolute Form
 *
 * Returns a new URI string (to be freed by the caller) or NULL in case
 *         of error.
 */
static char * uri_build(const char * uri, const char * base);

char * etpan_uri_build(const char * uri, const char * base)
{
  char * value;
  
  value = uri_build(uri, base);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}


static char * uri_build(const char * uri, const char * base)
{
  char *val = NULL;
  int ret, len, index, cur, out;
  struct etpan_uri * ref = NULL;
  struct etpan_uri * bas = NULL;
  struct etpan_uri * res = NULL;

  /*
   * 1) The URI reference is parsed into the potential four components and
   *    fragment identifier, as described in Section 4.3.
   *
   *    NOTE that a completely empty URI is treated by modern browsers
   *    as a reference to "." rather than as a synonym for the current
   *    URI.  Should we do that here?
   */
  if (uri == NULL) 
    ret = -1;
  else {
    if (*uri) {
      ref = etpan_uri_new();
      if (ref == NULL)
        goto done;
      ret = etpan_uri_parse_reference(ref, uri);
    }
    else
      ret = 0;
  }
  if (ret != 0)
    goto done;
  if (base == NULL)
    ret = -1;
  else {
    bas = etpan_uri_new();
    if (bas == NULL)
      goto done;
    ret = etpan_uri_parse_reference(bas, (const char *) base);
  }
  if (ret != 0) {
    if (ref)
      val = etpan_uri_render(ref);
    goto done;
  }
  if (ref == NULL) {
    /*
     * the base fragment must be ignored
     */
    if (bas->fragment != NULL) {
      free(bas->fragment);
      bas->fragment = NULL;
    }
    val = etpan_uri_render(bas);
    goto done;
  }

  /*
   * 2) If the path component is empty and the scheme, authority, and
   *    query components are undefined, then it is a reference to the
   *    current document and we are done.  Otherwise, the reference URI's
   *    query and fragment components are defined as found (or not found)
   *    within the URI reference and not inherited from the base URI.
   *
   *    NOTE that in modern browsers, the parsing differs from the above
   *    in the following aspect:  the query component is allowed to be
   *    defined while still treating this as a reference to the current
   *    document.
   */
  res = etpan_uri_new();
  if (res == NULL)
    goto done;
  if ((ref->scheme == NULL) && (ref->path == NULL) &&
      ((ref->authority == NULL) && (ref->server == NULL))) {
    if (bas->scheme != NULL)
      res->scheme = strdup(bas->scheme);
    if (bas->authority != NULL)
      res->authority = strdup(bas->authority);
    else if (bas->server != NULL) {
      res->server = strdup(bas->server);
      if (bas->user != NULL)
        res->user = strdup(bas->user);
      res->port = bas->port;		
    }
    if (bas->path != NULL)
      res->path = strdup(bas->path);
    if (ref->query != NULL)
      res->query = strdup(ref->query);
    else if (bas->query != NULL)
      res->query = strdup(bas->query);
    if (ref->fragment != NULL)
      res->fragment = strdup(ref->fragment);
    goto step_7;
  }
 
  if (ref->query != NULL)
    res->query = strdup(ref->query);
  if (ref->fragment != NULL)
    res->fragment = strdup(ref->fragment);

  /*
   * 3) If the scheme component is defined, indicating that the reference
   *    starts with a scheme name, then the reference is interpreted as an
   *    absolute URI and we are done.  Otherwise, the reference URI's
   *    scheme is inherited from the base URI's scheme component.
   */
  if (ref->scheme != NULL) {
    val = etpan_uri_render(ref);
    goto done;
  }
  if (bas->scheme != NULL)
    res->scheme = strdup(bas->scheme);

  /*
   * 4) If the authority component is defined, then the reference is a
   *    network-path and we skip to step 7.  Otherwise, the reference
   *    URI's authority is inherited from the base URI's authority
   *    component, which will also be undefined if the URI scheme does not
   *    use an authority component.
   */
  if ((ref->authority != NULL) || (ref->server != NULL)) {
    if (ref->authority != NULL)
      res->authority = strdup(ref->authority);
    else {
      res->server = strdup(ref->server);
      if (ref->user != NULL)
        res->user = strdup(ref->user);
      res->port = ref->port;		
    }
    if (ref->path != NULL)
      res->path = strdup(ref->path);
    goto step_7;
  }
  if (bas->authority != NULL)
    res->authority = strdup(bas->authority);
  else if (bas->server != NULL) {
    res->server = strdup(bas->server);
    if (bas->user != NULL)
      res->user = strdup(bas->user);
    res->port = bas->port;		
  }

  /*
   * 5) If the path component begins with a slash character ("/"), then
   *    the reference is an absolute-path and we skip to step 7.
   */
  if ((ref->path != NULL) && (ref->path[0] == '/')) {
    res->path = strdup(ref->path);
    goto step_7;
  }


  /*
   * 6) If this step is reached, then we are resolving a relative-path
   *    reference.  The relative path needs to be merged with the base
   *    URI's path.  Although there are many ways to do this, we will
   *    describe a simple method using a separate string buffer.
   *
   * Allocate a buffer large enough for the result string.
   */
  len = 2; /* extra / and 0 */
  if (ref->path != NULL)
    len += strlen(ref->path);
  if (bas->path != NULL)
    len += strlen(bas->path);
  res->path = malloc(len);
  if (res->path == NULL) {
    goto done;
  }
  res->path[0] = 0;

  /*
   * a) All but the last segment of the base URI's path component is
   *    copied to the buffer.  In other words, any characters after the
   *    last (right-most) slash character, if any, are excluded.
   */
  cur = 0;
  out = 0;
  if (bas->path != NULL) {
    while (bas->path[cur] != 0) {
      while ((bas->path[cur] != 0) && (bas->path[cur] != '/'))
        cur++;
      if (bas->path[cur] == 0)
        break;

      cur++;
      while (out < cur) {
        res->path[out] = bas->path[out];
        out++;
      }
    }
  }
  res->path[out] = 0;

  /*
   * b) The reference's path component is appended to the buffer
   *    string.
   */
  if (ref->path != NULL && ref->path[0] != 0) {
    index = 0;
    /*
     * Ensure the path includes a '/'
     */
    if ((out == 0) && (bas->server != NULL))
      res->path[out++] = '/';
    while (ref->path[index] != 0) {
      res->path[out++] = ref->path[index++];
    }
  }
  res->path[out] = 0;

  /*
   * Steps c) to h) are really path normalization steps
   */
  etpan_uri_normalize_path(res->path);

 step_7:

  /*
   * 7) The resulting URI components, including any inherited from the
   *    base URI, are recombined to give the absolute form of the URI
   *    reference.
   */
  val = etpan_uri_render(res);

 done:
  if (ref != NULL)
    etpan_uri_free(ref);
  if (bas != NULL)
    etpan_uri_free(bas);
  if (res != NULL)
    etpan_uri_free(res);
  return(val);
}




char * etpan_uri_get_scheme(struct etpan_uri * uri)
{
  return uri->scheme;
}

void etpan_uri_set_scheme(struct etpan_uri * uri, const char * scheme)
{
  if (scheme != uri->scheme) {
    free(uri->scheme);
    if (scheme != NULL) {
      uri->scheme = strdup(scheme);
      if (uri->scheme == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->scheme = NULL;
  }
}

char * etpan_uri_get_opaque(struct etpan_uri * uri)
{
  return uri->opaque;
}

void etpan_uri_set_opaque(struct etpan_uri * uri, const char * opaque)
{
  if (opaque != uri->opaque) {
    free(uri->opaque);
    if (opaque != NULL) {
      uri->opaque = strdup(opaque);
      if (uri->opaque == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->opaque = NULL;
  }
}

char * etpan_uri_get_authority(struct etpan_uri * uri)
{
  return uri->authority;
}

void etpan_uri_set_authority(struct etpan_uri * uri, const char * authority)
{
  if (authority != uri->authority) {
    free(uri->authority);
    if (authority != NULL) {
      uri->authority = strdup(authority);
      if (uri->authority == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->authority = NULL;
  }
}

char * etpan_uri_get_server(struct etpan_uri * uri)
{
  return uri->server;
}

void etpan_uri_set_server(struct etpan_uri * uri, const char * server)
{
  if (server != uri->server) {
    free(uri->server);
    if (server != NULL) {
      uri->server = strdup(server);
      if (uri->server == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->server = NULL;
  }
}

char * etpan_uri_get_user(struct etpan_uri * uri)
{
  return uri->user;
}

void etpan_uri_set_user(struct etpan_uri * uri, const char * user)
{
  if (user != uri->user) {
    free(uri->user);
    if (user != NULL) {
      uri->user = strdup(user);
      if (uri->user == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->user = NULL;
  }
}

int etpan_uri_get_port(struct etpan_uri * uri)
{
  return uri->port;
}

void etpan_uri_set_port(struct etpan_uri * uri, int port)
{
  uri->port = port;
}

char * etpan_uri_get_path(struct etpan_uri * uri)
{
  return uri->path;
}

void etpan_uri_set_path(struct etpan_uri * uri, const char * path)
{
  if (path != uri->path) {
    free(uri->path);
    if (path != NULL) {
      uri->path = strdup(path);
      if (uri->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->path = NULL;
  }
}

char * etpan_uri_get_query(struct etpan_uri * uri)
{
  return uri->query;
}

void etpan_uri_set_query(struct etpan_uri * uri, const char * query)
{
  if (query != uri->query) {
    free(uri->query);
    if (query != NULL) {
      uri->query = strdup(query);
      if (uri->query == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->query = NULL;
  }
}

char * etpan_uri_get_fragment(struct etpan_uri * uri)
{
  return uri->fragment;
}

void etpan_uri_set_fragment(struct etpan_uri * uri, const char * fragment)
{
  if (fragment != uri->fragment) {
    free(uri->fragment);
    if (fragment != NULL) {
      uri->fragment = strdup(fragment);
      if (uri->fragment == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      uri->fragment = NULL;
  }
}

void etpan_uri_log(struct etpan_uri * uri)
{
  ETPAN_LOG("scheme : %s", etpan_uri_get_scheme(uri));
  ETPAN_LOG("opaque : %s", etpan_uri_get_opaque(uri));
  ETPAN_LOG("authority : %s", etpan_uri_get_authority(uri));
  ETPAN_LOG("server : %s", etpan_uri_get_server(uri));
  ETPAN_LOG("user : %s", etpan_uri_get_user(uri));
  ETPAN_LOG("port : %i", etpan_uri_get_port(uri));
  ETPAN_LOG("path : %s", etpan_uri_get_path(uri));
  ETPAN_LOG("query : %s", etpan_uri_get_query(uri));
  ETPAN_LOG("fragment : %s", etpan_uri_get_fragment(uri));
}
