#include "etpan-utils.h"

#include <paths.h>
#include <unistd.h>
#include <limits.h>
#include <pthread.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "etpan-tc.h"
#include "etpan-error.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-outbox.h"
#include "etpan-log.h"
#include "etpan-imap-sync.h"
#include "etpan-pop-sync.h"

static char tmp_dir[PATH_MAX] = _PATH_TMP;
static pthread_mutex_t tmp_lock = PTHREAD_MUTEX_INITIALIZER;

void etpan_set_tmp_dir(char * tmp)
{
  strncpy(tmp_dir, tmp, sizeof(tmp_dir));
  tmp_dir[sizeof(tmp_dir) - 1] = '\0';
}

char * etpan_get_tmp_dir(void)
{
  return tmp_dir;
}

int etpan_get_tmp_fd(char * tmp_file, size_t size)
{
  int fd;
  
  etpan_make_dir(tmp_dir);
  
  snprintf(tmp_file, size, "%s/%s", tmp_dir, "etpan-XXXXXX");
  /* mkstemp() is not thread safe under some systems such as Mac OS X */
  pthread_mutex_lock(&tmp_lock);
  fd = mkstemp(tmp_file);
  pthread_mutex_unlock(&tmp_lock);
  
  return fd;
}

FILE * etpan_get_tmp_file(char * tmp_file, size_t size)
{
  int fd;
  FILE * f;
  
  fd = etpan_get_tmp_fd(tmp_file, size);
  if (fd < 0)
    return NULL;
  
  f = fdopen(fd, "w");
  if (f == NULL) {
    close(fd);
    unlink(tmp_file);
    return NULL;
  }
  
  return f;
}

int etpan_get_tmp_filename(char * tmp_file, size_t size)
{
  FILE * f;
  
  f = etpan_get_tmp_file(tmp_file, size);
  if (f == NULL)
    return -1;
  fclose(f);
  
  return 0;
}

int etpan_make_dir(char * path)
{
  return tcmkpath(path, 0700);
}

int etpan_get_tmp_path(char * tmp_path, size_t size)
{
  char * result;
  
  snprintf(tmp_path, size, "%s/%s", tmp_dir, "etpan-XXXXXX");
  /* mkstemp() is not thread safe under some systems such as Mac OS X */
  pthread_mutex_lock(&tmp_lock);
  result = mkdtemp(tmp_path);
  pthread_mutex_unlock(&tmp_lock);
  
  if (result == NULL)
    return -1;
  
  return 0;
}

int etpan_quote_filename(char * result, size_t size, char * path)
{
  char * p;
  char * result_p;
  size_t remaining;

  result_p = result;
  remaining = size;

  for(p = path ; * p != '\0' ; p ++) {
    int quote_not_needed;
    
    quote_not_needed = (* p != '\\') && (* p != '\'') && (* p != '\"');
    if (quote_not_needed) {
      if (remaining > 0) {
        * result_p = * p;
        result_p ++; 
        remaining --;
      }
      else {
        result[size - 1] = '\0';
        return -1;
      }
    }
    else { 
      if (remaining >= 2) {
        * result_p = '\\';
        result_p ++; 
        * result_p = * p;
        result_p ++; 
        remaining -= 2;
      }
      else {
        result[size - 1] = '\0';
        return -1;
      }
    }
  }
  if (remaining > 0) {
    * result_p = '\0';
  }
  else {
    result[size - 1] = '\0';
    return -1;
  }
  
  return 0;
}

int etpan_file_exists(char * filename)
{
  int fd;
  
  fd = open(filename, O_RDONLY);
  if (fd < 0)
    return 0;
  
  close(fd);
  
  return 1;
}

int etpan_write_file(char * filename, void * data, size_t length)
{
  FILE * f;
  int res;
  char * current;
  size_t remaining;
  size_t count;
  int r;
  
  f = fopen(filename, "wb");
  if (f == NULL) {
    perror("fopen");
    res = -1;
    goto err;
  }
  
  r = chmod(filename, 0600);
  if (r < 0) {
    perror("chmod");
    res = -1;
    goto close;
  }
  
  current = (char *) data;
  remaining = length;
  
  do {
    count = fwrite(current, 1, remaining, f);
    current += count;
    remaining -= count;
    if (remaining == 0)
      break;
  } while (count > 0);
  
  if (ferror(f) != 0) {
    perror("ferror");
    res = -1;
    goto close;
  }
  
  fclose(f);
  
  return 0;
  
 close:
  fclose(f);
 err:
  return res;
}

char * etpan_get_cache_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char cache_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(cache_path, sizeof(cache_path), "%s/%s/cache",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(cache_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_flags_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char flags_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(flags_path, sizeof(flags_path), "%s/%s/flags",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(flags_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_index_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char index_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(index_path, sizeof(index_path), "%s/%s/index",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(index_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}


char * etpan_get_filter_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char filter_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(filter_path, sizeof(filter_path), "%s/%s/filter",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(filter_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}


char * etpan_get_outbox_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char outbox_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(outbox_path, sizeof(outbox_path), "%s/%s/outbox/storage",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(outbox_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_outbox_sent_folder_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char outbox_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(outbox_path, sizeof(outbox_path),
      "%s/%s/outbox-sent-folder/storage",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(outbox_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_outbox_cache_dir(struct etpan_account_manager * manager,
    struct etpan_outbox * outbox)
{
  char outbox_path[PATH_MAX];
  char * s;
  (void) manager;
  
  snprintf(outbox_path, sizeof(outbox_path), "%s-cache",
      etpan_outbox_get_path(outbox));
  
  s = strdup(outbox_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_outbox_flags_dir(struct etpan_account_manager * manager,
    struct etpan_outbox * outbox)
{
  char outbox_path[PATH_MAX];
  char * s;
  (void) manager;
  
  snprintf(outbox_path, sizeof(outbox_path), "%s-flags",
      etpan_outbox_get_path(outbox));
  
  s = strdup(outbox_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_outbox_news_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char outbox_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(outbox_path, sizeof(outbox_path), "%s/%s/outbox-news/storage",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(outbox_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_order_dir(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char flags_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(flags_path, sizeof(flags_path), "%s/%s/order",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(flags_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_uuid_filename(struct etpan_account_manager * manager,
    struct etpan_account * account)
{
  char flags_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(flags_path, sizeof(flags_path), "%s/%s/uuid",
      etpan_account_manager_get_path(manager),
      etpan_account_get_id(account));
  
  s = strdup(flags_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_quote_mailbox(const char * mb)
{
  MMAPString * gstr;
  char * str;

  gstr = mmap_string_new("");
  if (gstr == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  while (* mb != 0) {
    char hex[3];

    if (((* mb >= 'a') && (* mb <= 'z')) ||
	((* mb >= 'A') && (* mb <= 'Z')) ||
	((* mb >= '0') && (* mb <= '9')))
      mmap_string_append_c(gstr, * mb);
    else {
      if (mmap_string_append_c(gstr, '%') == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      snprintf(hex, 3, "%02x", (unsigned char) (* mb));
      if (mmap_string_append(gstr, hex) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    mb ++;
  }

  str = strdup(gstr->str);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  mmap_string_free(gstr);
  
  return str;
}

char * etpan_get_imap_cache_dir(struct etpan_account_manager * manager,
    struct etpan_imap_sync * imap_sync)
{
  char cache_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(cache_path, sizeof(cache_path), "%s/%s/imap-cache",
      etpan_account_manager_get_path(manager),
      etpan_imap_sync_get_id(imap_sync));
  
  s = strdup(cache_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}

char * etpan_get_pop_cache_dir(struct etpan_account_manager * manager,
    struct etpan_pop_sync * pop_sync)
{
  char cache_path[PATH_MAX];
  char * s;
  
  if (etpan_account_manager_get_path(manager) == NULL)
    return NULL;
  
  snprintf(cache_path, sizeof(cache_path), "%s/%s/pop-cache",
      etpan_account_manager_get_path(manager),
      etpan_pop_sync_get_id(pop_sync));
  
  s = strdup(cache_path);
  if (s == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return s;
}
