#include "etpan-abook-config.h"

#include <string.h>
#include <stdlib.h>

#include "etpan-tc.h"
#include "etpan-abook.h"
#include "etpan-abook-vcard.h"
#include "etpan-abook-ldap.h"
#include "etpan-abook-manager.h"
#include "etpan-error.h"
#include "etpan-connection-types.h"
#include "etpan-config-types.h"
#include "etpan-log.h"
#include "etpan-nls.h"

#define DEFAULT_LDAP_TLS_PORT 636
#define DEFAULT_LDAP_PORT 389

static inline int get_int_entry(tcconf_section_t * conf, char * name)
{
  int r;
  int value;
  
  r = tcconf_getvalue(conf, name, "%i", &value);
  if (r <= 0)
    return -1;
  
  return value;
}

static inline char * get_entry(tcconf_section_t * conf, char * name)
{
  int r;
  char * value;
  
  r = tcconf_getvalue(conf, name, "%s", &value);
  if (r <= 0)
    return NULL;
  
  return value;
}

static void read_abook(struct etpan_abook_manager * manager,
    tcconf_section_t * abook_info, carray * error_list);

static int read_abook_info(tcconf_section_t * abook_info,
    struct etpan_abook_manager * manager,
    struct etpan_abook ** p_abook, carray * error_list);

static struct etpan_error * add_abook(struct etpan_abook_manager * manager,
    struct etpan_abook  * abook);

static void read_config(struct etpan_abook_manager * manager,
    tcconf_section_t * config, carray * error_list)
{
  void * iter;
  
  iter = NULL;
  do {
    tcconf_section_t * abook_info;
    
    abook_info = tcconf_nextsection(config, "abook", &iter);
    if (abook_info != NULL) {
      read_abook(manager, abook_info, error_list);
      tcfree(abook_info);
    }
    else {
      break;
    }
  } while (1);
}

static void read_abook(struct etpan_abook_manager * manager,
    tcconf_section_t * abook_info, carray * error_list)
{
  int r;
  struct etpan_abook * abook;
  struct etpan_error * error;
  
  r = read_abook_info(abook_info, manager, &abook, error_list);
  if (r < 0) {
    return;
  }
  
  error = add_abook(manager, abook);
  if (error != NULL) {
    etpan_abook_free(abook);
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

/*
  - type
  vcard
  ldap
  
  vcard :
  - filename
  
  ldap :
  - hostname
  - port
  - username
  - password
  - base
  - tls
*/

static void read_ldap_abook(tcconf_section_t * abook_info,
    struct etpan_abook_manager * manager,
    struct etpan_abook ** p_abook, carray * error_list);

static void read_vcard_abook(tcconf_section_t * abook_info,
    struct etpan_abook_manager * manager,
    struct etpan_abook ** p_abook, carray * error_list);

static int read_abook_info(tcconf_section_t * abook_info,
    struct etpan_abook_manager * manager,
    struct etpan_abook ** p_abook, carray * error_list)
{
  char * abook_type;
  int r;
  char * id;
  struct etpan_abook * abook;
  
  id = NULL;
  abook_type = NULL;
  
  id = get_entry(abook_info, "name");
  if (id == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Address book has no name"));
    etpan_error_strf_long_description(error, _("The address book in configuration has no name."));
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    goto free_str;
  }
  
  abook_type = get_entry(abook_info, "type");
  if (abook_type == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Address book has no type"));
    etpan_error_strf_long_description(error, _("The address book %s has no type."), id);
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    goto free_str;
  }
  
  if (strcasecmp(abook_type, "vcard") == 0) {
    read_vcard_abook(abook_info, manager, &abook, error_list);
  }
  else if (strcasecmp(abook_type, "ldap") == 0) {
    read_ldap_abook(abook_info, manager, &abook, error_list);
  }
  else {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Address book has an invalid type"));
    etpan_error_strf_long_description(error, _("The address book %s has an invalid type (%s)."), id, abook_type);
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    goto free_str;
  }
  
  etpan_abook_set_id(abook, id);
  
  free(id);
  free(abook_type);
  
  * p_abook = abook;
  
  return 0;
  
 free_str:
  free(id);
  free(abook_type);
  return -1;
}

static void read_vcard_abook(tcconf_section_t * abook_info,
    struct etpan_abook_manager * manager,
    struct etpan_abook ** p_abook, carray * error_list)
{
  struct etpan_abook * abook;
  char * filename;
  (void) manager;
  (void) error_list;
  
  abook = etpan_abook_vcard_new();
  
  filename = get_entry(abook_info, "path");
  etpan_abook_vcard_set_filename(abook, filename);
  free(filename);
  
  * p_abook = abook;
}


static void read_host_info(tcconf_section_t  * abook_info,
    char ** p_hostname, int * p_port, int * p_connection_type,
    char ** p_username, char ** p_password)
{
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  int tls;
  int starttls;
  
  hostname = NULL;
  port = -1;
  connection_type = ETPAN_CONNECTION_PLAIN;
  username = NULL;
  password = NULL;
  
  hostname = get_entry(abook_info, "hostname");
  
  tls = get_int_entry(abook_info, "tls");
  if (tls != -1) {
    if (tls)
      connection_type = ETPAN_CONNECTION_TLS;
  }
  
  starttls = get_int_entry(abook_info, "starttls");
  if (starttls != -1) {
    if (starttls)
      connection_type = ETPAN_CONNECTION_STARTTLS;
  }
  
  port = get_int_entry(abook_info, "port");
  
  username = get_entry(abook_info, "username");
  password = get_entry(abook_info, "password");
  
  * p_hostname = hostname;
  * p_port = port;
  * p_connection_type = connection_type;
  * p_username = username;
  * p_password = password;
}

static void read_ldap_abook(tcconf_section_t * abook_info,
    struct etpan_abook_manager * manager,
    struct etpan_abook ** p_abook, carray * error_list)
{
  struct etpan_abook * abook;
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  char * base;
  (void) manager;
  (void) error_list;
  
  abook = etpan_abook_ldap_new();
  
  read_host_info(abook_info,
      &hostname, &port, &connection_type,
      &username, &password);
  
  etpan_abook_ldap_set_hostname(abook, hostname);
  etpan_abook_ldap_set_username(abook, username);
  etpan_abook_ldap_set_password(abook, password);
  
  if (port != -1) {
    etpan_abook_ldap_set_port(abook, port);
  }
  else {
    if (connection_type == ETPAN_CONNECTION_TLS)
      etpan_abook_ldap_set_port(abook, DEFAULT_LDAP_TLS_PORT);
    else
      etpan_abook_ldap_set_port(abook, DEFAULT_LDAP_PORT);
  }
  
  etpan_abook_ldap_set_connection_type(abook, connection_type);

  base = get_entry(abook_info, "base");
  etpan_abook_ldap_set_base(abook, base);
  
  free(base);
  free(hostname);
  free(username);
  free(password);
  
  * p_abook = abook;
}

static struct etpan_error * add_abook(struct etpan_abook_manager * manager,
    struct etpan_abook  * abook)
{
  char * name;
  struct etpan_abook * existing_abook;
  struct etpan_error * error;
  
  name = etpan_abook_get_id(abook);
  
  existing_abook = etpan_abook_manager_get_abook(manager, name);
  if (existing_abook != NULL) {
    ETPAN_LOG("address book already exists with this name");
    etpan_crash();
  }
  
  error = etpan_abook_setup(abook);
  if (error != NULL)
    return error;
  
  etpan_abook_manager_add_abook(manager, abook);
  
  return NULL;
}


struct etpan_error *
etpan_abook_config_read(struct etpan_abook_manager * manager,
    char * filename)
{
  tcconf_section_t * config;
  carray * error_list;
  struct etpan_error * error;
  unsigned int i;
  
  config = tcconf_load_file(NULL, filename);
  if (config == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Address book configuration at %s could not be parsed."), filename);
    
    return error;
  }
  
  error_list = carray_new(4);
  if (error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  read_config(manager, config, error_list);
  
  tcfree(config);
  
  etpan_abook_manager_notify_modification(manager);
  
  if (carray_count(error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(error_list) == 1) {
    error = carray_get(error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(error_list);
  
  return error;
}

struct etpan_error * etpan_abook_config_read_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_abook_manager * manager;
  
  manager = etpan_abook_manager_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_ABOOK_FILE);
  
  return etpan_abook_config_read(manager, filename);
}

static void abook_ldap_save(FILE * f, struct etpan_abook * abook)
{
  char * hostname;
  int connection_type;
  int port;
  char * username;
  char * password;
  char * base;
  
  fprintf(f, "  type \"ldap\"\n");
  
  hostname = etpan_abook_ldap_get_hostname(abook);
  if (hostname != NULL) {
    char * quoted_hostname;
    
    quoted_hostname = etpan_quote_string(hostname);
    fprintf(f, "  hostname \"%s\"\n", quoted_hostname);
    free(quoted_hostname);
  }
  connection_type = etpan_abook_ldap_get_connection_type(abook);
  switch (connection_type) {
  case ETPAN_CONNECTION_TLS:
    fprintf(f, "  tls 1\n");
    break;
  case ETPAN_CONNECTION_STARTTLS:
    fprintf(f, "  starttls 1\n");
    break;
  }
  port = etpan_abook_ldap_get_port(abook);
  fprintf(f, "  port %i\n", port);
  
  username = etpan_abook_ldap_get_username(abook);
  if (username != NULL) {
    char * quoted_username;
    
    quoted_username = etpan_quote_string(username);
    fprintf(f, "  username \"%s\"\n", quoted_username);
    free(quoted_username);
  }
  
  password = etpan_abook_ldap_get_password(abook);
  if (password != NULL) {
    char * quoted_password;
    
    quoted_password = etpan_quote_string(password);
    fprintf(f, "  password \"%s\"\n", quoted_password);
    free(quoted_password);
  }
  
  base = etpan_abook_ldap_get_base(abook);
  if (base != NULL) {
    char * quoted_base;
    
    quoted_base = etpan_quote_string(base);
    fprintf(f, "  base \"%s\"\n", quoted_base);
    free(quoted_base);
  }
}

static void abook_vcard_save(FILE * f, struct etpan_abook * abook)
{
  char * path;
  
  fprintf(f, "  type \"vcard\"\n");
  
  path = etpan_abook_vcard_get_filename(abook);
  if (path != NULL) {
    char * quoted_path;
    
    quoted_path = etpan_quote_string(path);
    fprintf(f, "  path \"%s\"\n", quoted_path);
    free(quoted_path);
  }
}

static void abook_save(FILE * f, struct etpan_abook * abook)
{
  char * name;
  char * type;
  struct etpan_abook_driver * driver;
  
  fprintf(f, "abook [\n");
  
  name = etpan_abook_get_id(abook);
  if (name != NULL) {
    char * quoted_name;
    
    quoted_name = etpan_quote_string(name);
    fprintf(f, "  name \"%s\"\n", quoted_name);
    free(quoted_name);
  }
  
  driver = etpan_abook_get_driver(abook);
  type = driver->name;
  if (strcasecmp(type, "ldap") == 0) {
    abook_ldap_save(f, abook);
  }
  else if (strcasecmp(type, "vcard") == 0) {
    abook_vcard_save(f, abook);
  }
  
  fprintf(f, "]\n");
  fprintf(f, "\n");
}

struct etpan_error *
etpan_abook_config_save(struct etpan_abook_manager * manager,
    char * filename)
{
  carray * list;
  unsigned int i;
  FILE * f;
  
  f = fopen(filename, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Configuration file could not be written"));
    etpan_error_strf_long_description(error, _("Address book configuration file %s could not be written to disk."), filename);
    
    return error;
  }
  
  list = etpan_abook_manager_get_ordered_list(manager);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_abook * abook;
    
    abook = carray_get(list, i);
    abook_save(f, abook);
  }
  
  fclose(f);
  
  return NULL;
}

struct etpan_error * etpan_abook_config_save_default(void)
{
  char filename[PATH_MAX];
  char * home;
  struct etpan_abook_manager * manager;
  
  manager = etpan_abook_manager_get_default();
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_ABOOK_FILE);
  
  return etpan_abook_config_save(manager, filename);
}
